/**************************************************************************\
 gatos (General ATI TV and Overlay Software)

  Project Coordinated By Insomnia (Steaphan Greene)
  (insomnia@core.binghamton.edu)

  Copyright (C) 1999 Steaphan Greene, yvind Aabling, Octavian Purdila, 
	Vladimir Dergachev and Christian Lupien.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/

#define GATOS_FI12XX_C 1

#include "gatos.h"
#include "fi12xx.h"
#include "board.h"
#include "bt829.h"
#include "i2c.h"

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>

/* Private global vars */
static u32 tunerK=0, minK=0, maxK=0 ;
static struct {
  int type ; char *vendor ; char *ident ; char *system ; }
tuners[MAXTUNERTYPE+1] = {
  {  0, NULL,		NULL,		NULL },
  {  1, "Philips",	"FI1236",	"NTSC M/N" },
  {  2, "Philips",	"FI1236J",	"NTSC Japan" },
  {  3, "Philips",	"FI1216MK2",	"PAL B/G" },
  {  4, "Philips",	"FI1246",	"PAL I" },
  {  5, "Philips",	"FI1216MF",	"PAL B/G, SECAM L/L'" },
  {  6, "Philips",	"FI1236MK2",	"NTSC M/N" },
  {  7, "Philips",	"FI1256",	"SECAM D/K" },
  {  8, "Samsung",	"TCPN7082PC27A","NTSC M/N" },
  {  9, NULL,		NULL,		NULL },
  { 10, NULL,		NULL,		NULL },
  { 11, NULL,		NULL,		NULL },
  { 12, "Alps",		"TSCH5",	"NTSC M/N with FM" },
  { 13, NULL,		NULL,		NULL },
  { 14, NULL,		NULL,		NULL },
  { 15, NULL,		NULL,		NULL },
  { 16, NULL,		NULL,		NULL },
  { 17, NULL,		NULL,		NULL },
  { 18, "Alps(18)",	"TSCH5",	"NTSC M/N with FM" } } ;

/* Is the Alps tuner ID 12 (decimal) or 0x12 (hex) ?
 * ATI sample code says 0x12, but that could be wrong /AA */

/* ------------------------------------------------------------------------ */
/* Initialization routine */

int fi12xx_init(void) {
  gatos.fi12xx.addr = 0xFF ;
  switch (gatos.cardtype) {
    case CARD_STAND_ALONE:
    case CARD_NEC:
      if (i2c_device(0xC6)) fi12xx_register(0xC6,0) ;
      if (i2c_device(0xC0)) fi12xx_register(0xC0,1) ; break ;
    case CARD_INTEGRATED:  
    case CARD_EXTERNAL_POD:
    case CARD_ALL_IN_WONDER:
    case CARD_ALL_IN_WONDER_PRO:
    case CARD_ALL_IN_WONDER_128:
    case CARD_INTEL_BOARD:
      if (i2c_device(0xC0)) fi12xx_register(0xC0,0) ;
      if (i2c_device(0xC6)) fi12xx_register(0xC6,1) ; break ; }
  if (gatos.fi12xx.addr == 0xFF) RETURN(ENODEV) ; RETURN0 ; }

void fi12xx_setformat(void) {

  /* NTSC, PAL or SECAM ? */
  switch (gatos.format) {
    case 1: case 2: case 4:					/* NTSC */
      gatos.minfreq =  55.25 ; gatos.Fpc = 45.75 ;
      gatos.maxfreq = 801.25 ; gatos.ntsc = 1 ; break ;
    case 3: case 5: case 7:					/* PAL */
      /* PAL-D/K Fpc: ATI == 38.9, officially == 38.00 ? */
      gatos.minfreq =  48.25 ; gatos.Fpc = 38.90 ;
      gatos.maxfreq = 855.25 ; gatos.pal = 1 ; break ;
    case 6:							/* SECAM */
      /* SECAM-L1 Fpc: ATI == 33.95, officially == 32.70 ? */
      gatos.minfreq =  48.25 ; gatos.Fpc = 38.90 ;
      gatos.maxfreq = 855.25 ; gatos.secam = 1 ; break ; }

  minK = FREQ2K(gatos.minfreq) ; maxK = FREQ2K(gatos.maxfreq) ; }

/* ------------------------------------------------------------------------ */
/* Public routines */

/* Tune to frequency */
int fi12xx_tune(void) {

  u32 K ; int i ; u8 band, data[4] ;

  K = FREQ2K(gatos.freq) ; band = fi12xx_band(gatos.freq) ;
  gatos.freq = K2FREQ(K) ; i = (K>tunerK) ? 0 : 2 ; tunerK = K ;
  data[i+0] = (K>>8) & 0x7F ; data[i+1] = K & 0xFF ;
  data[2-i] = 0x8E ; data[3-i] = band ;
  i2c_write(gatos.fi12xx.addr,data,sizeof(data)) ;
  RETURN0 ; }

/* Channel scanner */
double fi12xx_scan(double start, double end,
  int (*addchan)(double, char*), int (*progress)(double)) {

  double freq ; int state=4, volume, ctr ; char name[16] ;
  u8 band, data[4], fi, bt, btmask=0xC0 ; u32 K, maxK ;

  if (gatos.usr1pid) kill(gatos.usr1pid,SIGSTOP) ;
  volume = gatos.volume ; if (volume) gatos_setvolume(0) ;

  if (start == gatos.minfreq) start -= 0.25 ;
  if (end == gatos.maxfreq)   end += 0.25 ;
  tunerK = K = FREQ2K(start) ; maxK = FREQ2K(end) ;
  if (gatos.senscan) btmask = 0x80 ;

  for ( ; tunerK<=maxK ; tunerK++ ) {

    freq = K2FREQ(tunerK) ; band = fi12xx_band(freq) ;

    if (progress && progress(freq)) {
      if (gatos.usr1pid) kill(gatos.usr1pid,SIGCONT) ;
      if (volume) gatos_setvolume(volume) ; return freq ; }

    data[0] = (tunerK>>8) & 0x7F ; data[1] = tunerK & 0xFF ;
    data[2] = 0xCE ; data[3] = band ;

    i2c_write(gatos.fi12xx.addr,data,sizeof(data)) ;
    usleep(10000) ; fi = FISTATUS&7 ;
    state = (state == 2 && fi <= 2) ? 5 : fi ;
    if (fi >= 2) K = tunerK ;
    if (state != 5) continue ;
    bt = BTREAD(STATUS)>>7 ;
    if (!bt) continue ;

    freq = K2FREQ(K) ; band = fi12xx_band(freq) ;
    data[2] = (K>>8) & 0x7F ; data[3] = K & 0xFF ;
    data[0] = 0xCE ; data[1] = band ;
    i2c_write(gatos.fi12xx.addr,data,sizeof(data)) ;
    usleep(20000) ; fi = FISTATUS ; bt = BTREAD(STATUS) ;
    if ( (fi&0x40) != 0x40 || (bt&btmask) != btmask ) continue ;

    if (addchan) {
      for ( ctr=0 ; ctr<gatos.numchannels
		&& ((freq+0.1)<gatos.channelinfo[ctr].frequency
		|| (freq-0.1)>gatos.channelinfo[ctr].frequency); ctr++ ) ;
      if (ctr>=gatos.numchannels) {
	snprintf(name,sizeof(name),"%.3f MHz",freq) ;
	addchan(freq,name) ; } } }

  if (gatos.usr1pid) kill(gatos.usr1pid,SIGCONT) ;
  if (volume) gatos_setvolume(volume) ; return 0.0 ; }

/* ------------------------------------------------------------------------ */
/* Private routines */

static u8 fi12xx_band(double freq) {
  int system=0 ;
  switch (gatos.tunertype) {
    case 1: case 2:				/* NTSC M/N and NTSC Japan */
      if (freq <= 160.00) return 0xA2 ;
      if (freq <= 451.25) return 0x94 ;
      if (freq <= 463.25) return 0x34 ;
                          return 0x31 ;
    case 3: case 4:				/* PAL B/G and PAL I */
      if (freq <= 140.25) return 0xA2 ;
      if (freq <= 168.25) return 0xA4 ;
      if (freq <= 447.25) return 0x94 ;
                          return 0x31 ;
    case 6:					/* NTSC M/N Mk2 */
      if (freq <= 160.00) return 0xA0 ;
      if (freq <= 454.00) return 0x90 ;
                          return 0x30 ;
    case 7:					/* SECAM D/K */
      if (freq <= 168.25) return 0xA0 ;
      if (freq <= 455.25) return 0x90 ;
                          return 0x30 ;
    case 5:					/* PAL B/G, SECAM L/L' */
      if (gatos.pal)        system = 1 ;
      if (gatos.secam == 2) system = 2 ;
      if (gatos.secam == 1) system = 3 ;
      if (freq <= 168.25) return 0xA0+system ;
      if (freq <= 447.25) return 0x90+system ;
                          return 0x30+system ;
    case 8:					/* NTSC M/N (Samsung) */
      if (gatos.cardtype == CARD_NEC) {
        if (freq <= 160.00) return 0xA0 ;
        if (freq <= 454.00) return 0x90 ;
                            return 0x30 ; }
      else if ((gatos.cardtype == CARD_EXTERNAL_POD && gatos.boardrev == 3) ||
               (gatos.cardtype == CARD_STAND_ALONE  && gatos.boardrev == 0)) {
        if (freq <= 157.25) return 0xA2 ;
        if (freq <= 451.25) return 0x94 ;
        if (freq <= 463.25) return 0x34 ;
                            return 0x31 ; }
      else {
        if (freq <= 127.25) return 0x01 ;
        if (freq <= 361.25) return 0x02 ;
                            return 0x08 ; }
    case 12: case 0x12:				/* NTSC M/N with FM */
      if (freq <= 130.00) return 0x14 ;
      if (freq <= 364.00) return 0x12 ;
                          return 0x11 ;
    default: return 0x00 ; } }

static int fi12xx_register(u8 addr, int use) {
  char *ident, name[64] ;
  gatos.fi12xx.deviceid = 0 ; gatos.fi12xx.revision = 0 ;
  if (gatos.tunertype) {
    ident = tuners[gatos.tunertype].ident ;
    if (ident==NULL) ident="Unknown" ;
    snprintf(name,sizeof(name),"%s Tuner Module (%s %s)%s",
             tuners[gatos.tunertype].system, tuners[gatos.tunertype].vendor,
             ident, (use)?"":" UNUSED") ; }
  else { ident = "Tuner" ; snprintf(name,sizeof(name),"Tuner Module") ; }
  if (use) gatos.fi12xx.addr = addr ;
  i2c_register(addr,ident,name) ; RETURN0 ; }

/* ------------------------------------------------------------------------ */
/* Debug and report routines */

#define DUMPREG	\
  fprintf(stderr,"%s: %-6s = 0x%02X\n", \
    i2c_ident(gatos.fi12xx.addr),"STATUS",FISTATUS)

void fi12xx_dumpregs(void) {
  DUMPREG ; }
