/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* Operations for files and file names. */

/* file operations ==========================================================*/

extern bool_t file_exists (string_t file_name);
/* Return TRUE iff file <file_name> exists. */

extern FILE *open_stream (string_t file_name, string_t stream_mode);
/* Open file <file_name> and create a stream from/to it in mode <stream_mode>.
 * Works like "fopen", but calls "error" if it doesn't work. */

extern void close_stream (FILE **stream_ptr, string_t file_name);
/* Close the stream <*stream_ptr> which is connected to the file <file_name>
 * and set <*stream_ptr> to NULL. Don't do anything if <*stream_ptr> == NULL.
 * Works like "fclose", but calls "error" if <file_name> != NULL and an error
 * occurs during closing. */

extern void write_vector (void *ptr, 
			  int_t item_size, 
			  int_t items, 
			  FILE *stream, 
			  string_t file_name);
/* Write <items> items, of size <item_size> each, stored at *<ptr>,
 * to <stream>, which is connected to file <file_name>.
 * Works like "fwrite", but calls "error" if it doesn't work. */

extern void read_vector (void *ptr, 
			 int_t item_size, 
			 int_t items, 
			 FILE *stream, 
			 string_t file_name);
/* Read <items> items, of size <item_size> each, from <stream>,
 * which is connected to file <file_name>, and store them at *<ptr>.
 * Works like "fread", but calls "error" if it doesn't work. */

extern void *read_new_vector (int_t item_size, 
			      int_t items, 
			      FILE *stream, 
			      string_t file_name);
/* Read <items> items, of size <items_size> each, from <stream>,
 * which is connected to file <file_name>, into allocated memory block,
 * and return a pointer to that block. */

extern void map_file (string_t file_name, void **address, int_t *length);
/* Map file "file_name" into the memory. It will be available in the 
 * memory region starting at *<address> and will occupy <length> bytes.
 * After usage, return the memory region via "unmap_file". */

extern void unmap_file (void **address, int_t length);
/* Return the memory region that has been allocated by "map_file".
 * The region starts at <*address> and occupies <length> bytes. */

/* file name operations =====================================================*/

extern string_t name_in_path (string_t path_name);
/* Return the file name in <path_name>, i.e. the name after the last "/". */

extern string_t absolute_path (string_t src_path, string_t relative_to);
/* Return the absolute path name which is equivalent to <src_path>.
 * If <src_path> starts with "~", it's replaced by the home directory of the
 * user whose login name is following (current user if no login name).
 * If the file name starts with "$<x>/", then "$<x>" is replaced by
 * the content of the environment variable <x>.
 * If <relative_to> is not NULL, <src_name> is relative to that path name.
 * <relative_to> must be an absolute path name (a directory or a file).
 * The returned path must be freed after use. */

extern string_t replace_extension (string_t file_name, string_t extension);
/* Replace extension of <file_name> by <extension> (append if it has none).
 * The new string must be freed after use. */

extern bool_t has_extension (string_t file_name, string_t extension);
/* Test if <file_name> has extension <extension>. */

extern void set_file_name (string_t *file_name_ptr, 
			   string_t file_name, 
			   string_t extension);
/* Set *<file_name_ptr> to <file_name>, but with extension <extension>.
 * Print an error if *<file_name_ptr> is already set.
 * If <extension> == NULL, the file name is not modified.
 * The created file name must be freed with "free". */

/* end of file ==============================================================*/
