/*
 * Copyright (C) 1998  Mark Baysinger (mbaysing@ucsd.edu)
 * Copyright (C) 1998,1999  Ross Combs (rocombs@cs.nmsu.edu)
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include "config.h"
#include "setup.h"
#define CONNECTION_INTERNAL_ACCESS
#include <stdio.h>
#include <stddef.h>
#ifdef STDC_HEADERS
# include <stdlib.h>
#else
# ifdef HAVE_MALLOC_H
#  include <malloc.h>
# endif
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include "compat/strdup.h"
#include <errno.h>
#include "compat/strerror.h"
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif
#include <sys/types.h>
#include <sys/socket.h>
#include "eventlog.h"
#include "account.h"
#include "channel.h"
#include "game.h"
#include "queue.h"
#include "tick.h"
#include "packet.h"
#include "tag.h"
#include "bn_type.h"
#include "message.h"
#include "version.h"
#include "prefs.h"
#include "util.h"
#include "list.h"
#include "watch.h"
#include "timer.h"
#include "connection.h"


static int      totalcount=0;
static t_list * conn_head=NULL;

static void conn_send_welcome(t_connection * c);
static void conn_test_latency(t_connection * c, time_t now, t_timer_data delta);


static void conn_send_welcome(t_connection * c)
{
    char const * filename;
    FILE *       fd;
    
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_send_welcome","got NULL connection");
	return;
    }
    if (c->welcomed)
	return;
    
    if ((filename = prefs_get_motdfile()))
	if ((fd = fopen(filename,"r")))
	{
	    message_send_file(c,fd);
	    fclose(fd);
	}
	else
	    eventlog(eventlog_level_error,"conn_send_welcome","could not open MOTD file \"%s\" for reading (fopen: %s)",filename,strerror(errno));
    
    c->welcomed = 1;
}


static void conn_test_latency(t_connection * c, time_t now, t_timer_data delta)
{
    t_packet * packet;
    
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_test_latency","got NULL connection");
        return;
    }
    
    /* FIXME: real Battle.net sends these even before login */
    if (conn_get_channel(c))
        if ((packet = packet_create(packet_class_normal)))
	{
	    packet_set_size(packet,sizeof(t_server_echoreq));
	    packet_set_type(packet,SERVER_ECHOREQ);
	    bn_int_set(&packet->u.server_echoreq.ticks,get_ticks());
	    queue_push_packet(conn_get_out_queue(c),packet);
	    packet_del_ref(packet);
	}
	else
	    eventlog(eventlog_level_error,"conn_test_latency","could not create packet");
    
    if (timerlist_add_timer(c,now+(time_t)delta.n,conn_test_latency,delta)<0)
	eventlog(eventlog_level_error,"conn_test_latency","could not add timer");
}




extern char const * conn_class_get_str(t_conn_class class)
{
    switch (class)
    {
    case conn_class_normal:
	return "normal";
    case conn_class_file:
	return "file";
    case conn_class_bot:
	return "bot";
    case conn_class_none:
	return "none";
    default:
	return "UNKNOWN";
    }
}


extern char const * conn_state_get_str(t_conn_state state)
{
    switch (state)
    {
    case conn_state_empty:
	return "empty";
    case conn_state_initial:
	return "initial";
    case conn_state_connected:
	return "connected";
    case conn_state_bot_username:
	return "bot_username";
    case conn_state_bot_password:
	return "bot_password";
    case conn_state_loggedin:
	return "loggedin";
    case conn_state_destroy:
	return "destroy";
    default:
	return "UNKNOWN";
    }
}


extern t_connection * conn_create(int sock, unsigned int addr, unsigned short port)
{
    t_connection * temp=NULL;
    
    if (sock<0)
    {
        eventlog(eventlog_level_error,"conn_create","got bad socket");
        return NULL;
    }
    
    if (!(temp = malloc(sizeof(t_connection))))
    {
        eventlog(eventlog_level_error,"conn_create","could not allocate memory for temp");
	return NULL;
    }
    
    temp->sock                  = sock;
    temp->port                  = port;
    temp->addr                  = addr;
    temp->class                 = conn_class_none;
    temp->state                 = conn_state_initial;
    temp->sessionkey            = ((unsigned int)rand())^((unsigned int)time(NULL));
    temp->flags                 = 0;
    temp->latency               = 0;
    temp->settings.dnd          = NULL;
    temp->settings.away         = NULL;
    temp->settings.ignore_list  = NULL;
    temp->settings.ignore_count = 0;
    temp->clienttag             = NULL;
    temp->clientver             = NULL;
    temp->account               = NULL;
    temp->channel               = NULL;
    temp->game                  = NULL;
    temp->outqueue              = NULL;
    temp->outsize               = 0;
    temp->inqueue               = NULL;
    temp->insize                = 0;
    temp->welcomed              = 0;
    temp->host                  = NULL;
    temp->user                  = NULL;
    temp->clientexe             = NULL;
    temp->owner                 = NULL;
    temp->cdkey                 = NULL;
    temp->botuser               = NULL;
    temp->protflag              = 0;
    
    if (list_prepend_item(&conn_head,temp)<0)
    {
	pfree(temp,sizeof(t_connection));
	eventlog(eventlog_level_error,"conn_create","could not prepend temp");
	return NULL;
    }
    
    eventlog(eventlog_level_info,"conn_create","[%d] session key is 0x%08x",temp->sock,temp->sessionkey);
    
    return temp;
}


extern void conn_destroy(t_connection * c)
{
    if (!c)
    {
	eventlog(eventlog_level_error,"conn_destroy","got NULL connection");
	return;
    }
    
    if (list_remove_item(&conn_head,c)<0)
    {
	eventlog(eventlog_level_error,"conn_destroy","could not remove item from list");
	return;
    }
    
    /* if this user in a channel, notify everyone that the user has left */
    if (c->channel)
	channel_del_connection(c->channel,c);
    
    conn_set_game(c,NULL,NULL,NULL,game_type_none,0);
    
    watchlist_del_all_events(c);
    timerlist_del_all_timers(c);
    
    c->state = conn_state_empty;
    
    if (c->settings.away)
	pfree((void *)c->settings.away,strlen(c->settings.away)+1); /* avoid warning */
    if (c->settings.dnd)
	pfree((void *)c->settings.dnd,strlen(c->settings.dnd)+1); /* avoid warning */
    
    if (c->clienttag)
	pfree((void *)c->clienttag,strlen(c->clienttag)+1); /* avoid warning */
    if (c->clientver)
	pfree((void *)c->clientver,strlen(c->clientver)+1); /* avoid warning */
    if (c->host)
	pfree((void *)c->host,strlen(c->host)+1); /* avoid warning */
    if (c->user)
	pfree((void *)c->user,strlen(c->user)+1); /* avoid warning */
    if (c->clientexe)
	pfree((void *)c->clientexe,strlen(c->clientexe)+1); /* avoid warning */
    if (c->owner)
	pfree((void *)c->owner,strlen(c->owner)+1); /* avoid warning */
    if (c->cdkey)
	pfree((void *)c->cdkey,strlen(c->cdkey)+1); /* avoid warning */
    if (c->botuser)
	pfree((void *)c->botuser,strlen(c->botuser)+1); /* avoid warning */
    
    if (c->settings.ignore_count>0)
	if (!c->settings.ignore_list)
	    eventlog(eventlog_level_error,"conn_destroy","found NULL ignore_list with ignore_count=%u",c->settings.ignore_count);
	else
	    pfree(c->settings.ignore_list,c->settings.ignore_count*sizeof(t_account *));
    
    if (c->account)
    {
	char const * tname;
	
	watchlist_notify_event(c->account,watch_event_logout);
	tname = account_get_name(c->account);
	eventlog(eventlog_level_info,"conn_destroy","[%d] \"%s\" logged out",c->sock,tname);
	account_unget_name(tname);
	c->account = NULL; /* the account code will free the memory later */
    }
    
    /* make sure the connection is closed */
    shutdown(c->sock,2);
    close(c->sock);
    
    /* clear out the packet queues */
    queue_clear(&c->inqueue);
    queue_clear(&c->outqueue);
    
    eventlog(eventlog_level_info,"conn_destroy","[%d] closed connection",c->sock);
    
    pfree(c,sizeof(t_connection));
}


extern int conn_match(t_connection const * c, char const * username)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_class","got NULL connection");
        return -1;
    }
    if (!username)
    {
        eventlog(eventlog_level_error,"conn_get_class","got NULL username");
        return -1;
    }
    
    if (!c->account)
	return 0;
    
    return account_match(c->account,username);
}


extern t_conn_class conn_get_class(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_class","got NULL connection");
        return conn_class_none;
    }
    
    return c->class;
}


extern void conn_set_class(t_connection * c, t_conn_class class)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_class","got NULL connection");
        return;
    }
    
    c->class = class;
    
    if (class==conn_class_normal)
    {
	t_timer_data  data;
	unsigned long delta;
	
	delta = prefs_get_latency();
	data.n = delta;
	if (timerlist_add_timer(c,time(NULL)+(time_t)delta,conn_test_latency,data)<0)
	    eventlog(eventlog_level_error,"conn_create","could not add timer");
    }
    
}


extern t_conn_state conn_get_state(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_state","got NULL connection");
        return conn_state_empty;
    }
    
    return c->state;
}


extern void conn_set_state(t_connection * c, t_conn_state state)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_state","got NULL connection");
        return;
    }
    
    c->state = state;
}


extern unsigned int conn_get_sessionkey(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_sessionkey","got NULL connection");
        return 0;
    }
    
    return c->sessionkey;
}


extern unsigned int conn_get_addr(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_addr","got NULL connection");
        return 0;
    }
    
    return c->addr;
}


extern unsigned short conn_get_port(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_port","got NULL connection");
        return 0;
    }
    
    return c->port;
}


extern unsigned int conn_get_game_addr(t_connection const * c)
{
    return conn_get_addr(c);
}


extern unsigned short conn_get_game_port(t_connection const * c)
{
    return prefs_get_gameport();
}


extern void conn_set_host(t_connection * c, char const * host)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_host","got NULL connection");
        return;
    }
    if (!host)
    {
        eventlog(eventlog_level_error,"conn_set_host","got NULL host");
        return;
    }
    
    if (c->host)
	pfree((void *)c->host,strlen(c->host)+1); /* avoid warning */
    if (!(c->host = strdup(host)))
	eventlog(eventlog_level_error,"conn_set_host","could not allocate memory for c->host");
}


extern void conn_set_user(t_connection * c, char const * user)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_user","got NULL connection");
        return;
    }
    if (!user)
    {
        eventlog(eventlog_level_error,"conn_set_user","got NULL user");
        return;
    }
    
    if (c->user)
	pfree((void *)c->user,strlen(c->user)+1); /* avoid warning */
    if (!(c->user = strdup(user)))
	eventlog(eventlog_level_error,"conn_set_user","could not allocate memory for c->user");
}


extern void conn_set_owner(t_connection * c, char const * owner)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_owner","got NULL connection");
        return;
    }
    if (!owner)
    {
        eventlog(eventlog_level_error,"conn_set_owner","got NULL owner");
        return;
    }
    
    if (c->owner)
	pfree((void *)c->owner,strlen(c->owner)+1); /* avoid warning */
    if (!(c->owner = strdup(owner)))
	eventlog(eventlog_level_error,"conn_set_owner","could not allocate memory for c->owner");
}


extern void conn_set_cdkey(t_connection * c, char const * cdkey)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_cdkey","got NULL connection");
        return;
    }
    if (!cdkey)
    {
        eventlog(eventlog_level_error,"conn_set_cdkey","got NULL cdkey");
        return;
    }
    
    if (c->cdkey)
	pfree((void *)c->cdkey,strlen(c->cdkey)+1); /* avoid warning */
    if (!(c->cdkey = strdup(cdkey)))
	eventlog(eventlog_level_error,"conn_set_cdkey","could not allocate memory for c->cdkey");
}


extern char const * conn_get_clientexe(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_clientexe","got NULL connection");
        return NULL;
    }
    
    if (!c->clientexe)
	return "";
    return c->clientexe;
}


extern void conn_set_clientexe(t_connection * c, char const * clientexe)
{
    char const * temp;
    
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_clientexe","got NULL connection");
        return;
    }
    if (!clientexe)
    {
        eventlog(eventlog_level_error,"conn_set_clientexe","got NULL clientexe");
        return;
    }
    
    if (!(temp = strdup(clientexe)))
    {
	eventlog(eventlog_level_error,"conn_set_clientexe","unable to allocate memory for clientexe");
	return;
    }
    if (c->clientexe)
	pfree((void *)c->clientexe,strlen(c->clientexe)+1); /* avoid warning */
    c->clientexe = temp;
}


extern char const * conn_get_clientver(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_clientver","got NULL connection");
        return NULL;
    }
    
    if (!c->clientver)
	return "";
    return c->clientver;
}


extern void conn_set_clientver(t_connection * c, char const * clientver)
{
    char const * temp;
    
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_clientver","got NULL connection");
        return;
    }
    if (!clientver)
    {
        eventlog(eventlog_level_error,"conn_set_clientver","got NULL clientver");
        return;
    }
    
    if (!(temp = strdup(clientver)))
    {
	eventlog(eventlog_level_error,"conn_set_clientver","unable to allocate memory for clientver");
	return;
    }
    if (c->clientver)
	pfree((void *)c->clientver,strlen(c->clientver)+1); /* avoid warning */
    c->clientver = temp;
}


extern char const * conn_get_clienttag(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_clienttag","got NULL connection");
        return NULL;
    }
    
    if (!c->clienttag)
	return "";
    return c->clienttag;
}


extern void conn_set_clienttag(t_connection * c, char const * clienttag)
{
    char const * temp;
    
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_clienttag","got NULL connection");
        return;
    }
    if (!clienttag)
    {
        eventlog(eventlog_level_error,"conn_set_clienttag","[%d] got NULL clienttag",conn_get_socket(c));
        return;
    }
    if (strlen(clienttag)!=4)
    {
        eventlog(eventlog_level_error,"conn_set_clienttag","[%d] got bad clienttag",conn_get_socket(c));
        return;
    }
    
    if (!c->clienttag || strcmp(c->clienttag,clienttag)!=0)
	eventlog(eventlog_level_info,"conn_set_clienttag","[%d] setting client type to \"%s\"",conn_get_socket(c),clienttag);
    
    if (!(temp = strdup(clienttag)))
    {
	eventlog(eventlog_level_error,"conn_set_clienttag","[%d] unable to allocate memory for clienttag",conn_get_socket(c));
	return;
    }
    if (c->clienttag)
	pfree((void *)c->clienttag,strlen(c->clienttag)+1); /* avoid warning */
    c->clienttag = temp;
}


extern void conn_set_account(t_connection * c, t_account * account)
{
    t_connection * other;
    unsigned int   now;
    char const *   tname;
    
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_account","got NULL connection");
        return;
    }
    if (!account)
    {
        eventlog(eventlog_level_error,"conn_set_account","got NULL account");
        return;
    }
    
    now = (unsigned int)time(NULL);
    
    if ((other = connlist_find_connection((tname = account_get_name(account)))))
    {
	eventlog(eventlog_level_info,"conn_set_account","[%d] forcing logout of previous login for \"%s\"",c->sock,tname);
	other->state = conn_state_destroy;
    }
    account_unget_name(tname);
    
    c->account = account;
    c->state = conn_state_loggedin;
    
    if (account_get_fl_time(c->account)==0)
    {
	account_set_fl_time(c->account,now);
	account_set_fl_connection(c->account,c->addr);
	account_set_fl_host(c->account,c->host);
	account_set_fl_user(c->account,c->user);
	account_set_fl_clientexe(c->account,c->clientexe);
	account_set_fl_clienttag(c->account,c->clienttag);
	account_set_fl_clientver(c->account,c->clientver);
	account_set_fl_owner(c->account,c->owner);
	account_set_fl_cdkey(c->account,c->cdkey);
    }
    account_set_ll_time(c->account,now);
    account_set_ll_connection(c->account,c->addr);
    account_set_ll_host(c->account,c->host);
    account_set_ll_user(c->account,c->user);
    account_set_ll_clientexe(c->account,c->clientexe);
    account_set_ll_clienttag(c->account,c->clienttag);
    account_set_ll_clientver(c->account,c->clientver);
    account_set_ll_owner(c->account,c->owner);
    account_set_ll_cdkey(c->account,c->cdkey);
    
    if (c->host)
    {
	pfree((void *)c->host,strlen(c->host)+1); /* avoid warning */
	c->host = NULL;
    }
    if (c->user)
    {
	pfree((void *)c->user,strlen(c->user)+1); /* avoid warning */
	c->user = NULL;
    }
    if (c->clientexe)
    {
	pfree((void *)c->clientexe,strlen(c->clientexe)+1); /* avoid warning */
	c->clientexe = NULL;
    }
    if (c->owner)
    {
	pfree((void *)c->owner,strlen(c->owner)+1); /* avoid warning */
	c->owner = NULL;
    }
    if (c->cdkey)
    {
	pfree((void *)c->cdkey,strlen(c->cdkey)+1); /* avoid warning */
	c->cdkey = NULL;
    }
    
    if (c->botuser)
    {
	pfree((void *)c->botuser,strlen(c->botuser)+1); /* avoid warning */
	c->botuser = NULL;
    }
    
    totalcount++;
    
    watchlist_notify_event(c->account,watch_event_login);
    
    return;
}


extern t_account * conn_get_account(t_connection const * c)
{
    if (!c)
    {
	eventlog(eventlog_level_error,"conn_get_account","got NULL connection");
	return NULL;
    }
    
    return c->account;
}


extern int conn_set_botuser(t_connection * c, char const * username)
{
    char const * temp;
    
    if (!c)
    {
	eventlog(eventlog_level_error,"conn_set_botuser","got NULL connection");
	return -1;
    }
    if (!username)
    {
	eventlog(eventlog_level_error,"conn_set_botuser","got NULL username");
	return -1;
    }
    
    if (!(temp = strdup(username)))
    {
	eventlog(eventlog_level_error,"conn_set_botuser","unable to duplicate username");
	return -1;
    }
    if (c->botuser)
	pfree((void *)c->botuser,strlen(c->botuser)+1); /* avoid warning */
    
    c->botuser = temp;
    
    return 0;
}


extern char const * conn_get_botuser(t_connection const * c)
{
    if (!c)
    {
	eventlog(eventlog_level_error,"conn_get_botuser","got NULL connection");
	return NULL;
    }
    
    return c->botuser;
}


extern unsigned int conn_get_flags(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_flags","got NULL connection");
        return 0;
    }
    
    return c->flags;
}


extern void conn_add_flags(t_connection * c, unsigned int flags)
{
    unsigned int oldflags;
    
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_add_flags","got NULL connection");
        return;
    }
    oldflags = c->flags;
    c->flags |= flags;
    
    if (oldflags!=c->flags && c->channel)
	channel_update_flags(c);
}


extern void conn_del_flags(t_connection * c, unsigned int flags)
{
    unsigned int oldflags;
    
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_del_flags","got NULL connection");
        return;
    }
    oldflags = c->flags;
    c->flags &= ~flags;
    
    if (oldflags!=c->flags && c->channel)
	channel_update_flags(c);
}


extern unsigned int conn_get_latency(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_latency","got NULL connection");
        return 0;
    }
    
    return c->latency;
}


extern void conn_set_latency(t_connection * c, unsigned int ms)
{
    if (!c)
    {
	eventlog(eventlog_level_error,"conn_set_latency","got NULL connection");
	return;
    }
    
    c->latency = ms;
    
    if (c->channel)
	channel_update_latency(c);
}


extern char const * conn_get_awaystr(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_awaystr","got NULL connection");
        return NULL;
    }
    
    return c->settings.away;
}


extern int conn_set_awaystr(t_connection * c, char const * away)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_awaystr","got NULL connection");
        return -1;
    }
    
    if (c->settings.away)
	pfree((void *)c->settings.away,strlen(c->settings.away)+1); /* avoid warning */
    if (!away)
        c->settings.away = NULL;
    else
        if (!(c->settings.away = strdup(away)))
	{
	    eventlog(eventlog_level_error,"conn_set_awaystr","could not allocate away string");
	    return -1;
	}
    
    return 0;
}


extern char const * conn_get_dndstr(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_dndstr","got NULL connection");
        return NULL;
    }
    
    return c->settings.dnd;
}


extern int conn_set_dndstr(t_connection * c, char const * dnd)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_dndstr","got NULL connection");
        return -1;
    }
    
    if (c->settings.away)
	pfree((void *)c->settings.away,strlen(c->settings.away)+1); /* avoid warning */
    if (!dnd)
        c->settings.dnd = NULL;
    else
        if (!(c->settings.dnd = strdup(dnd)))
	{
	    eventlog(eventlog_level_error,"conn_set_dndstr","could not allocate dnd string");
	    return -1;
	}
    
    return 0;
}


extern int conn_add_ignore(t_connection * c, t_account * account)
{
    t_account * * newlist;
    
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_add_ignore","got NULL connection");
        return -1;
    }
    if (!account)
    {
        eventlog(eventlog_level_error,"conn_add_ignore","got NULL account");
        return -1;
    }
    
    if (!(newlist = realloc(c->settings.ignore_list,sizeof(t_account const *)*(c->settings.ignore_count+1))))
    {
	eventlog(eventlog_level_error,"conn_add_ignore","could not allocate memory for newlist");
	return -1;
    }
    
    newlist[c->settings.ignore_count++] = account;
    c->settings.ignore_list = newlist;
    
    return 0;
}


extern int conn_del_ignore(t_connection * c, t_account const * account)
{
    t_account * * newlist;
    t_account *   temp;
    unsigned int  i;
    
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_del_ignore","got NULL connection");
        return -1;
    }
    if (!account)
    {
        eventlog(eventlog_level_error,"conn_del_ignore","got NULL account");
        return -1;
    }
    
    for (i=0; i<c->settings.ignore_count; i++)
	if (c->settings.ignore_list[i]==account)
	    break;
    if (i==c->settings.ignore_count)
	return -1; /* not in list */
    
    /* swap entry to be deleted with last entry */
    temp = c->settings.ignore_list[c->settings.ignore_count-1];
    c->settings.ignore_list[c->settings.ignore_count-1] = c->settings.ignore_list[i];
    c->settings.ignore_list[i] = temp;
    
    if (c->settings.ignore_count==1) /* some realloc()s are buggy */
    {
	pfree(c->settings.ignore_list,sizeof(t_account *));
	newlist = NULL;
    }
    else
	newlist = realloc(c->settings.ignore_list,sizeof(t_account const *)*(c->settings.ignore_count-1));
    
    c->settings.ignore_count--;
    c->settings.ignore_list = newlist;
    
    return 0;
}


extern int conn_add_watch(t_connection * c, t_account * account)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_add_watch","got NULL connection");
        return -1;
    }
    
    if (watchlist_add_events(c,account,watch_event_login|watch_event_logout|watch_event_joingame|watch_event_leavegame)<0)
	return -1;
    return 0;
}


extern int conn_del_watch(t_connection * c, t_account * account)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_add_watch","got NULL connection");
        return -1;
    }
    
    if (watchlist_del_events(c,account,watch_event_login|watch_event_logout|watch_event_joingame|watch_event_leavegame)<0)
	return -1;
    return 0;
}


extern int conn_check_ignoring(t_connection const * c, char const * me)
{
    unsigned int i;
    t_account *  temp;
    
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_check_ignoring","got NULL connection");
        return -1;
    }
    
    if (!me || !(temp = accountlist_find_account(me)))
	return -1;
    
    if (c->settings.ignore_list)
	for (i=0; i<c->settings.ignore_count; i++)
	    if (c->settings.ignore_list[i]==temp)
		return 1;
     
    return 0;
}


extern t_channel * conn_get_channel(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_channel","got NULL connection");
        return NULL;
    }
    
    return c->channel;
}


extern int conn_set_channel(t_connection * c, char const * channelname)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_channel","got NULL connection");
        return -1;
    }
    
    if (c->channel)
    {
	channel_del_connection(c->channel,c);
	c->channel = NULL;
    }
    
    if (channelname)
    {
	unsigned int created;
	
	/* if you're entering a channel, make sure they didn't exit a game without telling us */
	if (c->game)
	{
            game_del_player(conn_get_game(c),c);
	    c->game = NULL;
	}
	
	created = 0;
	if (!(c->channel = channellist_find_channel(channelname)))
	{
	    if (!(c->channel = channel_create(channelname,NULL,NULL,0,1,1)))
	    {
		eventlog(eventlog_level_error,"conn_set_channel","[%d] could not create channel on join \"%s\"",c->sock,channelname);
		return -1;
	    }
	    created = 1;
	}
	
	if (channel_add_connection(conn_get_channel(c),c)<0)
	{
	    if (created)
		channel_destroy(c->channel);
	    c->channel = NULL;
            return -1;
	}
	
	eventlog(eventlog_level_info,"conn_set_channel","[%d] joined channel \"%s\"",c->sock,channel_get_name(c->channel));
	conn_send_welcome(c);
    }
    
    return 0;
}


extern t_game * conn_get_game(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_game","got NULL connection");
        return NULL;
    }
    
    return c->game;
}


extern int conn_set_game(t_connection * c, char const * gamename, char const * gamepass, char const * gameinfo, t_game_type type, int version)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_game","got NULL connection");
        return -1;
    }
    
    if (c->game)
        game_del_player(conn_get_game(c),c);
    
    if (gamename)
    {
	if (!(c->game = gamelist_find_game(gamename,type)))
	    c->game = game_create(gamename,gamepass,gameinfo,type,version,conn_get_clienttag(c));
	if (c->game)
	    if (game_add_player(conn_get_game(c),gamepass,version,c)<0)
	    {
		c->game = NULL; /* bad password or version # */
		return -1;
	    }
    }
    else
	c->game = NULL;
    
    return 0;
}


extern t_queue * * conn_get_in_queue(t_connection * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_in_queue","got NULL connection");
        return NULL;
    }
    
    return &c->inqueue;
}


extern int conn_get_in_size(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_in_size","got NULL connection");
        return -1;
    }
    
    return c->insize;
}


extern void conn_set_in_size(t_connection * c, unsigned int size)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_in_size","got NULL connection");
        return;
    }
    
    c->insize = size;
}


extern t_queue * * conn_get_out_queue(t_connection * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_out_queue","got NULL connection");
        return NULL;
    }
    
    return &c->outqueue;
}


extern int conn_get_out_size(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_out_size","got NULL connection");
        return -1;
    }
    
    return c->outsize;
}


extern void conn_set_out_size(t_connection * c, unsigned int size)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_out_size","got NULL connection");
        return;
    }
    
    c->outsize = size;
}


extern char const * conn_get_username(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_username","got NULL connection");
        return NULL;
    }
    
    return account_get_name(c->account);
}


extern void conn_unget_username(char const * name)
{
    account_unget_name(name);
}


extern unsigned int conn_get_userid(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_userid","got NULL connection");
        return 0;
    }
    
    return account_get_uid(c->account);
}


extern int conn_get_socket(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_socket","got NULL connection");
        return -1;
    }
    
    return c->sock;
}


extern int conn_get_protflag(t_connection const * c)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_protflag","got NULL connection");
        return -1;
    }

    return c->protflag;
}


extern int conn_set_protflag(t_connection * c, int flag)
{
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_protflag","got NULL connection");
        return -1;
    }

    c->protflag = flag; 
    return 0;
}


extern char const * conn_get_playerinfo(t_connection const * c, char const * clienttag)
{
    static char playerinfo[1024];
    
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_get_playerinfo","got NULL connection");
        return NULL;
    }
    if (!clienttag || strlen(clienttag)!=4)
    {
	eventlog(eventlog_level_error,"conn_get_playerinfo","got bad clienttag");
	return NULL;
    }
    
    if (strcmp(clienttag,CLIENTTAG_BNCHATBOT)==0)
    {
	strcpy(playerinfo,""); /* shouldn't be called */
	eventlog(eventlog_level_warn,"conn_get_playerinfo","got CHAT clienttag");
    }
    else if (strcmp(clienttag,CLIENTTAG_STARCRAFT)==0)
    {
	sprintf(playerinfo,"RATS %u %u %u %u %u",
		account_get_ladder_rating(conn_get_account(c),clienttag),
		account_get_ladder_rank(conn_get_account(c),clienttag),
		account_get_normal_wins(conn_get_account(c),clienttag),
		0,0);
    }
    else if (strcmp(clienttag,CLIENTTAG_BROODWARS)==0)
    {
	sprintf(playerinfo,"PXES %u %u %u %u %u",
		account_get_ladder_rating(conn_get_account(c),clienttag),
		account_get_ladder_rank(conn_get_account(c),clienttag),
		account_get_normal_wins(conn_get_account(c),clienttag),
		0,0);
    }
    else if (strcmp(clienttag,CLIENTTAG_SHAREWARE)==0)
    {
	sprintf(playerinfo,"RHSS %u %u %u %u %u",
		account_get_ladder_rating(conn_get_account(c),clienttag),
		account_get_ladder_rank(conn_get_account(c),clienttag),
		account_get_normal_wins(conn_get_account(c),clienttag),
		0,0);
    }
    else if (strcmp(clienttag,CLIENTTAG_DIABLORTL)==0)
    {
	sprintf(playerinfo,"LTRD %u %u %u %u %u %u %u %u %u",
		account_get_normal_level(conn_get_account(c),clienttag),
		account_get_normal_class(conn_get_account(c),clienttag),
		account_get_normal_diablo_kills(conn_get_account(c),clienttag),
		account_get_normal_strength(conn_get_account(c),clienttag),
		account_get_normal_magic(conn_get_account(c),clienttag),
		account_get_normal_dexterity(conn_get_account(c),clienttag),
		account_get_normal_vitality(conn_get_account(c),clienttag),
		account_get_normal_gold(conn_get_account(c),clienttag),
		0);
    }
    else if (strcmp(clienttag,CLIENTTAG_DIABLOSHR)==0)
    {
	sprintf(playerinfo,"RHSD %u %u %u %u %u %u %u %u %u",
		account_get_normal_level(conn_get_account(c),clienttag),
		account_get_normal_class(conn_get_account(c),clienttag),
		account_get_normal_diablo_kills(conn_get_account(c),clienttag),
		account_get_normal_strength(conn_get_account(c),clienttag),
		account_get_normal_magic(conn_get_account(c),clienttag),
		account_get_normal_dexterity(conn_get_account(c),clienttag),
		account_get_normal_vitality(conn_get_account(c),clienttag),
		account_get_normal_gold(conn_get_account(c),clienttag),
		0);
    }
    else if (strcmp(clienttag,CLIENTTAG_WARCIIBNE)==0)
    {
	sprintf(playerinfo,"NB2W %u %u %u %u %u %u %u %u",
		0,
		0,
		account_get_normal_wins(conn_get_account(c),clienttag),
		0,
		0,
		0,
		0,
		0);
    }
    else
	playerinfo[0] = '\0';
    
    return playerinfo;
}


extern int conn_set_playerinfo(t_connection const * c, char const * playerinfo)
{
    char const * clienttag;
    
    if (!c)
    {
        eventlog(eventlog_level_error,"conn_set_playerinfo","got NULL connection");
        return -1;
    }
    if (!playerinfo)
    {
	eventlog(eventlog_level_error,"conn_set_playerinfo","got NULL playerinfo");
	return -1;
    }
    if (!(clienttag = conn_get_clienttag(c)))
    {
	eventlog(eventlog_level_error,"conn_set_playerinfo","no clienttag");
	return -1;
    }
    
    if (strcmp(clienttag,CLIENTTAG_DIABLORTL)==0)
    {
	unsigned int level;
	unsigned int class;
	unsigned int diablo_kills;
	unsigned int strength;
	unsigned int magic;
	unsigned int dexterity;
	unsigned int vitality;
	unsigned int gold;
	
	if (sscanf(playerinfo,"LTRD %u %u %u %u %u %u %u %u %*u",
		   &level,
		   &class,
		   &diablo_kills,
		   &strength,
		   &magic,
		   &dexterity,
		   &vitality,
		   &gold)!=8)
	{
	    eventlog(eventlog_level_error,"conn_set_playerinfo","got bad playerinfo");
	    return -1;
	}
	
	account_set_normal_level(conn_get_account(c),clienttag,level);
	account_set_normal_class(conn_get_account(c),clienttag,class);
	account_set_normal_diablo_kills(conn_get_account(c),clienttag,diablo_kills);
	account_set_normal_strength(conn_get_account(c),clienttag,strength);
	account_set_normal_magic(conn_get_account(c),clienttag,magic);
	account_set_normal_dexterity(conn_get_account(c),clienttag,dexterity);
	account_set_normal_vitality(conn_get_account(c),clienttag,vitality);
	account_set_normal_gold(conn_get_account(c),clienttag,gold);
    }
    else if (strcmp(clienttag,CLIENTTAG_DIABLOSHR)==0)
    {
	unsigned int level;
	unsigned int class;
	unsigned int diablo_kills;
	unsigned int strength;
	unsigned int magic;
	unsigned int dexterity;
	unsigned int vitality;
	unsigned int gold;
	
	if (sscanf(playerinfo,"RHSD %u %u %u %u %u %u %u %u %*u",
		   &level,
		   &class,
		   &diablo_kills,
		   &strength,
		   &magic,
		   &dexterity,
		   &vitality,
		   &gold)!=8)
	{
	    eventlog(eventlog_level_error,"conn_set_playerinfo","got bad playerinfo");
	    return -1;
	}
	
	account_set_normal_level(conn_get_account(c),clienttag,level);
	account_set_normal_class(conn_get_account(c),clienttag,class);
	account_set_normal_diablo_kills(conn_get_account(c),clienttag,diablo_kills);
	account_set_normal_strength(conn_get_account(c),clienttag,strength);
	account_set_normal_magic(conn_get_account(c),clienttag,magic);
	account_set_normal_dexterity(conn_get_account(c),clienttag,dexterity);
	account_set_normal_vitality(conn_get_account(c),clienttag,vitality);
	account_set_normal_gold(conn_get_account(c),clienttag,gold);
    }
    else
    {
	eventlog(eventlog_level_warn,"conn_set_playerinfo","setting playerinfo for client \"%s\" not supported",clienttag);
	return -1;
    }
    
    return 0;
}



extern void connlist_init(void)
{
    /* nothing to do for now */
}


extern t_connection * connlist_find_connection(char const * username)
{
    t_connection *         c;
    t_account const *      temp;
    t_list const * const * save;
    
    if (!username)
    {
	eventlog(eventlog_level_error,"connlist_find_connection","got NULL username");
	return NULL;
    }
    
    if (!(temp = accountlist_find_account(username)))
	return NULL;
    
    for (c=connlist_get_first(&save); c; c=connlist_get_next(&save))
	if (c->account==temp)
	    return c;
    
    return NULL;
}


extern t_connection * connlist_find_connection_by_sessionkey(unsigned int sessionkey)
{
    t_connection *         c;
    t_list const * const * save;
    
    for (c=connlist_get_first(&save); c; c=connlist_get_next(&save))
	if (c->sessionkey==sessionkey)
	    return c;
    
    return NULL;
}


extern int connlist_get_length(void)
{
    return list_get_length(conn_head);
}


extern int connlist_login_get_length(void)
{
    t_connection const *   c;
    int                    count;
    t_list const * const * save;
    
    for (count=0,c=connlist_get_first(&save); c; c=connlist_get_next(&save))
	if (c->state==conn_state_loggedin)
	    count++;
    
    return count;
}


extern t_connection * connlist_get_first(t_list const * const * * save)
{
    void * conn;
    
    if (!save)
    {
	eventlog(eventlog_level_error,"connlist_get_first","got NULL save");
	return NULL;
    }
    
    *save = (t_list const * const *)&conn_head; /* avoid warning */
    
    if (!**save)
    {
	*save = NULL;
	return NULL;
    }
    conn = list_get_item(**save);
    *save = list_get_next_const(**save);
    
    return conn;
}


extern t_connection * connlist_get_next(t_list const * const * * save)
{
    void * conn;
    
    if (!save)
    {
	eventlog(eventlog_level_error,"connlist_get_next","got NULL save");
	return NULL;
    }
    
    if (!*save || !**save)
    {
	*save = NULL;
	return NULL;
    }
    conn = list_get_item(**save);
    *save = list_get_next_const(**save);
    
    return conn;
}


extern int connlist_total_logins(void)
{
    return totalcount;
}
