// Simple utility to display the current batch status
// of the Darxite daemon.

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>

#include <darxite.h>

#define OPTION_STRING "vhac:w:"
#define AUTHOR        "Ashley Montanaro"
#define VERSION       "0.3"

int DaemonFd, RemotePort = 0, TermWidth = 80;
char RemoteHost[256];
BOOL Ansi;

void TruncatePath(char *path);
char *ReadCmdLine(int argc, char **argv);
void PrintVersion(void);
void Usage(char *prog_name);

void TruncatePath(char *path)
{
    char buffer[256];
    int last_pos, half_size = (TermWidth - 24) / 2;

    memset(buffer, 0, sizeof(buffer));
    if (strlen(path) > TermWidth - 24)
    {
        // make sure we pick up the filename
        last_pos = strrchr(path, '/') - path;
        if ((last_pos <= 0) || (last_pos >= strlen(path) - half_size))
        {
            last_pos = strlen(path) - half_size;
            strncpy(buffer, path, half_size);
            strcat(buffer, "...");
            strncat(buffer, path + strlen(path) - half_size, half_size);
        }
        else
        {
            if (strlen(path) - last_pos < TermWidth - 24)
                strncpy(buffer, path,
                        half_size - (strlen(path) - last_pos - half_size));
            else
                strncpy(buffer, path, 10);
            strcat(buffer, "...");
            strncat(buffer, path + last_pos, strlen(path) - last_pos);
        }
    }
    else
    {
        strcpy(buffer, path);
    }
    strcpy(path, buffer);
}

void PrintVersion(void)
{
    printf("DarxStat utility v%s, release %s (%s) by %s\n", VERSION,
           RELEASE_VER, RELEASE_NAME, AUTHOR);
    exit(0);
}

void Usage(char *prog_name)
{
    printf("Usage: %s [OPTIONS]\n"
           "where options are:\n"
           "-v, --version\t\t\tShow version and exit\n"
           "-h, --help\t\t\tShow some usage information\n"
           "-a, --ansi\t\t\tDisplay in ANSI colour\n"
           "-c<host>:<port> --connect<host>\tConnect to <host> on <port>\n"
           "-w<width>, --width <width>\tTerminal width is <width>\n",
           prog_name);
    exit(0);
}

char *ReadCmdLine(int argc, char **argv)
{
    char opt;
    int option_index;

    static struct option long_options[] = {
        { "version", 	0, 0, 'v' },
        { "help",		0, 0, 'h' },
        { "ansi",       0, 0, 'a' },
        { "connect",	1, 0, 'c' },
        { "width",		1, 0, 'w' },
        { 0, 			0, 0,  0  }
    };

    opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                      &option_index);
    while (opt != -1)
    {
        switch (opt)
        {
        case 'c':
            if (strchr(optarg, ':'))
            {
                memset(RemoteHost, 0, sizeof(RemoteHost));
                strncpy(RemoteHost, optarg, strchr(optarg, ':') - optarg);
                RemotePort = atoi(strchr(optarg, ':') + 1);
                if ((strlen(RemoteHost) == 0) || (RemotePort <= 0))
                {
                    printf("Invalid host/port to connect to daemon\n");
                    exit(0);
                }
            }
            else
            {
                printf("Syntax: -c <host>:<port>\n");
                exit(0);
            }
            break;
            
        case 'v':
            PrintVersion();
            break;

        case 'h':
            Usage(argv[0]);
            break;

        case 'a':
            Ansi = TRUE;
            break;

        case 'w':
            TermWidth = atoi(optarg);
            break;
        }
        opt = getopt_long(argc, argv, OPTION_STRING, long_options,
                          &option_index);        
    }
    
    if (optind < argc)
        return argv[optind];
    else
       return NULL;
}

int main(int argc, char *argv[])
{
    char *line, buffer[8192], activity[256];
    char protocol[20], server[256], path[256], dummy[10];
    int current_size, total_size, rate_overall;
    int total_time, file_count = 0, i, rc;
    char buffer2[256], buffer3[256], buffer4[256];
    float frac_done = 0.0;
    
    ReadCmdLine(argc, argv);
    
    if (strcmp(RemoteHost, "") && (RemotePort > 0))
    {
        sprintf(buffer, "Enter password for daemon on %s: ", RemoteHost);
        DaemonFd = DX_ConnectRemoteClient(RemoteHost, RemotePort,
                                          getpass(buffer), "darxstat");
    }
    else
    {
        DaemonFd = DX_ConnectClient("darxstat");
    }
    
    if (DaemonFd < 0)
    {
        fprintf(stderr, "Couldn't connect to daemon: %s\n",
                strerror(DX_errno));
        return 1;
    }
    
    strcpy(buffer, "FileStatus\n");
    write(DaemonFd, buffer, strlen(buffer));
    rc = DX_GetResponse(DaemonFd, buffer, sizeof(buffer) - 1, 10);
    buffer[sizeof(buffer) - 1] = '\0';
    if (rc == DX_LIB_ERROR)
    {
        fprintf(stderr, "Error reading response from daemon!\n");
        return 1;
    }
    line = strtok(buffer, "\n");
    if (Ansi)
    {
        printf("\x1B[38;44m");
        for (i = 0; i < TermWidth; i++)
            putchar('=');
    }
    else
    {
        for (i = 0; i < TermWidth; i++)
            putchar('+');
    }
    
    while (line)
    {
        file_count++;
        // skip new code where necessary
        if (atoi(line) == DX_FILE_STATUS)
            line += 4;
        DX_ParseFileStatus(line, protocol, server, sizeof(server),
                           path, sizeof(path), dummy, 0,
                           &current_size, &total_size, &total_time,
                           &rate_overall, activity, sizeof(activity), dummy);
        
        //total_size = 5678; strcpy(activity, "Transferring file");
        //current_size = 2800;
        
        if (!strcmp(activity, ""))
            strcpy(activity, "Unknown");
        if (total_size > 0)
        {
            frac_done = ((float)current_size) / ((float)total_size);
            if (total_size > 10 * 1024 * 1024)
            {
                sprintf(buffer2, "%.1f/%.1fMB",
                        (float)current_size / (1024.0 * 1024.0),
                        (float)total_size / (1024.0 * 1024.0));
            }
            else if (total_size > 10 * 1024)
            {
                sprintf(buffer2, "%d/%dK", current_size / 1024,
                        total_size / 1024);
            }
            else
            {
                sprintf(buffer2, "%d/%dB", current_size, total_size);
            }
        }
        if (!strcasecmp(activity, "Transferring file"))
        {
            if (rate_overall > 1024)
            {
                sprintf(buffer3, " (%.1fK/s)",
                        (float)rate_overall / 1024.0);
            }
            else
            {
                sprintf(buffer3, " (%dB/s)", rate_overall);
            }
            strcat(buffer2, buffer3);
            strcpy(buffer4, "");
            for (i = 0; i < (int)(frac_done *
                                  (TermWidth - strlen("0%%><100%%"))); i++)
                strcat(buffer4, Ansi ? "#" : "=");
            if (!Ansi)
                buffer4[strlen(buffer4) - 1] = '>';
            // make sure the extra line's displayed
            if (!strcmp(buffer4, ""))
                strcpy(buffer4, " ");
        }
        else if (!strcasecmp(activity, "Sleeping"))
        {
            frac_done = 0.5;
            sprintf(buffer3, "Sleeping (%s)", buffer2);
            strcpy(buffer2, buffer3);
            strcpy(buffer4, "");
        }
        else
        {
            frac_done = 0.5;
            strcpy(buffer2, activity);
            strcpy(buffer4, "");
        }
        if (file_count > 1)
        {
            if (Ansi)
                printf("\x1B[38;44m");
            
            for (i = 0; i < TermWidth; i++)
                putchar('-');
        }
        sprintf(buffer3, "\n%s://%s%s", protocol, server, path);
        TruncatePath(buffer3);
        strcat(buffer3, ": ");
        strcat(buffer3, buffer2);
        buffer3[TermWidth] = '\0';
        if (Ansi)
        {
            printf("\x1B[39;44m");
            // ensure that the whole line has a blue bg
            while (strlen(buffer3) <= TermWidth)
                strcat(buffer3, " ");
        }
        printf("%s\n", buffer3);
        // if we have a line of hashes to print
        if (strlen(buffer4) > 0)
        {
            printf("0%%>");
            if (Ansi)
            {
                if (frac_done < 0.34)
                    printf("\x1B[31;44m");
                else if (frac_done < 0.67)
                    printf("\x1B[36;44m");
                else
                    printf("\x1B[32;44m");
            }
            printf("%s", buffer4);
            if (Ansi)
                printf("\x1B[39;44m");
            strcpy(buffer3, "");
            for (i = strlen(buffer4); i < TermWidth - strlen("0%%><100%%" + 2);
                 i++)
            {
                strcat(buffer3, " ");
            }
            printf("%s<100%%\n", buffer3);
                
        }
        line = strtok(NULL, "\n");
    }
    if (Ansi)
        printf("\x1B[39;44m");
    if (file_count == 0)
    {
        strcpy(buffer, "There are no files being transferred.");
        while (strlen(buffer) < TermWidth)
            strcat(buffer, " ");
        printf("\n%s\n", buffer);
    }
    if (Ansi)
    {
        printf("\x1B[38;44m");
        for (i = 0; i < TermWidth; i++)
            putchar('=');
        printf("\x1B[0m\n");
    }
    else
    {
        for (i = 0; i < TermWidth; i++)
            putchar('+');
        putchar('\n');
    }
    if (rc == DX_LIB_MORE_DATA)
    {
        printf("There are more files in the batch than darxstat can "
               "display.\n");
    }
    
    DX_DisconnectClient(DaemonFd);
    
    return 0;
}
