// File containing HTTP-related functions

#include "http.h"
#include "sockets.h"
#include "network.h"
#include "files.h"
#include "clients.h"

BOOL HttpStartGettingFile(FileInfo *file)
{
    char range[256], get_string[256], buffer[4096];

    if (file->CurrentSize > 0)
    {
        if (file->TotalSize == SIZE_UNKNOWN)
        {
            sprintf(range, "Range: bytes=%d-\r\n", file->CurrentSize);
        }
        else
        {
            sprintf(range, "Range: bytes=%d-%d\r\n", file->CurrentSize,
                    file->TotalSize);
        }
    }
    else
    {
        strcpy(range, "");
    }
    // if it's a proxy we need to send the full url, otherwise we just send
    // the path (otherwise redirecting screws up)
    if (strcmp(DX_ProxyHostName, "") && (DX_ProxyPort > 0))
    {
        sprintf(get_string, "GET %s://%s%s HTTP/1.0", file->Protocol,
                file->ActualServer->Name, file->Path);
    }
    else
    {
        sprintf(get_string, "GET %s HTTP/1.0", file->Path);
    }
    sprintf(buffer,
            "%s\r\n"
            "Host: %s\r\n"
            "User-Agent: Darxite/%s\r\n"
            "%s\r\n",
            get_string, file->ActualServer->Name, RELEASE_VER, range);
    
    error(E_TRACE, "Sending HTTP command \"%s\"", buffer);
    if (write(file->ControlSocket->FileDes, buffer, strlen(buffer)) == -1)
    {
        error(E_WARN, "Couldn't write to socket: %s", strerror(errno));
        CancelStartGettingFile(file);
        return FALSE;
    }
    strcpy(file->Activity, "Transferring file");
    file->DataSocket->DataBuf = (char *)dxmalloc(DX_BufferSize + 1);
    file->DataSocket->DataLength = 0;
    file->DataSocket->Transferring = TRUE;
    file->DataSocket->Eof = FALSE;
    file->TimeOfLastRxTx = time(NULL);
    file->StartTime = time(NULL);
    file->GotHeader = FALSE;
    file->RxTxOverall = 0;
    file->Started = TRUE;
    file->Starting = FALSE;
    return TRUE;
}

// Parses an HTTP header.
void HttpParseHeader(FileInfo *file)
{
    char buffer[256], location[256], file_buf[256], dummy[256];
    BOOL got_length = FALSE, got_range = FALSE, redirect = FALSE;
    struct _clientInfo *client;
    FileInfo *new_file;
    FILE *local_file;
    int http_code, line_length, total_size;
    
    while (buf_hasline(file->DataSocket->DataBuf, DX_BufferSize))
    {
        line_length = buf_getline(file->DataSocket->DataBuf, DX_BufferSize,
                                  buffer, 255);
        // the buffer now has less stuff in it because buf_getline will move
        // the contents along
        file->DataSocket->DataLength -= line_length;
        
        error(E_TRACE, " header: %s", buffer);
        // if it's the end of the header
        if (!strcmp(buffer, ""))
        {
            file->GotHeader = TRUE;
            break;
        }
        // check for errors etc.
        sscanf(buffer, "HTTP/%*f %d", &http_code);
        switch (http_code)
        {
        case 301:
        case 302:
            error(E_TRACE, "File has been moved");
            redirect = TRUE;
            break;
            
        case 400:
            error(E_TRACE, "Bad request to server");
            break;
            
        case 401:
        case 403:
            error(E_TRACE, "Whoops! We're not allowed to do that");
            break;
            
        case 404:
            error(E_TRACE, "File does not exist");
            break;
            
        default:
            break;
        }
        http_code = 0;
        // do we have to cancel and start again elsewhere?
        if (redirect)
        {
            if (sscanf(buffer, "Location: %s", location))
            {
                error(E_TRACE, "New location: %s", location);
                CancelTransfer(file->ControlSocket, file->Protocol);
                // notify all clients that asked to be
                sprintf(file_buf, "%d \"%s://%s%s\" | \"%s\"",
                        DX_REDIRECT_EVENT, file->Protocol,
                        file->ActualServer->Name, file->Path, location);
                SendEvent(REDIRECT_EVENT, file_buf);
                // add a new file to the batch
                sprintf(file_buf, "\"%s\" | \"%s\" | %s | %s | %s", location,
                        file->LocalPath, file->LogIn, file->Password,
                        file->Flags);
                client = file->Client;
                DisconnectFile(file, FALSE);
                FileComplete(file, "File moved");
                new_file = AddFileToBatch(TRUE, file_buf, NULL);
                // if a client wanted to be notified of the old file, make
                // it be notified of the new one
                if (new_file)
                    new_file->Client = client;
                siglongjmp(MainJump, 1);
                // JUMP to main loop
            }
        }
        // did we resume successfully?
        if (file->CurrentSize > 0)
        {
            if (sscanf(buffer, "Content-%*cange: %s", dummy))
            {
                error(E_TRACE, "Read range OK");
                got_range = TRUE;
            }
        }
        // if the file's total size hasn't been specified in the batch, then
        // try and find it out now
        if (sscanf(buffer, "Content-%*cength: %d", &total_size))
        {
            got_length = TRUE;
        }
    }
    // if we resumed successfully
    if (file->CurrentSize > 0)
    {
        if (got_range)
        {
            error(E_TRACE, "Server supports file resume");
            file->ActualServer->SupportsFileResume = TRUE;
        }
        else
        {
            file->ActualServer->SupportsFileResume = FALSE;
            error(E_TRACE, "Server does NOT support file resume");
            file->CurrentSize = 0;
            if ((DX_EnableRenaming && !strchr(file->Flags, 'R')) ||
                strchr(file->Flags, 'r'))
            {
                sprintf(buffer, "%s.darxite", file->SpooledPath);
            }
            else
            {
                strcpy(buffer, file->SpooledPath);
            }
            local_file = fopen(buffer, "w");
            fclose(local_file);
        }
    }
    // we need to check the length after the range, because we reset the
    // current size to 0 if the server doesn't support file resume
    if (got_length)
    {
        if (file->TotalSize == SIZE_UNKNOWN)
        {
            if (file->CurrentSize > 0)
            {
                total_size += file->CurrentSize;
                // FIXME: when we ask to resume beyond the end of a file,
                // we get the whole thing. This works OK, but total size
                // is double what it should be, because current size has
                // not yet been set back to 0.
                error(E_TRACE, "Current size is %d so total is %d",
                      file->CurrentSize, total_size);
            }
            sprintf(buffer, "\"%s://%s%s\" | \"%s\" | %s | %s | %s | %d",
                    file->Protocol, file->Server->Name, file->Path,
                    file->LocalPath, file->LogIn, file->Password,
                    file->Flags, total_size);
            RewriteFileInfo(file, buffer);
            file->TotalSize = total_size;
        }
        // massive download stuff
        error(E_TRACE,
              "HTTP total size: %d limit: %d maxconns: %d, ratio: %d "
              "current: %d connectagain: %d",
              file->TotalSize, DX_MassiveLimit, DX_MassiveMaxConnections,
              (DX_MassiveMaxConnections > 0) ?
              (file->TotalSize / DX_MassiveMaxConnections) : -1,
              file->CurrentSize, DX_MassiveConnectAgainLimit);
        if ((DX_EnableMassiveDownload && !strchr(file->Flags, 'v') &&
             !strchr(file->Flags, 'E') &&
             (file->TotalSize > (DX_MassiveLimit * 1024))) ||
            strchr(file->Flags, 'e'))
        {
            StartMassiveDownload(file);
        }
    }
}

// goes through an HTML file, recursing the links therein
void HttpFindLinks(FileInfo *file, BOOL all_links)
{
    FILE *fp;
    char c, *end, path[256], link[256], buffer[1024];
    int count;

    error(E_TRACE, "Checking \"%s\" for links...", file->LocalPath);
    fp = fopen(file->LocalPath, "r");
    if (fp == NULL)
    {
        error(E_TRACE, "Couldn't open \"%s\": %s", file->LocalPath,
              strerror(errno));
        return;
    }
    // get the path of the file (minus the file name)
    memset(path, 0, sizeof(path));
    if (lastchr(file->Path) != '/')
    {
        strcpy(buffer, file->Path);
        end = strrchr(buffer, '/');
        strncpy(path, buffer, end - buffer + 1);
    }
    else
    {
        strcpy(path, file->Path);
    }
    error(E_TRACE, "path: %s", path);
    while ((c = fgetc(fp)) != EOF)
    {
        if (c != '<')
            continue;
        count = 0;
        memset(buffer, 0, sizeof(buffer));
        c = fgetc(fp);
        while (c != EOF && c != '>')
        {
            buffer[count] = c;
            c = fgetc(fp);
            count++;
        }
        
	/* FIXME: this isn't quite good enough: conceivably someone might
	 * do ie "<a target="_top" href="/foo.html">", or even
	 * "<a  href="foo">", but I don't feel like writing a parser for
	 * that right now
	 *  - glenn */
	if (strncasecmp(buffer, "a href=\"", 8)) continue;
	if(lastchr(buffer) != '"') continue;
	
	/* got a link */
	lastchr(buffer) = '\0';
	strcpy(link, buffer + 8);
	// don't follow mailtos, links to same page or links to parent
	// directories (stops infinite loops)
	// this won't stop two pages from linking to each other, or
	// absolute links to parent dirs, will it?
	if (!strncasecmp(link, "mailto:", 7) || strchr(link, '#') ||
	    strstr(link, "..")) continue;
	    
	error(E_TRACE, "Link: \"%s\"", link);
	if (!strstr(link, "://"))
	{
	    // FIXME: local paths
	    if (*link == '/')
	    {
		sprintf(buffer, "http://%s%s | | %s | %s | %s",
			file->ActualServer->Name, link,
			file->LogIn, file->Password, file->Flags);
	    }
	    else
	    {
		sprintf(buffer, "http://%s%s%s | | %s | %s | %s",
			file->ActualServer->Name, path, link,
			file->LogIn, file->Password, file->Flags);
	    }
	    AddFileToBatch(TRUE, buffer, NULL);
	}
	else
	{
	    // only download absolute paths if we're in link mode
	    if (strchr(file->Flags, 'l'))
	    {
		sprintf(buffer, "%s | | %s | %s | %s",
			link, file->LogIn, file->Password,
			file->Flags);
		AddFileToBatch(TRUE, buffer, NULL);
	    }
	}
    }
    fclose(fp);
}
