/*
 **************************************************************************
 *
 * Boot-ROM-Code to load an operating system across a TCP/IP network.
 *
 * Module:  arp.c
 * Purpose: ARP handling
 * Entries: init_arp, ip_resolve, addcache, set_gateway
 *
 **************************************************************************
 *
 * Copyright (C) 1995-1998 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <general.h>
#include <net.h>
#include <romlib.h>
#include "./netpriv.h"
#include "./arp.h"



/*
 **************************************************************************
 * 
 * Global variables
 */
#ifndef NOARP
static struct arphdr *arpbuf;			/* ARP send buffer	*/
#endif
static struct arp_cache arp_list[ARPNUM];	/* ARP address cache	*/
static int next_arp;				/* next table entry	*/
static t_ipaddr def_gw = IP_ANY;		/* default routing	*/



/*
 **************************************************************************
 * 
 * Find an IP address in the ARP cache. Return -1 if not found.
 */
static int findip(ip)
t_ipaddr ip;
{
  int i;

  for (i = 0; i < ARPNUM; i++)
	if (arp_list[i].ipaddr == ip)
		return(i);
  return(-1);
}



/*
 **************************************************************************
 * 
 * Add a new entry to the ARP cache.
 */
void addcache(ha, ip)
unsigned char *ha;
t_ipaddr       ip;
{
  int i;

  /*
   * If the IP address already exists in the ARP table then just replace
   * it's hardware address with the new onbe. Otherwise create a new ARP
   * table entry.
   */
  if ((i = findip(ip)) < 0) {
	i = next_arp++;
	if (next_arp >= ARPNUM)
		next_arp = 0;
	arp_list[i].ipaddr = ip;
  }
  memcpy(arp_list[i].hwaddr, ha, ETH_ALEN);
}



#ifndef NOARP
/*
 **************************************************************************
 * 
 * ARP receiver routine
 */
static int arp_recv(buf, bufsize, addr)
unsigned char *buf;
int            bufsize;
unsigned char *addr;
{
  register struct arphdr *ahp = (struct arphdr *)buf;

  /*
   * Do some checks on the received packet. It must be an ARP packet,
   * with an ethernet hardware address, and for the IP protocol. Other
   * packets just get discarded. Then save the sender's address.
   */
  if (bufsize < sizeof(struct arphdr) ||
      ahp->ar_hrd != htons(ARPHRD_ETHER) || ahp->ar_hln != ETH_ALEN ||
      ahp->ar_pro != htons(ETH_P_IP) || ahp->ar_pln != IP_ALEN)
	return(FALSE);

  addcache(ahp->ar_sha, _ntohl(ahp->ar_sip));

  /* If the packet is a request, send our own hardware address back */
  if (ahp->ar_op == htons(ARPOP_REQUEST) &&
      ahp->ar_tip == htonl(myipaddr)) {
		arpbuf->ar_hrd = htons(ARPHRD_ETHER);
		arpbuf->ar_pro = htons(ETH_P_IP);
		arpbuf->ar_hln = ETH_ALEN;
		arpbuf->ar_pln = IP_ALEN;
		arpbuf->ar_op  = htons(ARPOP_REPLY);
		arpbuf->ar_sip = htonl(myipaddr);
		arpbuf->ar_tip = ahp->ar_sip;
		memcpy(arpbuf->ar_sha, myhwaddr, ETH_ALEN);
		memcpy(arpbuf->ar_tha, ahp->ar_sha, ETH_ALEN);
		write_packet(sizeof(struct arphdr), htons(ETH_P_ARP), addr);
  }
  return(FALSE);
}
#endif



/*
 **************************************************************************
 * 
 * Resolve IP address and return pointer to hardware address.
 */
unsigned char *ip_resolve(ip)
t_ipaddr ip;
{
  int i;

#ifdef DEBUG
  printf("ARP: Resolving IP address %lx\n", ip);
#endif

  /* Broadcast addresses are always known */
  if (ip == IP_BROADCAST)
	return(bcasthw);

  /* Check for invalid IP addresses */
  if (ip == IP_ANY || ip == myipaddr || ip == IP_LOCALHOST)
	return(NULL);

  /* If the destination is not on subnet, send the packets through gateway */
  if ((ip ^ myipaddr) & mynetmask)
	if (def_gw != IP_ANY)
		ip = def_gw;
	else
		return(NULL);

  /* Check if we got the address in the cache already */
  if ((i = findip(ip)) >= 0)
	return(arp_list[i].hwaddr);

#ifndef NOARP
  /* No, we have to broadcast a request */
  arpbuf->ar_hrd = htons(ARPHRD_ETHER);
  arpbuf->ar_pro = htons(ETH_P_IP);
  arpbuf->ar_hln = ETH_ALEN;
  arpbuf->ar_pln = IP_ALEN;
  arpbuf->ar_op  = htons(ARPOP_REQUEST);
  arpbuf->ar_sip = htonl(myipaddr);
  arpbuf->ar_tip = htonl(ip);
  memcpy(arpbuf->ar_sha, myhwaddr, ETH_ALEN);
  memset(arpbuf->ar_tha, 0, ETH_ALEN);
  write_packet(sizeof(struct arphdr), htons(ETH_P_ARP), bcasthw);

#ifdef DEBUG
  printf("ARP: Request sent\n");
#endif

  /* Wait for someone to answer the broadcast request */
  set_timeout(ARP_TIMEOUT);
  while (!chk_timeout() && (i = findip(ip)) < 0) ;
  if (i >= 0)
	return(arp_list[i].hwaddr);

#ifdef DEBUG
  printf("ARP: Timed out\n");
#endif

#endif

  return(NULL);
}



/*
 **************************************************************************
 * 
 * Set the default gateway.
 */
void set_gateway(gw)
t_ipaddr gw;
{
  /* Just set the default gateway */
  if (gw != IP_BROADCAST && gw != myipaddr &&
      ((mynetmask == IP_POINTOPOINT) || !((gw ^ myipaddr) & mynetmask)))
	def_gw = gw;
}



/*
 **************************************************************************
 * 
 * Network intialization:
 */
int init_arp()
{
  /* Set name of module for error messages */
  net_module_name = "arp";

#ifndef NOARP
  /* Register ARP packet type and set send buffer pointer */
  if ((arpbuf = (struct arphdr *)reg_type(htons(ETH_P_ARP), arp_recv)) == NULL)
	return(FALSE);
#endif

  return(TRUE);
}

