/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * Authors: Matt Eagleson <e.Messiah@superpimp.org>
 *          Charles Kerr <charles@skywalker.ecn.ou.edu>
 *          Jason Leach <leach@wam.umd.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

#include <config.h>
#include <gnome.h>

#include "article-find.h"
#include "article-thread.h"
#include "article-toolbar.h"
#include "articlelist.h"
#include "globals.h"
#include "gui.h"
#include "gui-notebook.h"
#include "gui-paned.h"
#include "grouplist.h"
#include "log.h"
#include "message-filter.h"
#include "message-window.h"
#include "pan.h"
#include "prefs.h"
#include "print.h"
#include "status-item.h"
#include "status-item-view.h"
#include "save.h"
#include "text.h"
#include "util.h"

GtkTooltips *ttips = NULL;

/**
*** private function prototypes
**/

static void exit_cb (void);
static void gui_key_press_cb (GtkWidget*, GdkEventKey*, gpointer);
static int window_delete_event_cb (GtkWidget*, GdkEvent*, gpointer);
static void taskbar_validate (void);

static GnomeUIInfo file_menu[] = {
	GNOMEUIINFO_MENU_SAVE_AS_ITEM (save_current_article, NULL),
	GNOMEUIINFO_MENU_PRINT_ITEM (print_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_EXIT_ITEM(exit_cb, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo edit_menu[] = {
	//GNOMEUIINFO_MENU_CUT_ITEM (not_implemented, NULL),
	//GNOMEUIINFO_MENU_COPY_ITEM (not_implemented, NULL),
	//GNOMEUIINFO_MENU_PASTE_ITEM (not_implemented, NULL),
	GNOMEUIINFO_MENU_SELECT_ALL_ITEM (gui_select_all, NULL),
	//GNOMEUIINFO_MENU_CLEAR_ITEM (not_implemented, NULL),
	GNOMEUIINFO_SEPARATOR,
	{ GNOME_APP_UI_ITEM, N_("_Find"), N_("Find text in displayed header fields"), articlelist_find_text_cb,
	  NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_SEARCH, 'F', GDK_CONTROL_MASK, NULL},
	{ GNOME_APP_UI_ITEM, N_("Find _Next"), N_("Repeat the last search."), articlelist_find_next_cb,
	  NULL, NULL, GNOME_APP_PIXMAP_NONE, NULL, 'F', GDK_CONTROL_MASK|GDK_SHIFT_MASK, NULL},
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_PREFERENCES_ITEM(prefs_spawn, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo online_menu[] = {
	//FIXME{ GNOME_APP_UI_ITEM, N_("Go _Online"), N_("Go into online mode."), nntp_want_connect, 0 },
	//FIXME{ GNOME_APP_UI_ITEM, N_("Go O_ffline"), N_("Go into offline mode."), nntp_want_disconnect },
	//GNOMEUIINFO_SEPARATOR,
	//FIXME { GNOME_APP_UI_ITEM, N_("_Stop all Online Tasks"), N_("Stop all online tasks."), stop_all_mf,
	//  NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_STOP, 'S', GDK_CONTROL_MASK, NULL},
	//GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Get New for Subscribed Groups"),
		N_("Download all new headers from all subscribed groups."),
		grouplist_all_subscribed_download_new
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Get List of _All Groups"),
		N_("Refresh the listing of groups from the selected servers."),
		grouplist_get_all,
		NULL, NULL, GNOME_APP_PIXMAP_STOCK,
		GNOME_STOCK_PIXMAP_REFRESH, 0, 0, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Get List of _New Groups"),
		N_("Download a listing of new groups from the selected servers."),
		grouplist_get_new,
		NULL, NULL,
		GNOME_APP_PIXMAP_STOCK,
		GNOME_STOCK_PIXMAP_REFRESH, 0, 0, NULL
	},
	GNOMEUIINFO_END
};

GnomeUIInfo help_menu[] = {
	GNOMEUIINFO_ITEM_STOCK (N_("_About..."), N_("Display program information, version number, and copyright."),
				gui_about, GNOME_STOCK_MENU_ABOUT),
	GNOMEUIINFO_END
};


static gboolean dampen_feedback_sort = FALSE;

static void
sort_cb (GtkRadioMenuItem* item, gpointer data)
{
	if (!dampen_feedback_sort && GTK_CHECK_MENU_ITEM(item)->active)
		articlelist_set_sort_type (GPOINTER_TO_INT(data));
}
static void
show_state_filter_in_toolbar (GtkCheckMenuItem* item, gpointer data)
{
	article_toolbar_show_state_filter (item->active);
}
static void
thread_articles_cb (GtkCheckMenuItem* item, gpointer data)
{
	articlelist_set_threaded (item->active);
}
static void
rot_13_cb (GtkCheckMenuItem* item, gpointer data)
{
	rot13_enabled = item->active;
	text_refresh ();
}
static void
show_all_headers_cb (GtkCheckMenuItem* item, gpointer data)
{
	text_set_show_all_headers_in_body (item->active);
}


static gboolean dampen_feedback_layout = FALSE;

static void
layout_cb (GtkCheckMenuItem* item, gpointer data)
{
	if (!dampen_feedback_layout && GTK_CHECK_MENU_ITEM(item)->active)
		gui_set_layout (GPOINTER_TO_INT(data));
}

static GnomeUIInfo layout_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Paned Layout"),
		N_("Paned Layout"),
		layout_cb,
		GINT_TO_POINTER(GUI_PANED)
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Notebook Layout"),
		N_("Notebook Layout"),
		layout_cb,
		GINT_TO_POINTER(GUI_NOTEBOOK)
	},
	GNOMEUIINFO_END
};
static GnomeUIInfo view_menu[] =
{
	{
		GNOME_APP_UI_RADIOITEMS,
		N_("Layout"),
		N_("Layout"),
		(gpointer)&layout_menu
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("_Log Viewer"),
		N_("Open up the Log Viewer."),
		log_viewer_spawn,
		NULL, NULL, GNOME_APP_PIXMAP_STOCK,
		GNOME_STOCK_PIXMAP_PREFERENCES,
		'L', GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Filter in Article Toolbar"),
		N_("Show Filter in Article Toolbar"),
		show_state_filter_in_toolbar
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo groupview_menu[] = {
	{
		GNOME_APP_UI_ITEM,
		N_("All Groups"),
		N_("Show all available groups in the group list."),
		grouplist_set_view_mode,
		GINT_TO_POINTER (GROUP_ALL)
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Subscribed Groups"),
		N_("Show only the subscribed groups in the group list."),
		grouplist_set_view_mode,
		GINT_TO_POINTER (GROUP_SUBSCRIBED)
	},
 	{
		GNOME_APP_UI_ITEM,
		N_("New Groups"),
		N_("Show only the new groups in the group list."),
		grouplist_set_view_mode, GINT_TO_POINTER (GROUP_NEW)
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo group_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Subscribe to Selected"),
		N_("Subscribe to the selected group(s)"),
		grouplist_selected_subscribe
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Unsubscribe from Selected"),
		N_("Unsubscribe from the selected group(s)"),
		grouplist_selected_unsubscribe
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Next Group"),
		N_("Move to the Next Group."),
		grouplist_next_group,
		NULL, NULL,
		GNOME_APP_PIXMAP_NONE, NULL,
		'G', GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Next Unread Group"),
		N_("Move to the Next Group with Unread Messages."),
		grouplist_next_unread_group,
		NULL, NULL,
		GNOME_APP_PIXMAP_NONE, NULL,
		'G', GDK_CONTROL_MASK|GDK_SHIFT_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Previous Group"),
		N_("Move to the Previous Group."),
		grouplist_prev_group,
		NULL, NULL,
		GNOME_APP_PIXMAP_NONE, NULL,
		'B', GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Previous Unread Group"),
		N_("Move to the Previous Group with Unread Messages."),
		grouplist_prev_unread_group,
		NULL, NULL,
		GNOME_APP_PIXMAP_NONE, NULL,
		'B', GDK_SHIFT_MASK|GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Get All Article Headers"),
		N_("Get all article headers"),
		grouplist_selected_download_all
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Get New Article Headers"),
		N_("Get new article headers"),
		grouplist_selected_download_new
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Empty Group"),
		N_("Remove all articles from the selected group(s)."),
		grouplist_selected_empty
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Remove `New' Status"),
		N_("Remove selected group(s) from `new groups' list."),
		grouplist_selected_unnew
	},
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_SUBTREE (N_("_Show Groups"), &groupview_menu),
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("_Properties"),
		N_("Set the properties for the selected group."),
		grouplist_selected_properties
	},
	GNOMEUIINFO_END
};


static void
article_filter_cb (GtkCheckMenuItem* item, gpointer data)
{
	const guint flag = GPOINTER_TO_UINT(data);
	articlelist_poke_state_filter (flag, item->active);
}

static GnomeUIInfo article_sort_menu_2[] =
{
	{ GNOME_APP_UI_ITEM, N_("+ Author"), N_("+ Author"), sort_cb, GINT_TO_POINTER(+ARTICLE_SORT_AUTHOR) },
	{ GNOME_APP_UI_ITEM, N_("- Author"), N_("- Author"), sort_cb, GINT_TO_POINTER(-ARTICLE_SORT_AUTHOR) },
	{ GNOME_APP_UI_ITEM, N_("+ Date"), N_("+ Date"), sort_cb, GINT_TO_POINTER(+ARTICLE_SORT_DATE) },
	{ GNOME_APP_UI_ITEM, N_("- Date"), N_("- Date"), sort_cb, GINT_TO_POINTER(-ARTICLE_SORT_DATE) },
	{ GNOME_APP_UI_ITEM, N_("+ Line Count"), N_("+ Line Count"), sort_cb, GINT_TO_POINTER(+ARTICLE_SORT_LINES) },
	{ GNOME_APP_UI_ITEM, N_("- Line Count"), N_("- Line Count"), sort_cb, GINT_TO_POINTER(-ARTICLE_SORT_LINES) },
	{ GNOME_APP_UI_ITEM, N_("+ Subject"), N_("+ Subject"), sort_cb, GINT_TO_POINTER(+ARTICLE_SORT_SUBJECT) },
	{ GNOME_APP_UI_ITEM, N_("- Subject"), N_("- Subject"), sort_cb, GINT_TO_POINTER(-ARTICLE_SORT_SUBJECT) },
	{ GNOME_APP_UI_ITEM, N_("+ Unread Count"), N_("+ Unread Count"), sort_cb, GINT_TO_POINTER(+ARTICLE_SORT_UNREAD_CHILDREN) },
	{ GNOME_APP_UI_ITEM, N_("- Unread Count"), N_("- Unread Count"), sort_cb, GINT_TO_POINTER(-ARTICLE_SORT_UNREAD_CHILDREN) },
	GNOMEUIINFO_END
};
static GnomeUIInfo article_sort_menu[] =
{
	{
		GNOME_APP_UI_RADIOITEMS,
		N_("Sort"),
		N_("Sort"),
		article_sort_menu_2
	},
	GNOMEUIINFO_END
};
static GnomeUIInfo article_filter_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("Killfile"),
		N_("Thread Tracking and Killfile Management."),
		message_filter_ui
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Unread Messages"),
		N_("Show Unread Messages"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_UNREAD)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Read Messages"),
		N_("Show Read Messages"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_READ)
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Watched Threads"),
		N_("Show Watched Threads"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_WATCHED)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Killfiled Messages"),
		N_("Show Killfiled Messages"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_KILLFILE)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Others"),
		N_("Show Others"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_NORMAL_RANK)
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Complete Binary Articles"),
		N_("Show Complete Binary Articles"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_COMPLETE_BINARIES)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Incomplete Binary Articles"),
		N_("Show Incomplete Binary Articles"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_INCOMPLETE_BINARIES)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Non-Binary Articles"),
		N_("Show Non-Binary Articles"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_NONBINARIES)
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Saved Articles"),
		N_("Show Saved Articles"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_SAVED)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Queued Articles"),
		N_("Show Queued Articles"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_QUEUED)
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show Others"),
		N_("Show Others"),
		article_filter_cb,
		GUINT_TO_POINTER(STATE_FILTER_IDLE)
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo article_menu[] =
{
	{
		GNOME_APP_UI_ITEM,
		N_("_Post to newsgroup"),
		N_("Post a new message to the current group."),
		message_post_window
	},
	{
		GNOME_APP_UI_ITEM,
		N_("_Followup to newsgroup"),
		N_("Post a reply to the message on the news server."),
		message_followup_window
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Reply by _E-mail"),
		N_("Create a mail reply to the sender."),
		message_reply_window
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Reply with External Mailer"),
		N_("Create a mail reply to the sender with your favorite MUA."),
		message_reply_external
	},
		GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_SUBTREE,
		N_("Filter"),
		N_("Article Filter"),
		&article_filter_menu
	},
	{
		GNOME_APP_UI_SUBTREE,
		N_("Sort"),
		N_("Sort"),
		&article_sort_menu
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Thread Articles"),
		N_("Thread Articles"),
		thread_articles_cb
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Rot13"),
		N_("Rot13"),
		rot_13_cb
	},
	{
		GNOME_APP_UI_TOGGLEITEM,
		N_("Show All Headers in Body"),
		N_("Show All Headers in Body"),
		show_all_headers_cb
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("_Save Attachment"),
		N_("Download and decode attachments in the selected messages."),
		articlelist_selected_decode,
		NULL, NULL, GNOME_APP_PIXMAP_NONE,
		NULL, 'D', GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Save Attachment As..."),
		N_("Download and attachments in the selected message, and save as..."),
		articlelist_selected_decode_as
	},
	{
		GNOME_APP_UI_ITEM,
		N_("_Open Attachment"),
		N_("Download, decode, and open attachments in the selected messages."),
		articlelist_selected_open,
		NULL, NULL, GNOME_APP_PIXMAP_NONE,
		NULL, 'O', GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Next Article"),
		N_("Read the Next Article."),
		articlelist_read_next_article,
		NULL, NULL, GNOME_APP_PIXMAP_NONE,
		NULL, 'N', GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Next Unread Article"),
		N_("Read the Next Unread Article."),
		articlelist_read_next_unread_article,
		NULL, NULL, GNOME_APP_PIXMAP_NONE,
		NULL, 'N', GDK_SHIFT_MASK|GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Next Thread"),
		N_("Read the Next Thread."),
		articlelist_read_next_thread,
		NULL, NULL, GNOME_APP_PIXMAP_NONE,
		NULL, 'N', GDK_SHIFT_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Previous Article"),
		N_("Read the Previous Article."),
		articlelist_read_prev_article,
		NULL, NULL, GNOME_APP_PIXMAP_NONE,
		NULL, 'P', GDK_CONTROL_MASK, NULL
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Previous Unread Article"),
		N_("Read the Previous Unread Article."),
		articlelist_read_prev_unread_article,
		NULL, NULL, GNOME_APP_PIXMAP_NONE,
		NULL, 'P', GDK_SHIFT_MASK|GDK_CONTROL_MASK, NULL
	},
	{
	       	GNOME_APP_UI_ITEM,
	       	N_("Previous Thread"),
	       	N_("Read the Previous Thread."),
	       	articlelist_read_prev_thread,
		NULL, NULL, GNOME_APP_PIXMAP_NONE,
		NULL, 'P', GDK_SHIFT_MASK, NULL
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Download Entire Thread"),
		N_("Download Entire Thread."),
		articlelist_selected_thread_download
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("D_elete Message"),
		N_("Deletes the selected message from the database."),
		articlelist_selected_delete
	},
	{
		GNOME_APP_UI_ITEM,
		N_("Delete Thread"),
		N_("Deletes the entire thread from the database."),
		articlelist_selected_delete_thread
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Mark All _Read"),
		N_("Mark all articles read"),
		articlelist_all_mark_read
	},
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Select _All"),
		N_("Select all messages"),
		articlelist_select_all,
		NULL, NULL, GNOME_APP_PIXMAP_NONE,
		NULL, 'A', GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_END
};

static GnomeUIInfo pan_main_menu[] = {
	GNOMEUIINFO_MENU_FILE_TREE (file_menu),
	GNOMEUIINFO_MENU_EDIT_TREE (edit_menu),
	//GNOMEUIINFO_MENU_SETTINGS_TREE (settings_menu),
	GNOMEUIINFO_SUBTREE (N_("_View"), &view_menu),
	GNOMEUIINFO_SUBTREE (N_("_Online"), &online_menu),
	GNOMEUIINFO_SUBTREE (N_("_Group"), &group_menu),
	GNOMEUIINFO_SUBTREE (N_("_Message"), &article_menu),
	GNOMEUIINFO_MENU_HELP_TREE (help_menu),
	GNOMEUIINFO_END
};

#if 0
static GnomeUIInfo pan_main_toolbar[] = {
//	{ GNOME_APP_UI_ITEM, N_("New Mail"), N_("Create a new message."), message_post_window,
//	  NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_MAIL, 'N', GDK_CONTROL_MASK, NULL},
//	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_ITEM_STOCK (N_("Previous"), N_("Read the previous message"),
				message_previous, GNOME_STOCK_PIXMAP_UP),
	GNOMEUIINFO_ITEM_STOCK (N_("Next"), N_("Read the next message"),
				message_next, GNOME_STOCK_PIXMAP_DOWN),
	GNOMEUIINFO_SEPARATOR,
	{
		GNOME_APP_UI_ITEM,
		N_("Stop All"),
		N_("Stop all online tasks."),
		stop_all_mf,
		NULL, NULL, GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_STOP,
		'S', GDK_CONTROL_MASK, NULL
	},
	GNOMEUIINFO_END
};
#endif



/** 
 ** global variables
 **/

static GtkWidget *header_table = NULL;
static GtkWidget *taskbar = NULL;
static GtkWidget *appbar = NULL;
static GtkWidget *queue_qty_label = NULL;

static GHashTable *item_to_view_hashtable = NULL;

guint context_id;

GtkWidget *groups_vbox;
GtkWidget *articlelist_ctree;
GtkWidget *text_box;
GtkWidget *contents_vbox;

/*---[ gui_select_all ]-----------------------------------------------
 * Intelligently handle the Edit->Select All callback so the
 * active pane or page gets all it's items selected, in either layout.
 *--------------------------------------------------------------------*/
void
gui_select_all (void)
{
	if (Pan.viewmode == GUI_PANED)
		gui_paned_select_all ();
	else if (Pan.viewmode == GUI_NOTEBOOK)
		gui_notebook_select_all ();
}


/*---[ gui_about ]----------------------------------------------------
 * Display an informative about window popup
 *--------------------------------------------------------------------*/
void
gui_about (void)
{
	static GtkWidget *dialog = NULL;

	if (dialog != NULL)
	{
		g_assert (GTK_WIDGET_REALIZED (dialog));
		gdk_window_show (dialog->window);
		gdk_window_raise (dialog->window);
	}

	else
	{
		const gchar *authors[] = {
			"Matt Eagleson <e.Messiah@superpimp.org>",
			"Charles Kerr <charles@superpimp.org>",
			"Jason Leach <elerium@superpimp.org>",
			NULL
		};

		pan_lock();

		dialog = gnome_about_new ("Pan", VERSION,
					  _("Copyright (C) 1999-2000 the Pan Development Group"),
					  authors,
					  "The Pimp Ass Newsreader (PAN), a newsreader for GNOME.",
					  "xpm/pan_druid.xpm");
		gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
				    GTK_SIGNAL_FUNC (gtk_widget_destroyed),
				    &dialog);
		if (Pan.window)
			gnome_dialog_set_parent (GNOME_DIALOG (dialog),
						 GTK_WINDOW (Pan.window));

		gtk_widget_show (dialog);

		pan_unlock();
	}
}

static void
gui_create_appbar (GtkWidget* window)
{
	GtkWidget* frame = NULL;

	pan_lock();
	appbar = gtk_hbox_new (FALSE, 0);
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_container_set_border_width (GTK_CONTAINER(frame), 2);
	queue_qty_label = gtk_label_new ("");
	pan_unlock ();

	gui_set_queue_size (0);

	pan_lock ();
	gtk_container_add (GTK_CONTAINER(frame), queue_qty_label);
	gtk_box_pack_start (GTK_BOX(appbar), frame, FALSE, FALSE, 0);
	frame = gtk_frame_new (NULL);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_container_set_border_width (GTK_CONTAINER(frame), 3);
	taskbar = gtk_fixed_new ();
	gtk_widget_set_usize (GTK_WIDGET(taskbar), -1, 21); /* FIXME: hardcoded size */
#if 0
	gtk_signal_connect_after (
		GTK_OBJECT(taskbar), "size_allocate", size_allocated, NULL);
#endif

	gtk_container_add (GTK_CONTAINER(frame), taskbar);
	gtk_box_pack_start (GTK_BOX(appbar), frame, TRUE, TRUE, 0);
	gnome_app_set_statusbar (GNOME_APP(window), appbar);
	pan_unlock();
}

/*---[ gui_shutdown ]-------------------------------------------------
 * save geometry states into config files
 *--------------------------------------------------------------------*/

static void
exit_cb (void)
{
	gui_shutdown ();
	gtk_object_destroy ( GTK_OBJECT(Pan.window) );
}

static int
window_delete_event_cb (GtkWidget *widget,
			GdkEvent *event,
			gpointer  data)
{
	gui_shutdown ();
	return FALSE; /* causes "destroy" signal to be fired */
}


void
gui_shutdown (void)
{
	gint x = 0, y = 0;
	gint width = 640, height = 400;

	gdk_window_get_position (Pan.window->window, &x, &y);
	gdk_window_get_size (Pan.window->window, &width, &height);

	gnome_config_set_bool (
		"/Pan/Display/Show_Filter_In_Article_Toolbar",
		GTK_CHECK_MENU_ITEM(view_menu[4].widget)->active);
	gnome_config_set_bool (
		"/Pan/Display/Show_All_Headers_In_Body",
		GTK_CHECK_MENU_ITEM(article_menu[9].widget)->active);

	gnome_config_set_int ("/Pan/Geometry/pan_x", x);
	gnome_config_set_int ("/Pan/Geometry/pan_y", y);
	gnome_config_set_int ("/Pan/Geometry/width", width);
	gnome_config_set_int ("/Pan/Geometry/height", height);
	gnome_config_set_int ("/Pan/State/viewmode", Pan.viewmode);
	gnome_config_set_int ("/Pan/State/group_mode", grouplist_get_view_mode());

	if (Pan.viewmode == GUI_NOTEBOOK) {
		gnome_config_set_int ("/Pan/State/page", gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook)));
	}
	else if (Pan.viewmode == GUI_PANED) {
		gnome_config_set_int ("/Pan/Geometry/vpaned", articlelist_ctree->allocation.height);
		gnome_config_set_int ("/Pan/Geometry/hpaned", groups_vbox->allocation.width);
	}

	gui_save_column_widths (Pan.group_clist, "group");
	gui_save_column_widths (Pan.article_ctree, "articlelist");

	gnome_config_sync();
}

static int
gui_articlelist_group_changed_cb (gpointer call_object,
				  gpointer call_arg,
				  gpointer user_data)
{
	const group_data* gdata = articlelist_get_current_group();
	const gboolean have_group = gdata!=NULL;
	const gboolean have_article = articlelist_get_selected_adata()!=NULL;
	int i = -1;

	/* update menus */

	++i; /* post */
	++i; /* i is followup */
	gtk_widget_set_sensitive (article_menu[i].widget, have_article);
	++i; /* i is reply by mail */
	gtk_widget_set_sensitive (article_menu[i].widget, have_article);
	++i; /* i is reply by mail external */
	gtk_widget_set_sensitive (article_menu[i].widget, have_article);

	++i; /* separator */
	++i; /* article filter */
	++i; /* sort */
	++i; /* thread */
	++i; /* rot13 */
	++i; /* show all headers */

	++i; /* separator */
	++i; /* save attachment */
	gtk_widget_set_sensitive (article_menu[i].widget, have_article);
	++i; /* save attachment as */
	gtk_widget_set_sensitive (article_menu[i].widget, have_article);
	++i; /* open attachment */
	gtk_widget_set_sensitive (article_menu[i].widget, have_article);

	++i; /* separator */
	++i; /* next article */
	gtk_widget_set_sensitive (article_menu[i].widget, have_group);
	++i; /* next unread article */
	gtk_widget_set_sensitive (article_menu[i].widget, have_group);
	++i; /* next thread */
	gtk_widget_set_sensitive (article_menu[i].widget, have_group);
	++i; /* prev article */
	gtk_widget_set_sensitive (article_menu[i].widget, have_group);
	++i; /* prev unread article */
	gtk_widget_set_sensitive (article_menu[i].widget, have_group);
	++i; /* prev thread */
	gtk_widget_set_sensitive (article_menu[i].widget, have_group);

	++i; /* separator */
	++i; /* download entire thread*/
	gtk_widget_set_sensitive (article_menu[i].widget, have_article);

	++i; /* separator */
	++i; /* delete message */
	gtk_widget_set_sensitive (article_menu[i].widget, have_article);
	++i; /* delete thread */
	gtk_widget_set_sensitive (article_menu[i].widget, have_article);

	++i; /* separator */
	++i; /* mark all read */
	gtk_widget_set_sensitive (article_menu[i].widget, have_group);
	i++; /* select all */
	gtk_widget_set_sensitive (article_menu[i].widget, have_group);



	/* update the title */
	gui_set_title (gdata);

	return 0;
}

static int
gui_text_show_all_headers_changed_cb (
	gpointer call_object,
	gpointer call_arg,
	gpointer user_data)
{
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_menu[9].widget), call_arg!=NULL);

	return 0;
}

static int
gui_articlelist_selection_changed_cb (
	gpointer call_object,
	gpointer call_arg,
	gpointer user_data)
{
	gui_articlelist_group_changed_cb (NULL, NULL, NULL);
	return 0;
}


static int
thread_changed_cb (
	gpointer call_object,
	gpointer call_arg,
	gpointer user_data)
{
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_menu[7].widget), 
		call_arg!=NULL);

	return 0;
}

static int
gui_articlelist_sort_changed_cb (
	gpointer call_object,
	gpointer call_arg,
	gpointer user_data)
{
	int sort = GPOINTER_TO_INT(call_arg);
	int i = 0;
	GtkRadioMenuItem* w = NULL;

	switch (sort) {
		case ARTICLE_SORT_AUTHOR: i=0; break;
		case -ARTICLE_SORT_AUTHOR: i=1; break;
		case ARTICLE_SORT_DATE: i=2; break;
		case -ARTICLE_SORT_DATE: i=3; break;
		case ARTICLE_SORT_LINES: i=4; break;
		case -ARTICLE_SORT_LINES: i=5; break;
		case ARTICLE_SORT_SUBJECT: i=6; break;
		case -ARTICLE_SORT_SUBJECT: i=7; break;
		case ARTICLE_SORT_UNREAD_CHILDREN: i=8; break;
		case -ARTICLE_SORT_UNREAD_CHILDREN: i=9; break;
		default: pan_warn_if_reached(); break;
	}

	w = GTK_RADIO_MENU_ITEM(article_sort_menu_2[i].widget);
	dampen_feedback_sort = TRUE;
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(w), TRUE);
	dampen_feedback_sort = FALSE;

	return 0;
}

static int
gui_message_state_filter_changed_cb (gpointer call_object,
				     gpointer call_arg,
				     gpointer user_data)
{
	guint filter = GPOINTER_TO_UINT (call_arg);

	pan_lock();

	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[2].widget),
		filter & STATE_FILTER_UNREAD);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[3].widget),
		filter & STATE_FILTER_READ);

	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[5].widget),
		filter & STATE_FILTER_WATCHED);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[6].widget),
		filter & STATE_FILTER_KILLFILE);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[7].widget),
		filter & STATE_FILTER_NORMAL_RANK);

	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[9].widget),
		filter & STATE_FILTER_COMPLETE_BINARIES);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[10].widget),
		filter & STATE_FILTER_INCOMPLETE_BINARIES);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[11].widget),
		filter & STATE_FILTER_NONBINARIES);

	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[13].widget),
		filter & STATE_FILTER_SAVED);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[14].widget),
		filter & STATE_FILTER_QUEUED);
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(article_filter_menu[15].widget),
		filter & STATE_FILTER_IDLE);

	pan_unlock ();

	return 0;
}


/*---[ gui_construct ]------------------------------------------------
 * @geometry:
 *
 * initialize and place the basic interface components
 *--------------------------------------------------------------------*/
void
gui_construct (const gchar *geometry)
{
	GtkWidget *w, *x;
	//GtkWidget *hbox;
//	GtkStyle *boldstyle;
	/* int geo_x=0, geo_y=0; */

	ttips = gtk_tooltips_new();

	item_to_view_hashtable = g_hash_table_new (g_direct_hash, g_direct_equal);

	Pan.window = gnome_app_new ("Pan", "Pan");
	gtk_signal_connect (GTK_OBJECT(Pan.window), "delete_event", 
			    GTK_SIGNAL_FUNC (window_delete_event_cb), NULL);
	gtk_signal_connect (GTK_OBJECT(Pan.window), "destroy",
			    GTK_SIGNAL_FUNC (pan_shutdown), NULL);

	gtk_widget_realize (Pan.window);

	/** geometry parsing **/
	/* The following chunk of code will allow command line arguments
	 * to override the config. So if this was invoked as part of a 
	 * session we 'Do the right thing' :)
	 */
	if (geometry != NULL)
	  {
	    gint x, y, w, h;
	    if (gnome_parse_geometry (geometry, &x, &y, &w, &h) )
	      {
		if (x != -1)
		  gtk_widget_set_uposition (Pan.window, x, y);
		if (w != -1)
		  gtk_window_set_default_size (GTK_WINDOW (Pan.window), w, h);
	      }
	    else
	      {
		g_error (_("Unable to parse the geometry string '%s'"), geometry);
	      }
	  }
	else
	  {                  /* Lets see if the config has a geometry */
	    gint w, h;

	    w = gnome_config_get_int ("/Pan/Geometry/width");
	    h = gnome_config_get_int ("/Pan/Geometry/height");
	    if (w != 0 && h != 0)
		    gtk_window_set_default_size (GTK_WINDOW (Pan.window), w, h);
	    else
		    gtk_window_set_default_size (GTK_WINDOW (Pan.window), 640, 400);

/* While this is nice for some people, it's typically much better to let
 * the window manager do the position saving and restoring... we have similar
 * code at the bottom of this function...

	    x = gnome_config_get_int ("/Pan/Geometry/pan_x");
	    y = gnome_config_get_int ("/Pan/Geometry/pan_y");
	    if (x != 0 && y != 0)
	      gtk_widget_set_uposition (Pan.window, x, y);
*/
	  }

/* Groups vbox */
	groups_vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_set_usize (GTK_WIDGET (groups_vbox), 260, -1);
	x = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (groups_vbox), grouplist_create(), TRUE, TRUE, 1);
        gtk_signal_connect (GTK_OBJECT (Pan.group_clist), "key_press_event",
                            GTK_SIGNAL_FUNC (gui_key_press_cb),  NULL);
	gtk_box_pack_start (GTK_BOX (groups_vbox), x, FALSE, FALSE, 1);

/* Articlelist_ctree */
	articlelist_ctree = gtk_vbox_new (FALSE, 0);
	w = create_articlelist_ctree ();
	article_toolbar_new();
	gtk_box_pack_start (GTK_BOX (articlelist_ctree), w, TRUE, TRUE, 0);

/* Text table */
	text_box = gtk_vbox_new (FALSE, 0);

	header_table = gtk_table_new ( 0, 0, FALSE );
	gtk_box_pack_start (GTK_BOX(text_box), header_table, FALSE, FALSE, 0);

	gtk_box_pack_start (GTK_BOX (text_box), text_create(), TRUE, TRUE, 0);

	gui_create_appbar (Pan.window);

	gnome_app_create_menus (GNOME_APP (Pan.window), pan_main_menu);
//	gnome_app_create_toolbar (GNOME_APP (Pan.window), pan_main_toolbar);
	gnome_app_install_menu_hints (GNOME_APP (Pan.window), pan_main_menu);

	contents_vbox = gtk_vbox_new (FALSE, 0);
	gnome_app_set_contents (GNOME_APP (Pan.window), contents_vbox);

	/* connect signals */
        gtk_signal_connect (GTK_OBJECT (Pan.article_ctree), "key_press_event",
                            GTK_SIGNAL_FUNC (gui_key_press_cb), NULL);
        gtk_signal_connect (GTK_OBJECT (Pan.text), "key_press_event",
                            GTK_SIGNAL_FUNC (gui_key_press_cb), NULL);

        gui_set_layout ( gnome_config_get_int ( "Pan/State/viewmode=1" ) );

/* FIXME: this isn't useful until we can get it to automatically load the last
   selected group or message.  Otherwise, it remembers you were in the article
   list view, but then it's just blank, so you gotta switch back to the group
   list mode anyhow.
   if (Pan.viewmode == GUI_NOTEBOOK) {
   int last_page = gnome_config_get_int ("/Pan/State/page=0");
   tk_notebook_set_page (GTK_NOTEBOOK (notebook), last_page);
   }
*/

	/* FIXME: 0,0 shows up under the panel for me on IceWm
	 * I'm not sure we should mess with window placement
	 * until the user tells us to... 
	 geo_x = gnome_config_get_int ("/Pan/Geometry/pan_x=0");
	 geo_y = gnome_config_get_int ("/Pan/geometry/pan_y=0");
	 gdk_window_move (Pan.window->window, geo_x, geo_y);
	*/

	/* listen for changes to the articlelist "state filter"
	   so that we can update the menus accordingly */
	pan_callback_add (
		articlelist_state_filter_changed,
		gui_message_state_filter_changed_cb,
		NULL);

	/* listen for changes to the articlelist active group
	   so that we can sensitize/desensitize menus accordingly */
	gui_articlelist_group_changed_cb (
		NULL, articlelist_get_current_group(), NULL);
	pan_callback_add (
		articlelist_group_changed,
		gui_articlelist_group_changed_cb,
		NULL);

	pan_callback_add (
		articlelist_sort_changed,
		gui_articlelist_sort_changed_cb,
		NULL);

	thread_changed_cb (NULL, GINT_TO_POINTER(1), NULL);
	pan_callback_add (
		articlelist_thread_changed,
		thread_changed_cb,
		NULL);

	gui_articlelist_selection_changed_cb (
		NULL, articlelist_get_selected_adata(), NULL);
	pan_callback_add (
		articlelist_selection_changed,
		gui_articlelist_selection_changed_cb,
		NULL );


	gui_text_show_all_headers_changed_cb (
		NULL, GINT_TO_POINTER(show_all_headers_in_body), NULL);
	pan_callback_add (
		text_show_all_headers_changed,
		gui_text_show_all_headers_changed_cb,
		NULL );

	if (gnome_config_get_bool(
			"Pan/Display/Show_Filter_In_Article_Toolbar=true"))
		gtk_check_menu_item_set_active (
			GTK_CHECK_MENU_ITEM(view_menu[4].widget), TRUE);

	if (gnome_config_get_bool("Pan/Display/Show_All_Headers_In_Body=false"))
		gtk_check_menu_item_set_active (
			GTK_CHECK_MENU_ITEM(article_menu[9].widget), TRUE);

	gui_set_title (NULL);
}


/*---[ replace_checkbox ]---------------------------------------------
 * Update one of our homebrew checkboxes.
 * I copied the idea from the groups menu.
 * (We're using homebrew checkboxes to get the pretty checkbox icon?)
 * FIXME: if gnome user sets /Gnome/Icons/MenusUseIcons to false, this fails..
 *--------------------------------------------------------------------*/
static void
replace_checkbox ( const GnomeUIInfo* e, int is_checked, const gchar* menu_path )
{
	GnomeUIInfo entry[2];
	entry[0] = *e;
	if ( is_checked )
	{
		entry[0].pixmap_type = GNOME_APP_PIXMAP_STOCK;
		entry[0].pixmap_info = GNOME_STOCK_PIXMAP_CLOSE;
	}
	else
	{
		entry[0].pixmap_type = GNOME_APP_PIXMAP_NONE;
		entry[0].pixmap_info = NULL;
	}
	entry[1].type = GNOME_APP_UI_ENDOFINFO;

	pan_lock();
	gnome_app_insert_menus ( GNOME_APP(Pan.window), menu_path, entry );
	gnome_app_remove_menus ( GNOME_APP(Pan.window), menu_path, 1 );
	pan_unlock();
}




/*---[ gui_layout ]---------------------------------------------------
 * main menu callback
 * gui_set_layout (layout)
 *--------------------------------------------------------------------*/

void
gui_set_layout (int new_layout)
{
	int widget_index = 0;

	/* build the desired layout */
	switch ( new_layout ) {
		case GUI_PANED: gui_paned_construct (); break;
		case GUI_NOTEBOOK: gui_notebook_construct (); break;
	}

	/* update the menu */
	switch (new_layout) {
		case GUI_PANED: widget_index = 0; break;
		case GUI_NOTEBOOK: widget_index = 1; break;
		default: pan_warn_if_reached(); widget_index = 1; break;
	}
	gtk_check_menu_item_set_active (
		GTK_CHECK_MENU_ITEM(layout_menu[widget_index].widget),
		TRUE);
}


/*---[ gui_page_set ]-------------------------------------------------
 * flip to a different notebook page, setting keyboard focus on a 
 * widget in the new page
 *--------------------------------------------------------------------*/
void
gui_page_set (int page, GtkWidget *focus_item)
{
	if (Pan.viewmode == GUI_NOTEBOOK) {
		pan_lock();
		gtk_notebook_set_page (GTK_NOTEBOOK (notebook), page);
		gtk_widget_grab_focus (focus_item);
		pan_unlock();
	}
}


/*---[ gui_page_change ]----------------------------------------------
 * flip to a differnt notebook page, setting keyboard focus in new page.  
 * This is different from gui_page_set because it flips the page based on 
 * a offset, like -1 or 1, to flip back one page or forward one.  It is 
 * automatic in handling the focus grabbing too.
 *--------------------------------------------------------------------*/
void
gui_page_change (int change)
{
	int cur_page;

	if (Pan.viewmode == GUI_PANED)
		cur_page = gui_paned_get_current_pane ();
	if (Pan.viewmode == GUI_NOTEBOOK)
		cur_page = gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook));
	else
		return;

	cur_page += change;
	if ((cur_page < 0) || (cur_page > 2)) 
		return;

	if (change < 0) { /* Escape was pressed */
		switch (cur_page) {
		case GROUPS_PAGE:
			pan_lock();
			gtk_widget_grab_focus (Pan.group_clist);
			pan_unlock();
			break;
		case ARTICLELIST_PAGE:
			pan_lock();
			gtk_widget_grab_focus (Pan.article_ctree);
			pan_unlock();
			break;
		case MESSAGE_PAGE:
			pan_lock();
			gtk_widget_grab_focus (Pan.text);
			pan_unlock();
			break;
		default:
			break;
		}
	}
	else { /* enter key hit */
/* FIXME: segfaults if selection is zero, but when user hits escape, it
   automatically sends a "undo_selection" signal, deselecting all */
		switch (cur_page) {
		case GROUPS_PAGE:
			gtk_widget_grab_focus (Pan.group_clist);
			break;
		case ARTICLELIST_PAGE:
		{
			/* update the articlelist */
			server_data *sdata = grouplist_get_current_server();
			GSList *l = grouplist_get_selected_groups ();
			group_data *gdata = NULL;
			if (l!=NULL)
				gdata = (group_data*) l->data;
			if (sdata!=NULL && gdata!=NULL)
				articlelist_set_current_group (sdata, gdata);
			g_slist_free (l);

			/* show the articlelist */
			pan_lock();
			gtk_widget_grab_focus (Pan.article_ctree);
			pan_unlock();
			break;
		}
		case MESSAGE_PAGE:
		{
			const gchar* message_id = 
				articlelist_get_selected_message_id();
			if (message_id!=NULL) {
				text_set_from_message_id (
					articlelist_get_current_server(),
					articlelist_get_current_group (),
					message_id, NNTP_ARTICLE_READ, FALSE);
				pan_lock();
				gtk_widget_grab_focus (Pan.text);
				pan_unlock();
			}

			break;
		}
		default:
			break;
		}
	}

	if (Pan.viewmode == GUI_NOTEBOOK)
		gtk_notebook_set_page (GTK_NOTEBOOK (notebook), cur_page);
/*
  FIXME: this would be nice, there is a slight problem with the tabs at the top
  being refreshed after repeated flipping around.  But redrawing the entire
  notebook looks flickery and will waste a lot of resources.  Maybe redraw at
  some other less frequent place or redraw only the tab elements (if that is 
  possible).
  gtk_widget_draw (GTK_WIDGET (notebook), NULL);
*/
}

static void
gui_enter_clicked (void)
{
	if (Pan.viewmode == GUI_NOTEBOOK)
		gui_page_change (1);
	else if (Pan.viewmode == GUI_PANED)
		gui_paned_enter_clicked ();
}


/*---[ gui_set_title ]------------------------------------------------
 * @gdata:
 *
 * set Pan's title to reflect the current open group
 *--------------------------------------------------------------------*/
void
gui_set_title (const group_data *gdata)
{
	gchar *buf;
	
	if (gdata)
		buf = g_strdup_printf ("Pan %s [%s]", VERSION, gdata->name);
	else
		buf = g_strdup_printf ("Pan %s", VERSION);

	pan_lock();
	gtk_window_set_title (GTK_WINDOW (Pan.window), buf);
	pan_unlock();

	g_free (buf);

}


/*---[ gui_key_press_cb ]---------------------------------------------
 * key press callback, when a key is pressed in the group list, article
 * list, or in the message view.  This allows better keyboard navigation
 * and the potential for new key bindings to imitate Ag*nt.
 *--------------------------------------------------------------------*/
static void
gui_key_press_cb (GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	switch (event->keyval)
	{
	case GDK_Q:
	case GDK_q:
		gui_page_change (-1);
		break;
	case GDK_Delete:
		articlelist_selected_delete ();
		break;
	case GDK_Return:
//		gui_page_change (1);
		gui_enter_clicked ();
		break;
	default:
		break;
	}
}

/*---[ gui_popup_draw ]-----------------------------------------------
 * @dialog: The dialog or popup window widget
 * @parent: The window/widget that the dialog will be centered w.r.t.
 *
 * Use this to draw popup windows or dialogs, it will center them in the
 * exact center of the Pan window, show, and raise them.
 *--------------------------------------------------------------------*/
void
gui_popup_draw (GtkWidget *dialog, GtkWidget *parent)
{
	int x=0, y=0, width=640, height=400;
	int dlg_width, dlg_height;
	int dlg_x, dlg_y;

	pan_lock();

	if (parent==NULL)
		parent = Pan.window;

	gtk_window_set_transient_for (GTK_WINDOW(dialog), GTK_WINDOW(parent));

	gdk_window_get_position (Pan.window->window, &x, &y);
	gdk_window_get_size (Pan.window->window, &width, &height);

	gtk_widget_realize (dialog);
	gdk_window_get_size (dialog->window, &dlg_width, &dlg_height);

	dlg_x = x + width/2 - dlg_width/2;
	dlg_y = y + height/2 - dlg_height/2;
	gtk_widget_set_uposition (dialog, dlg_x, dlg_y);

	gtk_widget_show_all (dialog);
	gdk_window_raise (dialog->window);

	pan_unlock();
}

/*---[ gui_restore_column_widths ]------------------------------------
 * @clist:
 * @type:
 *
 * 
 *--------------------------------------------------------------------*/
void
gui_restore_column_widths (GtkWidget *clist, gchar *type)
{
	int i;
	GtkCList *list = GTK_CLIST (clist);
	const int cols = list->columns;

	for (i=0; i!=cols; ++i)
	{
		/* get width... */
		gchar* buf = g_strdup_printf ("/Pan/Geometry/%s_column_%d", type, i);
		const int width = gnome_config_get_int (buf);
		g_free (buf);

		/* if user had width set, use the setting */
		if (width) {
			pan_lock();
			gtk_clist_set_column_width (list, i, width);
			pan_unlock();
		}
	}
}

void
gui_save_column_widths (GtkWidget *clist, gchar *type)
{
	int i;
	int cols = GTK_CLIST (clist)->columns;
	
	for (i = 0; i < cols; i++) {
		gchar* buf = g_strdup_printf ( "/Pan/Geometry/%s_column_%d", type, i );
		gnome_config_set_int (buf, (int) GTK_CLIST (clist)->column[i].width);
		g_free ( buf );
	}
}

/*****
******
*****/

void
gui_set_queue_size (int size)
{
	static int old_size = -1;

	if (size != old_size)
	{
		gchar *str = g_strdup_printf ( _("Tasks: %2d"), size );
		old_size = size;

		pan_lock();
		gtk_label_set_text (GTK_LABEL(queue_qty_label), str);
		pan_unlock();

		g_free (str);
	}
}

static void
taskbar_validate (void)
{
	GList *kids = NULL;
	int child_qty = 0;
	int child_w = 0;
	int child_h = 0;
	int x = 0;

	kids = gtk_container_children (GTK_CONTAINER(taskbar));
	if (!kids)
		return;

	child_qty = g_list_length (kids);
	child_w = taskbar->allocation.width / child_qty;
	child_h = taskbar->allocation.height;
	for (x=0; kids!=NULL; kids=kids->next, x+=child_w) {
		gtk_widget_set_usize (GTK_WIDGET(kids->data), child_w, child_h);
		gtk_fixed_move (GTK_FIXED(taskbar), GTK_WIDGET(kids->data), x, 0);
	}
}

void
gui_add_status_item (StatusItem *item)
{
	GtkWidget *view = NULL;

	pan_lock();
	gtk_widget_hide (taskbar);
	view = status_item_view_new_with_item (item);
	gtk_fixed_put (GTK_FIXED(taskbar), GTK_WIDGET(view), 0, 0);
	taskbar_validate ();
	gtk_widget_show_all (taskbar);
	pan_unlock();

	g_hash_table_insert (item_to_view_hashtable, item, view);
}

void
gui_remove_status_item (StatusItem *item)
{
	GtkWidget *w = GTK_WIDGET(g_hash_table_lookup(item_to_view_hashtable, item));
	g_hash_table_remove (item_to_view_hashtable, item);

	pan_lock();
	gtk_widget_hide (taskbar);
	gtk_container_remove(GTK_CONTAINER(taskbar), w);
	taskbar_validate ();
	gtk_widget_show_all (taskbar);
	pan_unlock();
}


/*****
******
*****/


static void
showmore_cb (GtkButton *button, gpointer user_data)
{
	const gchar* showme = NULL;

	pan_lock();
	showme = (const gchar*) gtk_object_get_data ( GTK_OBJECT(button), "text" );
	gnome_ok_dialog_parented ( showme, GTK_WINDOW(Pan.window) );
	pan_unlock();
}

/* note: this is called inside a pan_lock() */
static void
add_header (
	GtkWidget* table,
	int row,
	const gchar* str1,
       	const gchar* str2,
	char delimiter )
{
	GtkTable *t = GTK_TABLE(table);
	GtkWidget *w1 = gtk_label_new (str1);
	GtkWidget *w2 = NULL;

	if ( delimiter && strchr(str2,delimiter)!=NULL )
	{
		GtkWidget *hbox = gtk_hbox_new (FALSE, 0);
		GtkWidget *l = NULL;
		GtkWidget *l2 = gtk_label_new (_("<More>"));
		gchar *full = NULL;
		gchar *pch = NULL;

		/* create the truncated label */
		pch = g_strdup(str2);
		*strchr(pch,delimiter) = '\0';
		l = gtk_label_new (pch);
		g_free (pch);

		/* add them both to the button */
		gtk_box_pack_start(GTK_BOX(hbox), l, FALSE, FALSE, 0);
		gtk_box_pack_end(GTK_BOX(hbox), l2, FALSE, FALSE, 0);

		/* break the line into easier-to-read pieces
		   for the popup dialog */
		full = g_strdup(str2);
		for (pch=full; *pch; ++pch)
			if (*pch == delimiter)
				*pch = '\n';

		/* create a button */
		w2 = gtk_button_new ( );
		gtk_button_set_relief (GTK_BUTTON(w2), GTK_RELIEF_NONE);
		gtk_container_add (GTK_CONTAINER(w2), hbox);
		gtk_widget_show (l);
		gtk_widget_show (l2);
		gtk_widget_show (hbox);
		gtk_object_set_data_full (
			GTK_OBJECT(w2), "text", full, g_free);
		gtk_signal_connect (
			GTK_OBJECT(w2), "clicked", showmore_cb, NULL);
	}
	else
	{
		/* string is short; show it all */
		w2 = gtk_label_new  (str2);
		gtk_misc_set_alignment (GTK_MISC(w2), 0.0, 0.5);
	}

	gtk_misc_set_alignment (GTK_MISC(w1), 1.0, 0.5);

	gtk_table_resize (t, row+1,2);

	gtk_table_attach (t, w1, 0,1, row, row+1, GTK_FILL, 0,2,1 );
	gtk_table_attach (t, w2, 1,2, row, row+1, GTK_EXPAND|GTK_FILL, 0,2,1 );
	gtk_widget_show (w1);
	gtk_widget_show (w2);
}

void
gui_set_headers (GtkWidget *table,
		 const article_data *adata,
		 gulong header_fields)
{
	int row = 0;

	if (!table)
		table = header_table;
	if (!adata)
		return;
	pan_lock();

	/* remove children */
	gtk_container_foreach (GTK_CONTAINER (header_table), GTK_SIGNAL_FUNC (gtk_widget_destroy), NULL);

	/* add new children */
	if (header_fields & HEADER_AUTHOR) {
		add_header (header_table, row++, _("From: "), adata->author, '\0');
	}
	/* if this is a crosspost, and newsgroups turned on, show newsgroups. */
	if (strchr(adata->newsgroups,',')!=NULL && (header_fields & HEADER_NEWSGROUPS)) {
		add_header (header_table, row++, _("Newsgroups: "), adata->newsgroups, ',');
	}
	if (header_fields & HEADER_MESSAGE_ID) {
		add_header (header_table, row++, _("Message-Id: "), adata->message_id, '\0');
	}
	if (*adata->references && (header_fields & HEADER_REFERENCES)) {
		add_header (header_table, row++, _("References: "), adata->references, ' ');
	}
	if (header_fields & HEADER_DATE) {
		char timebuf[40] = {'\0'};
		const struct tm *local_tm = localtime (&adata->date);
		strftime (timebuf, 40, "%A, %B %e, %Y %l:%M %p", local_tm);
		add_header (header_table, row++, _("Date: "), timebuf, '\0');
	}
	if ((header_fields&HEADER_REPLY_TO) && *adata->reply_to && strcmp(adata->reply_to,adata->author)) {
		add_header (header_table, row++, _("Reply-To: "), adata->reply_to, '\0');
	}
	if ((header_fields&HEADER_FOLLOWUP_TO)
	      && *adata->followup_to
	      && strcmp(adata->followup_to,adata->newsgroups)) {
		add_header (header_table, row++, _("Followup-To: "), adata->followup_to, ',');
	}
	if (header_fields & HEADER_SUBJECT) {
		add_header (header_table, row++, _("Subject: "), adata->subject, '\0');
	}

	pan_unlock();
}

void
gui_set_headers_default (GtkWidget* table,
			 const article_data* adata)
{
	gui_set_headers (table, adata,
			 (ulong)gnome_config_get_int_with_default("/Pan/State/Headers=243", NULL));
}


void
clist_unselect_all (GtkWidget *clist)
{
	GList *list;
	gint row;

	list = GTK_CLIST(clist)->selection;
	while (list) {
		row = GPOINTER_TO_INT (list->data);
		list = list->next;
		gtk_clist_unselect_row (GTK_CLIST(clist), row, 0);
	}
	g_list_free (list);
}

void
widget_set_font (GtkWidget* w,
                 const char* font_name)
{
        GtkStyle* style = NULL;
        GdkFont* font = NULL;

        g_return_if_fail (w!=NULL);
        if (!font_name || !*font_name)
                return;

        style = gtk_style_copy (gtk_widget_get_style(Pan.window));
        font = gdk_fontset_load ((char*)font_name);
        if ( font != NULL )
                style->font = font;
        else
                g_warning ("Couldn't load font ``%s''", font_name);
        gtk_widget_set_style (w, style);
}

/**
 * dialog_draw_centered:
 * @dialog: the dialog to draw
 * @window: the window to center the dialog w.r.t.
 *
 * Draw a dialog at the center of a window.
 **/
void
dialog_draw_centered (GtkWidget *dialog, GtkWidget *window)
{
        int x=0, y=0, width=640, height=400;
        int dlg_width, dlg_height;
        int dlg_x, dlg_y;

        if ( !window )
                window = Pan.window;

        gdk_window_get_position (GTK_WIDGET (window)->window, &x, &y);
        gdk_window_get_size (GTK_WIDGET (window)->window, &width, &height);

        gtk_widget_realize (dialog);
        gdk_window_get_size (dialog->window, &dlg_width, &dlg_height);

        dlg_x = x + width/2 - dlg_width/2;
        dlg_y = y + height/2 - dlg_height/2;

        gtk_widget_set_uposition (dialog, dlg_x, dlg_y);
        gtk_widget_show_all (dialog);

        gdk_window_raise (dialog->window);
}

