/*
 * Pan - A Newsreader for X
 * Copyright (C) 1999  Pan Development Team (pan@superpimp.org)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <glib.h>
#include <gtk/gtk.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-stock.h>

#include "fnmatch.h"

#include "article.h"
#include "articlelist.h"
#include "group.h"
#include "grouplist.h"
#include "message-filter.h"
#include "prefs.h" /* for data_dir */
#include "util.h"

#include "xpm/delete.xpm"
#include "xpm/arrow_up.xpm"
#include "xpm/arrow_up_create.xpm"
#include "xpm/arrow_down.xpm"

/*********************
**********************  Defines / Enumerated types
*********************/

/*********************
**********************  Macros
*********************/

/*********************
**********************  Structures / Typedefs
*********************/

typedef struct
{
	int unique_id;

	gchar* author;
	gchar* group;
	gchar* subject;
	gchar* thread;
	int crosspost_qty;

	MessageFilterAction onmatch_action;
	int lifespan_days;
}
rule_t;

typedef struct
{
	GnomeDialog *dialog;

	GtkWidget* kill_rb;
	GtkWidget* track_rb;

	GtkAdjustment* crosspost_data;
	GtkWidget* crosspost_tb;
	GtkWidget* crosspost_sb;

	GtkWidget* thread_tb;
	GtkWidget* thread_entry;

	GtkWidget* subject_tb;
	GtkWidget* subject_entry;
	GtkWidget* author_tb;
	GtkWidget* author_entry;
	GtkWidget* group_tb;
	GtkWidget* group_entry;

	GtkWidget* this_session_rb;
	GtkWidget* forever_rb;

	GtkWidget* rules_list;

	GtkWidget* edit_pixmap;
	GtkWidget* edit_button;
	GtkWidget* save_change_pixmap;
	GtkWidget* save_change_button;
	GtkWidget* save_new_pixmap;
	GtkWidget* save_new_button;
	GtkWidget* delete_pixmap;
	GtkWidget* delete_button;

	gboolean changes_made;

	GtkTooltips* tips;
}
MessageFilterUI;

/*********************
**********************  Private Function Prototypes
*********************/

/* ui */
static GtkWidget* create_rule_edit_page ( MessageFilterUI* ui );
static GtkWidget* create_filter_page ( MessageFilterUI* );
static void dialog_destroy_cb ( GtkObject*, gpointer );
static void update_controls ( MessageFilterUI* );

static void rule_ui_list_row_add ( MessageFilterUI*, const rule_t* );
static void rule_ui_list_row_refresh ( MessageFilterUI*, const rule_t* );
static void rule_ui_list_build ( MessageFilterUI* );
static rule_t* rule_ui_list_get_selected_rule ( MessageFilterUI* );
static gchar** rule_ui_list_get_text ( const rule_t* );

static void rule_to_ui ( const rule_t*, MessageFilterUI* );
static void ui_to_rule ( const MessageFilterUI*, rule_t* );

static void delete_button_clicked ( GtkButton*, gpointer );
static void save_change_button_clicked ( GtkButton*, gpointer );
static void edit_button_clicked ( GtkButton*, gpointer );
static void dialog_clicked_cb ( GnomeDialog*, gint, gpointer );

/* database */
static pan_db rule_db_open ( );
static void rule_db_save ( const rule_t* );
static void rule_db_save_new ( rule_t* );
static void rule_db_delete ( rule_t* );

/* utility */
static void rule_deserialize (rule_t*, const char* key, const char* val);
static void rule_clean (rule_t*);

/* init */
static void message_filter_init ( void );
static void message_filter_init_foreach ( const char* key, const char* val, gpointer );

static void dialog_clicked_cb ( GnomeDialog*, gint, gpointer );

/*********************
**********************  Constants
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

static const int cols = 7;
static const char* SEQUENCE_PROPERTY = "unique_id_sequence";

/*********************
**********************  Variables 
*********************/

/***********
************  Extern
***********/

/***********
************  Public
***********/

/***********
************  Private
***********/

static MessageFilterUI* _ui = NULL;

static gboolean _inited = FALSE;
static GSList* _rules = NULL;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

/************
*************  PUBLIC ROUTINES
************/

void message_filter_killfile_add (
	const char* author_str,
	const char* subject_str,
	const char* group_str,
	const char* thread_str,
	int crosspost_qty,
	int duration_days,
	MessageFilterAction action)
{
	rule_t *rule = NULL;

	if ( !_inited )
		message_filter_init ( );

	/* add the rule */
	rule = g_new0 (rule_t, 1);
	rule->author = g_strdup (author_str);
	rule->subject = g_strdup (subject_str);
	rule->group = g_strdup (group_str);
	rule->thread = g_strdup (thread_str);
	rule->crosspost_qty = crosspost_qty;
	rule->onmatch_action = action;
	rule->lifespan_days = duration_days;

	/* save the rule */
	rule_db_save_new (rule);
	_rules = g_slist_prepend (_rules, rule);
	if (_ui!=NULL)
		rule_ui_list_row_add (_ui, rule);
}


MessageFilterAction
message_filter_get_action (
	const group_data* gdata,
	const article_data* adata )
{
	GSList* l;

	if ( !_inited )
		message_filter_init ( );

	for ( l=_rules; l!=NULL; l=l->next )
	{
		const rule_t* rule = (const rule_t*)l->data;

		if (*rule->thread) {
			const size_t len = strlen (rule->thread);
			if ((strncmp(rule->thread, adata->references, len)) &&
			    (strncmp(rule->thread, adata->message_id, len)))
			{
				continue;
			}
		}

		if (*rule->subject && fnmatch(rule->subject,adata->subject,0))
			continue;

		if (*rule->group && fnmatch(rule->group,gdata->name,0))
			continue;

		if (*rule->author && fnmatch(rule->author,adata->author,0))
			continue;

		return rule->onmatch_action;
	}

	return 0;
}

void
message_filter_ui ( void )
{
	const group_data *gdata = NULL;
	const server_data *sdata = grouplist_get_current_server();

	g_return_if_fail (sdata);

	message_filter_init ( );

	if ( _ui != NULL ) {
                if ( !GTK_WIDGET_MAPPED(GTK_WIDGET(_ui->dialog)) )
                        gtk_widget_map(GTK_WIDGET(_ui->dialog));
		gdk_window_raise ( GTK_WIDGET(_ui->dialog)->window );
		return;
	}

	_ui = (MessageFilterUI*) g_malloc0 ( sizeof(MessageFilterUI) );
	_ui->changes_made = FALSE;
	_ui->tips = gtk_tooltips_new ( );
	_ui->dialog = GNOME_DIALOG( gnome_dialog_new ( _("Message Filter"), GNOME_STOCK_BUTTON_OK, NULL ) );
	gtk_widget_set (GTK_WIDGET(_ui->dialog), "allow_shrink", TRUE, "allow_grow", TRUE, NULL);
        gtk_box_pack_start (GTK_BOX(_ui->dialog->vbox), create_filter_page(_ui), TRUE, TRUE, 0);
        gtk_signal_connect (GTK_OBJECT(_ui->dialog), "clicked", dialog_clicked_cb, _ui);
        gtk_signal_connect (GTK_OBJECT(_ui->dialog), "destroy", dialog_destroy_cb, _ui);

	rule_ui_list_build ( _ui );

	if ( (gdata = articlelist_get_current_group()) )
	{
		const article_data* adata = NULL;
		gtk_entry_set_text (
			GTK_ENTRY(_ui->group_entry), gdata->name);

		adata = articlelist_get_selected_adata();
		if (adata != NULL)
		{
			gchar *pch = NULL;

			/* set the subject */
			gtk_entry_set_text (
				GTK_ENTRY(_ui->subject_entry),adata->subject);
			gtk_editable_set_position (
				GTK_EDITABLE(_ui->subject_entry), 0);

			/* set the author */
			gtk_entry_set_text (
				GTK_ENTRY(_ui->author_entry), adata->author);
			gtk_editable_set_position (
				GTK_EDITABLE(_ui->author_entry), 0);

			/* set the group name */
			gtk_entry_set_text (
				GTK_ENTRY(_ui->group_entry), gdata->name);
			gtk_editable_set_position (
				GTK_EDITABLE(_ui->group_entry), 0);

			/* set the top thread message-id */
			pch = article_get_thread_message_id(adata);
			gtk_entry_set_text (
				GTK_ENTRY(_ui->thread_entry), pch);
			gtk_editable_set_position (
				GTK_EDITABLE(_ui->thread_entry), 0);
			g_free (pch);
		}
	}

//	gui_popup_draw ( GTK_WIDGET (_ui->dialog) );
	gtk_window_set_position (GTK_WINDOW (_ui->dialog), GTK_WIN_POS_CENTER);
	gtk_widget_show_all ( GTK_WIDGET ( _ui->dialog ) );
}

/************
*************  PRIVATE ROUTINES
************/

/**
*** DATABASE
**/

static pan_db
rule_db_open (void)
{
	gchar* filename = g_strdup_printf ("/%s/killfile.db", data_dir);
	pan_db db = pan_db_open (filename);
	g_free (filename);
	return db;
}

static void
rule_db_save (
	const rule_t* rule )
{
	/* setup */
	pan_db db = rule_db_open ( );
	gchar *key, *val;

	/* store the rule */
	key = g_strdup_printf ( "%d", rule->unique_id );
	val = g_strdup_printf ( "%s\n%s\n%s\n%s\n%d\n%d",
		rule->subject ? rule->subject : "",
		rule->author ? rule->author : "",
		rule->group ? rule->group : "",
		rule->thread ? rule->thread : "",
		(unsigned)rule->onmatch_action,
		rule->lifespan_days);
	pan_db_put_value_str (db, key, val);

	/* cleanup */
	g_free (key);
	g_free (val);
	pan_db_close (db);
}

static void
rule_db_save_new (
	rule_t* rule )
{
	/* update sequence number */
	pan_db db = rule_db_open ();
	rule->unique_id = pan_db_get_value_i (db, SEQUENCE_PROPERTY);
	pan_db_add_value_i (db, SEQUENCE_PROPERTY, 1);
	pan_db_close (db);

	/* save  */
	rule_db_save ( rule );
}

static void
rule_db_delete (
	rule_t* rule )
{
	pan_db db = rule_db_open ();
	char key[16];
	sprintf (key, "%d", rule->unique_id);
	pan_db_erase (db, key);
	pan_db_close (db);
}

/**
*** UTIL
**/

static void
rule_clean (
	rule_t* rule)
{
	g_free (rule->author);
	rule->author = NULL;
	g_free (rule->subject);
	rule->subject = NULL;
	g_free (rule->group);
	rule->group = NULL;
	g_free (rule->thread);
	rule->thread = NULL;

	rule->crosspost_qty = 0;
	rule->lifespan_days = 0;
	rule->onmatch_action = MESSAGE_FILTER_ACTION_NONE;
}

static void
rule_deserialize (
	rule_t* rule,
	const gchar* key,
	const gchar* val)
{
        char** sd = g_strsplit (val, "\n", -1);

	rule->unique_id = (int) strtol (key, NULL, 10);
	rule->subject = g_strdup(sd[0]);
	rule->author = g_strdup(sd[1]);
	rule->group = g_strdup(sd[2]);
	rule->thread = g_strdup(sd[3]);
	rule->onmatch_action = (MessageFilterAction) atoi(sd[4]);
	rule->lifespan_days = atoi(sd[5]);

	g_strfreev (sd);
}

/**
*** INIT
**/

static void
message_filter_init_foreach (
	const char* key,
	const char* val,
	gpointer user_data )
{
	if ( strcmp ( SEQUENCE_PROPERTY, key ) ) 
	{
		rule_t* rule = g_new0 (rule_t, 1);
		rule_deserialize (rule, key, val);
		_rules = g_slist_prepend (_rules, rule);
	}
}

static void
message_filter_init (void)
{
	if (!_inited)
	{
		pan_db db = rule_db_open ();
		pan_db_foreach (db, message_filter_init_foreach, NULL);
		pan_db_close (db);

		_inited = TRUE;
	}
}

/**
*** UI
**/

static rule_t*
rule_ui_list_get_selected_rule (
	MessageFilterUI* ui )
{
	rule_t* returnme = NULL;
	GList* sel = GTK_CLIST(ui->rules_list)->selection;

	if ( sel )
	{
		int row_index = GPOINTER_TO_INT ( sel->data );
		g_return_val_if_fail ( row_index>=0, NULL ); /* no data */
		returnme = (rule_t*) gtk_clist_get_row_data ( GTK_CLIST(ui->rules_list), row_index );
	}

	return returnme;
}

static void
edit_button_clicked (
	GtkButton* button,
	gpointer user_data )
{
	MessageFilterUI* ui = (MessageFilterUI*) user_data;
	rule_t* rule = rule_ui_list_get_selected_rule ( ui );
	if ( rule != NULL )
	{
		rule_to_ui ( rule, ui );
	}
}

static gchar**
rule_ui_list_get_text (
	const rule_t* rule )
{
	gchar** ppch = g_new0 (gchar*, 8);
	const gchar* pch = NULL;
	const gchar* any = _("[Any]");
	int i = 0;

	ppch[i++] = g_strdup (*rule->author ? rule->author : any);

	ppch[i++] = g_strdup (*rule->subject ? rule->subject : any);

	ppch[i++] = rule->crosspost_qty
			? g_strdup_printf ("%d", rule->crosspost_qty)
			: g_strdup (any);

	ppch[i++] = g_strdup (*rule->thread ? rule->thread : any);

	ppch[i++] = g_strdup (*rule->group ? rule->group : any);

	switch ( rule->lifespan_days ) {
		case 1: pch = _("Session"); break;
		case -1: pch = _("Forever");  break;
		default: pan_warn_if_reached(); pch = _("BUG IN CODE"); break;
	}
	ppch[i++] = g_strdup (pch);

	if (rule->onmatch_action == MESSAGE_FILTER_ACTION_KILL)
		pch = _("Kill");
	else if (rule->onmatch_action == MESSAGE_FILTER_ACTION_WATCH)
		pch = _("Track");
	else {
		pan_warn_if_reached();
		pch = _("BUG IN CODE");
	}
	ppch[i++] = g_strdup ( pch );

	/* end */
	ppch[i++] = NULL;
	return ppch;
}

static void
rule_ui_list_build (
	MessageFilterUI* ui )
{
	GtkCList* list = GTK_CLIST ( ui->rules_list );
	GSList* l;

	gtk_clist_freeze ( list );
	gtk_clist_clear ( list );
	for ( l=_rules; l!=NULL; l=l->next )
		rule_ui_list_row_add ( ui, (const rule_t*)l->data );
	gtk_clist_thaw ( list );
}

static void
rule_ui_list_row_add (
	MessageFilterUI* ui,
	const rule_t* rule )
{
	GtkCList* list = GTK_CLIST ( ui->rules_list );

	/* add the row */
	gchar** ppch = rule_ui_list_get_text ( rule );
	int row = gtk_clist_append ( list, ppch );
	gtk_clist_set_row_data ( list, row, (gpointer)rule );

	/* cleanup */
	g_strfreev ( ppch );
}

static void
rule_ui_list_row_refresh (
	MessageFilterUI* ui,
	const rule_t* rule )
{
	GtkCList* list = GTK_CLIST ( ui->rules_list );
	gchar** ppch = rule_ui_list_get_text ( rule );
	gint row = gtk_clist_find_row_from_data ( list, (gpointer)rule );
	int i;

	gtk_clist_freeze (list);
	for (i=0; i!=cols; ++i)
		gtk_clist_set_text (list, row, i, ppch[i]);
	gtk_clist_thaw ( list );

	g_strfreev ( ppch );
}

static void
save_new_button_clicked (
	GtkButton* button,
	gpointer user_data )
{
	MessageFilterUI* ui = (MessageFilterUI*) user_data;
	rule_t* rule = g_new0 ( rule_t, 1 );
	ui_to_rule ( ui, rule );

	rule_db_save_new ( rule );
	ui->changes_made = TRUE;
	_rules = g_slist_prepend ( _rules, rule );
	rule_ui_list_row_add ( ui, rule );
}

static void
save_change_button_clicked (
	GtkButton* button,
	gpointer user_data )
{
	MessageFilterUI* ui = (MessageFilterUI*) user_data;
	rule_t* rule = rule_ui_list_get_selected_rule ( ui );
	if (rule != NULL)
	{
		ui_to_rule (ui, rule);
		rule_db_save (rule);
		rule_ui_list_row_refresh (ui, rule);
		ui->changes_made = TRUE;
	}
}

static void
delete_button_clicked (
	GtkButton* button,
	gpointer user_data )
{
	MessageFilterUI* ui = (MessageFilterUI*) user_data;
	rule_t* rule = rule_ui_list_get_selected_rule ( ui );
	if (rule != NULL)
	{
		int list_row;
		rule_db_delete (rule);
		_rules = g_slist_remove (_rules, rule);
		list_row = gtk_clist_find_row_from_data (
			GTK_CLIST(ui->rules_list), (gpointer)rule);
		gtk_clist_remove (GTK_CLIST(ui->rules_list), list_row);
		ui->changes_made = TRUE;

		/* memory cleanup */
		rule_clean (rule);
		g_free (rule);
	}
}

static void
rules_column_clicked (
	GtkCList* clist,
	int n )
{
	if ( clist->sort_column == n ) /* flip sort types on double click */
		clist->sort_type = clist->sort_type==GTK_SORT_ASCENDING
			? GTK_SORT_DESCENDING
			: GTK_SORT_ASCENDING;
	gtk_clist_set_sort_column ( clist, n );
	gtk_clist_sort ( clist );
}

static void
rules_selection_changed (
	GtkCList* clist,
	gint row,
	gint column,
	GdkEventButton* event,
	gpointer user_data )
{
	update_controls ( (MessageFilterUI*)user_data );
}

static GtkWidget*
create_filter_page (
	MessageFilterUI* ui )
{
	GtkWidget* bbox = NULL;
	GtkWidget* vbox = gtk_vbox_new (FALSE, 5);
	GtkWidget* rule_box = gtk_vbox_new (FALSE, 5);
	GtkWidget* swind = NULL;
	gchar* rule_list_titles [7];
       
	/* set the titles one-by-one so that gettext() can work them */
	rule_list_titles[0] = _("Author");
	rule_list_titles[1] = _("Subject");
	rule_list_titles[2] = _("Crosspost");
	rule_list_titles[3] = _("Thread");
	rule_list_titles[4] = _("Group");
	rule_list_titles[5] = _("Duration");
	rule_list_titles[6] = _("Action");

	/* rules list */
	ui->rules_list = gtk_clist_new_with_titles (cols, rule_list_titles);
        gtk_clist_set_column_width (GTK_CLIST(ui->rules_list), 0, 120);
        gtk_clist_set_column_width (GTK_CLIST(ui->rules_list), 1, 120 );
        gtk_clist_set_column_width (GTK_CLIST(ui->rules_list), 2, 70);
        gtk_clist_set_column_width (GTK_CLIST(ui->rules_list), 3, 70);
        gtk_clist_set_column_width (GTK_CLIST(ui->rules_list), 4, 100);
        gtk_clist_set_column_width (GTK_CLIST(ui->rules_list), 5, 90);
	swind = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(swind), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_widget_set_usize (swind, -1, 200);
	gtk_container_add (GTK_CONTAINER(swind), ui->rules_list);
	gtk_box_pack_start ( GTK_BOX(vbox), swind, TRUE, TRUE, 0  );
	gtk_signal_connect ( GTK_OBJECT(ui->rules_list), "click_column",
		GTK_SIGNAL_FUNC(rules_column_clicked), NULL );
	gtk_signal_connect ( GTK_OBJECT(ui->rules_list), "select-row",
		GTK_SIGNAL_FUNC(rules_selection_changed), (gpointer)ui );
	gtk_signal_connect ( GTK_OBJECT(ui->rules_list), "unselect-row",
		GTK_SIGNAL_FUNC(rules_selection_changed), (gpointer)ui );
	
	/* edit buttons */
	bbox = gtk_hbutton_box_new ( );
	/* edit button */
        ui->edit_pixmap = gnome_pixmap_new_from_xpm_d (arrow_down_xpm);
	ui->edit_button = gnome_pixmap_button ( ui->edit_pixmap, _("Edit"));
        gtk_signal_connect (GTK_OBJECT (ui->edit_button), "clicked", GTK_SIGNAL_FUNC(edit_button_clicked), ui);
	gtk_container_add (GTK_CONTAINER(bbox), ui->edit_button);
	/* save change button */
        ui->save_change_pixmap = gnome_pixmap_new_from_xpm_d (arrow_up_xpm);
	ui->save_change_button = gnome_pixmap_button (ui->save_change_pixmap, _("Save"));
        gtk_signal_connect (GTK_OBJECT (ui->save_change_button), "clicked", GTK_SIGNAL_FUNC(save_change_button_clicked), ui);
	gtk_container_add (GTK_CONTAINER(bbox), ui->save_change_button);
	/* save new button */
        ui->save_new_pixmap = gnome_pixmap_new_from_xpm_d (arrow_up_create_xpm);
	ui->save_new_button = gnome_pixmap_button (ui->save_new_pixmap, _("New"));
        gtk_signal_connect (GTK_OBJECT (ui->save_new_button), "clicked", GTK_SIGNAL_FUNC(save_new_button_clicked), ui);
	gtk_container_add ( GTK_CONTAINER(bbox), ui->save_new_button );
	/* delete button */
	ui->delete_pixmap = gnome_pixmap_new_from_xpm_d (delete_xpm);
	ui->delete_button = gnome_pixmap_button (ui->delete_pixmap, _("Delete"));
        gtk_signal_connect (GTK_OBJECT (ui->delete_button), "clicked", GTK_SIGNAL_FUNC(delete_button_clicked), ui);
	gtk_container_add (GTK_CONTAINER(bbox), ui->delete_button);
	gtk_box_pack_start (GTK_BOX(vbox), bbox, FALSE, TRUE, 0);

	/* edit window */
        gtk_box_pack_start (GTK_BOX(rule_box), create_rule_edit_page(ui), FALSE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), rule_box, FALSE, TRUE, 0);

	update_controls ( ui );

	return vbox;
}

static void
update_controls (
	MessageFilterUI* ui )
{
	const char* NO_RULE_SELECTED = _("  (Disabled: No Rule Selected)");
	const rule_t* selected_rule = rule_ui_list_get_selected_rule (ui);
	gboolean sensitive;
	gchar *tip=NULL, *tmp=NULL;

	/* edit button */
	sensitive = TRUE;
	tip = g_strdup (_("Edit Rule"));
	if (!selected_rule) {
		sensitive = FALSE;
		tmp = tip;
		tip = g_strconcat (tip, NO_RULE_SELECTED, NULL);
		g_free (tmp);
	}
	//gtk_widget_set_sensitive ( GTK_WIDGET(ui->edit_button), sensitive );
	gtk_tooltips_set_tip (ui->tips, ui->edit_button, tip, NULL);
	g_free (tip);
	
	/* save changed button */
	sensitive = TRUE;
	tip = g_strdup (_("Save Changed Rule"));
	if (!selected_rule) {
		sensitive = FALSE;
		tmp = tip;
		tip = g_strconcat (tip, NO_RULE_SELECTED, NULL);
		g_free (tmp);
	}
	//gtk_widget_set_sensitive ( GTK_WIDGET(ui->save_change_button), sensitive );
	gtk_tooltips_set_tip (ui->tips, ui->save_change_button, tip, NULL);
	g_free (tip);

	/* save new button */
	sensitive = TRUE;
	tip = g_strdup (_("Save New Rule"));
	//gtk_widget_set_sensitive ( GTK_WIDGET(ui->save_new_button), sensitive );
	gtk_tooltips_set_tip ( ui->tips, ui->save_new_button, tip, NULL );
	g_free ( tip );

	/* delete button */
	sensitive = TRUE;
	tip = g_strdup (_("Delete Selected Rule"));
	if ( !selected_rule ) {
		sensitive = FALSE;
		tmp = tip;
		tip = g_strconcat ( tip, NO_RULE_SELECTED, NULL );
		g_free ( tmp );
	}
	//gtk_widget_set_sensitive ( GTK_WIDGET(ui->delete_button), sensitive );
	gtk_tooltips_set_tip ( ui->tips, ui->delete_button, tip, NULL );
	g_free ( tip );
}

static void
togglebutton_changed_cb (
	GtkToggleButton* tb,
	gpointer user_data )
{
	update_controls ( (MessageFilterUI*)user_data );
}

static void
editable_changed_cb (
	GtkEditable* editable,
	gpointer user_data )
{
	update_controls ( (MessageFilterUI*)user_data );
}


static GtkWidget*
create_rule_edit_page (
	MessageFilterUI* ui )
{
	GtkWidget* vbox = gtk_vbox_new ( FALSE, 5 );
	GtkWidget* hbox;
	GtkWidget* frame;

	/* action to take */
	hbox = gtk_hbox_new (FALSE, 0);
	ui->kill_rb = gtk_radio_button_new_with_label (NULL, _("Kill"));
	gtk_box_pack_start (GTK_BOX(hbox), ui->kill_rb, FALSE, FALSE, 0 );
	ui->track_rb = gtk_radio_button_new_with_label_from_widget (
		GTK_RADIO_BUTTON(ui->kill_rb), _("Track"));
	gtk_box_pack_start (GTK_BOX(hbox), ui->track_rb, FALSE, FALSE, 0 );
	gtk_box_pack_start (GTK_BOX(hbox),
		gtk_label_new (_("articles which are:")),
		FALSE, FALSE, 0 );
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);


	/* crosspost threshold */
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox),
		ui->crosspost_tb = gtk_check_button_new_with_label (_("Posted to more than")),
		FALSE, FALSE, 0);
	ui->crosspost_data = GTK_ADJUSTMENT (gtk_adjustment_new (0, 1, 999, 1, 1, 30));
	ui->crosspost_sb = gtk_spin_button_new (ui->crosspost_data, 1, 0 );
	gtk_box_pack_start (GTK_BOX(hbox), ui->crosspost_sb, FALSE, FALSE, 0 );
	gtk_box_pack_start (GTK_BOX(hbox), gtk_label_new (_("groups")), FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
        gtk_signal_connect (
		GTK_OBJECT(ui->crosspost_tb), "toggled",
		GTK_SIGNAL_FUNC(togglebutton_changed_cb), ui);
        gtk_signal_connect (
		GTK_OBJECT(ui->crosspost_sb), "changed",
		GTK_SIGNAL_FUNC(editable_changed_cb), ui);

	/* thread */
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox),
		ui->thread_tb = gtk_check_button_new_with_label (
			_("The Thread this Article is in")),
		FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox),
		ui->thread_entry = gtk_entry_new (),
		TRUE, TRUE, 0);
        gtk_signal_connect (
		GTK_OBJECT(ui->thread_tb), "toggled",
		GTK_SIGNAL_FUNC(togglebutton_changed_cb), ui);
        gtk_signal_connect (
		GTK_OBJECT(ui->thread_entry), "changed",
		GTK_SIGNAL_FUNC(editable_changed_cb), ui);
	gtk_box_pack_start (
		GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	/* author */
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox),
		ui->author_tb = gtk_check_button_new_with_label (
			_("Written by")),
		FALSE, FALSE, 0 );
	gtk_box_pack_start (GTK_BOX(hbox),
		ui->author_entry = gtk_entry_new ( ),
		TRUE, TRUE, 0 );
	gtk_tooltips_set_tip (ui->tips, ui->author_entry,
		_("The article's author's name goes here. "
		  "By default this is an exact comparison, however, "
		  "wildcards are allowed anywhere in the line."
		  "\n"
		  "Sometimes an author won't get filtered out because "
		  "the rule is specified as \"daverhodes@first.spam.site.com <Dave Rhodes>\" "
		  "but an message is posted from \"daverhodes@second.spam.site.com <Dave Rhodes>\". "
		  "In such cases, specifying \"*Dave Rhodes*\" as the author does The Right Thing."),
		NULL);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
        gtk_signal_connect (
		GTK_OBJECT(ui->author_tb), "toggled",
		GTK_SIGNAL_FUNC(togglebutton_changed_cb), ui);
        gtk_signal_connect (
		GTK_OBJECT(ui->author_entry), "changed",
		GTK_SIGNAL_FUNC(editable_changed_cb), ui);

	/* subject */
	hbox = gtk_hbox_new ( FALSE, 0 );
	gtk_box_pack_start ( GTK_BOX(hbox),
		ui->subject_tb = gtk_check_button_new_with_label (
			_("When Subject Matches")),
		FALSE, FALSE, 0 );
	gtk_box_pack_start ( GTK_BOX(hbox),
		ui->subject_entry = gtk_entry_new ( ),
		TRUE, TRUE, 0 );
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0 );
        gtk_signal_connect (
		GTK_OBJECT(ui->subject_tb), "toggled",
		GTK_SIGNAL_FUNC(togglebutton_changed_cb), ui );
        gtk_signal_connect (
		GTK_OBJECT(ui->subject_entry), "changed",
		GTK_SIGNAL_FUNC(editable_changed_cb), ui );

	/* group */
	hbox = gtk_hbox_new ( FALSE, 0 );
	gtk_box_pack_start ( GTK_BOX(hbox),
		ui->group_tb = gtk_check_button_new_with_label (
			_("In Group")),
		FALSE, FALSE, 0 );
	gtk_box_pack_start ( GTK_BOX(hbox),
		ui->group_entry = gtk_entry_new ( ),
		TRUE, TRUE, 0 );
	gtk_box_pack_start ( GTK_BOX(vbox), hbox, FALSE, FALSE, 0 );
        gtk_signal_connect ( GTK_OBJECT(ui->group_tb), "toggled", GTK_SIGNAL_FUNC(togglebutton_changed_cb), ui );
        gtk_signal_connect ( GTK_OBJECT(ui->group_entry), "changed", GTK_SIGNAL_FUNC(editable_changed_cb), ui );

	/* lifespan */
	hbox = gtk_hbox_new ( FALSE, 0 );
	gtk_box_pack_start ( GTK_BOX(hbox),
		ui->this_session_rb = gtk_radio_button_new_with_label (
			NULL, _("This Session")),
		FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(hbox),
		ui->forever_rb = gtk_radio_button_new_with_label_from_widget (
			GTK_RADIO_BUTTON(ui->this_session_rb), _("Forever")),
		FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
        gtk_signal_connect (GTK_OBJECT(ui->this_session_rb), "toggled", GTK_SIGNAL_FUNC(togglebutton_changed_cb), ui);
        gtk_signal_connect (GTK_OBJECT(ui->forever_rb), "toggled", GTK_SIGNAL_FUNC(togglebutton_changed_cb), ui);

	/* signals */

	frame = gtk_frame_new (_("Article Filter Rule"));
	gtk_container_add ( GTK_CONTAINER(frame), vbox );

	return frame;
}


static void
rule_to_ui (
	const rule_t* rule,
	MessageFilterUI* ui )
{
	/* subject */
	gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON(ui->subject_tb), *rule->subject );
	gtk_entry_set_text ( GTK_ENTRY(ui->subject_entry), rule->subject );

	/* author */
	gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON(ui->author_tb), *rule->author );
	gtk_entry_set_text ( GTK_ENTRY(ui->author_entry), rule->author );

	/* group */
	gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON(ui->group_tb), *rule->group );
	gtk_entry_set_text ( GTK_ENTRY(ui->group_entry), rule->group );

	/* lifespan */
	switch ( rule->lifespan_days ) {
		case 1: gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON(ui->this_session_rb), TRUE ); break;
		case -1: gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON(ui->forever_rb), TRUE ); break;
		default: pan_warn_if_reached ();
	}

	/* crosspost */
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(ui->crosspost_tb),
		rule->crosspost_qty!=0);
	gtk_spin_button_set_value (
		GTK_SPIN_BUTTON(ui->crosspost_sb),
		(gfloat)rule->crosspost_qty);

	/* thread */
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON(ui->thread_tb),
		*rule->thread);

	/* onmatch */
	gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON(ui->kill_rb),
		rule->onmatch_action == MESSAGE_FILTER_ACTION_KILL );
	gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON(ui->track_rb),
		rule->onmatch_action == MESSAGE_FILTER_ACTION_WATCH );
}

static void
ui_to_rule (
	const MessageFilterUI* ui,
	rule_t* rule )
{
	rule_clean (rule);

	/* subject */
	rule->subject = g_strdup (
		gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(ui->subject_tb) )
		? gtk_entry_get_text ( GTK_ENTRY(ui->subject_entry) )
		: "" );

	/* author */
	rule->author = g_strdup (
		gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(ui->author_tb) )
		? gtk_entry_get_text ( GTK_ENTRY(ui->author_entry) )
		: "" );

	/* group */
	rule->group = g_strdup (
		gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(ui->group_tb) )
		? gtk_entry_get_text ( GTK_ENTRY(ui->group_entry) )
		: "" );

	/* crosspost qty */
	rule->crosspost_qty = 
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(ui->crosspost_tb))
		? gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(ui->crosspost_sb))
		: 0;

	/* thread */
	rule->thread = g_strdup (
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(ui->thread_tb))
		? gtk_entry_get_text (GTK_ENTRY(ui->thread_entry))
		: "");

	/* lifespan days */
	if ( gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(ui->this_session_rb) ) )
		rule->lifespan_days = 1;
	else // forever
		rule->lifespan_days = -1;

	/* action */
	rule->onmatch_action = MESSAGE_FILTER_ACTION_NONE;
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(ui->kill_rb)))
		rule->onmatch_action = MESSAGE_FILTER_ACTION_KILL;
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(ui->track_rb)))
		rule->onmatch_action = MESSAGE_FILTER_ACTION_WATCH;
}

static void
dialog_clicked_cb (
	GnomeDialog* dialog,
	gint button_index,
	gpointer user_data )
{
	MessageFilterUI* ui = (MessageFilterUI*) user_data;
	if (button_index==0)
	{
                gtk_widget_destroy (GTK_WIDGET(ui->dialog));
	}
}

static void
dialog_destroy_cb (
	GtkObject* object,
	gpointer user_data )
{
	if (_ui->changes_made)
		articlelist_queue_reload();

        gtk_widget_destroy (GTK_WIDGET(_ui->dialog));
	g_free (_ui);
        _ui = NULL;
}
