/*
** Copyright (C) 10 Feb 1999 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <stdlib.h>
#include <string.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>

#include "common_gtk.h"
#include "config_files.h"
#include "vector_commands.h"
#include "cdrecord_options.h"
#include "command.h"
#include "rip_audio.h"
#include "cdlabel.h"
#include "linebuffer.h"
#include "preferences.h"
#include "mainwindow.h"
#include "globals.h"

/* I expect to need to write to several config files, might as well use
 * a whole dir for gcombust */

#define GCOMBUST_CONFIG_DIR ".gcombust"
#define GCOMBUST_CONFIG_FILE_NAME "gcombustrc"

static void read_env(void) {
	char *s, *end;
	long int l;

	if ((s = getenv("CDR_SPEED")) != NULL) {
		l = strtol(s, (char **)NULL, 10);
		switch (l) {
			case 1: 
				gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(speed_x[0]), TRUE);
			break;
			case 2:
				gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(speed_x[1]), TRUE);
			break;
			case 4:
				gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(speed_x[2]), TRUE);
			break;
			case 6:
				gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(speed_x[3]), TRUE);
			break;
			case 8:
				gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(speed_x[4]), TRUE);
			break;
			default:
			break;
		}
	}
	if ((s = getenv("CDR_FIFOSIZE")) != NULL) {
		gtk_entry_set_text(GTK_ENTRY(fsentry), s);
	}
/* only handles the dev=(x,),y,z format, in a kind of ugly way */
	if ((s = getenv("CDR_DEVICE")) !=NULL) {
		l = strtol(s, &end, 10);
		if (strchr(s, ',') == strrchr(s, ',')) {  /* 1,0 */
			gtk_spin_button_set_value(
					GTK_SPIN_BUTTON(mainptr->cdr_option_info->scsibus), 0);
			end = s - 1;
		} else /* 0,1,0 */
			gtk_spin_button_set_value(
					GTK_SPIN_BUTTON(mainptr->cdr_option_info->scsibus), l);
		if ('\0' != *end) {
			s = end;
			l = strtol(s + 1, &end, 10);
			gtk_spin_button_set_value(
					GTK_SPIN_BUTTON(mainptr->cdr_option_info->target), l);
		}
		if ('\0' != *end) {
			s = end;
			l = strtol(s + 1, &end, 10);
			gtk_spin_button_set_value(
					GTK_SPIN_BUTTON(mainptr->cdr_option_info->lun), l);
		}
	}

	cdda_device_path = getenv("CDDA_DEVICE");
}

static char *get_config_dir(void) {
	char *config_dir, *config;

	if (NULL == (config_dir = getenv("HOME")))
		config_dir = ".";

	config = string_append(config_dir, NULL);
	config = string_append(config, "/");
	config = string_append(config, GCOMBUST_CONFIG_DIR);

	return config;
}

static char *get_config_file(void) {
	char *config;

	config = get_config_dir();
	config = string_append(config, "/");
	config = string_append(config, GCOMBUST_CONFIG_FILE_NAME);

	return config;
}

static void set_mkisofs_path(char *path) {
	g_free(mkisofs_path);
	mkisofs_path = g_strdup(path);
}
static void set_mkhybrid_path(char *path) {
	g_free(mkhybrid_path);
	mkhybrid_path = g_strdup(path);
}
static void set_cdrecord_path(char *path) {
	g_free(cdrecord_path);
	cdrecord_path = g_strdup(path);
}
static void set_cdlabelgen_path(char *path) {
	g_free(cdlabelgen_path);
	cdlabelgen_path = g_strdup(path);
}
static void set_diff_path(char *path) {
	g_free(diff_path);
	diff_path = g_strdup(path);
}
static void set_cdda2wav_path(char *path) {
	g_free(cdda2wav_path);
	cdda2wav_path = g_strdup(path);
}
static void set_iso_location(char *path) {
	g_free(def_image_path);
	def_image_path = g_strdup(path);
}
static void set_rip_location(char *path) {
	g_free(def_rip_path);
	def_rip_path = g_strdup(path);
}
static void set_enable_tooltips(char *path) {
	if (*(path) == '1')
		misc_prefs.enable_tooltips = TRUE;
	else
		misc_prefs.enable_tooltips = FALSE;
}
static void set_own_progresswin(char *path) {
	if (*(path) == '1')
		misc_prefs.own_progresswin = TRUE;
	else
		misc_prefs.own_progresswin = FALSE;
}
static void set_scsibus(char *path) {
	gtk_spin_button_set_value(
			GTK_SPIN_BUTTON(mainptr->cdr_option_info->scsibus), atoi(path));
}
static void set_target(char *path) {
	gtk_spin_button_set_value(
			GTK_SPIN_BUTTON(mainptr->cdr_option_info->target), atoi(path));
}
static void set_lun(char *path) {
	gtk_spin_button_set_value(
			GTK_SPIN_BUTTON(mainptr->cdr_option_info->lun), atoi(path));
}
static void set_driver(char *path) {
	gtk_entry_set_text(GTK_ENTRY(mainptr->cdr_option_info->driver), path);
}
static void set_device(char *path) {
	gtk_entry_set_text(GTK_ENTRY(mainptr->cdr_option_info->device), path);
}
static void set_swab(char *path) {
	if (*(path) == '1')
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mainptr->cdr_option_info->swab_audio), TRUE);
	else
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mainptr->cdr_option_info->swab_audio), FALSE);
}

static void set_speed(char *s) {
	int i;
	for (i = 0; i < NO_SPEED; i++) {
		if (*(s) == *(speed_string[i]))
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(speed_x[i]), TRUE);
	}
}
static void set_dummy(char *path) {
	if (*(path) == '1')
		misc_prefs.dummy = TRUE;
	else
		misc_prefs.dummy = FALSE;
}
static void set_eject(char *path) {
	if (*(path) == '1')
		misc_prefs.eject = TRUE;
	else
		misc_prefs.eject = FALSE;
}

static struct {
	char *string;
	void (*function)(char *arg);
} string_function[] = {
	{"mkisofs_path=", set_mkisofs_path},
	{"mkhybrid_path=", set_mkhybrid_path},
	{"cdrecord_path=", set_cdrecord_path},
	{"cdlabelgen_path=", set_cdlabelgen_path},
	{"gdiff_path=", set_diff_path},
	{"cdda2wav_path=", set_cdda2wav_path},
	{"iso_location=", set_iso_location},
	{"rip_location=", set_rip_location},
	{"enable_tooltips=", set_enable_tooltips},
	{"own_progresswin=", set_own_progresswin},
	{"dev_scsibus=", set_scsibus},
	{"dev_target=", set_target},
	{"dev_lun=", set_lun},
	{"dev_driver=", set_driver},
	{"dev_device=", set_device},
	{"dev_swab=", set_swab},
	{"dev_speed=", set_speed},
	{"dummy=", set_dummy},
	{"eject=", set_eject}
};

/* This is loosely based on an old which implementation by Paul Vixie
 * - license unclear, but it's not a plain copy so there shouldn't be
 * much of a problem anyway */
char *which(char* name) {
	char    test[256], *pc, save;
	int     len;

	pc = getenv("PATH");

	if (pc == NULL) {
		return g_strdup("NOT_FOUND");
	}

	while (*pc != '\0') {
		len = 0;
		while (*pc != ':' && *pc != '\0') {
			len++;
			pc++;
		}

		save = *pc;
		*pc = '\0';
		g_snprintf(test, 256, "%s/%s", pc-len, name);
		*pc = save;
		if (*pc)
			pc++;

		if (0 == access(test, X_OK))
			return (g_strdup(test));
	}

	return g_strdup("NOT_FOUND");
}


void read_config_file(void) {
	FILE *fp;
	char line[BUF_S];
	linebuffer file;
	char *config;

	read_env();

	if (NULL == mkisofs_path)
		mkisofs_path = which("mkisofs");
	if (NULL == mkhybrid_path)
		mkhybrid_path = which("mkhybrid");
	if (NULL == cdrecord_path)
		cdrecord_path = which("cdrecord");
	if (NULL == cdlabelgen_path)
		cdlabelgen_path = which("cdlabelgen");
	if (NULL == diff_path)
		diff_path = which("diff");
	if (NULL == cdda2wav_path)
		cdda2wav_path = which("cdda2wav");

	config = get_config_file();

	if (NULL == (fp = fopen(config, "r"))) {
		alert_user_of_error(_(" Could not open config file.\n"
					" Please configure gcombust in the File/Preferences menu \n"
					" (Preferences window opened, at least the path to \n"
					" mkisofs or mkhybrid need to be filled in) "));
		preferences_cmd_callback(NULL, NULL);
		free(config);
		return;
	}
	free(config);

	init_buffer(&file);

	while (NULL != fgets(line, BUF_S, fp)) {
		add_to_buffer(&file, line);
	}

	fclose(fp);
	while (TRUE == extract_line(&file, "\n")) {
		int i;
		file.newline[strlen(file.newline)-1] = '\0'; /* \n */
		for (i = 0; i < sizeof(string_function)/sizeof(string_function[0]); i++) {
			if (strlen(file.newline) > strlen(string_function[i].string)) {
				if (!(strncmp(string_function[i].string, file.newline,
								strlen(string_function[i].string)))) {
					string_function[i].function(
							file.newline + strlen(string_function[i].string));
				}
			}
		}
	}
	apply_misc_changes();
	apply_storage_changes();

	destroy_buffer(&file);
}

void save_config_file(config_cdr_data *cdr_info) {
	FILE *fp;
	char *config;
	int i;

	config = get_config_dir();
	mkdir(config, S_IRWXU | S_IRWXG);
	free(config);

	config = get_config_file();
	if (NULL == (fp = fopen(config, "w"))) {
		alert_user_of_error(_(" Could not open config file for writing! \n"));
		free(config);
		return;
	}
	free(config);

	fprintf(fp, "mkisofs_path=%s\n", mkisofs_path);
	fprintf(fp, "mkhybrid_path=%s\n", mkhybrid_path);
	fprintf(fp, "cdrecord_path=%s\n", cdrecord_path);
	fprintf(fp, "cdlabelgen_path=%s\n", cdlabelgen_path);
	fprintf(fp, "gdiff_path=%s\n", diff_path);
	fprintf(fp, "cdda2wav_path=%s\n", cdda2wav_path);
	fprintf(fp, "iso_location=%s\n", def_image_path);
	fprintf(fp, "rip_location=%s\n", def_rip_path);
	if (misc_prefs.enable_tooltips)
		fprintf(fp, "enable_tooltips=1\n");
	else
		fprintf(fp, "enable_tooltips=0\n");
	if (misc_prefs.own_progresswin)
		fprintf(fp, "own_progresswin=1\n");
	else
		fprintf(fp, "own_progresswin=0\n");
	if (misc_prefs.dummy)
		fprintf(fp, "dummy=1\n");
	else
		fprintf(fp, "dummy=0\n");

	if (misc_prefs.eject)
		fprintf(fp, "eject=1\n");
	else
		fprintf(fp, "eject=0\n");

	fprintf(fp, "dev_scsibus=%i\n", gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cdr_info->scsibus)));
	fprintf(fp, "dev_target=%i\n", gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cdr_info->target)));
	fprintf(fp, "dev_lun=%i\n", gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(cdr_info->lun)));
	fprintf(fp, "dev_driver=%s\n", gtk_entry_get_text(GTK_ENTRY(cdr_info->driver)));
	fprintf(fp, "dev_device=%s\n", gtk_entry_get_text(GTK_ENTRY(cdr_info->device)));

	if (GTK_TOGGLE_BUTTON(cdr_info->swab_audio)->active)
		fprintf(fp, "dev_swab=1\n");
	else
		fprintf(fp, "dev_swab=0\n");

	for (i = 0; i < NO_SPEED; i++) {
		if (GTK_TOGGLE_BUTTON(prefs_speed[i])->active)
			fprintf(fp, "dev_speed=%s\n", speed_string[i]);
	}

	fclose(fp);
}

