# Module: GetHardwareInfo.tcl 
# 12.11.96 tn
# 24.1.96 T.Niederreiter
# 
# External called funktions:
# - getscsidevices{}: fills the array scsidevices()
#   Elements of scsidevices(x,y):
#     gendevice,blockdevice,type,rmb,vendor,model,rev,date,typename,#,id-list
#
# - getidedevices{}: fills the array idedevices()
#   Elements of idedevices(x,y):
#     blockdevice,type,model,fwrev,serno,#
#
# - getdiskinfo{} : fills the arry diskinfo()
#   Elements of diskinfo(x,y):
#     blockdevice,name-string,scsi-id (-1 = IDE)
# 	
# - getcddrives{} : fills the arry cdinfo()
#   (includes IDE and CD-Writers)
#   Elements of cdinfo(x,y):
#     blockdevice,name-string,generic-device ("" = IDE), host,scsi-id,lun (-1 = IDE)
# 	
# - getpartinfo{} : fills the array partinfo()
#   Elements of partinfo(x,y):
#     partition,size(in blocks),id,id-string,mountpoint
#
# - updatemountinfo { }: update internal mount-table 
#
# - getcdinfo { }: gets the TOC and label of a CD 
#


# Call the inq program, that outputs information about all connected
# SCSI-devices and format that information in the global array 
# scsidevices(i,j)
# Also calculates the names of the corresponding block-devices 
# (e.g. /dev/sda) from the generic devices.

proc getscsidevices {} {
global scsidevices
global scanscsi_number

	set scsihdnum 0
	set scsicdnum 0
	set charlist { a b c d e f g h i j k l m n }
	set scsidevtypes { 
		"Direct-Access    " "Sequential-Access" "Printer          "
		"Processor        " "WORM             " "CD-ROM           "
		"Scanner          " "Optical Device   " "Medium Changer   "
		"Communications   "
	} 

	set inqout "x"
	if { $scanscsi_number != 0 } {
		set inqout [scsiinq -s $scanscsi_number];# Call inquire
	}
	set inqlist [split $inqout "\n"] 	;# Convert to list 

	# remove last empty element
	if { [lindex $inqlist end] == "" } { 
			catch { set inqlist [lreplace $inqlist end end] }
	}

	# if the first char of the output is an "x" then no scsi-devices
	# were found.
	if { [lindex $inqout 0] == "x" } {
		set scsidevices(0,0) "x" 
		return -1
	}

	set i 0
	foreach l $inqlist {

		# get inquire-line
		set gendev [string range $l 0 [expr [string first ":" $l]-1]]
		set gendevnr [string range $l 7 [expr [string first ":" $l]-1]]
		set loopdev [string range $l [expr [string first ":" $l]+1] end]
		set devtype [string index $loopdev 0]

		# accept direct-access and optical-dev as target
		if { $devtype == "0" || $devtype == "7" } {
			set auxdev "/dev/sd[lindex $charlist $scsihdnum]"
			incr scsihdnum
		} \
		elseif { $devtype == "4" || $devtype == "5" } {
			set auxdev "/dev/sr$scsicdnum"
			incr scsicdnum
		} \
		else {
			set auxdev "x"		;# Not a special block-dev
		}

		# generic-device name
		set scsidevices($i,0) $gendev 
		# block-device name or x, if none
		set scsidevices($i,1) $auxdev
		# device-type-number
		set scsidevices($i,2) $devtype
		# device removeable? (0 or 1)
		set scsidevices($i,3) [string index $loopdev 1]   
		# Vendor-string
		set scsidevices($i,4) [string range $loopdev 2 9]
		# Model-string
		set scsidevices($i,5) [string range $loopdev 10 25]
		# Revision-string
		set scsidevices($i,6) [string range $loopdev 26 29]
		# Date-string
		set tmpdate [string range $loopdev 30 37]
		if { [if_printable $tmpdate] == 1 } {
			set scsidevices($i,7) $tmpdate
		} else {
			set scsidevices($i,7) "        " 
		}	
		# device-type-name
		if { $devtype != "" } {	
			set scsidevices($i,8) [lindex $scsidevtypes $devtype]
		} \
		else {
			set scsidevices($i,8) ""
		}
		# incr number if two (or more) exact same devices are connected
		set scsidevices($i,9) "1"
		for { set k 0 } { $k < $i } { incr k } {
			if { $scsidevices($k,4) == $scsidevices($i,4) 
			  && $scsidevices($k,5) == $scsidevices($i,5) } {
				incr scsidevices($i,9)
			}	
		}
		# id-list "host channel id lun"
		set scsidevices($i,10) [getscsiid $gendevnr]

		incr i
	}
	# mark end of scsi-list
	set scsidevices($i,0) "x" 

	# Now convert the temporary number into a nicer format
	# A "1" will become the empty string, any other number will get
	# a hash in front of it.

	set i 0
	while { $scsidevices($i,0) != "x" } {
		if { $scsidevices($i,9) == "1" } {
			set scsidevices($i,9) ""
		} \
		else {
			set scsidevices($i,9) "#$scsidevices($i,9)"
		}
		incr i
	}
}


# fill the global array idedevices with the information about the 
# connected ide-devices. (up to 8 supported)

proc getidedevices { } {
global idedevices
global scanide_number

	# this devices are probed 
	set hddevs { "/dev/hda" "/dev/hdb" "/dev/hdc" "/dev/hdd" 
		     "/dev/hde" "/dev/hdf" "/dev/hdg" "/dev/hdh" }

	set count 0
	set numscanned 0
	foreach i $hddevs {
		# scan only scanide_number devices
		if { $numscanned == $scanide_number } {
			break
		}

		set hdout [queryide $i]
		#set hdout "Model=st3120AT-Seagate 2.5\"  {new}             , FwRev=rev :.40, SerialNo=00KH7434430000000000, Type=10"
		if { $hdout != "" } {

			set model [string range $hdout 6 45]
			set rev [string range $hdout 54 61]
			set serno [string range $hdout 73 92]
			set type [string trim [string range $hdout 100 102]]

			#puts "$i:$type:$model:$rev:$serno:"

			set idedevices($count,0) $i
			set idedevices($count,1) $type
			set idedevices($count,2) $model
			set idedevices($count,3) $rev
			set idedevices($count,4) $serno

			# incr number when more than one identical devices...
			set idedevices($count,5) "1"
			for { set k 0 } { $k < $count } { incr k } {
				if { $idedevices($k,2) == 
					$idedevices($count,2) } {
					incr idedevices($count,5)
				}
			}

			incr count
		}	
		incr numscanned
	}	

	# mark end of list
	set idedevices($count,0)  "x"

        # Now convert the temporary number into a nicer format
        # A "1" will become the empty string, any other number will get
        # a hash in front of it.

        set i 0
        while { $idedevices($i,0) != "x" } {
                if { $idedevices($i,5) == "1" } {
                        set idedevices($i,5) ""
                } \
                else {
                        set idedevices($i,5) "#$idedevices($i,5)"
                }
                incr i
        }


	# return -1 when no ide devices were found
	if { $count == 0 } {
		return -1
	}
}


# fill the global array "diskinfo" with all available harddrives
# in the system

proc getdiskinfo { } {
global diskinfo
global scsidevices
global idedevices

	set count 0
	set i 0
	while { $scsidevices($i,0) != "x" } {
		# is this a harddisk or optical device?
		if { $scsidevices($i,2) == "0" || $scsidevices($i,2) == "7" } {
			set tmpstr1 $scsidevices($i,4)
                	set tmpstr2 $scsidevices($i,5)
                	set tmpstr3 $scsidevices($i,9)
                	set tmpstr4 "$tmpstr1 $tmpstr2$tmpstr3"
			
			set diskinfo($count,0) $scsidevices($i,1)
			set diskinfo($count,1) $tmpstr4
			set diskinfo($count,2) [lindex $scsidevices($i,10) 2]
			incr count
		}
		incr i
	}

	set i 0
	while { $idedevices($i,0) != "x" } {
		# is this a harddisk? 0 and 5 are cdrom, 1 is tape
		if { $idedevices($i,1) != 0 && $idedevices($i,1) != 1 && \
		     $idedevices($i,1) != 5 } {
		
			set diskinfo($count,0) $idedevices($i,0)
			set diskinfo($count,1) [string trim $idedevices($i,2)]
			set diskinfo($count,1) "$diskinfo($count,1) $idedevices($i,5)"
			set diskinfo($count,2) "-1"
			incr count
		}
		incr i	
	}

	# mark end of list
	set diskinfo($count,0) "x"
}


# fill the global array "cdinfo" with all available cd-drivers
# (readers and writers) in the system

proc getcddrives { } {
global cdinfo
global scsidevices
global idedevices

	set count 0
	set i 0
	while { $scsidevices($i,0) != "x" } {
		# is this a CD-Rom or CD-Writer? 
		if { $scsidevices($i,2) == 4 || $scsidevices($i,2) == 5 } {
			set tmpstr1 $scsidevices($i,4)
                	set tmpstr2 $scsidevices($i,5)
                	set tmpstr3 $scsidevices($i,9)
                	set tmpstr4 "$tmpstr1 $tmpstr2$tmpstr3"
			
			set cdinfo($count,0) $scsidevices($i,1)
			set cdinfo($count,1) $tmpstr4
			set cdinfo($count,2) $scsidevices($i,0)
			#set cdinfo($count,3) [lindex $scsidevices($i,10) 2]

			set host [string range [lindex $scsidevices($i,10) 0] 4 end]
			set id [lindex $scsidevices($i,10) 2]
			set lun [lindex $scsidevices($i,10) 3]
			set cdinfo($count,3) "$host,$id,$lun"
			incr count
		}
		incr i
	}

	set i 0
	while { $idedevices($i,0) != "x" } {
		# is this a cdrom? 0 and 5 are cdrom, 1 is tape
		if { $idedevices($i,1) == 0 || $idedevices($i,1) == 5 } {
		
			set cdinfo($count,0) $idedevices($i,0)
			set cdinfo($count,1) [string trim $idedevices($i,2)]
			set cdinfo($count,1) "$cdinfo($count,1) $idedevices($i,5)"
			set cdinfo($count,2) ""
			set cdinfo($count,3) "-1"
			incr count
		}
		incr i	
	}

	# mark end of list
	set cdinfo($count,0) "x"
}


# Calls "mount" to look if the partition devname is mounted at the moment.
# The output of mount is parsed and if the partition is found, its
# mountpoint is extracted and returned.

proc getmountpoint { devname } {
global MOUNT

	set point ""
	set mountout [exec $MOUNT]
	set mountlist [split $mountout "\n"]

	set devname [string trim $devname]

	foreach i $mountlist {
		set tmpdev [string trim [string range $i 0 9]]
		if { $devname == $tmpdev } {
			# The variable $i looks like this:
			# DEVICE on MOUNTPOINT type ext2 (rw)
			# we extract MOUNTPOINT
			set tmpindex1 [string first "on " $i]
		        incr tmpindex1 3 	;# index of 1st char of mount-path
			set tmpindex2 [string last " type" $i]
			incr tmpindex2 -1	;# index of last char

			set point [string range $i $tmpindex1 $tmpindex2]
		}
	}
	return $point
}


# scans /proc/scsi/scsi to get the scsi-id (and more) for a given
# scsi-device (e.g. "0" means /dev/sg0 )
# returns: host channel id lun

proc getscsiid { devnr } {

	set opnflg [catch { set fileid [open "/proc/scsi/scsi" r] }]

	if { $opnflg != 0 } {
		puts "Error: failed to parse /proc/scsi/scsi"
		exit 1	
	}

	set count 0
	set hitline ""

	# get the n-th line starting with "Host:" 
	while { [gets $fileid procline] >= 0 } {
		if { [string range $procline 0 4] == "Host:" } {
			if { $count == $devnr } {
				set hitline $procline
				break;
			}
			incr count			
		}
	}
	close $fileid

	set host [lindex $hitline 1]
	set channel [lindex $hitline 3]
	set id [lindex $hitline 5]
	set lun [lindex $hitline 7]

	return "$host $channel $id $lun"
}


# Get the partitiontable for each disk and fill
# the global array partinfo with that information
# Also handles IDE-disks
 
proc getpartinfo {} {
global scsidevices
global partinfo

	set partout [queryfdisk -d]
	set partlist [split $partout "\n"];

	set partindex 0
	foreach j $partlist {

		# check if this is really a device-name
		if { [string index $j 0] != "/" } {
			continue
		}

		# partition-device (e.g. /dev/sda1)
		set partinfo($partindex,0) [string trim [string range $j 0 10]]
		set reststr [string trim [string range $j 11 end]]
		set restlist [split $reststr ","]

		# check if this partition is empty -> skip 
		set tmpst [string trim [lindex $restlist 2]]
		if { [string trim [string range $tmpst 3 end]] == "0" } {
			continue
		}

		# partition-size in blocks	
		set tmpst [string trim [lindex $restlist 1]]
		set partinfo($partindex,1) [string trim [string range $tmpst 5 end]]
		# partition-id
		set tmpst [string trim [lindex $restlist 2]]
		set partinfo($partindex,2) [string range $tmpst 3 end]

		# partition-id-string
		set tmpidx [string first "Name=" $reststr]
		set tmpst [string trim [string range $reststr $tmpidx end]]
		set endptr [expr [string length $tmpst]-2]
		# if the idstring is longer than 25 chars, skip it
		if { $endptr > 30 } {
			set endptr 30
		} 
		set partinfo($partindex,3) [
				string trim [string range $tmpst 6 $endptr]]

		# mountpoint of partition (if mounted)
		set partinfo($partindex,4) [
				getmountpoint $partinfo($partindex,0)]
	
		incr partindex
	}
	set partinfo($partindex,0) "x"		;# Mark end of array with "x"
}


# Updates the information in the internal mount-table.
# Syncs with real system mounttable

proc updatemountinfo { } {
global partinfo

	set i 0
	while { $partinfo($i,0) != "x" } {
		set partinfo($i,4) [getmountpoint $partinfo($i,0)]
		incr i
	}
}


# Call the getcdtoc-programm and return a formated list containing the
# information 

proc getcdinfo { blkname } {

	resetcd $blkname

	set stat [catch { set cdinfo [getcdtoc $blkname] }]

	if { $stat == 1 } {
		#error while executing getcdtoc?
		return "READ-ERROR Read-Error 00:00.00 0 |" 
	}

	set cdinfolist [split $cdinfo "\n"]

	set cdtoclist {}

	set tmpidx [lsearch -glob $cdinfolist "Type*"]
	lappend cdtoclist [string range [lindex $cdinfolist $tmpidx] 7 end]

	set tmpidx [lsearch -glob $cdinfolist "Label*"]
	lappend cdtoclist [string range [lindex $cdinfolist $tmpidx] 7 end]

	set tmpidx [lsearch -glob $cdinfolist "Time*"]
	lappend cdtoclist [string range [lindex $cdinfolist $tmpidx] 7 end]

	set tmpidx [lsearch -glob $cdinfolist "Tracks*"]
	lappend cdtoclist [string range [lindex $cdinfolist $tmpidx] 7 end]

	# Set a marker to show start of track-lists.     
	lappend cdtoclist "|"

	set tmpidx [lsearch -glob $cdinfolist "#track*"]
	incr tmpidx		; # tmpidx now points to first line of tracks

	# loop all tracks
	while { [set trkline [lindex $cdinfolist $tmpidx]] != "#" } {
		set trklinelist [split $trkline]
		set newtrklinelist {}

		# delete all empty list-elements
		foreach i $trklinelist {
			if { $i != {} } {
				lappend newtrklinelist $i
			}
		} 

		# preemphasis
		if { [lindex $newtrklinelist 1] == "no" } {
			lappend cdtoclist 0
		} else {
			lappend cdtoclist 1
		} 

		# copy-protected
		if { [lindex $newtrklinelist 2] == "no" } {
			lappend cdtoclist 0
		} else {
			lappend cdtoclist 1
		} 

		# track-type
		lappend cdtoclist [lindex $newtrklinelist 3]
		# track-length
		lappend cdtoclist [lindex $newtrklinelist 6]
		# track-frames
		lappend cdtoclist [lindex $newtrklinelist 7]

		incr tmpidx
	}

	return $cdtoclist
}


