
#include <stdio.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xmd.h>
#include <pthread.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkprivate.h>

#include "fullscreen.h"

#ifndef XF86VIDMODE

gboolean xmms_fullscreen_available(Display *dpy) { return FALSE; }
gboolean xmms_fullscreen_init(GtkWidget *win) { return FALSE; }

gboolean xmms_fullscreen_enter(GtkWidget *win, gint *w, gint *h) { return FALSE; }
void xmms_fullscreen_leave(GtkWidget *win) { return; }

gboolean xmms_fullscreen_in(GtkWidget *win) { return FALSE; }
gboolean xmms_fullscreen_mark(GtkWidget *win) { return FALSE; }
void xmms_fullscreen_unmark(GtkWidget *win) { return; }

void xmms_fullscreen_cleanup(GtkWidget *win) { return; }

GSList *xmms_fullscreen_modelist(GtkWidget *win) { return NULL; }
void xmms_fullscreen_modelist_free(GSList *modes) { return; }

#else /* XF86VIDMODE */

#include <X11/extensions/xf86vmode.h>
#include <X11/extensions/xf86vmstr.h>

gboolean xmms_fullscreen_available(Display *dpy) {
	int event_base, error_base, num_modes;
	XF86VidModeModeInfo **dummy;
	if (!XF86VidModeQueryExtension(dpy, &event_base, &error_base))
		return FALSE;

	XF86VidModeGetAllModeLines(dpy, DefaultScreen(dpy), &num_modes, &dummy);
	XFree(dummy);

	return (num_modes>1);
}

#define XA_WIN_LAYER	"_WIN_LAYER"

typedef enum
{
	WIN_LAYER_DESKTOP = 0,
	WIN_LAYER_BELOW = 2,
	WIN_LAYER_NORMAL = 4,
	WIN_LAYER_ONTOP = 6,
	WIN_LAYER_DOCK = 8,
	WIN_LAYER_ABOVE_DOCK = 10
}
WinLayer;

static Atom _XA_WIN_LAYER;

typedef struct {
	Display *display;
	XF86VidModeModeInfo **modes, *origmode;
	gboolean is_full, can_full;
	int xv, yv;
	int num_modes;
} fullscreen_display_t;
static fullscreen_display_t **displays = NULL;

typedef struct {
	GtkWidget *window;
	gint is_full;
	gint ox, oy, owidth, oheight;
	fullscreen_display_t *display;
} fullscreen_window_t;
static fullscreen_window_t **windows = NULL;

static pthread_mutex_t full_mutex;

#define FULL_LOCK() pthread_mutex_lock(&full_mutex);
#define FULL_UNLOCK() pthread_mutex_unlock(&full_mutex);

static gboolean full_have_mutex = 0;

static fullscreen_display_t *getdisplay(Display *dpy) {
	gint i;

	if (displays) {
		for (i = 0; displays[i]; i++) {
			if (displays[i]->display == dpy)
				return displays[i];
		}
		displays = g_realloc(displays, sizeof(*displays)*(i+2));
	} else {
		displays = g_malloc(sizeof(*displays)*2);
		i = 0;
	}
	displays[i+1] = NULL;
	displays[i] = g_malloc(sizeof(**displays));
	displays[i]->display = dpy;
	displays[i]->modes = NULL;
	displays[i]->origmode = NULL;
	displays[i]->num_modes = 0;
	displays[i]->is_full = FALSE;
	displays[i]->can_full = FALSE;
	displays[i]->xv = 0;
	displays[i]->yv = 0;
	return displays[i];
}

static fullscreen_window_t *getwindow(GtkWidget *win) {
	gint i;
	
	if (windows) {
		for (i = 0; windows[i]; i++) {
			if (windows[i]->window == win)
				return windows[i];
		}
		windows = g_realloc(windows, sizeof(*windows)*(i+2));
	} else {
		windows = g_malloc(sizeof(*windows)*2);
		i = 0;
	}
	windows[i+1] = NULL;
	windows[i] = g_malloc(sizeof(**windows));
	windows[i]->window = win;
	windows[i]->ox = 0;
	windows[i]->oy = 0;
	windows[i]->owidth = 0;
	windows[i]->oheight = 0;
	windows[i]->display = getdisplay(GDK_WINDOW_XDISPLAY(win->window));
	windows[i]->is_full = 0;
	return windows[i];
}

gboolean xmms_fullscreen_init(GtkWidget *win) {
	int event_base, error_base, dummy;
	fullscreen_window_t *fwin;
	gint i;
	XF86VidModeModeLine origmode;

	if (!full_have_mutex) {
		pthread_mutex_init(&full_mutex, NULL);
	}

	FULL_LOCK();
	fwin = getwindow(win);

	if (!XF86VidModeQueryExtension(fwin->display->display, &event_base, &error_base)) {
		FULL_UNLOCK();
		return FALSE;
	}

	if (!fwin->display->modes) {
		XF86VidModeGetAllModeLines(fwin->display->display, DefaultScreen(fwin->display->display), &fwin->display->num_modes, &fwin->display->modes);

		if (!fwin->display->origmode) {
			XF86VidModeGetModeLine(fwin->display->display, DefaultScreen(fwin->display->display), &dummy, &origmode);
			for (i = 0; i < fwin->display->num_modes; i++) {
				if (fwin->display->modes[i]->hdisplay == origmode.hdisplay
				 && fwin->display->modes[i]->vdisplay == origmode.vdisplay) {
					fwin->display->origmode = fwin->display->modes[i];
					break;
				}
			}

			if (!fwin->display->origmode) {
				fprintf(stderr, "ERROR: Could not determine original mode.\n");
				FULL_UNLOCK();
				return FALSE;
			}

		}

		fwin->display->can_full = (fwin->display->num_modes>1);
	}
	FULL_UNLOCK();
	return fwin->display->can_full;
}

/* ripped from xmms/hints.c */
static void gnome_wm_set_window_very_top(GtkWindow *window, gboolean ontop) {
	XEvent xev;
	GdkWindowPrivate *priv;
	gint prev_error;
	int layer = WIN_LAYER_ABOVE_DOCK;

	if (ontop == FALSE)
		layer = WIN_LAYER_NORMAL;

	gdk_error_warnings = 0;
	priv = (GdkWindowPrivate *) (GTK_WIDGET(window)->window);
	if (GTK_WIDGET_MAPPED(window))
	{
		xev.type = ClientMessage;
		xev.xclient.type = ClientMessage;
		xev.xclient.window = priv->xwindow;
		xev.xclient.message_type = _XA_WIN_LAYER;
		xev.xclient.format = 32;
		xev.xclient.data.l[0] = (CARD32) layer;
		xev.xclient.data.l[1] = gdk_time_get();

		XSendEvent(GDK_DISPLAY(), GDK_ROOT_WINDOW(), False,
			SubstructureNotifyMask, (XEvent *) & xev);
	}
	else
	{
		long data[1];

		data[0] = layer;
		XChangeProperty(GDK_DISPLAY(), priv->xwindow, _XA_WIN_LAYER,
			XA_CARDINAL, 32, PropModeReplace, (unsigned char *) data, 1);
	}
	gdk_error_warnings = prev_error;
}

gboolean xmms_fullscreen_enter(GtkWidget *win, gint *w, gint *h) {
	gint i, close = 0, how_close, t;
	gboolean retval = FALSE;
	gint offx, offy, dummy;
	fullscreen_window_t *fwin;

	FULL_LOCK();
	fwin = getwindow(win);

	if (!fwin->display->is_full && !fwin->is_full && fwin->display->can_full) {
		for (close = 0; close < fwin->display->num_modes; close++) {
			if ((fwin->display->modes[close]->hdisplay >= *w) &&
			    (fwin->display->modes[close]->vdisplay >= *h)) {
				how_close = fwin->display->modes[close]->hdisplay-*w;
				break;
			}
		}

		for (i = close+1; i < fwin->display->num_modes; i++) {
			if (fwin->display->modes[i]->vdisplay < *h) continue;
			t = fwin->display->modes[i]->hdisplay-*w;
			if (t >= 0 && t < how_close) {
				close = i;
				how_close = t;
			}
		}

		if (close < fwin->display->num_modes) {
			*w = fwin->display->modes[close]->hdisplay;
			*h = fwin->display->modes[close]->vdisplay;
			XF86VidModeSwitchToMode(fwin->display->display, DefaultScreen(fwin->display->display), fwin->display->modes[close]);

			gdk_window_get_geometry(fwin->window->window, &offx, &offy, &fwin->owidth, &fwin->oheight, &dummy);
			gdk_window_get_origin(fwin->window->window, &fwin->ox, &fwin->oy);
			fwin->ox -= offx; fwin->oy -= offy;
			gdk_window_move_resize(fwin->window->window, -offx, -offy, fwin->display->modes[close]->hdisplay, fwin->display->modes[close]->vdisplay);

			gnome_wm_set_window_very_top(GTK_WINDOW(fwin->window), TRUE);

			gdk_pointer_grab(fwin->window->window, TRUE, 0, fwin->window->window, NULL, GDK_CURRENT_TIME);
			gdk_keyboard_grab(fwin->window->window, TRUE, GDK_CURRENT_TIME);
			gdk_window_raise(fwin->window->window);
			gtk_window_activate_focus(GTK_WINDOW(fwin->window));

			XF86VidModeGetViewPort(fwin->display->display, DefaultScreen(fwin->display->display), &fwin->display->xv, &fwin->display->yv);
			XF86VidModeSetViewPort(fwin->display->display, DefaultScreen(fwin->display->display), 0, 0);
			retval = TRUE;

			fwin->is_full = TRUE;
			fwin->display->is_full = TRUE;
		}
	}

	FULL_UNLOCK();

	return retval;
}

void xmms_fullscreen_leave(GtkWidget *win) {
	fullscreen_window_t *fwin;

	FULL_LOCK();
	fwin = getwindow(win);

	if (fwin->is_full && fwin->display->is_full) {
		gdk_pointer_ungrab(GDK_CURRENT_TIME);
		gdk_keyboard_ungrab(GDK_CURRENT_TIME);
		gdk_window_move_resize(fwin->window->window, fwin->ox, fwin->oy, fwin->owidth, fwin->oheight);
		gnome_wm_set_window_very_top(GTK_WINDOW(fwin->window), FALSE);
		XF86VidModeSwitchToMode(fwin->display->display, DefaultScreen(fwin->display->display), fwin->display->origmode);
		fwin->display->is_full = FALSE;
	}
	fwin->is_full = FALSE;
	FULL_UNLOCK();
}

gboolean xmms_fullscreen_in(GtkWidget *win) {
	fullscreen_window_t *fwin;

	FULL_LOCK();
	fwin = getwindow(win);
	FULL_UNLOCK();

	if (fwin->display->is_full)
		return TRUE;
	else
		return FALSE;
}

gboolean xmms_fullscreen_mark(GtkWidget *win) {
	fullscreen_window_t *fwin;

	FULL_LOCK();
	fwin = getwindow(win);

	if (fwin->display->is_full) {
		FULL_UNLOCK();
		return FALSE;
	} else {
		fwin->is_full = TRUE;
		fwin->display->is_full = TRUE;
		FULL_UNLOCK();
		return TRUE;
	}
}

void xmms_fullscreen_unmark(GtkWidget *win) {
	fullscreen_window_t *fwin;

	FULL_LOCK();
	fwin = getwindow(win);

	if (fwin->is_full) {
		fwin->is_full = FALSE;
		fwin->display->is_full = FALSE;
	}
	FULL_UNLOCK();
}

void xmms_fullscreen_cleanup(GtkWidget *win) {
	gint i, j;
	fullscreen_display_t *display;

	FULL_LOCK();
	if (!windows) goto unlock_return;

	for (i = 0; windows[i]; i++) {
		if (windows[i]->window == win) {
			display = windows[i]->display;
			for (j = i+1; windows[j]; j++) ;
			windows[i] = windows[j-1];
			windows = g_realloc(windows, sizeof(*windows)*(j+1));
			windows[j] = NULL;

			for (i = 0; windows[i]; i++) {
				if (windows[i]->display == display) goto unlock_return;
			}
			/* bugger all, kill the display */
		for (i = 0; displays[i]; i++) {
			if (displays[i] == display) {
				XFree(displays[i]->modes);
				for (j = i+1; displays[j]; j++) ;
				displays[i] = displays[j-1];
				displays = g_realloc(displays, sizeof(*displays)*(j+1));
				displays[j] = NULL;
				break;
			}
		}
		}
	}
unlock_return:
	FULL_UNLOCK();
}

GSList *xmms_fullscreen_modelist(GtkWidget *win) {
	fullscreen_window_t *fwin;
	xmms_fullscreen_mode_t *ent;
	GSList *retlist = NULL;
	int i;

	FULL_LOCK();
	fwin = getwindow(win);

	for (i = 0; i < fwin->display->num_modes; i++) {
		ent = g_malloc(sizeof(*ent));
		ent->width = fwin->display->modes[i]->hdisplay;
		ent->height = fwin->display->modes[i]->vdisplay;
		retlist = g_slist_append(retlist, ent);
	}
	FULL_UNLOCK();

	return retlist;
}

void xmms_fullscreen_modelist_free(GSList *modes) {
	g_slist_foreach(modes, (GFunc)g_free, NULL);
	g_slist_free(modes);
}

#endif /* XF86VIDMODE */
