/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-1999  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "xmms.h"
#include <gdk/gdkprivate.h>
#include <ctype.h>

void textbox_generate_pixmap(TextBox * tb);

void textbox_draw(Widget * w)
{
	TextBox *tb = (TextBox *) w;
	gint cw;
	GdkPixmap *obj;
	GdkPixmap *src;

	if (tb->tb_text)
	{
		if (!tb->tb_pixmap_text)
			textbox_generate_pixmap(tb);
		else if (strcmp(tb->tb_text, tb->tb_pixmap_text))
			textbox_generate_pixmap(tb);
	}

	if (tb->tb_pixmap && get_skin_pixmap(tb->tb_skin_index))
	{
		if (is_new_skin(tb->tb_old_skin_path))
		{
			textbox_generate_pixmap(tb);
			if (tb->tb_old_skin_path)
			{
				g_free(tb->tb_old_skin_path);
				tb->tb_old_skin_path = NULL;
			}
			if (skin->path)
				tb->tb_old_skin_path = g_strdup(skin->path);
		}
		obj = tb->tb_widget.parent;
		src = tb->tb_pixmap;

		cw = tb->tb_pixmap_width - tb->tb_offset;
		if (cw > tb->tb_widget.width)
			cw = tb->tb_widget.width;
		gdk_draw_pixmap(obj, tb->tb_widget.gc, src, tb->tb_offset, 0, tb->tb_widget.x, tb->tb_widget.y, cw, 6);
		if (cw < tb->tb_widget.width)
			gdk_draw_pixmap(obj, tb->tb_widget.gc, src, 0, 0, tb->tb_widget.x + cw, tb->tb_widget.y, tb->tb_widget.width - cw, 6);
	}
}

gint textbox_scroll(gpointer data)
{
	TextBox *tb = (TextBox *) data;

	if (!tb->tb_is_dragging)
	{
		if (cfg.smooth_title_scroll)
			tb->tb_offset++;
		else
			tb->tb_offset += 5;
		if (tb->tb_offset >= tb->tb_pixmap_width)
			tb->tb_offset -= tb->tb_pixmap_width;
		draw_widget(tb);
	}
	return TRUE;
}

void textbox_button_press(GtkWidget * w, GdkEventButton * event, gpointer data)
{
	TextBox *tb = (TextBox *) data;

	if (event->button != 1)
		return;
	if (inside_widget(event->x, event->y, &tb->tb_widget) && tb->tb_scroll_allowed && tb->tb_pixmap_width > tb->tb_widget.width && tb->tb_is_scrollable)
	{
		tb->tb_is_dragging = TRUE;
		tb->tb_drag_off = tb->tb_offset;
		tb->tb_drag_x = event->x;
	}
}

void textbox_motion(GtkWidget * w, GdkEventMotion * event, gpointer data)
{
	TextBox *tb = (TextBox *) data;

	if (tb->tb_is_dragging)
	{
		if (tb->tb_scroll_allowed && tb->tb_pixmap_width > tb->tb_widget.width)
		{
			tb->tb_offset = tb->tb_drag_off - (event->x - tb->tb_drag_x);
			while (tb->tb_offset < 0)
				tb->tb_offset += tb->tb_pixmap_width;
			while (tb->tb_offset > tb->tb_pixmap_width)
				tb->tb_offset -= tb->tb_pixmap_width;
			draw_widget(tb);
		}
	}
}

void textbox_button_release(GtkWidget * w, GdkEventButton * event, gpointer data)
{
	TextBox *tb = (TextBox *) data;

	if (event->button == 1)
		tb->tb_is_dragging = FALSE;
}

void textbox_set_text(TextBox * tb, gchar * text)
{
	gint length, pad, i, wl;

	if (tb->tb_text)
	{
		if (tb->tb_is_scrollable && strlen(tb->tb_text) >= 7)
		{

			tb->tb_text[strlen(tb->tb_text) - 7] = '\0';

			if (!strcasecmp(text, tb->tb_text))
			{
				tb->tb_text[strlen(tb->tb_text)] = ' ';
				return;
			}
			tb->tb_text[strlen(tb->tb_text)] = ' ';
		}
		else if (!strcasecmp(text, tb->tb_text))
			return;
	}

	length = strlen(text);
	if (tb->tb_text)
		g_free(tb->tb_text);
	tb->tb_text = g_strdup(text);
	tb->tb_offset = 0;

	wl = tb->tb_widget.width / 5;
	if (wl * 5 != tb->tb_widget.width)
		wl++;

	if (length <= wl)
	{
		tb->tb_is_scrollable = FALSE;
		pad = wl - length;
		tb->tb_text = (gchar *) g_realloc(tb->tb_text, length + pad + 1);
		for (i = 0; i < pad; i++)
			strcat(tb->tb_text, " ");
		length += pad;
		if (tb->tb_timeout_tag)
		{
			gtk_timeout_remove(tb->tb_timeout_tag);
			tb->tb_timeout_tag = 0;
		}
	}
	else if (length * 5 > tb->tb_widget.width && tb->tb_scroll_allowed)
	{
		tb->tb_is_scrollable = TRUE;
		length += 7;
		tb->tb_text = (gchar *) g_realloc(tb->tb_text, length + 1);
		strcat(tb->tb_text, "  ***  ");
		if (tb->tb_scroll_enabled && !tb->tb_timeout_tag)
		{
			if (cfg.smooth_title_scroll)
				tb->tb_timeout_tag = gtk_timeout_add(TEXTBOX_SCROLL_SMOOTH_TIMEOUT, textbox_scroll, tb);
			else
				tb->tb_timeout_tag = gtk_timeout_add(TEXTBOX_SCROLL_TIMEOUT, textbox_scroll, tb);
		}
	}
	draw_widget(tb);
}

void textbox_generate_pixmap(TextBox * tb)
{
	gint length, i, x, y;
	GdkGC *gc;

	if (tb->tb_pixmap)
		gdk_pixmap_unref(tb->tb_pixmap);

	if (tb->tb_pixmap_text) ;
	g_free(tb->tb_pixmap_text);
	tb->tb_pixmap_text = g_strdup(tb->tb_text);

	length = strlen(tb->tb_text);

	tb->tb_pixmap_width = length * 5;
	tb->tb_pixmap = gdk_pixmap_new(mainwin->window, tb->tb_pixmap_width, 6, gdk_rgb_get_visual()->depth);
	gc = tb->tb_widget.gc;

	for (i = 0; i < length; i++)
	{
		x = y = -1;
		tb->tb_text[i] = toupper(tb->tb_text[i]);
		if (tb->tb_text[i] >= 'A' && tb->tb_text[i] <= 'Z')
		{
			x = 5 * (tb->tb_text[i] - 'A');
			y = 0;
		}
		else if (tb->tb_text[i] >= '0' && tb->tb_text[i] <= '9')
		{
			x = 5 * (tb->tb_text[i] - '0');
			y = 6;
		}
		else
		{
			switch (tb->tb_text[i])
			{
				case '"':
					x = 130;
					y = 0;
					break;
				case ':':
					x = 60;
					y = 6;
					break;
				case '(':
					x = 65;
					y = 6;
					break;
				case ')':
					x = 70;
					y = 6;
					break;
				case '-':
					x = 75;
					y = 6;
					break;
				case '`':
				case '\'':
					x = 80;
					y = 6;
					break;
				case '!':
					x = 85;
					y = 6;
					break;
				case '_':
					x = 90;
					y = 6;
					break;
				case '+':
					x = 95;
					y = 6;
					break;
				case '\\':
					x = 100;
					y = 6;
					break;
				case '/':
					x = 105;
					y = 6;
					break;
				case '[':
					x = 110;
					y = 6;
					break;
				case ']':
					x = 115;
					y = 6;
					break;
				case '^':
					x = 120;
					y = 6;
					break;
				case '&':
					x = 125;
					y = 6;
					break;
				case '%':
					x = 130;
					y = 6;
					break;
				case '.':
				case ',':
					x = 135;
					y = 6;
					break;
				case '=':
					x = 140;
					y = 6;
					break;
				case '$':
					x = 145;
					y = 6;
					break;
				case '#':
					x = 150;
					y = 6;
					break;
				case '':
				case '':
					x = 0;
					y = 12;
					break;
				case '':
				case '':
					x = 5;
					y = 12;
					break;
				case '':
				case '':
					x = 10;
					y = 12;
					break;
				case '':
				case '':
					x = 100;
					y = 0;
					break;
				case '?':
					x = 15;
					y = 12;
					break;
				case '*':
					x = 20;
					y = 12;
					break;
				default:
					x = 145;
					y = 0;
					break;
			}
		}
		gdk_draw_pixmap(tb->tb_pixmap, gc, get_skin_pixmap(tb->tb_skin_index), x, y, i * 5, 0, 5, 6);
	}

	
}

void textbox_set_scroll(TextBox * tb, gboolean s)
{
	tb->tb_scroll_enabled = s;
	if (tb->tb_scroll_enabled && tb->tb_is_scrollable && tb->tb_scroll_allowed)
	{
		if (cfg.smooth_title_scroll)
			tb->tb_timeout_tag = gtk_timeout_add(TEXTBOX_SCROLL_SMOOTH_TIMEOUT, textbox_scroll, tb);
		else
			tb->tb_timeout_tag = gtk_timeout_add(TEXTBOX_SCROLL_TIMEOUT, textbox_scroll, tb);
	}
	else
	{
		if (tb->tb_timeout_tag)
		{
			gtk_timeout_remove(tb->tb_timeout_tag);
			tb->tb_timeout_tag = 0;
		}
		tb->tb_offset = 0;
		draw_widget(tb);
	}

}

TextBox *create_textbox(GList ** wlist, GdkPixmap * parent, GdkGC * gc, gint x, gint y, gint w, gboolean allow_scroll, SkinIndex si)
{
	TextBox *tb;

	tb = (TextBox *) g_malloc0(sizeof (TextBox));
	tb->tb_widget.parent = parent;
	tb->tb_widget.gc = gc;
	tb->tb_widget.x = x;
	tb->tb_widget.y = y;
	tb->tb_widget.width = w;
	tb->tb_widget.height = 6;
	tb->tb_widget.visible = 1;
	tb->tb_widget.button_press_cb = textbox_button_press;
	tb->tb_widget.button_release_cb = textbox_button_release;
	tb->tb_widget.motion_cb = textbox_motion;
	tb->tb_widget.draw = textbox_draw;
	tb->tb_scroll_allowed = allow_scroll;
	tb->tb_scroll_enabled = TRUE;
	tb->tb_skin_index = si;
	add_widget(wlist, tb);
	return tb;
}

void free_textbox(TextBox * tb)
{
	if (tb->tb_pixmap)
		gdk_pixmap_unref(tb->tb_pixmap);
	if (tb->tb_text)
		g_free(tb->tb_text);
	g_free(tb);
}
