/*
 * GdiffOverview widget module
 * This widget takes care of (diff)ranges, i.e. rectangles that represents differences of the text.
 * In addition, a slider, which acts as one on scrollbar.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#include "gdiffoverview.h"

/* Constant number */
enum {
	ARG_0,
	ARG_ADJUSTMENT,
};
#define MIN_SLIDER_LENGTH	20


/* Internal data structure */
typedef struct _PaintRange {
	gdouble begin;/* 0.0 - 1.0 */
	gdouble end;/* 0.0 - 1.0 */
} PaintRange;

/* Private function declarations */
static void gdiff_overview_class_init(GdiffOverviewClass *klass);
static void gdiff_overview_init(GdiffOverview *overview);

static void gdiff_overview_set_arg(GtkObject *object, GtkArg *arg, guint arg_id);
static void gdiff_overview_get_arg(GtkObject *object, GtkArg *arg, guint arg_id);
static void gdiff_overview_finalize(GtkObject *object);

static void gdiff_overview_realize(GtkWidget *widget);
static void gdiff_overview_unrealize(GtkWidget *widget);
static void gdiff_overview_size_allocate(GtkWidget *widget, GtkAllocation *allocation);
static gint gdiff_overview_expose(GtkWidget *widget, GdkEventExpose *event);

static void gdiff_overview_adjustment_changed(GtkAdjustment *adjustment, gpointer data);
static void gdiff_overview_adjustment_value_changed(GtkAdjustment *adjustment, gpointer data);

static void gdiff_overview_draw_slider(GdiffOverview *overview, const GdkRectangle *area);
static void gdiff_overview_clear_background(GtkWidget *widget, const GdkRectangle *area);
static void gdiff_overview_draw_ranges(GdiffOverview *overview, const GdkRectangle *area);


static GtkWidgetClass *parent_class = NULL;

GtkType
gdiff_overview_get_type(void)
{
  static GtkType overview_type = 0;

  if (!overview_type) {
      static const GtkTypeInfo overview_info = {
		  "GdiffOverview",
		  sizeof(GdiffOverview),
		  sizeof(GdiffOverviewClass),
		  (GtkClassInitFunc)gdiff_overview_class_init,
		  (GtkObjectInitFunc)gdiff_overview_init,
		  /* reserved_1 */ NULL,
		  /* reserved_2 */ NULL,
		  (GtkClassInitFunc)NULL,
      };
      overview_type = gtk_type_unique(GTK_TYPE_WIDGET, &overview_info);
  }
  
  return overview_type;
}

static void
gdiff_overview_class_init(GdiffOverviewClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	gtk_object_add_arg_type("GdiffOverview::adjustment",
							GTK_TYPE_ADJUSTMENT,
							GTK_ARG_READWRITE | GTK_ARG_CONSTRUCT,
							ARG_ADJUSTMENT);
	object_class = (GtkObjectClass*)class;
	widget_class = (GtkWidgetClass*)class;

	parent_class = gtk_type_class(GTK_TYPE_WIDGET);

	object_class->set_arg = gdiff_overview_set_arg;
	object_class->get_arg = gdiff_overview_get_arg;
	object_class->finalize = gdiff_overview_finalize;

	widget_class->realize = gdiff_overview_realize;
	widget_class->unrealize = gdiff_overview_unrealize;
	widget_class->size_allocate = gdiff_overview_size_allocate;
	widget_class->expose_event = gdiff_overview_expose;
}

static void
gdiff_overview_init(GdiffOverview *overview)
{
	overview->slider_y = 0;
	overview->slider_length = MIN_SLIDER_LENGTH;
	overview->adjustment = NULL;
	overview->old_value = 0.0;
	overview->old_lower = 0.0;
	overview->old_upper = 0.0;
	overview->old_page_size = 0.0;
	overview->xor_gc = NULL;
	overview->range_gc = NULL;
	overview->range_fg = NULL;
	overview->range_bg = NULL;
	
	overview->range_list = NULL;
}


static void
gdiff_overview_set_arg(GtkObject *object,
					   GtkArg *arg,
					   guint arg_id)
{
	GdiffOverview *overview;
	
	overview = GDIFF_OVERVIEW(object);
  
	switch (arg_id) {
	case ARG_ADJUSTMENT:
		gdiff_overview_set_adjustment(GDIFF_OVERVIEW(overview), GTK_VALUE_POINTER(*arg));
		break;
	default:
		break;
	}
}

static void
gdiff_overview_get_arg(GtkObject *object,
					   GtkArg *arg,
					   guint arg_id)
{
	GdiffOverview *overview;
  
	overview = GDIFF_OVERVIEW(object);
  
	switch (arg_id) {
    case ARG_ADJUSTMENT:
		GTK_VALUE_POINTER(*arg) = GDIFF_OVERVIEW(overview);
		break;
    default:
		arg->type = GTK_TYPE_INVALID;
		break;
    }
}

static void
gdiff_overview_finalize(GtkObject *object)
{
	GSList *list;
	GdiffOverview *overview;

	overview = GDIFF_OVERVIEW(object);

	if (overview->adjustment) {
		gtk_signal_disconnect_by_data(GTK_OBJECT(overview->adjustment),
									  (gpointer)overview);
		gtk_object_unref(GTK_OBJECT(overview->adjustment));
	}

	if (overview->range_fg)
		gdk_color_free(overview->range_fg);
	if (overview->range_bg)
		gdk_color_free(overview->range_bg);

	for (list = overview->range_list; list; list = list->next) {
		g_free(list->data);
	}
	g_slist_free(overview->range_list);

	(*GTK_OBJECT_CLASS(parent_class)->finalize)(object);
}

/**
 * gdiff_overview_new:
 * _Exported interface_
 **/
GtkWidget*
gdiff_overview_new(GtkAdjustment *adjustment)
{
	GtkWidget *overview;
  
	overview = gtk_widget_new(GDIFF_TYPE_OVERVIEW,
							  "adjustment", adjustment,
							  NULL);
  
	return overview;
}

/**
 * gdiff_overview_size:
 * _Exported interface_
 * Set size of the widget.
 **/
void
gdiff_overview_size(GdiffOverview *overview,
					gint width,
					gint height)
{
	g_return_if_fail(overview != NULL);
	g_return_if_fail(GDIFF_IS_OVERVIEW(overview));

	GTK_WIDGET(overview)->requisition.width = width;
	GTK_WIDGET(overview)->requisition.height = height;
}

/**
 * gdiff_overview_set_adjustment:
 * _Exported interface_
 **/
void
gdiff_overview_set_adjustment(GdiffOverview *overview,
							  GtkAdjustment *adjustment)
{
	g_return_if_fail(overview != NULL);
	g_return_if_fail(GDIFF_IS_OVERVIEW(overview));

	if (!adjustment)
		adjustment = GTK_ADJUSTMENT(gtk_adjustment_new(0.0, 0.0, 0.0, 0.0, 0.0, 0.0));
	else
		g_return_if_fail(GTK_IS_ADJUSTMENT(adjustment));

	if (overview->adjustment != adjustment) {
		if (overview->adjustment) {
			gtk_signal_disconnect_by_data(GTK_OBJECT(overview->adjustment),
										  (gpointer)overview);
			gtk_object_unref(GTK_OBJECT(overview->adjustment));
		}
		
		overview->adjustment = adjustment;
		gtk_object_ref(GTK_OBJECT(adjustment));
		gtk_object_sink(GTK_OBJECT(adjustment));
      
		gtk_signal_connect(GTK_OBJECT(adjustment), "changed",
						   GTK_SIGNAL_FUNC(gdiff_overview_adjustment_changed),
						   (gpointer)overview);
		gtk_signal_connect(GTK_OBJECT(adjustment), "value_changed",
						   GTK_SIGNAL_FUNC(gdiff_overview_adjustment_value_changed),
						   (gpointer)overview);

		overview->old_value = adjustment->value;
		overview->old_lower = adjustment->lower;
		overview->old_upper = adjustment->upper;
		overview->old_page_size = adjustment->page_size;

		gdiff_overview_adjustment_changed(adjustment, (gpointer)overview);
    }
}

/**
 * gdiff_overview_insert_paintrange:
 * _Exported interface_
 **/
void
gdiff_overview_insert_paintrange(GdiffOverview *overview, gdouble begin, gdouble end)
{
	PaintRange *pr;

	g_return_if_fail(begin >= 0);
	g_return_if_fail(end >= 0);

	/* begin > 1 or end > 1 can happen, when the both last one line is different.
	   I'm not sure I can get around this by a better logic. */
	if (begin < 0) begin = 0;
	if (begin > 1) begin = 1;
	if (end < 0) end = 0;
	if (end > 1) end = 1;
	
	pr = g_new(PaintRange, 1);
	pr->begin = begin;
	pr->end = end;
	overview->range_list = g_slist_append(overview->range_list, pr);
}

/**
 * gdiff_overview_set_foreground:
 * _Exported interface_
 **/
void gdiff_overview_set_foreground(GdiffOverview *overview, GdkColor *color)
{
	if (overview->range_fg)
		gdk_color_free(overview->range_fg);
	overview->range_fg = gdk_color_copy(color);
}

/**
 * gdiff_overview_set_background:
 * _Exported interface_
 **/
void gdiff_overview_set_background(GdiffOverview *overview, GdkColor *color)
{
	if (overview->range_bg)
		gdk_color_free(overview->range_bg);
	overview->range_bg = gdk_color_copy(color);
}



static void
gdiff_overview_realize(GtkWidget *widget)
{
	GdiffOverview *overview;
	GdkWindowAttr attributes;
	gint attributes_mask;

	g_return_if_fail(widget != NULL);
	g_return_if_fail(GDIFF_IS_OVERVIEW(widget));

	overview = GDIFF_OVERVIEW(widget);
	GTK_WIDGET_SET_FLAGS(widget, GTK_REALIZED);

	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.x = widget->allocation.x;
	attributes.y = widget->allocation.y;
	attributes.width = widget->allocation.width;
	attributes.height = widget->allocation.height;
	attributes.wclass = GDK_INPUT_OUTPUT;
	attributes.visual = gtk_widget_get_visual(widget);
	attributes.colormap = gtk_widget_get_colormap(widget);
	attributes.event_mask = gtk_widget_get_events(widget) | GDK_EXPOSURE_MASK;

	attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

	widget->window = gdk_window_new(gtk_widget_get_parent_window(widget), &attributes, attributes_mask);
	gdk_window_set_user_data(widget->window, overview);

	widget->style = gtk_style_attach(widget->style, widget->window);
	gtk_style_set_background(widget->style, widget->window, GTK_STATE_NORMAL);

	overview->range_gc = gdk_gc_new(widget->window);
	if (overview->range_fg)
		gdk_gc_set_foreground(overview->range_gc, overview->range_fg);
	if (overview->range_bg)
		gdk_gc_set_background(overview->range_gc, overview->range_bg);

	/* Use color of fg_gc, which a user can specify in gtkdiffrc file. */
	overview->xor_gc = gdk_gc_new(widget->window);
	gdk_gc_copy(overview->xor_gc, widget->style->fg_gc[GTK_STATE_NORMAL]);
	gdk_gc_set_function(overview->xor_gc, GDK_XOR);
}

static void
gdiff_overview_unrealize(GtkWidget *widget)
{
	GdiffOverview *overview;

	overview = GDIFF_OVERVIEW(widget);
	if (overview->range_fg) {
		gdk_gc_unref(overview->range_gc);
		overview->range_gc = NULL;
	}
	if (overview->xor_gc) {
		gdk_gc_unref(overview->xor_gc);
		overview->xor_gc = NULL;
	}		

	if (GTK_WIDGET_CLASS(parent_class)->unrealize)
		(*GTK_WIDGET_CLASS(parent_class)->unrealize)(widget);
}

static void
gdiff_overview_size_allocate(GtkWidget *widget,
							 GtkAllocation *allocation)
{
	g_return_if_fail(widget != NULL);
	g_return_if_fail(GDIFF_IS_OVERVIEW(widget));
	g_return_if_fail(allocation != NULL);
	
	widget->allocation = *allocation;

	if (GTK_WIDGET_REALIZED(widget)) {
		gdk_window_move_resize(widget->window,
							   allocation->x, allocation->y,
							   allocation->width, allocation->height);
    }
}

static gint
gdiff_overview_expose(GtkWidget *widget,
					  GdkEventExpose *event)
{
	gdiff_overview_clear_background(widget, &event->area);
    gdiff_overview_draw_ranges(GDIFF_OVERVIEW(widget), &event->area);
    gdiff_overview_draw_slider(GDIFF_OVERVIEW(widget), &event->area);

	return TRUE;
}


static void
gdiff_overview_adjustment_changed(GtkAdjustment *adjustment, gpointer data)
{
	GdiffOverview *overview;

	g_return_if_fail(adjustment != NULL);
	g_return_if_fail(data != NULL);
	
	overview = GDIFF_OVERVIEW(data);
	
	if (((overview->old_lower != adjustment->lower) ||
		 (overview->old_upper != adjustment->upper) ||
		 (overview->old_page_size != adjustment->page_size)) &&
		(overview->old_value == adjustment->value)) {
		if ((adjustment->lower == adjustment->upper) ||
			(overview->old_lower == (overview->old_upper - overview->old_page_size))) {
			adjustment->value = adjustment->lower;
			gtk_signal_emit_by_name(GTK_OBJECT(adjustment), "value_changed");
		}
    }

	if ((overview->old_value != adjustment->value) ||
		(overview->old_lower != adjustment->lower) ||
		(overview->old_upper != adjustment->upper) ||
		(overview->old_page_size != adjustment->page_size))	{
		overview->old_value = adjustment->value;
		overview->old_lower = adjustment->lower;
		overview->old_upper = adjustment->upper;
		overview->old_page_size = adjustment->page_size;
	}
}

static void
gdiff_overview_adjustment_value_changed(GtkAdjustment *adjustment, gpointer data)
{
	GdiffOverview *overview;

	g_return_if_fail(adjustment != NULL);
	g_return_if_fail(data != NULL);

	overview = GDIFF_OVERVIEW(data);

	if (overview->old_value != adjustment->value) {
		GdkRectangle rect;
		gint w, h;

		gdk_window_get_size(GTK_WIDGET(overview)->window, &w, &h);
		rect.x = rect.y = 0;
		rect.width = w;
		rect.height = h;

		/* redraw all areas */
		gdiff_overview_clear_background(GTK_WIDGET(overview), &rect);
		gdiff_overview_draw_ranges(overview, &rect);
		gdiff_overview_draw_slider(overview, &rect);
			
		overview->old_value = adjustment->value;
	}
}


static void
gdiff_overview_draw_slider(GdiffOverview *overview, const GdkRectangle *area)
{
	gint width;
	gint height;
    gint top;
	gint bottom;
	GdkRectangle s_rect;/* slider rectangle */
	GdkRectangle i_rect;/* intersect between s_rect and the area to draw */
	GtkWidget *widget = GTK_WIDGET(overview);

	g_return_if_fail(overview != NULL);
	g_return_if_fail(GDIFF_IS_OVERVIEW(overview));
	g_return_if_fail(GTK_WIDGET(overview)->window != NULL);

	gdk_window_get_size(GTK_WIDGET(overview)->window, &width, &height);

	/* default values */
	s_rect.x = 0;
	s_rect.y = overview->slider_y;
	s_rect.width = width;
	s_rect.height = overview->slider_length;

	/*XXX: calc slider height(length) */
	if ((overview->adjustment->page_size > 0) &&
		(overview->adjustment->lower != overview->adjustment->upper)) {
		if (overview->adjustment->page_size >
			(overview->adjustment->upper - overview->adjustment->lower))
            overview->adjustment->page_size = overview->adjustment->upper - overview->adjustment->lower;
		
		s_rect.height = (height * overview->adjustment->page_size /
						 (overview->adjustment->upper - overview->adjustment->lower));
          
		if (s_rect.height < MIN_SLIDER_LENGTH)
			s_rect.height = MIN_SLIDER_LENGTH;
	}
      
	/* calc slider pos y */
	top = 0;
	bottom = height - overview->slider_length;
	if (overview->adjustment->lower != (overview->adjustment->upper - overview->adjustment->page_size)) {
		s_rect.y = ((bottom - top) * (overview->adjustment->value - overview->adjustment->lower) /
					(overview->adjustment->upper - overview->adjustment->lower - overview->adjustment->page_size));
	}
	/* adjustment for special cases */
	if (s_rect.y < top)
		s_rect.y = top;
	else if (s_rect.y > bottom)
		s_rect.y = bottom;

	if (gdk_rectangle_intersect(area, &s_rect, &i_rect)) {
		gdk_draw_rectangle(widget->window, overview->xor_gc,
						   1, i_rect.x, i_rect.y, i_rect.width, i_rect.height);
	}
	overview->slider_y = s_rect.y;
	overview->slider_length = s_rect.height;
}

static void
gdiff_overview_clear_background(GtkWidget *widget, const GdkRectangle *area)
{
	g_return_if_fail(widget->window != NULL);
	gdk_window_clear_area (widget->window, area->x, area->y, area->width, area->height);
}

/**
 * gdiff_overview_draw_ranges:
 * Draw (diff)ranges, i.e. rectagles related to each differences.
 * Input:
 * GdiffOverview *overview;
 * const GdkRectangle *area; The area to draw, (usually)the exposed area.
 **/
static void
gdiff_overview_draw_ranges(GdiffOverview *overview, const GdkRectangle *area)
{
	GSList *node;
	PaintRange *prange;
	GdkRectangle r_rect;/* range rectangle */
	GdkRectangle i_rect;/* intersect between r_rect and the area to draw */
	gint w, h;

	gdk_window_get_size(GTK_WIDGET(overview)->window, &w, &h);
	r_rect.x = 0;
	r_rect.width = w;
	for (node = overview->range_list; node; node = node->next) {
		prange = node->data;
		r_rect.y = h * prange->begin;
		r_rect.height = h * (prange->end - prange->begin);

		/* adjustment for special cases */
		if (r_rect.height < 1)
			r_rect.height = 1;
		if (r_rect.y == h)
			r_rect.y--;
		
		if (gdk_rectangle_intersect(area, &r_rect, &i_rect)) {
			gdk_draw_rectangle(GTK_WIDGET(overview)->window,
							   overview->range_gc,
							   1,
							   i_rect.x, i_rect.y,
							   i_rect.width, i_rect.height);
		}
	}
}
