/********************************************************************\
 * AccWindow.c -- window for creating new accounts for xacc         *
 *                (X-Accountant)                                    *
 * Copyright (C) 1997 Robin D. Clark                                *
 * Copyright (C) 1997, 1998, 1999 Linas Vepstas                     *
 *                                                                  *
 * This program is free software; you can redistribute it and/or    *
 * modify it under the terms of the GNU General Public License as   *
 * published by the Free Software Foundation; either version 2 of   *
 * the License, or (at your option) any later version.              *
 *                                                                  *
 * This program is distributed in the hope that it will be useful,  *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of   *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the    *
 * GNU General Public License for more details.                     *
 *                                                                  *
 * You should have received a copy of the GNU General Public License*
 * along with this program; if not, write to the Free Software      *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.        *
 *                                                                  *
 *   Author: Rob Clark                                              *
 * Internet: rclark@cs.hmc.edu                                      *
 *  Address: 609 8th Street                                         *
 *           Huntington Beach, CA 92648-4632                        *
\********************************************************************/

#include <string.h>

#include <Xm/Xm.h>
#include <Xm/DialogS.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/RowColumn.h>
#include <Xm/LabelGP.h>
#include <Xm/ToggleB.h>
#include <Xm/PushB.h>
#include <Xm/Text.h>

#include "config.h"

#include "Account.h"
#include "AccountMenu.h"
#include "AccWindow.h"
#include "BuildMenu.h"
#include "Group.h"
#include "top-level.h"
#include "enriched-messages.h"
#include "MainWindow.h"
#include "PriceMenu.h"
#include "RegWindow.h"
#include "Refresh.h"
#include "FileDialog.h"
#include "TextBox.h"
#include "util.h"
#include "xtutil.h"

/* this global parameter determines whether the user is allowed to make
 * unsafe edits on the data file.  This includes changing the name of the 
 * currency and/or the security -- since any chagnes here can seriously scrmble
 * currency and security trading accounts.
 * hack alert --  this should be a user-configurable parameter.
 * see also unsafe_ops in the engine ... 
 */
int unsafe_edits = 1;

/* this define sets how many digits should be used in the pre-guessed account
 * codes.  Default to three; -- hack alert -- this should be a user-configurable
 * paramter.
 */
#define XACC_ACC_CODE_DIGITS 3

struct _accwindow {
                         /* The account type buttons:               */
  Widget dialog;
  Widget type_widgets[NUM_ACCOUNT_TYPES];

                         /* The text fields:                        */
  Widget name;           /* The account name text field             */
  Widget desc;           /* Account description text field          */
  Widget currency;       /* Account denomination field              */
  Widget security;       /* Account stock/security field            */
  Widget sec_label;      /* Account stock/security label            */
  Widget psource;        /* Account stock price source field        */
  Widget psrc_label;     /* Account stock price sourc label         */
  Widget code;           /* Account code field                      */

  AccountMenu  *accMenu;
  PriceMenu    *prcMenu;

  Account      *newacc;  /* tmp account for editing */
  AccountGroup *topgrp;  /* top of account heirarchy */

};

struct _editaccwindow {
  Widget dialog;
                         /* The text fields:                        */
  Widget  name;          /* The account name text field             */
  Widget  desc;          /* Account description text field          */
  Widget  currency;      /* Account denomination field              */
  Widget  security;      /* Account stock/security field            */
  Widget  psource;       /* Account stock price source field        */
  Widget  code;          /* Account code field                      */
  
  Account *account;      /* The account to edit                     */
  PriceMenu    *prcMenu;

};

struct _editnoteswindow {
  TextBox *tb;
  Account *account;      /* The account to edit                     */
};


static EditAccWindow   ** editAccList   = NULL;
static EditNotesWindow ** editNotesList = NULL;

/* This static indicates the debugging module that this .o belongs to.  */
static short module = MOD_GUI;

/** PROTOTYPES ******************************************************/
static void closeAccWindow      ( Widget mw, XtPointer cd, XtPointer cb );
static void closeEditAccWindow  ( Widget mw, XtPointer cd, XtPointer cb );
static void notesCB             ( Widget mw, XtPointer cd, XtPointer cb );
static void editNotesCB         ( Widget mw, XtPointer cd, XtPointer cb );
static void createCB            ( Widget mw, XtPointer cd, XtPointer cb );
static void finishEditCB        ( Widget mw, XtPointer cd, XtPointer cb );
static void selectAccountCB     ( Widget mw, XtPointer cd, XtPointer cb );
static void priceSourceCB       ( Widget mw, XtPointer cd, XtPointer cb );
static void selectTypeCB        ( Widget mw, XtPointer cd, XtPointer cb );
static void closeEditNotesWindow( Widget mw, XtPointer cd, XtPointer cb );

EditNotesWindow * editNotesWindow (Account *acc);

/********************************************************************\
 * accWindow                                                        *
 *   opens up a window to create a new account... the account is    * 
 *   actually created in the "create" callback                      * 
 *                                                                  * 
 * Args:   parent   - the parent of the window to be created        * 
 * Return: none                                                     *
\********************************************************************/
AccWindow *
accWindow (AccountGroup *grp)
  {
  int i;
  Widget    dialog, form, frame, rc, widget, 
            label, buttonform, group_menu, topwid;
  AccWindow *accData;
  char *tmp;
  Widget parent = gnc_get_ui_data();
  
  ENTER ("accWindow() \n");
  gnc_set_busy_cursor( parent );
  
  accData = (AccWindow *)_malloc(sizeof(AccWindow));

  accData->topgrp = grp;
  accData->newacc = xaccMallocAccount();
  
  /* force the size of the dialog so it is not resizable */
  dialog = XtVaCreatePopupShell( "dialog", 
				 xmDialogShellWidgetClass, parent,
				 XmNtitle,            SETUP_ACCT_STR,
				 XmNdeleteResponse,   XmDESTROY,
                                 /*
                                  * Let the window find its own size,
                                  * based on the font size.
				  * XmNwidth,     350,
				  * XmNminWidth,  350,
				  * XmNmaxWidth,  350,
				  * XmNheight,    340,
				  * XmNminHeight, 340,
				  * XmNmaxHeight, 340,
                                  */
                                 XmNresizable, FALSE,
                                 XmNallowShellResize, FALSE,
                                 /* allow window to be repositioned */
                                 /* XmNtransient, FALSE, */ 
				 NULL );
  
  XtAddCallback( dialog, XmNdestroyCallback, 
		 closeAccWindow, (XtPointer)accData );
  accData->dialog = dialog;
  
  /* The form to put everything in the dialog in */
  form = XtVaCreateWidget( "form", xmFormWidgetClass, dialog, 
                                 XmNrubberPositioning, TRUE,
                                 NULL );
  

  /******************************************************************\
   * The account type area                                          *
  \******************************************************************/
  
  /* Label tells the user what this area is */
  widget = XtVaCreateManagedWidget( ACC_TYPE_STR,
				    xmLabelGadgetClass, form,
				    XmNtopAttachment,   XmATTACH_FORM,
				    XmNtopOffset,       10,
				    XmNleftAttachment,  XmATTACH_FORM,
				    XmNleftOffset,      20,
				    NULL );
  
  /* Makes a nice looking frame around the radio buttons */
  frame = XtVaCreateManagedWidget( "frame", 
				   xmFrameWidgetClass, form,
				   XmNtopAttachment,   XmATTACH_WIDGET,
				   XmNtopWidget,       widget,
				   XmNleftAttachment,  XmATTACH_FORM,
				   XmNleftOffset,      20,
				   XmNrightAttachment, XmATTACH_FORM,
				   XmNrightOffset,     20,
				   NULL);
  
  /* A RowCol goes in the frame, to place the buttons */ 
  rc = XtVaCreateManagedWidget( "rowcol", 
				xmRowColumnWidgetClass, frame, 
				/*XmNentryAlignment,      XmALIGNMENT_CENTER,*/
				XmNorientation,         XmVERTICAL,
				XmNmarginHeight,        10,
				XmNmarginWidth,         10,
				/*XmNpacking,             XmPACK_TIGHT,*/
				XmNradioBehavior,       True,
				XmNnumColumns,          3,
				NULL );
  
  /* Create the buttons */
  for (i=0; i<NUM_ACCOUNT_TYPES; i++) {
    accData->type_widgets[i] = 
    XtVaCreateManagedWidget( xaccAccountGetTypeStr (i),
			     xmToggleButtonWidgetClass, rc,
			     XmNindicatorType,   XmONE_OF_MANY,
			     XmNset,             False,
			     NULL);
    XtAddCallback( accData->type_widgets[i], XmNarmCallback, 
		 selectTypeCB, (XtPointer)accData );  
  
  }
  
  /******************************************************************\
   * Text fields....                                                *
  \******************************************************************/
  
    
  label = 
    XtVaCreateManagedWidget( ACC_NAME_C_STR,
			     xmLabelGadgetClass, form,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       frame,
			     XmNtopOffset,       10, 
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition,   35,        /* 35% */
			     NULL );
  
  accData->name = 
    XtVaCreateManagedWidget( "nameText",
			     xmTextWidgetClass,  form,
			     XmNmaxLength,       40,
			     XmNcolumns,         25,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       frame,
			     XmNtopOffset,       10, 
			     XmNleftAttachment,  XmATTACH_POSITION,
			     XmNleftPosition,    35,        /* 35% */
			     NULL );
  
  label = 
    XtVaCreateManagedWidget( DESC_C_STR,
			     xmLabelGadgetClass, form,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       accData->name, 
			     XmNtopOffset,       10,
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition,   35,        /* 35% */
			     NULL );
  
  accData->desc = 
    XtVaCreateManagedWidget( "descText",
			     xmTextWidgetClass,  form,
			     XmNmaxLength,       40,
			     XmNcolumns,         30,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       accData->name,
			     XmNtopOffset,       10, 
			     XmNleftAttachment,  XmATTACH_POSITION,
			     XmNleftPosition,    35,        /* 35% */
			     NULL );
  
  label = 
    XtVaCreateManagedWidget( CURRENCY_C_STR,
			     xmLabelGadgetClass, form,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       accData->desc, 
			     XmNtopOffset,       10,
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition,   35,        /* 35% */
			     NULL );
  
  accData->currency = 
    XtVaCreateManagedWidget( "currencyText",
			     xmTextWidgetClass,  form,
			     XmNmaxLength,       20,
			     XmNcolumns,         15,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       accData->desc,
			     XmNtopOffset,       10, 
			     XmNleftAttachment,  XmATTACH_POSITION,
			     XmNleftPosition,    35,        /* 35% */
			     NULL );

  topwid = accData->currency;

  accData->sec_label = 
    XtVaCreateManagedWidget( SECURITY_C_STR,
			     xmLabelGadgetClass, form,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       topwid, 
			     XmNtopOffset,       10,
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition,   35,        /* 35% */
			     NULL );
  
  accData->security = 
    XtVaCreateManagedWidget( "securityText",
			     xmTextWidgetClass,  form,
			     XmNmaxLength,       20,
			     XmNcolumns,         15,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       topwid,
			     XmNtopOffset,       10, 
			     XmNleftAttachment,  XmATTACH_POSITION,
			     XmNleftPosition,    35,        /* 35% */
			     NULL );
  
  topwid = accData->security;

  label = 
    XtVaCreateManagedWidget( ACC_CODE_C_STR,
			     xmLabelGadgetClass, form,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       topwid, 
			     XmNtopOffset,       10,
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition,   35,        /* 35% */
			     NULL );
  
  tmp = xaccGroupGetNextFreeCode (grp, XACC_ACC_CODE_DIGITS);
  accData->code = 
    XtVaCreateManagedWidget( "codeText",
			     xmTextWidgetClass,  form,
			     XmNmaxLength,       20,
			     XmNcolumns,         15,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       topwid,
			     XmNtopOffset,       10, 
			     XmNleftAttachment,  XmATTACH_POSITION,
			     XmNleftPosition,    35,        /* 35% */
                             XmNvalue,           tmp,
			     NULL );
  free (tmp);
  topwid = accData->code;
  
  /* ---------------------------------------------------------- */
  label = 
    XtVaCreateManagedWidget( PARENT_ACC_C_STR,
			     xmLabelGadgetClass, form,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       topwid,
			     XmNtopOffset,       10,
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition,   35,        /* 35% */
			     NULL );
  
  /* put up a pulldown menu to let user choose an account */
  accData->accMenu = xaccBuildAccountMenu (grp, form, PICK_ONE_STR);
  group_menu = xaccGetAccountMenuWidget (accData->accMenu);
  xaccAccountMenuAddCallback (accData->accMenu, selectAccountCB, (XtPointer) accData);

  XtVaSetValues( group_menu,
                             XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       topwid,
			     XmNtopOffset,       10, 
			     XmNleftAttachment,  XmATTACH_POSITION,
			     XmNleftPosition,    35,        /* 35% */
                             NULL );
  
  XtManageChild (group_menu); 
  topwid = group_menu;

  /* ---------------------------------------------------------- */
  /* create another menu box to allow user to select a data source */
  accData->psrc_label = 
    XtVaCreateManagedWidget( QUOTE_SRC_C_STR,
			     xmLabelGadgetClass, form,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       topwid, 
			     XmNtopOffset,       10,
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition,   35,        /* 35% */
			     NULL );
  
  accData->prcMenu = xaccBuildPriceMenu (form, "");
  accData->psource = xaccGetPriceMenuWidget (accData->prcMenu);
  xaccPriceMenuAddCallback (accData->prcMenu, priceSourceCB, 
                            (XtPointer) accData->newacc);

  XtVaSetValues( accData->psource,
                             XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       topwid,
			     XmNtopOffset,       10, 
			     XmNleftAttachment,  XmATTACH_POSITION,
			     XmNleftPosition,    35,        /* 35% */
                             NULL );
  
  XtManageChild (accData->psource); 
  topwid = accData->psource;

  /******************************************************************\
   * The buttons at the bottom...                                   *
  \******************************************************************/

  buttonform = XtVaCreateWidget( "form", 
				 xmFormWidgetClass,   form,
				 XmNfractionBase,     5,
				 XmNtopAttachment,    XmATTACH_WIDGET,
				 XmNtopWidget,        topwid, 
				 XmNtopOffset,        10, 
				 XmNbottomAttachment, XmATTACH_FORM,
				 XmNbottomOffset,     10,
				 XmNleftAttachment,   XmATTACH_FORM,
				 XmNrightAttachment,  XmATTACH_FORM,
				 NULL );
  
  
  /* The "Notes" button opens a window to a few lines of notes about
   * the account */
  widget = XtVaCreateManagedWidget( NOTES_STR,
				    xmPushButtonWidgetClass, buttonform,
				    XmNtopAttachment,      XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       1,
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      2,  
				    XmNshowAsDefault,      True,
				    NULL );
  XtAddCallback( widget, XmNactivateCallback, 
		 notesCB, (XtPointer)accData );  
  
  /* The "Cancel" button */
  widget = XtVaCreateManagedWidget( CANCEL_STR,
				    xmPushButtonWidgetClass, buttonform,
				    XmNtopAttachment,      XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       2, 
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      3,  
				    XmNshowAsDefault,      True,
                                    XmNnavigationType,     XmTAB_GROUP, 
				    NULL );
  
  /* We need to do something to clean up memory too! */
  XtAddCallback( widget, XmNactivateCallback, 
		 destroyShellCB, (XtPointer)dialog );  
  
  /* The "Create" button creates the new account with the data 
   * that the user entered */
  widget = XtVaCreateManagedWidget( CREATE_STR,
				    xmPushButtonWidgetClass, buttonform,
				    XmNtopAttachment,      XmATTACH_FORM,
				    XmNleftAttachment,     XmATTACH_POSITION,
				    XmNleftPosition,       3, 
				    XmNrightAttachment,    XmATTACH_POSITION,
				    XmNrightPosition,      4,  
				    XmNshowAsDefault,      True,
                                    XmNnavigationType,     XmTAB_GROUP, 
				    NULL );
  
  XtAddCallback( widget, XmNactivateCallback, 
		 createCB, (XtPointer)accData );
  /* We need to do something to clean up memory too! */
  /* this is done at end of dialog.
   * XtAddCallback( widget, XmNactivateCallback, 
   * 		 destroyShellCB, (XtPointer)dialog );  
   */
    
  /******************************************************************/
  /* some configuration after all teh widgets have been created. */
  XtVaSetValues (accData->type_widgets[BANK],
                             XmNset,             True,
                             NULL);
  XtSetSensitive (accData->security,   False);
  XtSetSensitive (accData->sec_label,  False);
  XtSetSensitive (accData->psource,    False);
  XtSetSensitive (accData->psrc_label, False);

  XtManageChild(buttonform);
  
  /******************************************************************/
  XtManageChild(form);
  
  XtPopup( dialog, XtGrabNone );
  
  gnc_unset_busy_cursor( parent );

  LEAVE ("accWindow() \n");
  return accData;
  }

/********************************************************************\
 * closeAccWindow                                                   *
 *   frees memory allocated for an accWindow, and other cleanup     * 
 *   stuff                                                          * 
 *                                                                  * 
 * Args:   mw -                                                     * 
 *         cd - accData - the struct for the accWindow that is      * 
 *              being closed                                        * 
 *         cb -                                                     * 
 * Return: none                                                     *
\********************************************************************/
static void 
closeAccWindow( Widget mw, XtPointer cd, XtPointer cb )
  {
  AccWindow *accData = (AccWindow *)cd;
  ENTER("closeAccWindow\n");

  if(accData->newacc) xaccFreeAccount (accData->newacc);  

  xaccFreeAccountMenu (accData->accMenu);
  xaccFreePriceMenu (accData->prcMenu);
  accData->topgrp = NULL;
  _free(accData);
  LEAVE("closeAccWindow\n");
  }

/********************************************************************\
 * editAccWindow                                                    *
 *   opens up a window to edit an account                           * 
 *                                                                  * 
 * Args:   parent   - the parent of the window to be created        * 
 *         account  - the account to edit                           * 
 * Return: none                                                     *
\********************************************************************/
EditAccWindow *
editAccWindow( Account *acc )
{

  Widget dialog, form, widget, label, buttonform, lasttext;
  EditAccWindow *editAccData;
  char * str;
  int acctype;
  Widget parent = gnc_get_ui_data();
  ENTER("editAccWindow\n");
  
  FETCH_FROM_LIST (EditAccWindow, editAccList, acc, account, editAccData);

  gnc_set_busy_cursor( parent );
  
  /* force the size of the dialog so it is not resizable */
  dialog = XtVaCreatePopupShell( "dialog", 
				 xmDialogShellWidgetClass, parent,
				 XmNtitle,            EDIT_ACCT_STR,
				 XmNdeleteResponse,   XmDESTROY,
                                 XmNresizable, FALSE,
                                 XmNallowShellResize, FALSE,
                                 XmNtransient, FALSE,  /* allow window to be repositioned */
				 NULL );
  editAccData->dialog = dialog;
  
  XtAddCallback( dialog, XmNdestroyCallback, 
		 closeEditAccWindow, (XtPointer)editAccData );
  
  /* The form to put everything in the dialog in */
  form = XtVaCreateWidget( "form", xmFormWidgetClass, dialog, 
                                 XmNrubberPositioning, TRUE,
                                 NULL );
  /******************************************************************\
   * Text fields....                                                *
  \******************************************************************/
  
  label = 
    XtVaCreateManagedWidget( ACC_NAME_C_STR,
			     xmLabelGadgetClass, form,
			     XmNtopAttachment,   XmATTACH_FORM,
			     XmNtopOffset,       10,
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition,   35,        /* 35% */
			     NULL );
  
  str = xaccAccountGetName (acc);
  editAccData->name = 
    XtVaCreateManagedWidget( "nameText",
			     xmTextWidgetClass,  form,
			     XmNmaxLength,       40,
			     XmNcolumns,         25,
			     XmNvalue,           str,
			     XmNeditable,        True,
			     XmNtopAttachment,   XmATTACH_FORM,
			     XmNtopOffset,       10,
			     XmNleftAttachment,  XmATTACH_POSITION,
			     XmNleftPosition,    35,        /* 35% */
			     NULL );
  
  label = 
    XtVaCreateManagedWidget( DESC_C_STR,
			     xmLabelGadgetClass, form,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       editAccData->name,
			     XmNtopOffset,       10,
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition,   35,        /* 35% */
			     NULL );
  
  str = xaccAccountGetDescription (acc);
  editAccData->desc = 
    XtVaCreateManagedWidget( "descText",
			     xmTextWidgetClass,  form,
			     XmNmaxLength,       40,
			     XmNcolumns,         30,
			     XmNvalue,           str,
			     XmNeditable,        True,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       editAccData->name,
			     XmNtopOffset,       10,
			     XmNleftAttachment,  XmATTACH_POSITION,
			     XmNleftPosition,    35,        /* 35% */
			     NULL );
  
  label = 
    XtVaCreateManagedWidget( CURRENCY_C_STR,
			     xmLabelGadgetClass, form,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       editAccData->desc, 
			     XmNtopOffset,       10,
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition,   35,        /* 35% */
			     NULL );
  
  str = xaccAccountGetCurrency (acc);
  if (!str) str = "";
  editAccData->currency = 
    XtVaCreateManagedWidget( "currencyText",
			     xmTextWidgetClass,  form,
			     XmNmaxLength,       20,
			     XmNcolumns,         15,
			     XmNvalue,           str,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       editAccData->desc,
			     XmNtopOffset,       10, 
			     XmNleftAttachment,  XmATTACH_POSITION,
			     XmNleftPosition,    35,        /* 35% */
			     NULL );
  
  if (unsafe_edits) {
     /* normally, user is not allowed to change this value. */
     XtSetSensitive (editAccData->currency,  True);
     XtSetSensitive (label,  True);
  } else
  if (0x0 != str[0]) {
    /* allow user to change this value only if not yet set */
    XtSetSensitive (editAccData->currency,  False);
    XtSetSensitive (label,  False);
  }
  lasttext = editAccData->currency;

  /* ------------------------------------------------------------ */
  /* the security field should only be displayed for stock trading 
   * accounts, and currency trading accounts 
   */

  acctype = xaccAccountGetType (acc);
  if ((STOCK == acctype) || (MUTUAL == acctype) || (CURRENCY== acctype)) {
    label = 
      XtVaCreateManagedWidget( SECURITY_C_STR,
			     xmLabelGadgetClass, form,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       lasttext, 
			     XmNtopOffset,       10,
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition,   35,        /* 35% */
			     NULL );
  
    str = xaccAccountGetSecurity (acc);
    if (!str) str = "";
    editAccData->security = 
      XtVaCreateManagedWidget( "securityText",
			     xmTextWidgetClass,  form,
			     XmNmaxLength,       20,
			     XmNcolumns,         15,
			     XmNvalue,           str,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       lasttext,
			     XmNtopOffset,       10, 
			     XmNleftAttachment,  XmATTACH_POSITION,
			     XmNleftPosition,    35,        /* 35% */
			     NULL );
  
    if (unsafe_edits) {
       /* normally, user is not allowed to change this value. */
       XtSetSensitive (editAccData->security,  True);
       XtSetSensitive (label,  True);
    } else
    if (0x0 != str[0]) {
      /* allow user to change this value only if not yet set */
      XtSetSensitive (editAccData->security,  False);
      XtSetSensitive (label,  False);
    }


    lasttext = editAccData->security;
    /* ------------------------------------------------------------ */
    label = 
      XtVaCreateManagedWidget( QUOTE_SRC_C_STR,
			     xmLabelGadgetClass, form,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       lasttext, 
			     XmNtopOffset,       10,
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition,   35,        /* 35% */
			     NULL );
  
    editAccData->prcMenu = xaccBuildPriceMenu (form, "");
    editAccData->psource = xaccGetPriceMenuWidget (editAccData->prcMenu);
    xaccPriceMenuAddCallback (editAccData->prcMenu, priceSourceCB, 
                            (XtPointer) editAccData->account);

    {
    /* make sure the the menu is showing the correct choice */
    AccInfo *accinfo = xaccAccountGetAccInfo (editAccData->account);
    InvAcct *invacct = xaccCastToInvAcct (accinfo);
    str = "";
    if (invacct) str = xaccInvAcctGetPriceSrc (invacct);
    xaccPriceMenuSetValue (editAccData->prcMenu, str);
    }
    XtVaSetValues( editAccData->psource,
                             XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       lasttext,
			     XmNtopOffset,       10, 
			     XmNleftAttachment,  XmATTACH_POSITION,
			     XmNleftPosition,    35,        /* 35% */
                             NULL );
  
    XtManageChild (editAccData->psource); 
    lasttext = editAccData->psource;

  } else {
    editAccData->security = 0;
    editAccData->psource = 0;
    editAccData->prcMenu = NULL;
  }

  label = 
    XtVaCreateManagedWidget( ACC_CODE_C_STR,
			     xmLabelGadgetClass, form,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       lasttext, 
			     XmNtopOffset,       10,
			     XmNrightAttachment, XmATTACH_POSITION,
			     XmNrightPosition,   35,        /* 35% */
			     NULL );
  
  str = xaccAccountGetCode (acc);
  editAccData->code = 
    XtVaCreateManagedWidget( "codeText",
			     xmTextWidgetClass,  form,
			     XmNmaxLength,       20,
			     XmNcolumns,         15,
			     XmNvalue,           str,
			     XmNeditable,        True,
			     XmNtopAttachment,   XmATTACH_WIDGET,
			     XmNtopWidget,       lasttext,
			     XmNtopOffset,       10,
			     XmNleftAttachment,  XmATTACH_POSITION,
			     XmNleftPosition,    35,        /* 35% */
			     NULL );
  lasttext = editAccData->code;
  
  /******************************************************************\
   * The buttons at the bottom...                                   *
  \******************************************************************/
  
  buttonform = 
    XtVaCreateWidget( "form", 
		      xmFormWidgetClass,   form,
		      XmNfractionBase,     5,
		      XmNtopAttachment,    XmATTACH_WIDGET,
		      XmNtopWidget,        lasttext,
		      XmNtopOffset,        10,
		      XmNbottomAttachment, XmATTACH_FORM,
		      XmNbottomOffset,     10,
		      XmNleftAttachment,   XmATTACH_FORM,
		      XmNrightAttachment,  XmATTACH_FORM,
		      NULL );
  
  /* The "Notes" button opens a window to a few lines of notes about
   * the account */
  widget = 
    XtVaCreateManagedWidget( NOTES_STR, 
			     xmPushButtonWidgetClass, buttonform,
			     XmNtopAttachment,      XmATTACH_FORM,
			     XmNleftAttachment,     XmATTACH_POSITION,
			     XmNleftPosition,       1,
			     XmNrightAttachment,    XmATTACH_POSITION,
			     XmNrightPosition,      2,
			     XmNshowAsDefault,      True,
			     NULL );
  
  XtAddCallback( widget, XmNactivateCallback, 
		 editNotesCB, (XtPointer)editAccData );  
  
  /* The "Cancel" button */
  widget = 
    XtVaCreateManagedWidget( CANCEL_STR, 
			     xmPushButtonWidgetClass, buttonform,
			     XmNtopAttachment,      XmATTACH_FORM,
			     XmNleftAttachment,     XmATTACH_POSITION,
			     XmNleftPosition,       2,
			     XmNrightAttachment,    XmATTACH_POSITION,
			     XmNrightPosition,      3,
			     XmNshowAsDefault,      True,
			     NULL );
  
  /* We need to do something to clean up memory too! */
  XtAddCallback( widget, XmNactivateCallback, 
		 destroyShellCB, (XtPointer)dialog );  
  
  /* The "OK" button commits changes to the new account with the data 
   * that the user entered */
  widget = 
    XtVaCreateManagedWidget( OK_STR,
			     xmPushButtonWidgetClass, buttonform,
			     XmNtopAttachment,      XmATTACH_FORM,
			     XmNleftAttachment,     XmATTACH_POSITION,
			     XmNleftPosition,       3,
			     XmNrightAttachment,    XmATTACH_POSITION,
			     XmNrightPosition,      4,
			     XmNshowAsDefault,      True,
			     NULL );
  
  XtAddCallback( widget, XmNactivateCallback, 
		 finishEditCB, (XtPointer)editAccData );
  /* We need to do something to clean up memory too! */
  XtAddCallback( widget, XmNactivateCallback, 
		 destroyShellCB, (XtPointer)dialog );  
    
  XtManageChild(buttonform);
  
  /******************************************************************/
  XtManageChild(form);
  
  XtPopup( dialog, XtGrabNone );
  
  gnc_unset_busy_cursor( parent );

  LEAVE("editAccWindow\n");
  return editAccData;
}

/********************************************************************\
 * Don't delete any structures -- the close callback wil do this    *
\********************************************************************/

void xaccDestroyEditAccWindow (Account * acc)
{
  EditAccWindow *editAccData;

  FIND_IN_LIST (EditAccWindow,editAccList,acc,account,editAccData); 
  if (!editAccData) return;
  XtDestroyWidget (editAccData->dialog);
}

/********************************************************************\
 * closeEditAccWindow                                               *
 *   frees memory allocated for an editAccWindow, and other cleanup * 
 *   stuff                                                          * 
 *                                                                  * 
 * Args:   mw -                                                     * 
 *         cd - editAccData - the struct for the editAccWindow      * 
 *              that is being closed                                * 
 *         cb -                                                     * 
 * Return: none                                                     *
\********************************************************************/
static void 
closeEditAccWindow( Widget mw, XtPointer cd, XtPointer cb )
  {
  EditAccWindow *editAccData = (EditAccWindow *)cd;
  Account *acc = editAccData->account;

  ENTER("closeEditAccWindow\n");
  xaccFreePriceMenu (editAccData->prcMenu);
  REMOVE_FROM_LIST (EditAccWindow,editAccList,acc,account); 
  free(editAccData);
  LEAVE("closeEditAccWindow\n");
  }

/********************************************************************\
 * notesCB -- called when the user presses the "Notes" Button       * 
 *                                                                  * 
 * Args:   mw - the widget that called us                           * 
 *         cd - accData - the struct that has the notes text in it  * 
 *         cb -                                                     * 
 * Return: none                                                     * 
 * Global: toplevel    - the toplevel widget                        *
\********************************************************************/
static void
notesCB( Widget mw, XtPointer cd, XtPointer cb )
{
  AccWindow *accData = (AccWindow *)cd;
  Account *acc = accData -> newacc;
  editNotesWindow (acc);
  /* hack alert -- should raise window to the top */
}

/********************************************************************\
 * editNotesCB -- called when the user presses the "Notes" Button   * 
 *                                                                  * 
 * Args:   mw - the widget that called us                           * 
 *         cd - accData - the struct that has the notes text in it  * 
 *         cb -                                                     * 
 * Return: none                                                     * 
 * Global: toplevel    - the toplevel widget                        *
\********************************************************************/
static void
editNotesCB( Widget mw, XtPointer cd, XtPointer cb )
{
  EditAccWindow *editAccData = (EditAccWindow *)cd;
  Account *acc = editAccData -> account;
  editNotesWindow (acc);
  /* hack alert -- should raise window to the top */
}

/********************************************************************\
 * createCB -- creates the new account from data in the newaccount  * 
 *   dialog window                                                  * 
 *                                                                  * 
 * Args:   mw - the widget that called us                           * 
 *         cd - accData - the struct of data associated with this   * 
 *              accWindow.                                          * 
 *         cb -                                                     * 
 * Return: none                                                     * 
 * Global: data        - the data from the datafile                 *
 *         toplevel    - the toplevel widget                        *
\********************************************************************/
static void
createCB( Widget mw, XtPointer cd, XtPointer cb )
  {
  int acctype;
  Transaction *trans;
  Account     *acc, *parent_acc;
  AccWindow   *accData = (AccWindow *)cd;
  Boolean set = False;

  String name, tmp;
  
  ENTER ("createCB for newacct\n");
  name = XmTextGetString(accData->name);

  /* The account has to have a name! */
  if( strcmp( name, "" ) == 0 ) {
    gnc_error_dialog (ACC_NO_NAME_MSG);
    XtFree (name);
    return;
  }
  
  acc = accData->newacc;
  accData->newacc = NULL;

  xaccAccountBeginEdit (acc, 0);
  xaccAccountSetName (acc, name);
  XtFree (name);

  tmp = XmTextGetString(accData->desc);
  xaccAccountSetDescription (acc, tmp);
  XtFree (tmp);
  
  tmp = XmTextGetString(accData->code);
  xaccAccountSetCode (acc, tmp);
  XtFree (tmp);
  
  tmp = XmTextGetString(accData->currency);
  xaccAccountSetCurrency (acc, tmp);
  XtFree (tmp);
  
  /* figure out account type */
  for (acctype=0; acctype<NUM_ACCOUNT_TYPES; acctype++) {
    XtVaGetValues( accData->type_widgets[acctype], XmNset, &set, NULL );
    if(set) {
       xaccAccountSetType (acc, acctype);
       break;
    }
  }

  /* can declare a security for only certain account types */
  acctype = xaccAccountGetType (acc);
  if ((STOCK == acctype) || (MUTUAL == acctype) || (CURRENCY== acctype)) {
     tmp = XmTextGetString(accData->security);
     xaccAccountSetSecurity (acc, tmp);
     XtFree (tmp);
  }
  
  /* Add an opening balance transaction (as the first transaction) */
  trans = xaccMallocTransaction();
 
  xaccTransBeginEdit (trans, 1);
  xaccTransSetDateToday (trans);
  xaccTransSetDescription (trans, OPEN_BALN_STR);
  xaccTransCommitEdit (trans);

  /* add the new transaction to the account */
  xaccAccountInsertSplit (acc, xaccTransGetSplit (trans, 0) );
  
  /* once the account is set up, add it to account group 
   * If the user indicated a parent acccount, make it a 
   * sub account of that */
  parent_acc = xaccGetAccountMenuSelection (accData->accMenu);
  if (parent_acc) {
    xaccInsertSubAccount (parent_acc, acc);
  } else {
    AccountGroup * topmostgroup = xaccGetAccountMenuTop (accData->accMenu);
    xaccGroupInsertAccount( topmostgroup, acc );
  }
  xaccAccountCommitEdit (acc);
  
  /* make sure the accountlist is updated to reflect the new account */
  gnc_refresh_main_window();
  gnc_group_ui_refresh(gncGetCurrentGroup());

  /* open up the account window for the user */
  regWindowSimple (acc);

  /* if we got to here, tear down the dialog window */
  XtDestroyWidget (accData->dialog);
  LEAVE ("createCB for newacct\n");
  }

/********************************************************************\
 * finishEditCB -- records the edits made in the editAccWindow      * 
 *                                                                  * 
 * Args:   mw - the widget that called us                           * 
 *         cd - editAccData - the struct of data associated with    *
 *              the EditAccWindow                                   * 
 *         cb -                                                     * 
 * Return: none                                                     * 
 * Global: data        - the data from the datafile                 *
\********************************************************************/
static void
finishEditCB( Widget mw, XtPointer cd, XtPointer cb )
  {
  EditAccWindow *editAccData = (EditAccWindow *)cd;
  Account *acc = editAccData->account;
  String name, tmp;
  char * oldcurr;
  
  ENTER ("finishEditC\n");
  xaccAccountBeginEdit (acc, 0);

  /* The account has to have a name! */
  name = XmTextGetString(editAccData->name);
  if( strcmp( name, "" ) != 0 )
    {
    xaccAccountSetName (acc, name);
    }
  XtFree(name);
  
  tmp = XmTextGetString(editAccData->desc);
  xaccAccountSetDescription (acc, tmp);
  XtFree(tmp);

  tmp = XmTextGetString(editAccData->code);
  xaccAccountSetCode (acc, tmp);
  XtFree(tmp);

  oldcurr = xaccAccountGetCurrency (acc);
  if (!oldcurr) oldcurr = "";
  if ((0x0 == oldcurr[0]) || unsafe_edits) {
    char * newcurr = XmTextGetString (editAccData->currency);
    if (0x0 == oldcurr[0]) {
       xaccAccountSetCurrency (acc, newcurr);
    } else 
    if (strcmp (oldcurr, newcurr)) {
       char buff[1000];
       sprintf (buff, EDIT_CURRENCY_MSG, oldcurr, newcurr);
       if (gnc_verify_dialog (buff, GNC_T)) {
          xaccAccountSetCurrency (acc, newcurr);
       }
    }
    XtFree(newcurr);
  }

  /* if this field wasn't displayed, then don't try to deal with it */
  if (editAccData->security) {
     char * oldsec = xaccAccountGetSecurity (acc);
     if (!oldsec) oldsec = "";
     if ((0x0 == oldsec[0]) || unsafe_edits) {
       char * newsec = XmTextGetString (editAccData->security);
       if (0x0 == oldsec[0]) {
          xaccAccountSetSecurity (acc, newsec);
       } else 
       if (strcmp (oldsec, newsec)) {
          char buff[1000];
          sprintf (buff, EDIT_SECURITY_MSG, oldsec, newsec);
          if (gnc_verify_dialog (buff, GNC_T)) {
             xaccAccountSetSecurity (acc, newsec);
          }
       }
       XtFree(newsec);
     }
  }  
  xaccAccountCommitEdit (acc);
  
  gnc_refresh_main_window();
  gnc_group_ui_refresh(gncGetCurrentGroup());
  LEAVE ("finishEditC\n");
  }

/********************************************************************\
 * selectTypeCB -- checks the use account selection              * 
 * 
\********************************************************************/
static void
selectTypeCB( Widget mw, XtPointer cd, XtPointer cb )
{
  int i, but=0;
  AccWindow *menu = (AccWindow *) cd;

  /* figure out which radio button might be set */
  for (i=0; i<NUM_ACCOUNT_TYPES; i++) {
    if (mw == menu->type_widgets[i]) { but = i; break; }
  }

  /* may set a security type only for stocks, mutual funds,
   * and currency trading accounts
   */
  switch (but) {
       case STOCK:
       case MUTUAL:
       case CURRENCY:
          XtSetSensitive (menu->security,   True);
          XtSetSensitive (menu->sec_label,  True);
          XtSetSensitive (menu->psource,    True);
          XtSetSensitive (menu->psrc_label, True);
          break;

       case BANK:
       case CASH:
       case ASSET:
       case CHECKING:
       case SAVINGS:
       case MONEYMRKT:
       case CREDITLINE:
       case LIABILITY:
       case CREDIT:
       case INCOME:
       case EXPENSE:
       case EQUITY:
       default:
          XtSetSensitive (menu->security,   False);
          XtSetSensitive (menu->sec_label,  False);
          XtSetSensitive (menu->psource,    False);
          XtSetSensitive (menu->psrc_label, False);
  }
}

/********************************************************************\
 * priceSourceCB -- set the source for stock & mutual fund pricing info   * 
 *                                                                  * 
\********************************************************************/
static void
priceSourceCB( Widget mw, XtPointer cd, XtPointer cb )
{
  char * src = (char *) cb;
  Account *acc = (Account *) cd;
  AccInfo *accinfo = xaccAccountGetAccInfo (acc);
  InvAcct *invacct;

  invacct = xaccCastToInvAcct (accinfo);
  if (invacct) {
    xaccInvAcctSetPriceSrc (invacct, src);
  }
}

/********************************************************************\
 * selectAccountCB -- checks the use account selection              * 
 * 
 * Basically, sub-account *must* be of the same category as thier 
 * parent accounts, otherwise chaos will errupt.  The five basic 
 * categories are asset, liability, income,. expense, and equity.
 *
 * Currently, there are four subcategories for asset accounts:
 * banks, cash, stocks, bonds, mutual funds.
 *
 *                                                                  * 
\********************************************************************/
static void
selectAccountCB( Widget mw, XtPointer cd, XtPointer cb )
{
  int i, but=0;
  Boolean set;
  AccWindow *menu = (AccWindow *) cd;
  Account *acc = (Account *) cb;
  int acc_type;

  /* unset any pressed radio buttons in preparation for 
   * setting insensitive of some of them. 
   */

  /* figure out which radio button might be set */
  for (i=0; i<NUM_ACCOUNT_TYPES; i++) {
    XtVaGetValues( menu->type_widgets[i], XmNset, &set, NULL );
    if(set) { but = i; break; }
  }

  if (acc) {
    char * tmp;
    /* gues an account code, and fill it in */
    tmp = xaccAccountGetNextChildCode (acc, XACC_ACC_CODE_DIGITS);
    XtVaSetValues (menu->code, XmNvalue, tmp, NULL);
    free (tmp);

    acc_type = xaccAccountGetType (acc);
    switch (acc_type) {
       case BANK:
       case CASH:
       case ASSET:
       case STOCK:
       case MUTUAL:
       case CURRENCY:
       case CHECKING:
       case SAVINGS:
       case MONEYMRKT:
       case CREDITLINE:
          XtSetSensitive (menu->type_widgets[BANK],      True);
          XtSetSensitive (menu->type_widgets[CASH],      True);
          XtSetSensitive (menu->type_widgets[ASSET],     True);
          XtSetSensitive (menu->type_widgets[STOCK],     True);
          XtSetSensitive (menu->type_widgets[MUTUAL],    True);
          XtSetSensitive (menu->type_widgets[CURRENCY],  True);
          XtSetSensitive (menu->type_widgets[LIABILITY], False);
          XtSetSensitive (menu->type_widgets[CREDIT],    False);
          XtSetSensitive (menu->type_widgets[INCOME],    False);
          XtSetSensitive (menu->type_widgets[EXPENSE],   False);
          XtSetSensitive (menu->type_widgets[EQUITY],    False);
#ifdef FUTURE_STUFF
          XtSetSensitive (menu->type_widgets[CHECKING],  True);
          XtSetSensitive (menu->type_widgets[SAVINGS],   True);
          XtSetSensitive (menu->type_widgets[MONEYMRKT], True);
          XtSetSensitive (menu->type_widgets[CREDITLINE],True);
#endif /* FUTURE_STUFF */

          /* unset unavailable buttons */
          XtVaSetValues (menu->type_widgets[LIABILITY], XmNset, False, NULL);
          XtVaSetValues (menu->type_widgets[CREDIT],    XmNset, False, NULL);
          XtVaSetValues (menu->type_widgets[INCOME],    XmNset, False, NULL);
          XtVaSetValues (menu->type_widgets[EXPENSE],   XmNset, False, NULL);
          XtVaSetValues (menu->type_widgets[EQUITY],    XmNset, False, NULL);

          /* set a default, if an inapporpriate button is pushed */
          if ((BANK   != but) && (CASH     != but) &&
              (ASSET  != but) && (STOCK    != but) &&
              (MUTUAL != but) && (CURRENCY != but) ) {
             XtVaSetValues (menu->type_widgets[acc_type], XmNset, True, NULL);
          }
          break;

       case LIABILITY:
       case CREDIT:
          XtSetSensitive (menu->type_widgets[BANK],      False);
          XtSetSensitive (menu->type_widgets[CASH],      False);
          XtSetSensitive (menu->type_widgets[ASSET],     False);
          XtSetSensitive (menu->type_widgets[STOCK],     False);
          XtSetSensitive (menu->type_widgets[MUTUAL],    False);
          XtSetSensitive (menu->type_widgets[CURRENCY],  False);
          XtSetSensitive (menu->type_widgets[LIABILITY], True);
          XtSetSensitive (menu->type_widgets[CREDIT],    True);
          XtSetSensitive (menu->type_widgets[INCOME],    False);
          XtSetSensitive (menu->type_widgets[EXPENSE],   False);
          XtSetSensitive (menu->type_widgets[EQUITY],    False);
#ifdef FUTURE_STUFF
          XtSetSensitive (menu->type_widgets[CHECKING],  False);
          XtSetSensitive (menu->type_widgets[SAVINGS],   False);
          XtSetSensitive (menu->type_widgets[MONEYMRKT], False);
          XtSetSensitive (menu->type_widgets[CREDITLINE],False);
#endif /* FUTURE_STUFF */

          /* unset unavailable buttons */
          for (i=0; i<NUM_ACCOUNT_TYPES; i++) {
             XtVaSetValues (menu->type_widgets[i],      XmNset, False, NULL);
          }

          /* set a default, if an inapporpriate button is pushed */
          if ((LIABILITY != but) && (CREDIT != but)) {
             XtVaSetValues (menu->type_widgets[acc_type], XmNset, True, NULL);
          }
          break;

       case INCOME:
          XtSetSensitive (menu->type_widgets[BANK],      False);
          XtSetSensitive (menu->type_widgets[CASH],      False);
          XtSetSensitive (menu->type_widgets[ASSET],     False);
          XtSetSensitive (menu->type_widgets[STOCK],     False);
          XtSetSensitive (menu->type_widgets[MUTUAL],    False);
          XtSetSensitive (menu->type_widgets[CURRENCY],  False);
          XtSetSensitive (menu->type_widgets[LIABILITY], False);
          XtSetSensitive (menu->type_widgets[CREDIT],    False);
          XtSetSensitive (menu->type_widgets[INCOME],    True);
          XtSetSensitive (menu->type_widgets[EXPENSE],   False);
          XtSetSensitive (menu->type_widgets[EQUITY],    False);
#ifdef FUTURE_STUFF
          XtSetSensitive (menu->type_widgets[CHECKING],  False);
          XtSetSensitive (menu->type_widgets[SAVINGS],   False);
          XtSetSensitive (menu->type_widgets[MONEYMRKT], False);
          XtSetSensitive (menu->type_widgets[CREDITLINE],False);
#endif /* FUTURE_STUFF */

          /* unset unavailable buttons */
          for (i=0; i<NUM_ACCOUNT_TYPES; i++) {
             XtVaSetValues (menu->type_widgets[i],      XmNset, False, NULL);
          }

          /* set a default, if an inapporpriate button is pushed */
          XtVaSetValues (menu->type_widgets[acc_type], XmNset, True, NULL);
          break;

       case EXPENSE:
          XtSetSensitive (menu->type_widgets[BANK],      False);
          XtSetSensitive (menu->type_widgets[CASH],      False);
          XtSetSensitive (menu->type_widgets[ASSET],     False);
          XtSetSensitive (menu->type_widgets[STOCK],     False);
          XtSetSensitive (menu->type_widgets[MUTUAL],    False);
          XtSetSensitive (menu->type_widgets[CURRENCY],  False);
          XtSetSensitive (menu->type_widgets[LIABILITY], False);
          XtSetSensitive (menu->type_widgets[CREDIT],    False);
          XtSetSensitive (menu->type_widgets[INCOME],    False);
          XtSetSensitive (menu->type_widgets[EXPENSE],   True);
          XtSetSensitive (menu->type_widgets[EQUITY],    False);
#ifdef FUTURE_STUFF
          XtSetSensitive (menu->type_widgets[CHECKING],  False);
          XtSetSensitive (menu->type_widgets[SAVINGS],   False);
          XtSetSensitive (menu->type_widgets[MONEYMRKT], False);
          XtSetSensitive (menu->type_widgets[CREDITLINE],False);
#endif /* FUTURE_STUFF */

          /* unset unavailable buttons */
          for (i=0; i<NUM_ACCOUNT_TYPES; i++) {
             XtVaSetValues (menu->type_widgets[i],      XmNset, False, NULL);
          }

          /* set a default, if an inapporpriate button is pushed */
          XtVaSetValues (menu->type_widgets[acc_type], XmNset, True, NULL);
          break;

       case EQUITY:
          XtSetSensitive (menu->type_widgets[BANK],      False);
          XtSetSensitive (menu->type_widgets[CASH],      False);
          XtSetSensitive (menu->type_widgets[ASSET],     False);
          XtSetSensitive (menu->type_widgets[STOCK],     False);
          XtSetSensitive (menu->type_widgets[MUTUAL],    False);
          XtSetSensitive (menu->type_widgets[CURRENCY],  False);
          XtSetSensitive (menu->type_widgets[LIABILITY], False);
          XtSetSensitive (menu->type_widgets[CREDIT],    False);
          XtSetSensitive (menu->type_widgets[INCOME],    False);
          XtSetSensitive (menu->type_widgets[EXPENSE],   False);
          XtSetSensitive (menu->type_widgets[EQUITY],    True);
#ifdef FUTURE_STUFF
          XtSetSensitive (menu->type_widgets[CHECKING],  False);
          XtSetSensitive (menu->type_widgets[SAVINGS],   False);
          XtSetSensitive (menu->type_widgets[MONEYMRKT], False);
          XtSetSensitive (menu->type_widgets[CREDITLINE],False);
#endif /* FUTURE_STUFF */

          /* unset unavailable buttons */
          for (i=0; i<NUM_ACCOUNT_TYPES; i++) {
             XtVaSetValues (menu->type_widgets[i],      XmNset, False, NULL);
          }

          /* set a default, if an inapporpriate button is pushed */
          XtVaSetValues (menu->type_widgets[acc_type], XmNset, True, NULL);
          break;

    }
  } else {
    char * tmp;
    /* guess an account code, and fill it in */
    tmp = xaccGroupGetNextFreeCode (menu->topgrp, XACC_ACC_CODE_DIGITS);
    XtVaSetValues (menu->code, XmNvalue, tmp, NULL);
    free (tmp);

     XtSetSensitive (menu->type_widgets[BANK],      True);
     XtSetSensitive (menu->type_widgets[CASH],      True);
     XtSetSensitive (menu->type_widgets[ASSET],     True);
     XtSetSensitive (menu->type_widgets[CREDIT],    True);
     XtSetSensitive (menu->type_widgets[LIABILITY], True);
     XtSetSensitive (menu->type_widgets[STOCK],     True);
     XtSetSensitive (menu->type_widgets[MUTUAL],    True);
     XtSetSensitive (menu->type_widgets[CURRENCY],  True);
     XtSetSensitive (menu->type_widgets[INCOME],    True);
     XtSetSensitive (menu->type_widgets[EXPENSE],   True);
     XtSetSensitive (menu->type_widgets[EQUITY],    True);
#ifdef FUTURE_STUFF
     XtSetSensitive (menu->type_widgets[CHECKING],  True);
     XtSetSensitive (menu->type_widgets[SAVINGS],   True);
     XtSetSensitive (menu->type_widgets[MONEYMRKT], True);
     XtSetSensitive (menu->type_widgets[CREDITLINE],True);
#endif /* FUTURE_STUFF */
  }
}

/********************************************************************\
 *                                                                  * 
\********************************************************************/

EditNotesWindow *
editNotesWindow (Account *acc)
{
  EditNotesWindow *enw;
  char *notes;

  FETCH_FROM_LIST (EditNotesWindow, editNotesList, acc, account, enw);
  
  notes = xaccAccountGetNotes (acc);
  enw->tb = textBox( gnc_get_ui_data(), NOTES_STR, 
                     &notes,
                     closeEditNotesWindow, enw);
  return enw;
}

/********************************************************************\
 * don't delete any structures; the close callack will do this       *
\********************************************************************/

void 
xaccDestroyEditNotesWindow (Account *acc)
{
  EditNotesWindow *edwin;

  FIND_IN_LIST (EditNotesWindow,editNotesList,acc,account,edwin) 
  if (!edwin) return;

  xaccDestroyTextBox (edwin->tb);
}

/********************************************************************\
\********************************************************************/

static void 
closeEditNotesWindow( Widget mw, XtPointer cd, XtPointer cb )
{
  EditNotesWindow *enw = (EditNotesWindow *) cd;
  Account * acc = enw->account;
  char * newnotes = (char *) cb;
  char *notes;

  /* don't bother updating the database if the value of the field
   * literally did not change.  Note that this callback is called 
   * for commit and cancel buttons ... 
   */
  notes = xaccAccountGetNotes (acc);
  if (safe_strcmp (notes, newnotes)) {
    xaccAccountBeginEdit (acc, 1);
    xaccAccountSetNotes (acc, newnotes);
    xaccAccountCommitEdit (acc);
  }

  REMOVE_FROM_LIST (EditNotesWindow,editNotesList,acc,account) 

  xaccDestroyTextBox (enw->tb);
  free (enw);

  DEBUG("close EditNotesWindow");
}

/********************** END OF FILE *********************************\
\********************************************************************/
