/* gxset - GTK interface to xset(1)

   Copyright (C) 1999 Ren Seindal (rene@seindal.dk)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <math.h>
#include <string.h>
#include <stdlib.h>

#include <gtk/gtk.h>

#include "support.h"
#include "callbacks.h"
#include "interface.h"

#include "main.h"
#include "guistuff.h"

GtkWidget *
find_widget(GtkWidget *w, const char *name)
{
    GtkWidget *tmp = lookup_widget(w, name);

    if (tmp == NULL)
	g_error(_("Widget lookup failure %s - shouldn't happen!"), name);
    return tmp;
}

ApplicationState *
get_application_state(GtkWidget *w)
{
    return (ApplicationState *)gtk_object_get_user_data(GTK_OBJECT(w));
}



typedef void (*UpdateFunc)(GtkToggleButton *w, gpointer user_data);

static void
toggle_button_set(GtkWidget *w, gchar *name, gboolean val, UpdateFunc func)
{
    GtkToggleButton *button = GTK_TOGGLE_BUTTON(find_widget(w, name));
    gtk_toggle_button_set_active(button, val);

    if (func)
	(*func)(button, w);
}

static gboolean
toggle_button_get(GtkWidget *w, gchar *name)
{
    GtkToggleButton *button = GTK_TOGGLE_BUTTON(find_widget(w, name));
    return gtk_toggle_button_get_active(button);
}


static void
spin_button_set(GtkWidget *w, gchar *name, gint val)
{
    GtkSpinButton *button = GTK_SPIN_BUTTON(find_widget(w, name));
    gtk_spin_button_set_value(button, val);
}

static gint
spin_button_get(GtkWidget *w, gchar *name)
{
    GtkSpinButton *button = GTK_SPIN_BUTTON(find_widget(w, name));
    return gtk_spin_button_get_value_as_int(button);
}

static void
spin_button_set_fraction(GtkWidget *w, gchar *name, gint num, gint div)
{
    GtkSpinButton *button = GTK_SPIN_BUTTON(find_widget(w, name));
    gtk_spin_button_set_value(button, ((gfloat)(num))/div);
}

static void
spin_button_get_fraction(GtkWidget *w, gchar *name, gint *nump, gint *divp)
{
    GtkSpinButton *button = GTK_SPIN_BUTTON(find_widget(w, name));
    gfloat val = gtk_spin_button_get_value_as_float(button);

    /* This assumes max one decimal in the spin button */
    *divp = 10;			/* divisors 2 and 5 */
    *nump = floor((val * 10) + 0.5);

    while (*divp%2 == 0 && *nump%2 == 0) {
	*divp /= 2;
	*nump /= 2;
    }

    while (*divp%5 == 0 && *nump%5 == 0) {
	*divp /= 5;
	*nump /= 5;
    }
}


static void
text_entry_set(GtkWidget *w, gchar *name, gchar *val)
{
    GtkEntry *entry = GTK_ENTRY(find_widget(w, name));
    gtk_entry_set_text(entry, val);
}

static gchar *
text_entry_get(GtkWidget *w, gchar *name)
{
    GtkEntry *entry = GTK_ENTRY(find_widget(w, name));
    return gtk_entry_get_text(entry);
}


static void
clist_set_rows(GtkWidget *w, gchar *name, GPtrArray *vec)
{
    int i;
    gchar *tmp[1];
    GtkCList *clist = GTK_CLIST(find_widget(w, name));

    gtk_clist_freeze(clist);
    gtk_clist_clear(clist);

    for (i = 0; i < vec->len; i++) {
	tmp[0] = g_ptr_array_index(vec, i);
	gtk_clist_append(clist, tmp);
    }

    gtk_clist_thaw(clist);
}



void
adjust_font_buttons(GtkWidget *gxset, ApplicationState *appstate)
{
    int row = appstate->font_path_selected;

    gtk_widget_set_sensitive(lookup_widget(gxset, "font_button_edit"), 
			     row >= 0);
    gtk_widget_set_sensitive(lookup_widget(gxset, "font_button_up"), 
			     row > 0);
    gtk_widget_set_sensitive(lookup_widget(gxset, "font_button_down"), 
			     (row >= 0 
			      && row < appstate->cur->font_path->len-1));
    gtk_widget_set_sensitive(lookup_widget(gxset, "font_button_remove"), 
			     row >= 0);

}

void
set_font_path(GtkWidget *gxset, ApplicationState *appstate)
{
    clist_set_rows(gxset, "font_path", appstate->cur->font_path);

    if ((appstate->font_path_selected >= 0
	 && appstate->font_path_selected < appstate->cur->font_path->len))
	gtk_clist_select_row(GTK_CLIST(find_widget(gxset, "font_path")),
			     appstate->font_path_selected, 0);

    adjust_font_buttons(gxset, appstate);
}

void
set_gui_state(GtkWidget *w, ApplicationState *appstate)
{
    XsetInfo *info = appstate->cur;

    /* Key auto-repeat */
    toggle_button_set(w, "keyrepeat_onoff", info->keyrepeat_onoff,
		      on_keyrepeat_onoff_toggled);
    spin_button_set(w, "keyrepeat_delay", info->keyrepeat_delay);
    spin_button_set(w, "keyrepeat_rate", info->keyrepeat_rate);

    /* Key click */
    toggle_button_set(w, "keyclick_onoff", info->keyclick_volume > 0,
		      on_keyclick_onoff_toggled);
    spin_button_set(w, "keyclick_volume", info->keyclick_volume);

    /* Bell */
    toggle_button_set(w, "bell_onoff", info->bell_volume != 0,
		      on_bell_onoff_toggled);
    spin_button_set(w, "bell_volume", info->bell_volume);
    spin_button_set(w, "bell_pitch", info->bell_pitch);
    spin_button_set(w, "bell_duration", info->bell_duration);

    /* Mouse */
    spin_button_set_fraction(w, "mouse_accel", 
			     info->mouse_acceleration,
			     info->mouse_divisor);
    spin_button_set(w, "mouse_threshold", info->mouse_threshold);

    /* Screen saver */
    toggle_button_set(w, "scrsaver_onoff", info->scrsaver_onoff,
		      on_scrsaver_onoff_toggled);
    toggle_button_set(w, "scrsaver_blank", info->scrsaver_blank, 0);
    toggle_button_set(w, "scrsaver_expose", info->scrsaver_expose, 0);

    spin_button_set(w, "scrsaver_delay", info->scrsaver_delay);
    spin_button_set(w, "scrsaver_cycle", info->scrsaver_cycle);

    /* DPMS */
    toggle_button_set(w, "dpms_onoff", info->dpms_onoff != 0,
		      on_dpms_onoff_toggled);

    spin_button_set(w, "dpms_standby", info->dpms_standby);
    spin_button_set(w, "dpms_suspend", info->dpms_suspend);
    spin_button_set(w, "dpms_off", info->dpms_off);

    set_font_path(w, appstate);
}

void
get_gui_state(GtkWidget *w, ApplicationState *appstate)
{
    XsetInfo *info = appstate->cur;

    info->keyrepeat_onoff = toggle_button_get(w, "keyrepeat_onoff");
    info->keyrepeat_delay = spin_button_get(w, "keyrepeat_delay");
    info->keyrepeat_rate = spin_button_get(w, "keyrepeat_rate");

    info->keyclick_volume = spin_button_get(w, "keyclick_volume");

    if (toggle_button_get(w, "bell_onoff"))
	info->bell_volume = spin_button_get(w, "bell_volume");
    else
	info->bell_volume = 0;
    info->bell_pitch = spin_button_get(w, "bell_pitch");
    info->bell_duration = spin_button_get(w, "bell_duration");

    spin_button_get_fraction(w, "mouse_accel", 
			     &info->mouse_acceleration,
			     &info->mouse_divisor);
    info->mouse_threshold = spin_button_get(w, "mouse_threshold");

    info->scrsaver_onoff = toggle_button_get(w, "scrsaver_onoff");
    info->scrsaver_blank = toggle_button_get(w, "scrsaver_blank");
    info->scrsaver_expose = toggle_button_get(w, "scrsaver_expose");

    info->scrsaver_delay = (info->scrsaver_onoff 
			    ? spin_button_get(w, "scrsaver_delay")
			    : 0);
    info->scrsaver_cycle = spin_button_get(w, "scrsaver_cycle");

    info->dpms_onoff = toggle_button_get(w, "dpms_onoff");
    info->dpms_standby = spin_button_get(w, "dpms_standby");
    info->dpms_suspend = spin_button_get(w, "dpms_suspend");
    info->dpms_off = spin_button_get(w, "dpms_off");
}




gchar *
font_dialog_get_data(GtkWidget *font_dialog)
{
    gchar *new_path;
    GtkNotebook *nb;

    nb = (GtkNotebook *)find_widget(font_dialog, "notebook");
    if (gtk_notebook_get_current_page(nb) == 0) { /* directory */
	gchar *file;
	gboolean allow_scaling;

	file = text_entry_get(font_dialog, "font_dir_entry");
	allow_scaling = toggle_button_get(font_dialog, "font_dir_scaling");

	if (!(file && *file))
	    new_path = NULL;
	else {
	    file = g_strdup(file); /* GtkEntry doesn't malloc for us */

	    if (!g_path_is_absolute(file)) {
		gchar *cwd = g_get_current_dir();
		gchar *abs_path = g_strjoin("/", cwd, file, NULL);

		g_free(cwd);
		g_free(file);

		file = abs_path;
	    }

	    if (allow_scaling)
		new_path = g_strdup(file);
	    else
		new_path = g_strdup_printf("%s:unscaled", file);

	    g_free(file);
	}
    } else {		/* font server */
	gchar *host;
	gint port;

	host = text_entry_get(font_dialog, "font_host_entry");
	port = spin_button_get(font_dialog, "font_port");

	if (host && *host)
	    new_path = g_strdup_printf("%s/:%d", host, port);
	else
	    new_path = NULL;
    }
    return new_path;
}

GtkWidget *
font_dialog_new(gchar *old_path)
{
    GtkWidget *font_dialog;
    GtkNotebook *nb;

    gint len;
    gchar *tmp;
    gchar *portp;
    gint port;

    font_dialog = create_new_font();
    nb = (GtkNotebook *)find_widget(font_dialog, "notebook");

    if (old_path == NULL) {	/* New entry */
	gtk_notebook_set_page(nb, 0);
    } else {
	tmp = g_strdup(old_path);

	if (g_path_is_absolute(old_path)) {	/* Directory entry */
	    len = strlen(tmp);

	    if (len > 9 && strcmp(tmp+len-9, ":unscaled") == 0) {
		toggle_button_set(font_dialog, "font_dir_scaling", 
				  FALSE, NULL);
		tmp[len-9] = '\0';
	    } else {
		toggle_button_set(font_dialog, "font_dir_scaling", 
				  TRUE, NULL);
	    }

	    text_entry_set(font_dialog, "font_dir_entry", tmp);
	    gtk_notebook_set_page(nb, 0);
	} else if ((portp = strstr(tmp, "/:")) != NULL) { /* Font server */
	    *portp = '\0';
	    portp += 2;

	    port = atoi(portp);

	    if (port > 0)
		spin_button_set(font_dialog, "font_port", port);

	    text_entry_set(font_dialog, "font_host_entry", tmp);
	    gtk_notebook_set_page(nb, 1);
	} else {
	    text_entry_set(font_dialog, "font_dir_entry", tmp);
	    gtk_notebook_set_page(nb, 0);
	}

	g_free(tmp);
    }

    return font_dialog;
}
