/*  Screem:  cssWizard.c,
 *  A css1 wizard
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <gmodule.h>
#include <gnome.h>

#include <glade/glade.h>

#include "site.h"
#include "page.h"
#include "editor.h"
#include "support.h"

#include "link.xpm"

#include "css.h"

extern GtkWidget *app;

static GladeXML *xml;
static GtkWidget *window = NULL;
static GtkCTreeNode *node = NULL;
static GtkCTreeNode *pnode = NULL;
static gint current_page = FONT;

static gboolean closing = FALSE;

static void remove_styles( GtkCTreeNode *n );

static gchar* create_css( GtkCTree *tree, GtkCTreeNode *n );

static void store_current( void );
static void set_current( void );

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	g_print("cssWizard: check-init\n");
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
	g_print( "cssWizard: unloaded\n" );
}

G_MODULE_EXPORT void init() 
{
	GtkWidget *linkButton;
	GtkWidget *toolbar;
 
	GnomeUIInfo menuinfo[] = { 
                {
                        GNOME_APP_UI_ITEM, N_( "CaSSIUS" ),
                        N_( "Open Stylesheet editor" ),
                        cssWizard, NULL, NULL,
                        GNOME_APP_PIXMAP_STOCK,
                        GNOME_STOCK_MENU_BLANK,
                        0,
                        GDK_CONTROL_MASK, NULL
                },
                GNOMEUIINFO_END
        };
	
	toolbar = gtk_object_get_data( GTK_OBJECT( app ), "wizardbar" );
  
	/* place menu item after image under insert */
        gnome_app_insert_menus( GNOME_APP( app ),
                                _("_Tools/"), menuinfo);

	/* place a button on the wizards toolbar */
	linkButton = gnome_pixmap_new_from_xpm_d( LINK_XPM );
	gtk_toolbar_append_item( GTK_TOOLBAR( toolbar ), "",
				 _("CSS Wizard"), "", linkButton ,
				 cssWizard, 0 );

	g_print( "cssWizard: initialised\n" );
}

void cssWizard()
{
  	Page *page;

        if( window ) {
                gdk_window_raise( window->window );
                gdk_window_show( window->window );
                return;
        }

	xml = glade_xml_new( GLADE_PATH"/css.glade", 
			     "CaSSIUS" );
	glade_xml_signal_autoconnect( xml );

	window = glade_xml_get_widget( xml, "CaSSIUS" );

	node = NULL;
	pnode = NULL;
	current_page = FONT;
}

void cassius_about()
{
	GladeXML *about_xml;
	GtkWidget *dialog;

	about_xml = glade_xml_new( GLADE_PATH"/css.glade",
				   "about1" );
	glade_xml_signal_autoconnect( about_xml );

	dialog = glade_xml_get_widget( about_xml, "about1" );
}

void css_close()
{
	GtkWidget *tree;
	GtkCTreeNode *n;

	tree = glade_xml_get_widget( xml, "styles" );

	pnode = gtk_ctree_node_nth( GTK_CTREE( tree ), 0 );
	remove_style();

	closing = TRUE;
	gtk_widget_destroy( window );
	window = NULL;
}

void css_dump_stylesheet()
{
	GtkWidget *tree;
	GtkCTreeNode *n;
	gchar *data;

	tree = glade_xml_get_widget( xml, "styles" );

	n = gtk_ctree_node_nth( GTK_CTREE( tree ), 0 );

	store_current();

	/* recursivley create css data */
	data = create_css( GTK_CTREE( tree ), n );

	screem_editor_insert( -1, data );

	g_free( data );
}

void css_dump_style()
{
	GtkWidget *tree;
	GtkCTreeNode *n;
	gchar *data;
	Style *style;

	g_return_if_fail( node != NULL );

	tree = glade_xml_get_widget( xml, "styles" );

	store_current();

	style = gtk_ctree_node_get_row_data( GTK_CTREE( tree ), node );

	data = css_style_output( style );

	screem_editor_insert( -1, data );

	g_free( data );
}

void style_selected( GtkCTree *tree, GtkCTreeNode *row, gint column )
{
	GtkWidget *notebook;

	/* store details of currently selected node */
	store_current();
	
	/* enter details of clicked node */
	node = row;
	set_current();

	notebook = glade_xml_get_widget( xml, "notebook1" );
	gtk_widget_set_sensitive( notebook, TRUE );
}

void style_unselected( GtkCTree *tree, GtkCTreeNode *row, gint column )
{
	GtkWidget *notebook;

	node = NULL;
	pnode = NULL;
	
	notebook = glade_xml_get_widget( xml, "notebook1" );
	gtk_widget_set_sensitive( notebook, FALSE );
}

void style_clicked( GtkWidget *widget, GdkEventButton *event )
{
	GladeXML *menuxml;
	GtkWidget *menu;

	gint row;
	gint col;

	gtk_clist_get_selection_info( GTK_CLIST( widget ),
                                      event->x, event->y, &row, &col );
	
	if( row != - 1 )
		pnode = gtk_ctree_node_nth( GTK_CTREE( widget ), (guint)row );
	else
		pnode = NULL;

	if( event->button == 3 ) {
		/* popup menu */
		gtk_signal_emit_stop_by_name( GTK_OBJECT( widget ),
                                              "button_press_event" );

		menuxml = glade_xml_new( GLADE_PATH"/css.glade", "menu1" );
		glade_xml_signal_autoconnect( menuxml );
		menu = glade_xml_get_widget( menuxml, "menu1" );
		gnome_popup_menu_do_popup_modal( menu, 0, 0, event, 0 );
		gtk_widget_destroy( menu );
	}
}

void add_style()
{
	GladeXML *dialogxml;
	GtkWidget *dialog;
	GtkWidget *entry;
	gchar *name[ 4 ] = { NULL, NULL, NULL, NULL };
	gint button;
	GtkCTreeNode *new_node;
	GtkWidget *tree;
	Style *style;
	GtkWidget *notebook;

	dialogxml = glade_xml_new( GLADE_PATH"/css.glade", "dialog1" );
	glade_xml_signal_autoconnect( dialogxml );

	dialog = glade_xml_get_widget( dialogxml, "dialog1" );
	tree = glade_xml_get_widget( xml, "styles" );

	button = gnome_dialog_run( GNOME_DIALOG( dialog ) );

	if( button == 0 ) {
		entry = glade_xml_get_widget( dialogxml, "style_name" );
		name[ 0 ] = gtk_entry_get_text( GTK_ENTRY( entry ) );
		entry = glade_xml_get_widget( dialogxml, "style_class" );
		name[ 1 ] = gtk_entry_get_text( GTK_ENTRY( entry ) );
		entry = glade_xml_get_widget( dialogxml,"style_pseudo_class" );
		name[ 2 ] = gtk_entry_get_text( GTK_ENTRY( entry ) );
		/* add new tree node */
		new_node = gtk_ctree_insert_node( GTK_CTREE( tree ), 
						  pnode, 
						  NULL, name,
						  3, NULL, NULL, NULL, NULL, 
						  (gboolean)node, FALSE );
		if( pnode )
			gtk_ctree_expand( GTK_CTREE( tree ), pnode);

		/* create a new style struct */
		style = css_style_new();
		style->name = g_strdup( name[ 0 ] );
		style->class_name = g_strdup( name[ 1 ] );
		style->pseudo = g_strdup( name[ 2 ] );

		/* set it as this nodes row data */
		gtk_ctree_node_set_row_data( GTK_CTREE( tree ), new_node,
					     style );

		/* store details of currently selected node */
		store_current();

		gtk_ctree_select( GTK_CTREE( tree ), new_node );
     	}

	gtk_widget_destroy( dialog );
}

void remove_style()
{
	Style *style;
	GtkWidget *tree;
	GtkCTreeNode *n;

	if( ! pnode )
		return;

	tree = glade_xml_get_widget( xml, "styles" );

	n = pnode;

	if( pnode == node ) {
		style_unselected( GTK_CTREE( tree ), node, 0 );
	}

	/* need to recursively destroy its child styles */
	remove_styles( GTK_CTREE_ROW( n )->children );
	/* destroy its own style */
	style = gtk_ctree_node_get_row_data( GTK_CTREE( tree ), n );
	css_style_destroy( style );

	/* remove the GtkCTreeNode */
	gtk_ctree_remove_node( GTK_CTREE( tree ), n );
}

static void remove_styles( GtkCTreeNode *n )
{
	Style *style;
	GtkWidget *tree;

	g_return_if_fail( n != NULL );

	tree = glade_xml_get_widget( xml, "styles" );

	style = gtk_ctree_node_get_row_data( GTK_CTREE( tree ), n );
	css_style_destroy( style );	

	remove_styles( GTK_CTREE_ROW( n )->children );
	remove_styles( GTK_CTREE_ROW( n )->sibling );
}

gboolean page_switch( GtkNotebook *notebook, GtkNotebookPage *page, 
		      guint page_num )
{
	if( closing )
		return TRUE;

	store_current();

	current_page = page_num;
	set_current();

	return TRUE;
}

void color_set( GnomeColorPicker *cp, guint r, guint g, guint b, guint a,
		gchar *entry )
{
	gchar *colour;
	GtkWidget *e;

	colour = g_strdup_printf( "#%.2x%.2x%.2x", r >> 8, g >> 8, b >> 8 );

	e = glade_xml_get_widget( xml, entry );

	gtk_entry_set_text( GTK_ENTRY( e ), colour );

	g_free( colour );
}

void css_new()
{

}

void css_load()
{

}

void css_save()
{
	GtkWidget *tree;
	GtkCTreeNode *n;
	gchar *filename;
	gchar *data;
	FILE *file;

	tree = glade_xml_get_widget( xml, "styles" );

	filename = file_select( _( "Select filename for site template" ) );

        if( ! filename )
                return;

	n = gtk_ctree_node_nth( GTK_CTREE( tree ), 0 );

	store_current();

	/* recursivley create css file */
	data = create_css( GTK_CTREE( tree ), n );

	/* we have the style sheet, save to the named file */
	if( ( file = fopen( filename, "w" ) ) ) {
		fwrite( data, 1, strlen( data ), file );
		fclose( file );
	}
}

static gchar* create_css( GtkCTree *tree, GtkCTreeNode *n )
{
	gchar *data;
	gchar *temp;
       	gchar *retval;
	Style *style;

	g_return_val_if_fail( n != NULL, NULL );

	/* create the style text for this node */
	style = gtk_ctree_node_get_row_data( GTK_CTREE( tree ), n );
	data = css_style_output( style );

	temp = create_css( tree, GTK_CTREE_ROW( n )->children );

	retval = g_strconcat( data, temp, NULL );
	g_free( temp );

	temp = create_css( tree, GTK_CTREE_ROW( n )->sibling );

	retval = g_strconcat( data, temp, NULL );
	g_free( temp );

	return retval;
}

static void store_current()
{
	Style *style;
	GtkWidget *tree;
	GtkWidget *widget;
	gint i;

	if( ! node )
		return;

	tree = glade_xml_get_widget( xml, "styles" );
	style = gtk_ctree_node_get_row_data( GTK_CTREE( tree ), node );

	/* store details from previous page */
     	switch( current_page ) {
	case FONT:
		if( ! style->font )
			style->font = css_font_style_new();
		/* fill in fields */
		widget = glade_xml_get_widget( xml, "font-family" );
		g_free( style->font->family );
		style->font->family = gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "font-style" );
		g_free( style->font->style );
		style->font->style = gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "font-variant" );
		g_free( style->font->variant );
		style->font->variant = gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "font-weight" );
		g_free( style->font->weight );
		style->font->weight = gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "font-size" );
		g_free( style->font->size );
		style->font->size = gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "color" );
		g_free( style->font->color );
		style->font->color = gtk_entry_get_text( GTK_ENTRY(widget) );

		style->font->family = g_strdup( style->font->family );
		style->font->style = g_strdup( style->font->style );
		style->font->variant = g_strdup( style->font->variant );
		style->font->weight = g_strdup( style->font->weight );
		style->font->size = g_strdup( style->font->size );
		style->font->color = g_strdup( style->font->color );
		break;
	case BACKGROUND:
		if( ! style->back )
			style->back = css_back_style_new();
		/* fill in fields */
		widget = glade_xml_get_widget( xml, "background-color" );
		g_free( style->back->color );
		style->back->color = gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "background-image" );
		g_free( style->back->image );
		style->back->image = gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "background-repeat" );
		g_free( style->back->repeat );
		style->back->repeat = gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "background-attachment" );
		g_free( style->back->attachment );
		style->back->attachment = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "background-position" );
		g_free( style->back->position );
		style->back->position = 
			gtk_entry_get_text( GTK_ENTRY(widget) );

		style->back->color = g_strdup( style->back->color );
		style->back->image = g_strdup( style->back->image );
		style->back->repeat = g_strdup( style->back->repeat );
		style->back->attachment = g_strdup( style->back->attachment );
		style->back->position = g_strdup( style->back->position );
		break;
	case TEXT:
		if( ! style->text )
			style->text = css_text_style_new();
		/* fill in fields */
		widget = glade_xml_get_widget( xml, "word-spacing" );
		g_free( style->text->word_spacing );
		style->text->word_spacing = 
			gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "letter-spacing" );
		g_free( style->text->letter_spacing );
		style->text->letter_spacing = 
			gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "text-decoration" );
		g_free( style->text->decoration );
		style->text->decoration = 
			gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "vertical-align" );
		g_free( style->text->vertical_align );
		style->text->vertical_align = 
			gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "text-transform" );
		g_free( style->text->transform );
		style->text->transform = 
			gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "text-align" );
		g_free( style->text->align );
		style->text->align = gtk_entry_get_text( GTK_ENTRY(widget) );
	
		widget = glade_xml_get_widget( xml, "text-indent" );
		g_free( style->text->indent );
		style->text->indent = gtk_entry_get_text( GTK_ENTRY(widget) );

		widget = glade_xml_get_widget( xml, "line-height" );
		g_free( style->text->height );
		style->text->height = gtk_entry_get_text( GTK_ENTRY(widget) );

		style->text->word_spacing = 
			g_strdup( style->text->word_spacing );
		style->text->letter_spacing = 
			g_strdup( style->text->letter_spacing );
		style->text->decoration = g_strdup( style->text->decoration );
		style->text->vertical_align = 
			g_strdup( style->text->vertical_align );
		style->text->transform = g_strdup( style->text->transform );
		style->text->align = g_strdup( style->text->align );
		style->text->indent = g_strdup( style->text->indent );
		style->text->height = g_strdup( style->text->height );
		break;
	case BOX: 
		if( ! style->box )
			style->box = css_box_style_new();
		/* fill in fields */
		widget = glade_xml_get_widget( xml, "margin-top" );
		g_free( style->box->margin[ TOP ] );
		style->box->margin[ TOP ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		widget = glade_xml_get_widget( xml, "margin-right" );
		g_free( style->box->margin[ RIGHT ] );
		style->box->margin[ RIGHT ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );
	
		widget = glade_xml_get_widget( xml, "margin-bottom" );
		g_free( style->box->margin[ BOTTOM ] );
		style->box->margin[ BOTTOM ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "margin-left" );
		g_free( style->box->margin[ LEFT ] );
		style->box->margin[ LEFT ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "padding-top" );
		g_free( style->box->padding[ TOP ] );
		style->box->padding[ TOP ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );
		
		widget = glade_xml_get_widget( xml, "padding-right" );
		g_free( style->box->padding[ RIGHT ] );
		style->box->padding[ RIGHT ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );
	
		widget = glade_xml_get_widget( xml, "padding-bottom" );
		g_free( style->box->padding[ BOTTOM ] );
		style->box->padding[ BOTTOM ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "padding-left" );
		g_free( style->box->padding[ LEFT ] );
		style->box->padding[ LEFT ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		for( i = 0; i < 4; i ++ ) {
			style->box->margin[ i ] = 
				g_strdup( style->box->margin[ i ] );
			style->box->padding[ i ] = 
				g_strdup( style->box->padding[ i ] );
		}
		break;
		
	case BORDER: 
		if( ! style->box )
			style->box = css_box_style_new();
		/* fill in fields */
		widget = glade_xml_get_widget( xml, "top-color" );
		g_free( style->box->border_color[ TOP ] );
		style->box->border_color[ TOP ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "right-color" );
		g_free( style->box->border_color[ RIGHT ] );
		style->box->border_color[ RIGHT ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );
	
		widget = glade_xml_get_widget( xml, "bottom-color" );
		g_free( style->box->border_color[ BOTTOM ] );
		style->box->border_color[ BOTTOM ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "left-color" );
		g_free( style->box->border_color[ LEFT ] );
		style->box->border_color[ LEFT ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "top-width" );
		g_free( style->box->border_width[ TOP ] );
		style->box->border_width[ TOP ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "right-width" );
		g_free( style->box->border_width[ RIGHT ] );
		style->box->border_width[ RIGHT ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );
	
		widget = glade_xml_get_widget( xml, "bottom-width" );
		g_free( style->box->border_width[ BOTTOM ] );
		style->box->border_width[ BOTTOM ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "left-width" );
		g_free( style->box->border_width[ LEFT ] );
		style->box->border_width[ LEFT ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "top-style" );
		g_free( style->box->border_style[ TOP ] );
		style->box->border_style[ TOP ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "right-style" );
		g_free( style->box->border_style[ RIGHT ] );
		style->box->border_style[ RIGHT ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );
	
		widget = glade_xml_get_widget( xml, "bottom-style" );
		g_free( style->box->border_style[ BOTTOM ] );
		style->box->border_style[ BOTTOM ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "left-style" );
		g_free( style->box->border_style[ LEFT ] );
		style->box->border_style[ LEFT ] = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		for( i = 0; i < 4; i ++ ) {
		style->box->border_color[ i ] = 
			g_strdup( style->box->border_color[ i ] );
		style->box->border_width[ i ] = 
			g_strdup( style->box->border_width[ i ] );
		style->box->border_style[ i ] = 
			g_strdup( style->box->border_style[ i ] );
		}
		break;
	case SIZEPLACE:
		if( ! style->box )
			style->box = css_box_style_new();
		/* fill in fields */
		widget = glade_xml_get_widget( xml, "width" );
		g_free( style->box->width );
		style->box->width = gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "height" );
		g_free( style->box->height );
		style->box->height = gtk_entry_get_text( GTK_ENTRY( widget ) );
	
		widget = glade_xml_get_widget( xml, "float" );
		g_free( style->box->float_ );
		style->box->float_ = gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "clear" );
		g_free( style->box->clear );
		style->box->clear = gtk_entry_get_text( GTK_ENTRY( widget ) );

		style->box->width = g_strdup( style->box->width );
		style->box->height = g_strdup( style->box->height );
		style->box->float_ = g_strdup( style->box->float_ );
		style->box->clear = g_strdup( style->box->clear );
		break;
	case CLASSIFICATION:
		if( ! style->class )
			style->class = css_class_style_new();
		/* fill in fields */
		widget = glade_xml_get_widget( xml, "display" );
		g_free( style->class->display );
		style->class->display = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "white-space" );
		g_free( style->class->white_space );
		style->class->white_space = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "list-style-type" );
		g_free( style->class->list_style_type );
		style->class->list_style_type = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "list-style-image" );
		g_free( style->class->list_style_image );
		style->class->list_style_image = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "list-style-position" );
		g_free( style->class->list_style_position );
		style->class->list_style_position = 
			gtk_entry_get_text( GTK_ENTRY( widget ) );

		style->class->display = g_strdup( style->class->display );
		style->class->white_space = 
			g_strdup( style->class->white_space );
		style->class->list_style_type = 
			g_strdup( style->class->list_style_type );
		style->class->list_style_image = 
			g_strdup( style->class->list_style_image );
		style->class->list_style_position = 
			g_strdup( style->class->list_style_position );
		break;
	}
}

static void set_current()
{
	Style *style;
	GtkWidget *tree;
	GtkWidget *widget;

	g_return_if_fail( node != NULL );

	tree = glade_xml_get_widget( xml, "styles" );
	style = gtk_ctree_node_get_row_data( GTK_CTREE( tree ), node );

	/* insert details for current_page */
	switch( current_page ) {
	case FONT:
		widget = glade_xml_get_widget( xml, "font-family" );
		if( style->font )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->family );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
 
		widget = glade_xml_get_widget( xml, "font-style" );
		if( style->font )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->style );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "font-variant" );
		if( style->font )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->variant );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "font-weight" );
		if( style->font )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->weight );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "font-size" );
		if( style->font )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->size );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "color" );
		if( style->font )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->font->color );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		break;
	case BACKGROUND:
		widget = glade_xml_get_widget( xml, "background-color" );
		if( style->back )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->color );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "background-image" );
		if( style->back )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->image );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "background-repeat" );
		if( style->back )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->repeat );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "background-attachment" );
		if( style->back )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->attachment );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "background-position" );
		if( style->back )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->back->position );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		break;
	case TEXT:
		widget = glade_xml_get_widget( xml, "word-spacing" );
		if( style->text )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->word_spacing );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "letter-spacing" );
		if( style->text )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->letter_spacing );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "text-decoration" );
		if( style->text )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->decoration );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "vertical-align" );
		if( style->text )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->vertical_align );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "text-transform" );
		if( style->text )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->transform );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "text-align" );
		if( style->text )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->align );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "text-indent" );
		if( style->text )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->indent );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "line-height" );
		if( style->text )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->text->height );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		break;
	case BOX: 
		widget = glade_xml_get_widget( xml, "margin-top" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->margin[ TOP ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "margin-right" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->margin[ RIGHT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "margin-bottom" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->margin[ BOTTOM ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "margin-left" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->margin[ LEFT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "padding-top" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->padding[ TOP ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "padding-right" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->padding[ RIGHT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "padding-bottom" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->padding[ BOTTOM ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "padding-left" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->padding[ LEFT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		break;
	case BORDER: 
		widget = glade_xml_get_widget( xml, "top-color" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_color[ TOP ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "right-color" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_color[ RIGHT ]);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "bottom-color" );
		if( style->box )
			gtk_entry_set_text(GTK_ENTRY( widget ),
					   style->box->border_color[ BOTTOM ]);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "left-color" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_color[ LEFT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "top-width" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_width[ TOP ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "right-width" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_width[ RIGHT ]);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "bottom-width" );
		if( style->box )
			gtk_entry_set_text(GTK_ENTRY( widget ),
					   style->box->border_width[ BOTTOM ]);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "left-width" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_width[ LEFT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "top-style" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_style[ TOP ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		
		widget = glade_xml_get_widget( xml, "right-style" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_style[ RIGHT ]);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "bottom-style" );
		if( style->box )
			gtk_entry_set_text(GTK_ENTRY( widget ),
					   style->box->border_style[ BOTTOM ]);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "left-style" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->border_style[ LEFT ] );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		break;
	case SIZEPLACE:
		widget = glade_xml_get_widget( xml, "width" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->width );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );	

		widget = glade_xml_get_widget( xml, "height" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->height );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "float" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->float_ );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "clear" );
		if( style->box )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->box->clear );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		break;
	case CLASSIFICATION:
		widget = glade_xml_get_widget( xml, "display" );
		if( style->class )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->display );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "white-space" );
		if( style->class )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->white_space );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "list-style-type" );
		if( style->class )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->list_style_type );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "list-style-image" );
		if( style->class )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->list_style_image );
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );

		widget = glade_xml_get_widget( xml, "list-style-position" );
		if( style->class )
			gtk_entry_set_text( GTK_ENTRY( widget ),
					    style->class->list_style_position);
		else
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		break;
	}
}




Style* css_style_new()
{
	Style *style;

	style = g_new( Style, 1 );

	/* we don't want to waste memory by allocating space for
	   structs which may never be used, we will create the structs
	   as and when we need them */
	style->font = NULL;
	style->back = NULL;
	style->text = NULL;
	style->box = NULL;
	style->class = NULL;

	style->name = NULL;
	style->class_name = NULL;
	style->pseudo = NULL;

	return style;
}

FontStyle* css_font_style_new()
{
	FontStyle *style;

	style = g_new( FontStyle, 1 );

	style->family = NULL;
	style->style = NULL;
	style->variant = NULL;
	style->weight = NULL;
	style->size = NULL;
	style->color = NULL;

	return style;
}

BackStyle* css_back_style_new()
{
	BackStyle *style;

	style = g_new( BackStyle, 1 );

	style->color = NULL;
	style->image = NULL;
	style->repeat = NULL;
	style->attachment = NULL;
	style->position = NULL;

	return style;
}

TextStyle* css_text_style_new()
{
	TextStyle *style;

	style = g_new( TextStyle, 1 );

	style->word_spacing = NULL;
	style->letter_spacing = NULL;
	style->decoration = NULL;
	style->vertical_align = NULL;
	style->transform = NULL;
	style->align = NULL;
	style->indent = NULL;
	style->height = NULL;

	return style;
}

BoxStyle* css_box_style_new()
{
	BoxStyle *style;
	int i;

	style = g_new( BoxStyle, 1 );

	for( i = 0; i < 4; i ++ ) {
		style->margin[ i ] = NULL;
		style->padding[ i ] = NULL;
		style->border_color[ i ] = NULL;
		style->border_width[ i ] = NULL;
		style->border_style[ i ] = NULL;
	}
	style->width = NULL;
	style->height = NULL;
	style->float_ = NULL;
	style->clear = NULL;

	return style;
}

ClassStyle* css_class_style_new()
{
	ClassStyle *style;

	style = g_new( ClassStyle, 1 );

	style->display = NULL;
	style->white_space = NULL;
	style->list_style_type = NULL;
	style->list_style_image = NULL;
	style->list_style_position = NULL;

	return style;
}

void css_style_destroy( Style *style )
{
	g_return_if_fail( style != NULL );

	css_font_style_destroy( style->font );
	css_back_style_destroy( style->back );
	css_text_style_destroy( style->text );
	css_box_style_destroy( style->box );
	css_class_style_destroy( style->class );

	g_free( style->name );
	g_free( style->class_name );
	g_free( style->pseudo );

	g_free( style );
}

void css_font_style_destroy( FontStyle *style )
{
	g_return_if_fail( style != NULL );

	g_free( style->family );
	g_free( style->style );
	g_free( style->variant );
	g_free( style->weight );
	g_free( style->size );
	g_free( style->color );

	g_free( style );
}

void css_back_style_destroy( BackStyle *style )
{
	g_return_if_fail( style != NULL );

	g_free( style->color );
	g_free( style->image );
	g_free( style->repeat );
	g_free( style->attachment );
	g_free( style->position );

	g_free( style );
}

void css_text_style_destroy( TextStyle *style )
{
	g_return_if_fail( style != NULL );

	g_free( style->word_spacing );
	g_free( style->letter_spacing );
	g_free( style->decoration );
	g_free( style->vertical_align );
	g_free( style->transform );
	g_free( style->align );
	g_free( style->indent );
	g_free( style->height );

	g_free( style );
}

void css_box_style_destroy( BoxStyle *style )
{
	gint i;

	g_return_if_fail( style != NULL );

	for( i = 0; i < 4; i ++ ) {
		g_free( style->margin[ i ] );
		g_free( style->padding[ i ] );
		g_free( style->border_color[ i ] );
		g_free( style->border_width[ i ] );
		g_free( style->border_style[ i ] );
	}
	g_free( style->width );
	g_free( style->height );
	g_free( style->float_ );
	g_free( style->clear );

	g_free( style );
}

void css_class_style_destroy( ClassStyle *style )
{
	g_return_if_fail( style != NULL );

	g_free( style->display );
	g_free( style->white_space );
	g_free( style->list_style_type );
	g_free( style->list_style_image );
	g_free( style->list_style_position );

	g_free( style );
}

gchar* css_style_output( Style *style )
{
	gchar *data;
	gchar *temp;
	gchar *data2;
	gchar *name;
	
	g_return_val_if_fail( style != NULL, NULL );

	/* add the class on */
	if( strlen( style->class_name ) > 0 )
		name = g_strdup_printf("%s.%s", style->name,style->class_name);
	else
		name = g_strdup( style->name );

	/* add the pseudo class on */
	if( strlen( style->pseudo ) > 0 ) {
		data = name;
		name = g_strdup_printf( "%s:%s", name, style->pseudo );
		g_free( data );
	}

	if( ! ( temp = css_font_style_output( style->font ) ) )
		temp = g_strdup( "" );

	data = temp;

	if( ! ( temp = css_back_style_output( style->back ) ) )
		temp = g_strdup( "" );
	
	data2 = g_strconcat( data, temp, NULL );
	g_free( data );
	g_free( temp );

	if( ! ( temp = css_text_style_output( style->text ) ) )
		temp = g_strdup( "" );

	data = g_strconcat( data2, temp, NULL );
	g_free( data2 );
	g_free( temp );

	if( ! ( temp = css_box_style_output( style->box ) ) )
		temp = g_strdup( "" );

	data2 = g_strconcat( data, temp, NULL );
	g_free( data );
	g_free( temp );


	if( ! ( temp = css_class_style_output( style->class ) ) )
		temp = g_strdup( "" );

	data = g_strconcat( data2, temp, NULL );
	g_free( data2 );
	g_free( temp );

	data2 = g_strdup_printf( "%s {%s\n}", name, data );

	g_free( data );
	g_free( name );

	return data2;
}

gchar* css_font_style_output( FontStyle *style )
{
	gchar *data;
	gchar *temp;
	gchar *temp2;

	g_return_val_if_fail( style != NULL, NULL );

	if( ( style->family ) && ( strlen( style->family ) > 0 ) )
		data = g_strdup_printf( "\n\tfont-family: %s;", style->family );
	else
		data = g_strdup( "" );

	temp = data;

	if( ( style->style ) && ( strlen( style->style ) > 0 ) )
		data = g_strdup_printf( "\n\tfont-style: %s;", style->style );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->variant ) && ( strlen( style->variant ) > 0 ) )
		data = g_strdup_printf("\n\tfont-variant: %s;", style->variant);
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->weight ) && ( strlen( style->weight ) > 0 ) )
		data = g_strdup_printf( "\n\tfont-weight: %s;", style->weight );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->size ) && ( strlen( style->size ) > 0 ) )
		data = g_strdup_printf( "\n\tfont-size: %s;", style->size );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->color ) && ( strlen( style->color ) > 0 ) )
		data = g_strdup_printf( "\n\tcolor: %s;", style->color );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp; 
}

gchar* css_back_style_output( BackStyle *style )
{
	gchar *data;
	gchar *temp;
	gchar *temp2;

	g_return_val_if_fail( style != NULL, NULL );

	if( ( style->color ) && ( strlen( style->color ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-color: %s;", 
					style->color );
	else
		data = g_strdup( "" );

	temp = data;

	if( ( style->image ) && ( strlen( style->image ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-image: %s;", 
					style->image );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->repeat ) && ( strlen( style->repeat ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-repeat: %s;", 
					style->repeat );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->attachment ) && ( strlen( style->attachment ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-attachment: %s;", 
					style->attachment );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->position ) && ( strlen( style->position ) > 0 ) )
		data = g_strdup_printf( "\n\tbackground-position: %s;", 
					style->position );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp; 
}

gchar* css_text_style_output( TextStyle *style )
{
	gchar *data;
	gchar *temp;
	gchar *temp2;

	g_return_val_if_fail( style != NULL, NULL );

	if( ( style->word_spacing ) && ( strlen( style->word_spacing ) > 0 ) )
		data = g_strdup_printf( "\n\tword-spacing: %s;", 
					style->word_spacing );
	else
		data = g_strdup( "" );

	temp = data;

	if( ( style->letter_spacing ) && 
	    ( strlen( style->letter_spacing ) > 0 ) )
		data = g_strdup_printf( "\n\tletter-spacing: %s;", 
					style->letter_spacing );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->decoration ) && ( strlen( style->decoration ) > 0 ) )
		data = g_strdup_printf( "\n\ttext-decoration: %s;", 
					style->decoration );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->vertical_align ) && 
	    ( strlen( style->vertical_align ) > 0 ) )
		data = g_strdup_printf( "\n\tvertical-align: %s;", 
					style->vertical_align );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->transform ) && ( strlen( style->transform ) > 0 ) )
		data = g_strdup_printf( "\n\ttext-transform: %s;", 
					style->transform );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->align ) && ( strlen( style->align ) > 0 ) )
		data = g_strdup_printf( "\n\ttext-align: %s;", style->align );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->indent ) && ( strlen( style->indent ) > 0 ) )
		data = g_strdup_printf( "\n\ttext-indent: %s;", style->indent );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->height ) && ( strlen( style->height ) > 0 ) )
		data = g_strdup_printf( "\n\tline-height: %s;", style->height );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp; 	
}

gchar* css_box_style_output( BoxStyle *style )
{
	gchar *data;
	gchar *temp;
	gchar *temp2;
	gint i;

	gchar temp3[ 4 ];
	gboolean not_empty = TRUE;

	gchar *one;
	gchar *two;
	gchar *three;

	const gchar *side[] = { "top", "right", "bottom", "left" };

	g_return_val_if_fail( style != NULL, NULL );

	temp = g_strdup( "" );

	for( i = 0; i < 4; i ++ ) {
		if( ( style->margin[ i ] ) && 
		    ( strlen( style->margin[ i ] ) > 0 ) )
			data = g_strdup_printf( "\n\tmargin-%s %s;",
						side[ i ],
						style->margin[ i ] );
		else
			data = g_strdup( "" );
		temp2 = temp;
		temp = g_strconcat( temp2, data, NULL );
		g_free( temp2 );
		g_free( data );
	}

	for( i = 0; i < 4; i ++ ) {
		if( ( style->padding[ i ] ) && 
		    ( strlen( style->padding[ i ] ) > 0 ) )
			data = g_strdup_printf( "\n\tpadding-%s %s;",
						side[ i ],
						style->padding[ i ] );
		else
			data = g_strdup( "" );
		temp2 = temp;
		temp = g_strconcat( temp2, data, NULL );
		g_free( temp2 );
		g_free( data );
	}

	/* border-width border-style border-color */
	for( i = 0; i < 4; i ++ ) {
		not_empty = FALSE;
		if( style->border_width[ i ] && 
		    ( strlen( style->border_width[ i ] ) > 0 ) )
			not_empty = (gboolean)
				one = g_strconcat( " ", 
						   style->border_width[ i ],
						   NULL );
		else
			one = g_strdup( "" );
		if( style->border_style[ i ] && 
		    ( strlen( style->border_style[ i ] ) > 0 ) )
			not_empty = (gboolean)
				two = g_strconcat( " ",
						   style->border_style[ i ],
						   NULL );
		else
			two = g_strdup( "" );
		if( style->border_color[ i ] && 
		    ( strlen( style->border_color[ i ] ) > 0 ) ) 
			not_empty = (gboolean)
				three = g_strconcat( " ",
						     style->border_color[ i ],
						     NULL );
		else
			three = g_strdup( "" );

		if( not_empty ) {
			data = g_strdup_printf( "\n\tborder-%s %s%s%s;",
						side[ i ],
						one, two, three );
			
			temp2 = temp;
			temp = g_strconcat( temp2, data, NULL );
			g_free( temp2 );
			g_free( data );
		}
	}

	if( ( style->width ) && ( strlen( style->width ) > 0 ) )
		data = g_strdup_printf( "\n\twidth: %s;", style->width );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->height ) && ( strlen( style->height ) > 0 ) )
		data = g_strdup_printf( "\n\theight: %s;", style->height );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->float_ ) && ( strlen( style->float_ ) > 0 ) )
		data = g_strdup_printf( "\n\tfloat: %s;", style->float_ );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->clear ) &&  ( strlen( style->clear ) > 0 ) )
		data = g_strdup_printf( "\n\tclear: %s;", style->clear );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp;      
}

gchar* css_class_style_output( ClassStyle *style )
{
 	gchar *data;
	gchar *temp;
	gchar *temp2;

	g_return_val_if_fail( style != NULL, NULL );

	if( ( style->display ) && ( strlen( style->display ) > 0 ) )
		data = g_strdup_printf( "\n\tdisplay: %s;", style->display );
	else
		data = g_strdup( "" );

	temp = data;

	if( ( style->white_space ) && ( strlen( style->white_space ) > 0 ) )
		data = g_strdup_printf( "\n\twhite-space: %s;", 
					style->white_space );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->list_style_type ) && 
	    ( strlen( style->list_style_type ) > 0 ) )
		data = g_strdup_printf( "\n\tlist-style-type: %s;", 
					style->list_style_type );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->list_style_image ) && 
	    ( strlen( style->list_style_image ) > 0 ) )
		data = g_strdup_printf( "\n\tlist-style-image: %s;", 
					style->list_style_image );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	if( ( style->list_style_position ) && 
	    ( strlen( style->list_style_position ) > 0 ) )
		data = g_strdup_printf( "\n\tlist-style-position: %s;", 
					style->list_style_position );
	else
		data = g_strdup( "" );

	temp2 = temp;
	temp = g_strconcat( temp2, data, NULL );
	g_free( temp2 );
	g_free( data );

	return temp;      
}
