/*  Screem:  page.c,
 *  a tidied up page.c, separating ui from backend
 * 
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <ctype.h>
#include <fcntl.h>
#include <gnome.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "page.h"
#include "html.h"

/**
 * screem_page_new:
 *
 * Creates and initialises a new page struct
 *
 * return values: a Page
 */
Page *screem_page_new()
{
	Page *p;

	p = (Page*)g_malloc( sizeof( Page ) );

	p->pathname = NULL;
	p->data = NULL;
	p->tab = -1;
			
	return p;
}

/**
 * screem_page_destroy:
 * @page:  the page to destroy
 *
 * destroys the page struct and any memory it may be taking up
 *
 * return values: none
 */
void screem_page_destroy( Page *page )
{
        g_return_if_fail( page != NULL );

	g_free( page->pathname );
	g_free( page->data );
	g_free( page );
}

/**
 * screem_page_set_pathname:
 * @page:  the page to set
 * @path:  the pathname to set to
 *
 * sets the given pages pathname to path
 *
 * return values: none
 */
void screem_page_set_pathname( Page *page, const gchar *path )
{
	g_return_if_fail( page != NULL );

	if( page->pathname )
		g_free( page->pathname );

	page->pathname = g_strdup( path );
}

/**
 * screem_page_get_pathname:
 * @page:  the page
 *
 * obtain the pathname for a page
 *
 * return values: the pathname
 */
const gchar *screem_page_get_pathname( Page *page )
{
	g_return_val_if_fail( page != NULL, NULL );

	return page->pathname;
}

/**
 * screem_page_set_tab:
 * @page:  the page to set
 * @tab:   the tab number
 *
 * stores the notebook tab number of the page
 *
 * return values: none
 */
void screem_page_set_tab( Page *page, gint num )
{
	g_return_if_fail( page != NULL );

	page->tab = num;
}

/**
 * screem_page_get_tab:
 * @page:  the page to set
 *
 * retrieves the notebook tab number of the page
 *
 * return values: none
 */
gint screem_page_get_tab( Page *page )
{
	g_return_val_if_fail( page != NULL, -1 );

	return page->tab;
}

/**
 * screem_page_read:
 * @page:  the page to read into
 * @fd:    the file descriptor to read from
 *
 * reads from file descriptor fd into page->data
 * returns TRUE if successful, FALSE on error
 * The fd is not closed, even if an error occurs, it is down to the calling
 * function to do this.
 *
 * return values: boolean
 */
#define MAX_SIZE 8196
gboolean screem_page_read( Page *page, int fd )
{
	int size = 1;
	int buffer_size = MAX_SIZE;
 	gchar buffer[ MAX_SIZE + 1];

        g_return_val_if_fail( page != NULL, FALSE );

	page->data = "";

	while( size ) {
		size = read( fd, buffer, MAX_SIZE );
		if( size == -1 ) {
			/* there was an error with the read */
			g_free( page->data );
			page->data = NULL;
			return FALSE;
		}
		buffer[ size ] = 0;
		page->data = g_strconcat( page->data, buffer, NULL );
	}

	return TRUE;
}

/**
 * screem_page_load:
 * @page:  the page to load
 *
 * If the page isn't already in memory then this function will
 * load it.
 * returns FALSE on error, TRUE if successful.
 * the page already being loaded is not counted as an error
 *
 * return values: boolean
 */
gboolean screem_page_load( Page *page )
{
	int fd;
	gboolean ret;

        g_return_val_if_fail( page, FALSE );

	if( page->data )
		return TRUE;

	/* load the page */
	fd = open( screem_page_get_pathname( page ), O_RDONLY );

	if( fd == -1 ) {
		/* failed to open the page */
		return FALSE;
	}

	ret = screem_page_read( page, fd );

	close( fd );
	return ret;
}

/**
 * screem_page_revert:
 * @page:  the page to revert
 *
 * reverts the page to the version on disk
 *
 * return values: boolean
 */
gboolean screem_page_revert( Page *page )
{
	g_return_val_if_fail( page, FALSE );

	if( page->data )
		g_free( page->data );
	
	return screem_page_load( page );
}

/**
 * screem_page_save:
 * @page:  the page to save
 *
 * saves the given page, if the page wasn't loaded then we just return
 *
 * return values: boolean
 */
gboolean screem_page_save( Page *page )
{
	int size;
	FILE *out;

	g_return_val_if_fail( page, FALSE );

	if( ! page->data )
		return TRUE;

	size = strlen( page->data );

	out = fopen( screem_page_get_pathname( page ), "w" );
	if( ! out )
		return FALSE;

	fwrite( page->data, 1, size, out );
	fclose( out );

	return TRUE;
}

Page *screem_page_from_tab_num( GList *list, gint num )
{
	Page *page = NULL;

	g_return_val_if_fail( num > -1, NULL );

	for( ; list && !page; list = list->next ) {
		if( num == screem_page_get_tab( list->data ) )
			page = list->data;
	}
	
	return page;
}

void screem_page_update_tab_nums( GList *list, gint num )
{
	Page *page = NULL;
	gint num2;

	g_return_if_fail( num > -1 );

	for( ; list; list = list->next ) {
		page = (Page*)list->data;
		num2 = screem_page_get_tab( page );
		if( num2 > num ) 
			screem_page_set_tab( page, --num2 );
		else if( num2 == num )
			screem_page_set_tab( page, -1 ); /* the closed one */
	}
}
