/*  Screem:  editMenu.c,
 *  This file handles find/replace
 *
 *  Copyright (C) 1999  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <gnome.h>

#include <libgnome/gnome-regex.h>

#include "editor.h"
#include "editMenu.h"
#include "site.h"
#include "support.h"
#include "page.h"
#include "pageUI.h"
#include "preferences.h"

/* the maximum number of regular expressions for the cache */
#define MAX_EXPRESSIONS 2

extern GtkWidget *app;
extern Preferences *cfg;
extern Site *current_site;
extern Page *current_page;

extern GList *icon_list;

/* Lots of globals this is messy */
static gchar *find_string = NULL;
static gchar *replace_string = NULL;
static gint find_pos;
static gboolean use_regex = TRUE;
static gboolean check_all = FALSE;
static const size_t nmatch = 1;
static regmatch_t pmatch[ 1 ];
static gboolean setup = FALSE;
static gboolean replace_all = FALSE;

static gint search( gchar *text, gchar *f );
static void find_replace( gboolean r );
static void find_file_sel( GtkCList *clist, gint row, gint column,
			 GdkEventButton *event, gpointer data );

void set_use_regex( gboolean s )
{
	use_regex = s;
}

gchar* find_text( gchar *text, gchar *f, gchar *r )
{
	gint retval;
	gint findlen;
	gchar *chunk1;
	gchar *chunk2;
	gchar *newChunk;

	find_pos = 0;
       	retval = search( text, f );

	if( retval == -1 ) {
		return NULL;
	} else {
		/* replace the text if needed */
		if( use_regex ) {
			findlen = pmatch[ 0 ].rm_eo - 
				pmatch[ 0 ].rm_so;
		} else
			findlen = strlen( f );
		if( r ) {
			chunk1 = g_strndup( text, retval );
			chunk2 = g_strdup( text + retval +
					   findlen );
			newChunk = g_strdup_printf( "%s%s%s",
						    chunk1,
						    r,
						    chunk2 );
			g_free( chunk1 );
			g_free( chunk2 );
			g_free( text );
			text = newChunk;
		} else
			text = text + retval;
	}
	return text;
}
/*************************************************************************/
/* we pass parameters here rather than just using the global strings so
   that we can search all from else where in the program,
   dialog is NULL or the find/replace dialog */
void search_all( gchar *f, gchar *r, GtkWidget *dialog )
{
	GList *list;
	GList *matches; /* a list of pages that contain matches */
     	Page *p;
	gint retval;
	GtkWidget *hitList = NULL;
	gchar *item[ 2 ];
	gint row;

	gint findlen;

	gchar *chunk1;
	gchar *chunk2;
	gchar *newChunk;

	const gchar *mime_type;
	const gchar *page_path;
	Icons *icon;

	Site *site = current_site;

	g_return_if_fail( site != NULL );

	matches = NULL;
	/* clear the old matches from the clist if we were called
	   from the find/replace dialog */
	if( dialog ) {
		hitList = gtk_object_get_data(GTK_OBJECT(dialog), "matchList");
		gtk_clist_clear( GTK_CLIST( hitList ) );
	}

	for( list = screem_site_get_pages( site ); list; list = list->next ) {
		p = (Page*)list->data;
		if( p == screem_site_get_current_page( site )){
				/* grab the current editor text to search
				   through */
			screem_editor_buffer_text();
		}
		/* load the page, function deals with it already
		   being loaded */
		screem_page_load( p );
		/* now search data */
		find_pos = 0;
		while( ( retval = search( p->data, f ) ) != -1 ) {
			/* we got a match so add it to the list */
			if( ! g_list_find( matches, p ) )
				matches = g_list_append( matches, p );
			/* get find length */
			if( use_regex ) {
				findlen = pmatch[ 0 ].rm_eo - 
					pmatch[ 0 ].rm_so;
			} else
				findlen = strlen( f );
			if( r && replace_all ) {
				chunk1 = g_strndup( p->data, retval );
				chunk2 = g_strdup( p->data + retval +
						   findlen );
				newChunk = g_strdup_printf( "%s%s%s",
							    chunk1,
							    r,
							    chunk2 );
				g_free( chunk1 );
				g_free( chunk2 );
				g_free( p->data );
				p->data = newChunk;
			}
			find_pos = retval;
				/* if we aren't replacing all then break out
				   of the loop */
			if( ! r || ! replace_all )
				break;
		}
		if( r && replace_all && 
		    ( p == screem_site_get_current_page( site ) ) ) {
				/* refresh display */
			screem_editor_clear();
			screem_editor_insert( 0, p->data );
		}
	}
	
	/* scanned all pages, show the list of "hits" if we came from
	   the find/replace dialog, or perform the replace if we
	   are acting non-interactivly */
	if( dialog ) {
		setup = TRUE;
		row = 0;
		gtk_clist_freeze( GTK_CLIST( hitList ) );
		for( list = matches; list; list = list->next ) {
			page_path = screem_page_get_pathname( list->data );
			mime_type = gnome_mime_type( page_path );
			icon = icon_from_mime_type( mime_type );
			item[ 0 ] = NULL;
			item[ 1 ] = g_strdup( page_path );
			gtk_clist_append( GTK_CLIST( hitList ), item );
			gtk_clist_set_pixmap( GTK_CLIST( hitList ),
					      row, 0, icon->open,
					      icon->open_mask );
			row ++;
		}
		gtk_clist_columns_autosize( GTK_CLIST( hitList ) );
		gtk_clist_thaw( GTK_CLIST( hitList ) );
		setup = FALSE;
	}
	g_list_free( matches );
}
/*************************************************************************/
static gboolean replace_text( gchar *text, gint pos )
{
	if( ! replace_string )
		return FALSE;

	if( use_regex )
		screem_editor_delete_forward( pos, 
					      pmatch[ 0 ].rm_eo -
					      pmatch[ 0 ].rm_so );
	else
		screem_editor_delete_forward( pos, strlen( find_string ) );
	
	screem_editor_insert( pos, replace_string );
	pos += strlen( replace_string );
	screem_editor_set_pos( pos );

	return TRUE;
}
/*************************************************************************/
void select_all()
{
       	Page *page;

	if( current_site )
		page = screem_site_get_current_page( current_site );
	else
		page = current_page;

	g_return_if_fail( page != NULL );

	screem_editor_select_region( 0, screem_editor_get_length() );
}

void cut()
{
	screem_editor_cut();
}

void copy()
{
	screem_editor_copy();
}

void paste()
{
	screem_editor_paste();
}

void clear()
{
	screem_editor_clear_selection();
}


/*************************************************************************/
void find()
{
	find_replace( FALSE );
}
/*************************************************************************/
/* only on single pages */
void find_again()
{
	gchar *text = screem_editor_get_text( 0, screem_editor_get_length() );
	gint retval;
	gint findLen;

	if( ! find_string )
		return;

	retval = search( text, find_string );
	if( retval == -1 ) {
		find_pos =  screem_editor_get_pos();
	} else {
		if( use_regex )
			findLen =  pmatch[0].rm_eo - pmatch[0].rm_so;
		else
			findLen = strlen( find_string);

		/* replace the text if needed */
		if( ! replace_text( text, retval ) ) {
			/* we aren't replacing so select */
			screem_editor_set_pos( retval );
			screem_editor_select_region( retval, retval + findLen);
		}
		find_pos = retval + findLen;
	}

	g_free( text );
}
/*************************************************************************/
void replace()
{
	find_replace( TRUE );
}
/*************************************************************************/
void replace_again()
{

}
/*************************************************************************/
static void closed( GtkWidget *widget, GdkEvent *event, gpointer data )
{
	GtkWidget **d;

	d = gtk_object_get_data( GTK_OBJECT( widget ), "dialog" );
	if( d )
		*d = NULL;
}
/*************************************************************************/
static void clicked( GtkWidget *widget, gint button, gpointer data )
{
	GnomeEntry      *fEntry;
	GtkEntry        *fileEntry;
	GnomeEntry      *rEntry;
	GtkEntry        *replaceEntry;
	GtkToggleButton *from;
	GtkToggleButton *regex;
	GtkToggleButton *checkall;
	GtkToggleButton *replaceall;

	GtkWidget  **d;

	gchar *f;
	gchar *r;
	gint ret;

	d = gtk_object_get_data( GTK_OBJECT( widget ), "dialog" );
	fEntry = GNOME_ENTRY( gtk_object_get_data( GTK_OBJECT( widget ),
						   "findEntry" ) );
	rEntry = GNOME_ENTRY( gtk_object_get_data( GTK_OBJECT( widget ),
						   "replaceEntry" ) );
	from = GTK_TOGGLE_BUTTON( gtk_object_get_data( GTK_OBJECT( widget ), 
						      "searchpos" ) );
	regex = GTK_TOGGLE_BUTTON( gtk_object_get_data( GTK_OBJECT( widget ),
							"regex" ) );
	checkall = GTK_TOGGLE_BUTTON( gtk_object_get_data(GTK_OBJECT( widget ),
							  "check_all" ) );
	replaceall = GTK_TOGGLE_BUTTON( gtk_object_get_data(GTK_OBJECT(widget),
							    "replace_all" ) );

	if( button < 2 ) {
		fileEntry = GTK_ENTRY(  gnome_entry_gtk_entry( fEntry ) );
		f = gtk_entry_get_text( fileEntry );
		if( find_string ) {
			if( strcmp( f, find_string ) ) {
				g_free( find_string );
				find_string = NULL;
			}
		}
		if( ! find_string )
			find_string = g_strdup( f );

		if( GTK_WIDGET_IS_SENSITIVE( rEntry ) ) {
			replaceEntry=GTK_ENTRY(gnome_entry_gtk_entry(rEntry) );
			r = gtk_entry_get_text( replaceEntry );
			if( replace_string ) {
				if( strcmp( r, replace_string ) ) {
					g_free( replace_string );
					replace_string = NULL;
				}
			}
			if( ! replace_string )
				replace_string = g_strdup( r );
		} else {
			if( replace_string )
				g_free( replace_string );
			replace_string = NULL;
		}

		use_regex = gtk_toggle_button_get_active( regex );
		check_all = gtk_toggle_button_get_active( checkall );
		replace_all = gtk_toggle_button_get_active( replaceall );
		
		if( check_all )
			search_all( find_string, replace_string, widget );
		else {
			find_again();
		}
	}

	if( ( button == 0 && ! check_all ) || button == 2 ) {
		gtk_widget_destroy( widget );
		if( d )
			*d = NULL;
	}
}
/*************************************************************************/
static void find_replace( gboolean r )
{
	static GtkWidget *dialog = NULL;
	GtkWidget *dialog_vbox;
	GtkWidget *table;
	GtkWidget *label;
	GtkWidget *findEntry;
	GtkWidget *replaceEntry;
	GtkWidget *regex;
	GtkWidget *posStart;
	GtkWidget *posCursor;
	GtkWidget *matchList;
	GtkWidget *checkall;
	GtkWidget *replaceall;
	GtkWidget *replaceCheck;
	GtkWidget *sw;
	GtkTooltips *tooltips;
	GSList *group = NULL;

	if( dialog ) {
		replaceCheck = gtk_object_get_data( GTK_OBJECT( dialog ),
						    "replaceCheck" );
                gdk_window_raise( dialog->window );
                gdk_window_show( dialog->window );
		/* activate the replace entry if needed */
		if( r != gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( replaceCheck ) ) )
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(replaceCheck), r );
		return;
        }

	dialog = gnome_dialog_new( _("Find Replace dialog" ), 
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_APPLY,
				   GNOME_STOCK_BUTTON_CLOSE,
				   NULL );
	dialog_vbox = GNOME_DIALOG( dialog )->vbox;
     
	table = gtk_table_new( 2, 6, FALSE );
	gtk_box_pack_start( GTK_BOX( dialog_vbox ), table, TRUE, TRUE, 
			    GNOME_PAD );

	label = gtk_label_new( _( "Find" ) );
	gtk_table_attach( GTK_TABLE( table ), label, 0, 1, 0, 1,
			  0, 0, GNOME_PAD, GNOME_PAD );
	findEntry = gnome_entry_new( "findEntry" );
	gtk_combo_set_case_sensitive(GTK_COMBO(&GNOME_ENTRY(findEntry)->combo),
				      TRUE );
	gtk_object_set_data( GTK_OBJECT( dialog ), "findEntry", findEntry );
	gtk_table_attach( GTK_TABLE( table ), findEntry, 1, 2, 0, 1,
			  GTK_EXPAND | GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	replaceEntry = gnome_entry_new( "replaceEntry" );
	gtk_combo_set_case_sensitive( GTK_COMBO( &GNOME_ENTRY( replaceEntry )->combo ), TRUE );
	gtk_object_set_data( GTK_OBJECT( dialog ), "replaceEntry",
			     replaceEntry );
	gtk_table_attach( GTK_TABLE( table ), replaceEntry, 1, 2, 1, 2,
			  GTK_EXPAND | GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	regex = gtk_check_button_new_with_label( _("Use regular expressions"));
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( regex ), use_regex );
	gtk_object_set_data( GTK_OBJECT( dialog ), "regex", regex );
	gtk_table_attach( GTK_TABLE( table ), regex, 0, 1, 2, 3,
			  GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	replaceall = gtk_check_button_new_with_label( _( "Replace all" ) );
	tooltips = gtk_tooltips_new ();
	gtk_tooltips_set_tip( tooltips, replaceall, _("When replacing automatically replace all occurences of the find string" ), NULL );
	gtk_object_set_data( GTK_OBJECT( dialog ), "replace_all", replaceall );
	gtk_table_attach( GTK_TABLE( table ), replaceall, 1, 2, 2, 3,
			  GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	posStart = gtk_radio_button_new_with_label( group, _( "From start" ) );
	group = gtk_radio_button_group( GTK_RADIO_BUTTON( posStart ) );
	posCursor = gtk_radio_button_new_with_label(group, _( "From current position" )); 	gtk_table_attach( GTK_TABLE( table ), posStart, 0, 1, 3, 4,
			  GTK_FILL, 0, GNOME_PAD, GNOME_PAD );
	gtk_table_attach( GTK_TABLE( table ), posCursor, 1, 2, 3, 4,
			  GTK_FILL, 0, GNOME_PAD, GNOME_PAD );
	/* only need to add data for one of them as if it isn't selected
	   then the other one must be */
	gtk_object_set_data( GTK_OBJECT( dialog ), "searchpos", posStart );

	sw = gtk_scrolled_window_new( NULL, NULL );
        gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
                                        GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC );
	matchList = gtk_clist_new( 2 );
	gtk_scrolled_window_add_with_viewport( GTK_SCROLLED_WINDOW( sw ),
                                               matchList );
	gtk_object_set_data( GTK_OBJECT( dialog ), "matchList", matchList );
	gtk_table_attach( GTK_TABLE( table ), sw, 0, 2, 5, 6,
			  GTK_FILL, GTK_EXPAND | GTK_FILL, GNOME_PAD, 
			  GNOME_PAD );
	gtk_signal_connect( GTK_OBJECT( matchList ), "select-row",
			    find_file_sel, 0 );

	gtk_clist_set_column_width( GTK_CLIST( matchList ), 0, 16 );
       	gtk_clist_set_selection_mode( GTK_CLIST( matchList ),
				      GTK_SELECTION_BROWSE );
	gtk_clist_column_titles_hide( GTK_CLIST( matchList ) );


	checkall = gtk_check_button_new_with_label( _( "Check all files" ) );
	tooltips = gtk_tooltips_new ();
	gtk_tooltips_set_tip( tooltips, checkall, _("When searching multiple files, any  which contain a match are shown below, selecting one will open the page in the editor" ), NULL );
	gtk_object_set_data( GTK_OBJECT( dialog ), "check_all", checkall );
	gtk_table_attach( GTK_TABLE( table ), checkall, 0, 2, 4, 5,
			  GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	replaceCheck = gtk_check_button_new_with_label( _( "Replace" ) );
	gtk_object_set_data( GTK_OBJECT( dialog ), "replaceCheck", 
			     replaceCheck );
	gtk_table_attach( GTK_TABLE( table ), replaceCheck, 0, 1, 1, 2,
			  GTK_FILL, 0, GNOME_PAD, GNOME_PAD );

	gtk_widget_set_sensitive( replaceEntry, FALSE );
	gtk_signal_connect_object( GTK_OBJECT( replaceCheck ), "toggled",
                                   change_state, (gpointer)replaceEntry );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(replaceCheck), r );

	gtk_widget_set_usize( sw, -1, 100 );

	gtk_signal_connect( GTK_OBJECT( dialog ), "clicked",
			    clicked, 0 );
	gtk_signal_connect( GTK_OBJECT( dialog ), "delete_event",
			    closed, 0 );

	gtk_object_set_data( GTK_OBJECT( dialog ), "dialog", &dialog );

	if( find_string ) {
		g_free( find_string );
		find_string = NULL;
	}
	if( replace_string ) {
		g_free( replace_string );
		replace_string = NULL;
	}

	find_pos =  screem_editor_get_pos();
	
	gtk_widget_show_all( dialog );
}
/*************************************************************************/
static void find_file_sel( GtkCList *clist, gint row, gint column,
			   GdkEventButton *event, gpointer data ) 
{
	static guint32 t = -1;
	gchar *info[ 1 ];
	Page *p;

	if( setup )
		return;

	if( t != event->time ) {
		t = event->time;
		return;
	}

	gtk_clist_get_text( clist, row, 1, &info[ 0 ] );

	p = screem_site_locate_page( current_site, info[ 0 ] );

	if( ! p )
		return;
	
	screem_page_insert( p );

	/* open the page in the editor and perform a single page search */
   	find_pos = 0;
	while( gtk_events_pending() )
		gtk_main_iteration();
	find_again();
}
/*************************************************************************/
/* text = text to search
   f    = the string to find 
   returns the position at which a match was found or -1
*/
static gint search( gchar *text, gchar *f )
{
	gchar *pos;
	GnomeRegexCache* regexCache;
	regex_t *expressions;

	gint retval = find_pos;

	if( use_regex ) {
		regexCache = gnome_regex_cache_new();
		gnome_regex_cache_set_size( regexCache, MAX_EXPRESSIONS );
		expressions = gnome_regex_cache_compile( regexCache, f, 
							 0 );
		if( regexec(expressions, text + find_pos, nmatch, pmatch, 0 )) {
			/* no match */
			retval = -1;
		} else {
			/* match at pmatch[ 0 ].rm_so ending at 
			   pmatch[ 0 ].rm_eo */
			retval += pmatch[ 0 ].rm_so;
			gnome_regex_cache_destroy( regexCache );
		}
	} else {
		/* not using regular expressions */
		pos = strstr( text + retval, f );
		if( ! pos )
			retval = -1;
		else
			retval = pos - text;
	}

	return retval;
}
/*************************************************************************/
