/* prims.c: -*- C -*-  Primitive internal functions. */

/* Author: Brian J. Fox (bfox@ai.mit.edu) Sat Jul 20 17:22:47 1996.

   This file is part of <Meta-HTML>(tm), a system for the rapid deployment
   of Internet and Intranet applications via the use of the Meta-HTML
   language.

   Copyright (c) 1995, 1996, Brian J. Fox (bfox@ai.mit.edu).
   Copyright (c) 1996, Universal Access Inc. (http://www.ua.com).

   Meta-HTML is free software; you can redistribute it and/or modify
   it under the terms of the UAI Free Software License as published
   by Universal Access Inc.; either version 1, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   UAI Free Software License for more details.

   You should have received a copy of the UAI Free Software License
   along with this program; if you have not, you may obtain one by
   writing to:

   Universal Access Inc.
   129 El Paseo Court
   Santa Barbara, CA
   93101  */

/*
################### CAUTION ##################################
# This file has been modified for inclusion into WML
# D. Barbier   16/01/1999
##############################################################
*/

#include "language.h"
#include "symdump.h"

#if defined (__cplusplus)
extern "C"
{
#endif

static void pf_eval (PFunArgs);
static void pf_quote_for_setvar (PFunArgs);
static void pf_after_page_return (PFunArgs);
static void pf_apply (PFunArgs);

static PFunDesc func_table[] =
{
  { "%%EVAL",			0, 0, pf_eval },
  { "%%QUOTE-FOR-SET-VAR",	0, 0, pf_quote_for_setvar },
  { "%%AFTER-PAGE-RETURN",	1, 0, pf_after_page_return },
  { "APPLY",			-1, 0, pf_apply },

  { (char *)NULL,		0, 0, (PFunHandler *)NULL }
};

PACKAGE_INITIALIZER (initialize_primitive_functions)
DEFINE_SECTION (PRIMITIVE-OPERATORS, builtins; primitives; low-level,
"While most functions in <Meta-HTML> are used to create or manipulate Web
pages, there may be times when you will wish to manipulate the <Meta-HTML>
language itself, the low-level operation of the server, or have direct
access to the current page that is executing, and information pertaining
to the <Meta-HTML> parser itself.

The functions described in this section allow just this type of low-level
access.  Programming wizards may find these functions useful -- many of
them are here to allow the implementation of core functionality in <Meta-HTML>
in the <Meta-HTML> language itself.

All of the primitive language operators begin with the two-character sequence
of double percent signs (\"%%\") in order to distinguish them from the
other, higher-level functions in <Meta-HTML>.", "")

DEFUNX (pf_%%eval, &rest body,
"Evaluate the result of evaluating <var body> and return that value.

You may use this function to call another function on some arguments,
where the other function is determined dynamically.  For example:
<example>
<if <set-in-session>
    <set-var func=set-session-var>
  <set-var func=set-var>>
.blank
<%%eval <<get-var func> <get-var name> = <get-var value>>>
</example>")
static void
pf_eval (PFunArgs)
{
  char *result = body ? body->buffer : (char *)NULL;

  if (result != (char *)NULL)
    {
      char *expr = mhtml_evaluate_string (result);
      result = mhtml_evaluate_string (expr);

      if (!empty_string_p (result))
	{
	  bprintf_insert (page, start, "%s", result);
	  *newstart = start + strlen (result);
	}

      if (expr != (char *)NULL) free (expr);
      if (result != (char *)NULL) free (result);
    }
}

DEFUNX (pf_%%quote_for_set_var, &rest body,
"After evaluating <var body>, the results are quoted in such a way that
Meta-HTML will treat it as one argument.  Used internally by the function
invoker.")
static void
pf_quote_for_setvar (PFunArgs)
{
  if (body->buffer)
    {
      register int i;
      char *value;

      for (i = 0; i < body->bindex && whitespace (body->buffer[i]); i++);
      value = mhtml_evaluate_string (body->buffer + i);

      if (value != (char *)NULL)
	{
	  bprintf_insert (page, start, "%s", quote_for_setvar (value));
	  free (value);
	}
    }
}

DEFMACROX (pf_%%after_page_return, ,
"Store <var body> for execution at a later time, specifically, after the top
level process is completed.  For the <Meta-HTML> server or engine, this is
after the requested page has been successfully delivered; for <code>mhc</code>,
this is after the main document has finished processing, and the results have
been returned.

I would be interested if anybody actually needs this function -- if you do,
please drop me a line showing how it made your life easier.")

static PAGE *after_page_return_buffer = (PAGE *)NULL;
static void
pf_after_page_return (PFunArgs)
{
  if ((body != (PAGE *)NULL) && (!empty_string_p (body->buffer)))
    {
      if (after_page_return_buffer == (PAGE *)NULL)
	after_page_return_buffer = page_create_page ();

      bprintf (after_page_return_buffer, "%s", body->buffer);
    }
}

PAGE *
get_after_page_return_buffer (void)
{
  return (after_page_return_buffer);
}
      
DEFUN (pf_apply, func &rest args,
"Apply <var func> to <var args>.

This <i>weak</i>macro can either be used as a simple tag or as a
complex tag -- its usage is dependent on the function being called.

Using <tag apply> as a simple tag:
<example>
<apply add 3 4 5>         --> 12
<defun foo &key bar baz>
  <get-var bar>, <get-var baz>
</defun>
<apply foo \"bar=this baz=2\"> --> this,2
</example>

Using <tag apply> as a complex tag:
<example>
<defmacro upcase-text &key bold? &rest body>
   <if <get-var-once bold?> <b>>
   <upcase %qbody>
   <if <get-var-once bold?> </b>>
</defmacro>
<apply upcase-text> This is a list of things to \"change case of\" </apply>
<apply upcase-text bold?=true> And this is upcased <i>and</i> bold </apply>
</example>")
{
  char *func = mhtml_evaluate_string (get_positional_arg (vars, 0));
  char *result = (char *)NULL;

  if (!empty_string_p (func))
    {
      BPRINTF_BUFFER *funcall = bprintf_create_buffer ();
      int arg_index = 1;
      char *raw_arg;
      UserFunction *uf = mhtml_find_user_function (func);
      PFunDesc *desc = pagefunc_get_descriptor (func);
      int macro_call = 0;

      if (uf != (UserFunction *)NULL)
	{
	  if (uf->type == user_MACRO)
	    macro_call = 1;
	}
      else if (desc != (PFunDesc *)NULL)
	{
	  if (desc->complexp != 0)
	    macro_call = 1;
	}

      bprintf (funcall, "<%s", func);

      while ((raw_arg = get_positional_arg (vars, arg_index)) != (char *)NULL)
	{
	  char *arg = mhtml_evaluate_string (raw_arg);

	  if (empty_string_p (arg))
	    bprintf (funcall, " \"\"");
	  else
	    bprintf (funcall, " %s", arg);

	  xfree (arg);
	  arg_index++;
	}

      /* Now pass in keyword arguments. */
      {
	Symbol **syms = symbols_of_package (vars);

	if (syms != (Symbol **)NULL)
	  {
	    register int i;	    
	    Symbol *sym;

	    for (i = 0; (sym = syms[i]) != (Symbol *)NULL; i++)
	      {
		if ((sym->type == symtype_STRING) &&
		    (strcmp (sym->name, "*PVARS*") != 0) &&
		    (strcmp (sym->name, "*PVALS*") != 0) &&
		    (sym->values_index != 0))
		  {
		    char *val = mhtml_evaluate_string (sym->values[0]);

		    bprintf (funcall, " %s=%s", sym->name,
			     empty_string_p (val) ? "" : 
			     quote_for_setvar (val));
		    xfree (val);
		  }
	      }
	  }
      }

      bprintf (funcall, ">");

      /* If this is a macro call, we must print the body here. */
      if (macro_call)
	{
	  bprintf (funcall, "%s", body->buffer);
	  bprintf (funcall, "</%s>", func);
	}

      if (debug_level > 5)
	page_debug ("Apply: [%s]", funcall->buffer);

      result = mhtml_evaluate_string (funcall->buffer);
      bprintf_free_buffer (funcall);
    }

  if (!empty_string_p (result))
    {
      int len = strlen (result);
      bprintf_insert (page, start, "%s", result);
      *newstart += len;
    }

  xfree (func);
  xfree (result);
}

#if defined (__cplusplus)
}
#endif

