/* GKrellM
|  Copyright (C) 1999 Bill Wilson
|
|  Author:	Bill Wilson		bill@gkrellm.net
|  Latest versions might be found at:
|		http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation, Inc.,
|  59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "gkrellm.h"

#include <mntent.h>
#include <sys/vfs.h>

#include <linux/cdrom.h>
#include <linux/fd.h>


typedef struct
	{
	gchar	*directory;
	gchar	*device;
	gchar	*type;
	gchar	*options;
	}
	Mount;

typedef struct
	{
	GtkWidget		*vbox;
	gint			idx;
	Panel			panel;
	Decal			mount_decal,
					label_decal;
	gchar			*label;
	gint			label_is_data;

	Mount			mount;
	gint			enable_mounting;
	gint			in_drawer;

	gint			is_mounted;
	gint			eject_pending;
	struct statfs	stat_fs;
	}
	FSmon;

static GList		*fs_mon_list,
					*proc_mounts_list;

static GtkWidget	*fs_main_vbox,
					*fs_drawer_vbox;

static gint			fs_drawer_is_shown;

static gint			n_fs_monitors;
static gint			need_fs_check;
static gint			label_data_toggle;		/* Show bavail or total blocks */

static GList		*fstab_list;


static void
get_fstab_list()
	{
	FILE			*f;
	struct mntent	*me;
	Mount			*m;

	while (fstab_list)
		{
		m = (Mount *) fstab_list->data;
		g_free(m->device);
		g_free(m->directory);
		g_free(m->type);
		g_free(m->options);
		g_free(m);
		fstab_list = g_list_remove(fstab_list, fstab_list->data);
		}
	if ((f = setmntent("/etc/fstab", "r")) == NULL)
		return;
	while ((me = getmntent(f)) != NULL)
		{
		if (   strcmp(me->mnt_type, "devpts") == 0
			|| strcmp(me->mnt_type, "swap") == 0
			|| strcmp(me->mnt_type, "proc") == 0
/*			|| strcmp(me->mnt_type, "nfs") == 0 */
			|| strcmp(me->mnt_type, "ignore") == 0
		   )
			continue;
		m = g_new0(Mount, 1);
		m->device = g_strdup(me->mnt_fsname);
		m->directory = g_strdup(me->mnt_dir);
		m->type = g_strdup(me->mnt_type);
		m->options = g_strdup(me->mnt_opts);

		fstab_list = g_list_append(fstab_list, m);
		}
	endmntent(f);
	}

static Mount *
in_fstab_list(gchar *s)
	{
	GList	*list;
	Mount	*m;

	for (list = fstab_list; list; list = list->next)
		{
		m = (Mount *)list->data;
		if (strcmp(s, m->directory) == 0)
			return m;
		}
	return NULL;
	}

static void
get_proc_mounts_list()
	{
	FILE		*f;
	Mount		*m;
	gchar		*s, buf[128], dev[64], dir[128], type[32];

	while (proc_mounts_list)
		{
		m = (Mount *) proc_mounts_list->data;
		g_free(m->directory);
		g_free(m->device);
		g_free(m->type);
		g_free(proc_mounts_list->data);
		proc_mounts_list =
				g_list_remove(proc_mounts_list, proc_mounts_list->data);
		}
	if ((f = fopen("/proc/mounts", "r")) == NULL)
		return;
	while (fgets(buf, sizeof(buf), f))
		{
		sscanf(buf, "%s %s %s", dev, dir, type);
		if (   strcmp(type, "devpts") == 0
			|| strcmp(type, "proc") == 0
/*			|| strcmp(type, "nfs") == 0	*/
		   )
			continue;
		/* Strip trailing / from the directory.
		*/
		s = strrchr(dir, (int) '/');
		if (s && s != dir && *(s+1) == '\0')
			*s = '\0';
		m = g_new0(Mount, 1);
		m->directory = g_strdup(dir);
		m->device = strdup(dev);
		m->type = g_strdup(type);

		proc_mounts_list = g_list_append(proc_mounts_list, m);
		}
	fclose(f);
	}

static gint
fs_is_mounted(FSmon *fs)
	{
	Mount	*m_fs, *m_mounted;
	GList	*list;

	fs->is_mounted = FALSE;
	m_fs = &fs->mount;
	for (list = proc_mounts_list; list; list = list->next)
		{
		m_mounted = (Mount *) list->data;
		if (strcmp(m_fs->directory, m_mounted->directory) == 0)
			fs->is_mounted = TRUE;
		}
	return fs->is_mounted;
	}

static void
fs_format_label_data(FSmon *fs, gchar *buf)
	{
	struct statfs	*st;
	gfloat			bytes;

	st = &fs->stat_fs;
	if (label_data_toggle == 0)
		bytes = (gfloat) st->f_blocks * (gfloat) st->f_bsize;
	else
		bytes = (gfloat) st->f_bavail * (gfloat) st->f_bsize;

	if (bytes < 10e6)	/* Ramdisk, floppy */
		sprintf(buf, "%.2f M", bytes / 1e6);
	else if (bytes < 1e9)
		sprintf(buf, "%.0f M", bytes / 1e6);
	else if (bytes < 10e9)
		sprintf(buf, "%.2f G", bytes / 1e9);
	else if (bytes < 20e9)
		sprintf(buf, "%.1f G", bytes / 1e9);
	else if (bytes < 1000e9)
		sprintf(buf, "%.0f G", bytes / 1e9);
	else
		sprintf(buf, "%.1f T", bytes / 1e12);
	}

  /* Draw the fs label or toggle the fs total blocks and blocks avail.
  */
static void
draw_fs_label_decal(FSmon *fs, gint value)
	{
	Panel	*p;
	Decal	*d;
	Style	*style;
	gchar	buf[64];
	gint	w, x;

	p = &fs->panel;
	d = &fs->label_decal;
	if (value == 0 && value == d->value)	/* Label is shown */
		return;

	if (value == 0)
		sprintf(buf, "%s", fs->label);
	else
		fs_format_label_data(fs, buf);

	style = GK.meter_style[FS_IMAGE];

	w = gdk_string_measure(d->text_style.font, buf);
	x = UC.chart_width * p->label.position / LABEL_MAX;
	x -= style->border_panel.left;
	if (p->label.position > 50)
		x -= fs->mount_decal.w;
	x -= w / 2;

	if (x > d->w - w)
		x = d->w - w;
	if (x < 0)
		x = 0;
	
	gdk_draw_pixmap(d->pixmap, GK.draw1_GC, p->background,
			d->x, d->y,  0, 0,  d->w, d->h);
	draw_string(d->pixmap, d->text_style.font, &d->text_style.color,
			d->text_style.effect, x, d->y_baseline, buf);
	d->value = value;
	d->modified = TRUE;
	}

  /* Try to eject~ CDROMs.  If not a cdrom, the ioctl will just fail.
  |  Can't do zip or jazz without getting into SCSI commands.  Will
  |  consider that later.
  */
static void
eject_cdrom(FSmon *fs)
	{
	Mount	*m;
	gint	d, n;

	m = in_fstab_list(fs->mount.directory);
	if (m != NULL)
		{
		if ((d = open(m->device, (O_RDONLY | O_NONBLOCK))) >= 0)
			{
			extern int errno;

			n = ioctl(d, CDROMEJECT);
			if (GK.debug)
				printf("fs eject %s errno = %d\n", m->device, errno);
/*			ioctl(d, FDEJECT); */
			close(d);
			}
		else if (GK.debug)
			printf("could not open %s\n", m->device);
		}
	}

static void
mount_command(FSmon *fs)
	{
	gchar	cmd[256];

	if (fs->enable_mounting == FALSE)
		return;
	if (fs->is_mounted)
		{
		sprintf(cmd, "umount %s &", fs->mount.directory);
		fs->label_is_data = FALSE;
		draw_fs_label_decal(fs, 0);
		system(cmd);
		if (UC.enable_auto_eject)
			fs->eject_pending = GK.timer_ticks + 5;	/* at least 1/2 sec delay*/
		}
	else
		{
		sprintf(cmd, "mount %s &", fs->mount.directory);
		system(cmd);
		}
	/* Update fs mon at 1/2 second intervals for next 5 seconds.
	*/
	need_fs_check = 50;
	}

void
update_fs(void)
	{
	FSmon			*fs;
	Panel			*p;
	struct statfs	*st;
	GList			*list;

	if (fs_mon_list == NULL)
		return;
	if (need_fs_check > 0 && (need_fs_check-- % 5) != 0)
		return;
	else if (! GK.two_second_tick && need_fs_check == 0)
		return;

	if (GK.two_second_tick && need_fs_check == 0)
		label_data_toggle = 1 - label_data_toggle;

	get_proc_mounts_list();
	for (list = fs_mon_list; list; list = list->next)
		{
		fs = (FSmon *) list->data;
		p  = &fs->panel;
		st = &fs->stat_fs;
		p->krell->previous = 0;
		if (fs_is_mounted(fs))
			{
			if (fs->enable_mounting)
				draw_decal(p, p->decal, DECAL_FS_M_MOUNTABLE);
			else
				draw_decal(p, p->decal, DECAL_FS_M);
			if (statfs(fs->mount.directory, st) != 0)
				continue;
			p->krell->full_scale = st->f_blocks;
			update_krell(p, p->krell, st->f_blocks - st->f_bavail);
			}
		else
			{
			if (fs->enable_mounting)
				draw_decal(p, p->decal, DECAL_FS_UM_MOUNTABLE);
			else
				draw_decal(p, p->decal, DECAL_FS_UM);
			p->krell->full_scale = 100;		/* Arbitrary > 0 */
			update_krell(p, p->krell, 0);
			}
		if (fs->label_is_data)
			draw_fs_label_decal(fs, (gint) st->f_bavail);
		else if (fs->label_decal.value < 0)
			draw_fs_label_decal(fs, 0);

		draw_layers(p);
		if (fs->eject_pending && fs->eject_pending < GK.timer_ticks)
			{
			eject_cdrom(fs);
			fs->eject_pending = 0;
			}
		}
	}


  /* Real pixmaps are always in chart_minute even if chart_hour is displayed.
  */
static gint
fs_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	FSmon	*fs;
	GList	*list;

	for (list = fs_mon_list; list; list = list->next)
		{
		fs = (FSmon *) list->data;
		if (widget == fs->panel.drawing_area)
			gdk_draw_pixmap(widget->window,GK.draw1_GC, fs->panel.pixmap,
					ev->area.x, ev->area.y, ev->area.x, ev->area.y,
					ev->area.width, ev->area.height);
		}
	return FALSE;
	}

static gint
cb_fs_panel_click(GtkWidget *widget, GdkEventButton *ev)
	{
	FSmon	*fs;
	GList	*list;
	Decal	*d;
	gint	h_adjust_factor;

	/* Button 2 in any fs monitor toggles drawer open/shut
	*/
	if (ev->button == 2)
		{
		if (fs_drawer_is_shown)
			{
			gtk_widget_hide(fs_drawer_vbox);
			h_adjust_factor = -1;
			}
		else
			{
			gtk_widget_show(fs_drawer_vbox);
			h_adjust_factor = 1;
			}
		fs_drawer_is_shown = 1 - fs_drawer_is_shown;
		for (list = fs_mon_list; list; list = list->next)
			{
			fs = (FSmon *) list->data;
			if (fs->in_drawer)
				GK.monitor_height += h_adjust_factor * fs->panel.h;
			}
		pack_side_frames();
		return TRUE;
		}
	for (list = fs_mon_list; list; list = list->next)
		{
		fs = (FSmon *) list->data;
		d = fs->panel.decal;

		/* Just a strip, so no need to check y
		*/
		if (widget == fs->panel.drawing_area)
			{
			if (ev->button == 1)
				{
				if (ev->x > d->x && ev->x <= d->x + d->w)
					mount_command(fs);
				else if (fs->is_mounted)
					{
					label_data_toggle = 0;
					fs->label_is_data = 1 - fs->label_is_data;
					if (! fs->label_is_data)
						draw_fs_label_decal(fs, 0);
					need_fs_check = 8;
					}
				}
			break;
			}
		}
	return TRUE;
	}

static void
create_fs_monitor(GtkWidget *vbox1, FSmon *fs, gint index)
	{
	GtkWidget	*vbox;
	Style		*style;
	Panel		*p;
	Decal		*d;
	gint		l, r, w, a, dd;

	if (GK.debug)
		printf("Creating fs monitor\n");

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox1), vbox);
	fs->vbox = vbox;

	fs->idx = index;
	++n_fs_monitors;
	p = &fs->panel;

	style = GK.meter_style[FS_IMAGE];

	d = &fs->mount_decal;
	p->decal = d;
	if (style->label_position <= 50)
		d->x = UC.chart_width - GK.decal_width - style->border_panel.right;
	else
		d->x = style->border_panel.left;
	d->y = style->border_panel.top;
	d->w = GK.decal_width;
	d->h = GK.decal_height;
	d->pixmap = GK.decal_pixmap;
	d->mask = GK.decal_mask;
	d->value = -1;			/* Force initial draw */
	d->type = DECAL_TRANSPARENCY;

	/* Make the label a decal so I can toggle writing the label and fs
	|  used/avail info without rerendering the panel background.
	*/
	d->next = &fs->label_decal;
	d = &fs->label_decal;
	d->next = NULL;
	d->y = style->border_panel.top;
	default_textstyle(&d->text_style, TEXTSTYLE_METER);
	gdk_string_extents(GK.label_font, "Ap0", &l, &r, &w, &a, &dd);
	d->y_baseline = a;
	d->h = a + dd + d->text_style.effect;
	if (style->label_position <= 50)
		d->x = style->border_panel.left;
	else
		d->x = style->border_panel.left + fs->mount_decal.w;
	d->w = UC.chart_width - fs->mount_decal.w
			- style->border_panel.left - style->border_panel.right;
	d->pixmap = gdk_pixmap_new(top_window->window, d->w, d->h, -1);
	d->value = -1;			/* Force initial draw */
	d->type = DECAL_OPAQUE;

	create_krell("fs", GK.krell_meter_image[FS_IMAGE], &p->krell, style);
	default_textstyle(&p->label.textstyle, TEXTSTYLE_METER);

	configure_panel(p, NULL, style);

	create_panel_area(vbox, p, GK.bg_meter_image[FS_IMAGE]);

	if (!fs->in_drawer || (fs->in_drawer && fs_drawer_is_shown))
		GK.monitor_height += p->h;

	gtk_signal_connect(GTK_OBJECT(p->drawing_area), "expose_event",
			(GtkSignalFunc) fs_expose_event, NULL);
	gtk_signal_connect(GTK_OBJECT(p->drawing_area),"button_release_event",
			(GtkSignalFunc) cb_fs_panel_click, NULL);

	gtk_widget_show_all(vbox);
	draw_fs_label_decal(fs, 0);
	}

void
destroy_fs_monitor(FSmon *fs)
	{
	g_free(fs->label);
	g_free(fs->mount.directory);
	GK.monitor_height -= fs->panel.h;
	destroy_panel(&fs->panel);
	destroy_krell(fs->panel.krell);
	gtk_widget_destroy(fs->vbox);
	g_free(fs);
	--n_fs_monitors;
	}

void
create_fs(GtkWidget *vbox)
	{
	GList		*list;
	FSmon		*fs;
	gint		i;

	fs_main_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), fs_main_vbox);
	gtk_widget_show(fs_main_vbox);

	fs_drawer_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), fs_drawer_vbox);
	fs_drawer_is_shown = FALSE;

	get_fstab_list();
	for (i = 0, list = fs_mon_list; list; ++i, list = list->next)
		{
		fs = (FSmon *)list->data;
		create_fs_monitor(fs->in_drawer ? fs_drawer_vbox : fs_main_vbox,fs, i);
		}
	}

void
write_fs_config(FILE *f)
	{
	GList	*list;
	FSmon	*fs;

	for (list = fs_mon_list; list; list = list->next)
		{
		fs = (FSmon *) list->data;
		fprintf(f, "fs %s %s %d %d\n",
				fs->label, fs->mount.directory,
				fs->enable_mounting, fs->in_drawer);

		}
	}

void
load_fs_config(gchar *arg)
	{
	FSmon	*fs;
	gchar	label[16], dir[256];

	fs = g_new0(FSmon, 1);
	sscanf(arg, "%s %s %d %d", label, dir,
							&fs->enable_mounting, &fs->in_drawer);
	fs->label           = g_strdup(label);
	fs->mount.directory = g_strdup(dir);
	fs_mon_list = g_list_append(fs_mon_list, fs);
	}


/* --------------------------------------------------------------------- */

static GtkWidget	*primary_label_entry,
					*primary_dir_combo,
					*secondary_label_entry,
					*secondary_dir_combo;

static GtkWidget	*primary_clist,
					*secondary_clist;

static GtkWidget	*primary_mounting_button,
					*secondary_mounting_button;
static GtkWidget	*enable_auto_eject_button;

static gint			primary_selected_row	= -1;
static gint			secondary_selected_row	= -1;

static gint			fs_list_modified;
static gint			uid;


  /* Watch what is going into the directory combo entry, compare it to
  |  fstab entries and accordingly set sensitivity of the mounting_button.
  */
static void
cb_combo_changed(GtkWidget *widget, gpointer data)
	{
	GtkWidget	*dir_combo,
				*mounting_button;
	Mount		*m;
	gchar		*s;
	gint		drawer = (gint) data;

	dir_combo = drawer ? secondary_dir_combo : primary_dir_combo;
	mounting_button = drawer ? secondary_mounting_button : primary_mounting_button;

	s = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(dir_combo)->entry));
	m = in_fstab_list(s);
	if (m != NULL && (strstr(m->options, "user") != NULL || uid == 0))
		gtk_widget_set_sensitive(mounting_button, TRUE);
	else
		{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mounting_button), FALSE);
		gtk_widget_set_sensitive(mounting_button, FALSE);
		}
	}

static void
reset_fs_entries(GtkWidget *entry, GtkWidget *combo, GtkWidget *button)
	{
	gtk_entry_set_text(GTK_ENTRY(entry), "");
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry), "");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), FALSE);
	}

static void
cb_fs_clist_selected(GtkWidget *clist, gint row, gint column,
			GdkEventButton *bevent, gpointer data)
	{
	GtkWidget	*dir_combo,
				*label_entry,
				*mounting_button;
	gchar		*s;
	gint		state, *selected_row;
	gint		drawer = (gint) data;

	label_entry = drawer ? secondary_label_entry : primary_label_entry;
	dir_combo = drawer ? secondary_dir_combo : primary_dir_combo;
	mounting_button = drawer ? secondary_mounting_button : primary_mounting_button;
	selected_row = drawer ? &secondary_selected_row : &primary_selected_row;

	gtk_clist_get_text(GTK_CLIST(clist), row, 0, &s);
	gtk_entry_set_text(GTK_ENTRY(label_entry), s);

	gtk_clist_get_text(GTK_CLIST(clist), row, 1, &s);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(dir_combo)->entry), s);

	gtk_clist_get_text(GTK_CLIST(clist), row, 2, &s);
	state = (strcmp(s, "yes") == 0) ? TRUE : FALSE;
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(mounting_button), state);

	*selected_row = row;
	}

static void
cb_fs_clist_unselected(GtkWidget *clist, gint row, gint column,
			GdkEventButton *bevent, gpointer data)
	{
	GtkWidget	*dir_combo,
				*label_entry,
				*mounting_button;
	gint		*selected_row;
	gint		drawer = (gint) data;

	label_entry = drawer ? secondary_label_entry : primary_label_entry;
	dir_combo = drawer ? secondary_dir_combo : primary_dir_combo;
	mounting_button = drawer ? secondary_mounting_button : primary_mounting_button;
	selected_row = drawer ? &secondary_selected_row : &primary_selected_row;

	reset_fs_entries(label_entry, dir_combo, mounting_button);
	*selected_row = -1;
	}

static void
cb_enter_fs(GtkWidget *widget, gpointer data)
	{
	GtkWidget	*dir_combo,
				*label_entry,
				*mounting_button,
				*clist;
	gchar		*buf[4];
	gint		*selected_row;
	gint		drawer = (gint) data;

	label_entry = drawer ? secondary_label_entry : primary_label_entry;
	dir_combo = drawer ? secondary_dir_combo : primary_dir_combo;
	mounting_button = drawer ? secondary_mounting_button : primary_mounting_button;
	clist = drawer ? secondary_clist : primary_clist;
	selected_row = drawer ? &secondary_selected_row : &primary_selected_row;

	buf[0] = gtk_entry_get_text(GTK_ENTRY(label_entry));
	buf[1] = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(dir_combo)->entry));
	buf[2] = GTK_TOGGLE_BUTTON(mounting_button)->active ? "yes" : "---";
	buf[3] = NULL;
	if (*(buf[0]) == '\0' || *(buf[1]) == '\0')	/* validate we have input */
		return;
	if (*selected_row >= 0)
		{
		gtk_clist_set_text(GTK_CLIST(clist), *selected_row, 0, buf[0]);
		gtk_clist_set_text(GTK_CLIST(clist), *selected_row, 1, buf[1]);
		gtk_clist_set_text(GTK_CLIST(clist), *selected_row, 2, buf[2]);
		gtk_clist_unselect_row(GTK_CLIST(clist), *selected_row, 0);
		*selected_row = -1;
		}
	else
		gtk_clist_append(GTK_CLIST(clist), buf);
	reset_fs_entries(label_entry, dir_combo, mounting_button);
	fs_list_modified = TRUE;
	}

static void
cb_delete_fs(GtkWidget *widget, gpointer data)
	{
	GtkWidget	*dir_combo,
				*label_entry,
				*mounting_button,
				*clist;
	gint		*selected_row;
	gint		drawer = (gint) data;

	label_entry = drawer ? secondary_label_entry : primary_label_entry;
	dir_combo = drawer ? secondary_dir_combo : primary_dir_combo;
	mounting_button = drawer ? secondary_mounting_button : primary_mounting_button;
	clist = drawer ? secondary_clist : primary_clist;
	selected_row = drawer ? &secondary_selected_row : &primary_selected_row;

	reset_fs_entries(label_entry, dir_combo, mounting_button);
	if (*selected_row >= 0)
		{
		gtk_clist_remove(GTK_CLIST(clist), *selected_row);
		fs_list_modified = TRUE;
		*selected_row = -1;
		}
	}

  /* Transfer the state of the config tab clists to the fs_mon_list glist.
  |  There are separate primary and secondary clists while only one glist,
  |  so the clists are merged into a new glist which replaces the old
  |  fs_mon_list.  FSmon structs from the old fs_mon_list are reused if
  |  possible.
  */
void
apply_fs_config()
	{
	GList	*new_fs_list, *old_fs_list;
	FSmon	*fs;
	gchar	*s;
	gint	row, need_new_monitor;

	UC.enable_auto_eject = GTK_TOGGLE_BUTTON(enable_auto_eject_button)->active;
	if (!fs_list_modified)
		return;
	new_fs_list = NULL;
	old_fs_list = fs_mon_list;

	/* Append primary_clist entries to the new_fs_list and reuse old primary
	|  structures as long as possible.
	*/
	for (row = 0; row < GTK_CLIST(primary_clist)->rows; ++row)
		{
		if (old_fs_list && ((FSmon *) old_fs_list->data)->in_drawer == FALSE)
			{
			fs = (FSmon *) old_fs_list->data;
			g_free(fs->label);
			g_free(fs->mount.directory);
			fs->label_decal.value = -1;
			old_fs_list = old_fs_list->next;
			need_new_monitor = FALSE;
			}
		else
			{
			fs = g_new0(FSmon, 1);
			need_new_monitor = TRUE;
			}
		gtk_clist_get_text(GTK_CLIST(primary_clist), row, 0, &s);
		fs->label = g_strdup(s);
		gtk_clist_get_text(GTK_CLIST(primary_clist), row, 1, &s);
		fs->mount.directory = g_strdup(s);
		gtk_clist_get_text(GTK_CLIST(primary_clist), row, 2, &s);
		fs->enable_mounting = (strcmp(s, "yes") == 0) ? TRUE : FALSE;
		fs->in_drawer = FALSE;
		new_fs_list = g_list_append(new_fs_list, fs);
		if (need_new_monitor)
			create_fs_monitor(fs_main_vbox, fs, n_fs_monitors);
		}

	/* Destroy any remaining primary monitors left in the old fs_mon_list.
	*/
	while (old_fs_list && ((FSmon *) old_fs_list->data)->in_drawer == FALSE)
		{
		destroy_fs_monitor((FSmon *) old_fs_list->data);
		old_fs_list = old_fs_list->next;
		}

	/* Append the secondary clist monitors to the new_fs_list.
	*/
	for (row = 0; row < GTK_CLIST(secondary_clist)->rows; ++row)
		{
		if (old_fs_list && ((FSmon *) old_fs_list->data)->in_drawer == TRUE)
			{
			fs = (FSmon *) old_fs_list->data;
			g_free(fs->label);
			g_free(fs->mount.directory);
			fs->label_decal.value = -1;
			old_fs_list = old_fs_list->next;
			need_new_monitor = FALSE;
			}
		else
			{
			fs = g_new0(FSmon, 1);
			need_new_monitor = TRUE;
			}
		gtk_clist_get_text(GTK_CLIST(secondary_clist), row, 0, &s);
		fs->label = g_strdup(s);
		gtk_clist_get_text(GTK_CLIST(secondary_clist), row, 1, &s);
		fs->mount.directory = g_strdup(s);
		gtk_clist_get_text(GTK_CLIST(secondary_clist), row, 2, &s);
		fs->enable_mounting = (strcmp(s, "yes") == 0) ? TRUE : FALSE;
		fs->in_drawer = TRUE;
		new_fs_list = g_list_append(new_fs_list, fs);
		if (need_new_monitor)
			create_fs_monitor(fs_drawer_vbox, fs, n_fs_monitors);
		}
	/* Destroy any remaining secondary monitors left in the old fs_mon_list.
	*/
	while (old_fs_list && ((FSmon *) old_fs_list->data)->in_drawer == TRUE)
		{
		destroy_fs_monitor((FSmon *) old_fs_list->data);
		old_fs_list = old_fs_list->next;
		}
	/* And destroy the old fs_mon_list glist and reassign it to the new list.
	*/
	while (fs_mon_list)
		fs_mon_list = g_list_remove(fs_mon_list, fs_mon_list->data);
	fs_mon_list = new_fs_list;
	pack_side_frames();
	fs_list_modified = FALSE;
	}

static gchar *fs_title[3] = { "Label", "Mount Point", "Enable mount/umount" };

static void
fill_fs_tab(GtkWidget *vbox, GtkWidget **label_entry, GtkWidget **dir_combo,
			GtkWidget **mount_button, GtkWidget **fs_clist, gint drawer)
	{
	GtkWidget		*table;
	GtkWidget		*hbox;
	GtkWidget		*label;
	GtkWidget		*button;
	GtkWidget		*separator;
	GtkWidget		*scrolled;
	GList			*list, *combo_list;
	FSmon			*fs;
	gchar			*buf[4];
	gint			row;

	table = gtk_table_new(2, 2, FALSE /*homogeneous*/);
	gtk_table_set_col_spacings(GTK_TABLE(table), 5);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 2);

	label = gtk_label_new("Label");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
	label = gtk_label_new("Mount Point");
	gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 0, 1);

	*label_entry = gtk_entry_new_with_max_length(10);
	gtk_widget_set_usize(*label_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), *label_entry, 0, 1, 1, 2);

	*dir_combo = gtk_combo_new();
	gtk_table_attach_defaults(GTK_TABLE(table), *dir_combo, 1, 2, 1, 2);
	combo_list = NULL;
	for (list = fstab_list; list; list = list->next)
		combo_list = g_list_append(combo_list,
					((Mount *)list->data)->directory);
	gtk_combo_set_popdown_strings( GTK_COMBO(*dir_combo), combo_list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(*dir_combo)->entry), "");
	gtk_signal_connect(GTK_OBJECT(GTK_COMBO(*dir_combo)->entry), "changed",
				GTK_SIGNAL_FUNC (cb_combo_changed), (gpointer) drawer);

	hbox = gtk_hbox_new(FALSE, 3);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);

	*mount_button = gtk_check_button_new_with_label(
					"Enable mount/umount commands");
	gtk_box_pack_start(GTK_BOX(hbox), *mount_button, TRUE, TRUE, 10);

	button = gtk_button_new_with_label("Enter");
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) cb_enter_fs, (gpointer) drawer);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 4);

	button = gtk_button_new_with_label("Delete");
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) cb_delete_fs, (gpointer) drawer);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 4);

	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 2);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	*fs_clist = gtk_clist_new_with_titles(3, fs_title);
	gtk_clist_set_shadow_type (GTK_CLIST(*fs_clist), GTK_SHADOW_OUT);
	gtk_clist_set_column_width (GTK_CLIST(*fs_clist), 0, 60);
	gtk_clist_set_column_width (GTK_CLIST(*fs_clist), 1, 210);

	gtk_signal_connect (GTK_OBJECT(*fs_clist), "select_row",
			(GtkSignalFunc) cb_fs_clist_selected, (gpointer) drawer);
	gtk_signal_connect (GTK_OBJECT(*fs_clist), "unselect_row",
			(GtkSignalFunc) cb_fs_clist_unselected, (gpointer) drawer);
	gtk_container_add (GTK_CONTAINER (scrolled), *fs_clist);
	for (list = fs_mon_list; list; list = list->next)
		{
		fs = (FSmon *) list->data;
		if (fs->in_drawer == drawer)
			{
			buf[0] = fs->label;
			buf[1] = fs->mount.directory;
			buf[2] = fs->enable_mounting ? "yes" : "---";
			buf[3] = NULL;
			row = gtk_clist_append(GTK_CLIST(*fs_clist), buf);
			}
		}
	}


static gchar *fs_info_text =
"Enter file system mount points to monitor.  The krell shows the\n"
"ratio of blocks used to total blocks available.  If a mount point\n"
"is in your /etc/fstab and you have mount permission then mount and\n"
"umount commands can be enabled and executed for that mount point.\n"
"Mount table entries in /etc/fstab must have the \"user\" option set\n"
"to enable mounting permission unless GKrellM is run as root.\n"
"For example, if you run GKrellM as a normal user and you want to be\n"
"able to mount your floppy, your /etc/fstab could have either of:\n"
"    /dev/fd0  /mnt/floppy  ext2   user,noauto,rw,exec  0  0\n"
"or\n"
"    /dev/fd0  /mnt/floppy  ext2   user,defaults  0  0\n\n"

"File system monitors can be created as primary (always visible)\n"
"or secondary which can be hidden and then shown when they are of\n"
"interest.  For example, you might make primary file system monitors\n"
"for root, home, or user so they will be always visible, but make\n"
"secondary monitors for less frequently used mount points such as\n"
"floppy, zip, backup partitions, foreign file system types, etc.\n"
"A standard cdrom mount will show as 100% full but a monitor for it\n"
"could be created with mounting enabled just to have the\n"
"mount/umount convenience.\n\n"

"    User Interface\n"
" * Left click on a fs monitor label to toggle display of fs capacity\n"
" and free space.\n"
" * Left click on a drive decal to apply mount/umount commands (if\n"
" enabled) to the mount point being monitored.\n"
" * Middle click on a fs panel to show/hide secondary fs monitors.\n\n"

"At least one primary fs monitor must exist to click on in order to show\n"
"secondary ones.\n"
	;

void
create_fs_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*vbox;
	GtkWidget		*scrolled;
	GtkWidget		*text;

	fs_list_modified = FALSE;
	get_fstab_list();
	uid = geteuid();		/* If root will allow mount/umount always */

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

	vbox = create_tab(tabs, "Primary");
	fill_fs_tab(vbox, &primary_label_entry, &primary_dir_combo,
			&primary_mounting_button, &primary_clist, FALSE);

	vbox = create_tab(tabs, "Secondary");
	fill_fs_tab(vbox, &secondary_label_entry, &secondary_dir_combo,
			&secondary_mounting_button, &secondary_clist, TRUE);

	vbox = create_tab(tabs, "Options");
	enable_auto_eject_button = gtk_check_button_new_with_label(
		"Enable eject when CD-ROMs are unmounted");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_auto_eject_button),
			UC.enable_auto_eject);
	gtk_box_pack_start(GTK_BOX(vbox), enable_auto_eject_button, TRUE, TRUE, 0);

/* --Info tab */
	vbox = create_tab(tabs, "Info");
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	text = gtk_text_new(NULL, NULL);
	gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL, fs_info_text, -1);
	gtk_text_set_editable(GTK_TEXT(text), FALSE);
	gtk_container_add(GTK_CONTAINER(scrolled), text);
	}
