/***************************************************************************
                          grain.cpp  -  grain entry
                             -------------------                                         
    begin                : Sat Oct 9 1999                                           
    copyright            : (C) 1999 by David Johnson                         
    email                : arandir@meer.net                                     

    This software licensed under the Berkeley Software Distribution License
 ***************************************************************************/

#include <math.h>

#include <qstring.h>
#include <qstringlist.h>

#include "grain.h"

Grain::Grain() { name = "grain", quantity = extract = color = 0; use = GRAIN_OTHER; }

Grain::Grain(const QString &gname, const unsigned &gquant, const unsigned &gextr,
			const unsigned &gcolor, const int &guse)
{
	name = gname;
	quantity = gquant;
	extract = gextr;
	color = gcolor;
	use = guse;
	yield = (double(quantity) / 100.0) * double(extract - 1000);
	hcu = (double(quantity) / 100.0) * double(color / 10.0);
}

Grain::Grain(const Grain &g)
{
	name = g.name;
	quantity = g.quantity;
	extract = g.extract;
	color = g.color;
	use = g.use;
	yield = g.yield;
	hcu = g.hcu;
}

Grain Grain::operator=(const Grain &g)
{
	name = g.name;
	quantity = g.quantity;
	extract = g.extract;
	color = g.color;
	use = g.use;
	yield = g.yield;
	hcu = g.hcu;
	return *this;
}

bool Grain::operator==(const Grain &g)
{
	return (name == g.name)
		&& (quantity == g.quantity)
		&& (extract == g.extract)
		&& (color == g.color)
		&& (use == g.use);
}

Grain::~Grain() {}

///////////////////////////////////////////////////////////////////////////////
// member access

QString Grain::getName() { return name; }

unsigned Grain::getQuantity() { return quantity; }

unsigned Grain::getExtract() { return extract; }

unsigned Grain::getColor() { return color; }

int Grain::getUse() { return use; }
	
QString Grain::getQuantityString() { return formatValue(quantity, 2); }

QString Grain::getExtractString() { return formatValue(extract, 3); }

QString Grain::getColorString() { return formatValue(color, 1); }

QString Grain::getUseString()
{
	switch (use) {
		case GRAIN_OTHER: return QObject::tr("other");
		case GRAIN_MASHED: return QObject::tr("mashed");
		case GRAIN_STEEPED:return QObject::tr("steeped");
		case GRAIN_EXTRACT: return QObject::tr("extract");
	}
	return QObject::tr("other");
}

double Grain::getYield() { return yield; }

double Grain::getHCU() { return hcu; }

QStringList Grain::getUseStringList()
{
	QStringList stemp = QObject::tr("extract");
	stemp += QObject::tr("mashed");
	stemp += QObject::tr("steeped");
	stemp += QObject::tr("other");
	return stemp;
}

///////////////////////////////////////////////////////////////////////////////
// utility functions

QString Grain::formatValue(const unsigned &value, const unsigned &fix)
{
	unsigned divisor = unsigned(pow(10.0, double(fix)));
	unsigned leftuint = value / divisor;
	QString leftstring = QString::number(leftuint) + ".";
	QString rightstring = QString::number(value - (leftuint * divisor));
	leftstring += rightstring.rightJustify(fix, '0', true);
	return leftstring;
}

int Grain::useStringToEnum(const QString &uname)
{
	if ((uname.lower() == "mashed") || (uname.lower() == "mash")) return GRAIN_MASHED;
	if ((uname.lower() == "steeped") || (uname.lower() == "steep")) return GRAIN_STEEPED;
	if (uname.lower() == "extract") return GRAIN_EXTRACT;
	return GRAIN_OTHER;
}

///////////////////////////////////////////////////////////////////////////////
// serialization

QString Grain::serializeOut()
{
	QString text;
	text = QString::number(quantity) + ", " + QString::number(extract) + ", ";
	text += QString::number(color) + ", " + getUseString();
	return text;
}

void Grain::serializeIn(const QString &gname, const QString &gvalue)
{
	QString value = gvalue;
	QStringList tokens;
	int pos;
	
	// get tokens
	while (value.length() > 0) {
		pos = value.find(',');
		if (pos > 0) {
			tokens += value.left(pos);
			value.remove(0, pos+1);
			value = value.stripWhiteSpace();
		} else {
			// get last token
			tokens += value;
			value = "";
		}
	}
	// create new style
	name = gname;
	quantity	= tokens[0].toUInt();
	extract	= tokens[1].toUInt();
	color	= tokens[2].toUInt();
	/** do a bit more work on use */
	use = useStringToEnum(tokens[3]);
	yield = (double(quantity) / 100.0) * double(extract - 1000);
	hcu = (double(quantity) / 100.0) * double(color / 10.0);
}

///////////////////////////////////////////////////////////////////////////////
// GrainListItem class

GrainListItem::GrainListItem(QListView * parent, QString a, QString b = QString::null,
			QString c = QString::null, QString d = QString::null, QString e = QString::null,
			QString f = QString::null, QString g = QString::null, QString h = QString::null )
			: QListViewItem(parent, a, b, c, d, e, f, g, h) { git = NULL; };

GrainListItem::~GrainListItem() {}

GrainIterator GrainListItem::getGrainIterator() { return git; }

void GrainListItem::setGrainIterator(GrainIterator &it) { git = it; }

void GrainListItem::setToIterator()
{
	setText(0, (*git).getName());
	setText(1, (*git).getQuantityString());
	setText(2, (*git).getExtractString());
	setText(3, (*git).getColorString());
	setText(4, (*git).getUseString());
}












