/* sfm -- Simple File Manager
   Copyright (C) 1998 Pixel (Pascal Rigaux)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */
#include "io.h"
#include "sfm_server.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/types.h>

static const int listInitSize = 2;

#define elements_malloc(size) (struct elements_ *) malloc((size) * sizeof(struct elements_))
#define elements_realloc(p, size) *(p) = (struct elements_ *) realloc((void *) *(p), (size) * sizeof(struct elements_))

static int name_cmp(struct elements_ *a, struct elements_ *b);
static int extension_cmp(struct elements_ *a, struct elements_ *b);

static int (*mycmp)(struct elements_ *a, struct elements_ *b) = name_cmp;

static struct { int pid; char *msg; } pids[tmpSize];
static int size_pids = 0;



static int name_cmp(struct elements_ *a, struct elements_ *b)
{
  boolean ba = (lastchar(a->name) == '/');
  boolean bb = (lastchar(b->name) == '/');
  if (ba ^ bb) return ba ? -1 : 1;
  else return strcmp(a->name, b->name);
}

static int extension_cmp(struct elements_ *a, struct elements_ *b)
{
  char *pa, *pb;
  int r;
  boolean ba = (lastchar(a->name) == '/');
  boolean bb = (lastchar(b->name) == '/');
  if (ba ^ bb) return ba ? -1 : 1;

  pa = strrchr(a->name, '.');
  pb = strrchr(b->name, '.');
  if (pa && pb) {
    if ((r = strcmp(pa, pb))) return r;
  } else if (pa || pb) return (pa == NULL) ? -1 : 1;
  return strcmp(a->name, b->name);
}

extern int exec_it(char **args)
{
  int pid;

  if ((pid = fork()) == 0) {
    execvp(args[0], args);
    fprintf(stderr, "executable %s not found\n", args[0]);
    X_exit(1);
  }
  return pid;
}

extern void exec_it_and_mem(char **args, char *msg)
{
  int i, pid;

  view_warn(strdup(msg));
  pid = exec_it(args);
  for (i = 0; i < size_pids; i++) if (pids[i].pid == 0) break;
  if (i == size_pids) {
    if (i == tmpSize) return;
    size_pids++;
  }
  pids[i].pid = pid;
  pids[i].msg = msg;
}

extern int exec_it_and_wait(char **args)
{
  int pid, status;
  void received_sigint() { kill(pid, SIGINT); myexit(1); }

  install_signal_handler(SIGINT, received_sigint);
  pid = exec_it(args);
  if (waitpid(pid, &status, 0) == -1) die("can't get status of my child !");
  return status;
}

extern boolean any_child_left()
{
  int i;
  for (i = 0; i < size_pids; i++) if (pids[i].pid) return true;
  return false;
}

extern void kill_children_left()
{
  int i;
  for (i = 0; i < size_pids; i++) if (pids[i].pid) {
    kill(pids[i].pid, SIGINT);
    pids[i].pid = 0;
  }
}


extern void update_views()
{
  if (kill(getppid(), SIGUSR2)) die("kill (child)");
}

extern void received_child_ended(int pid, int status)
{
  int i;
  for (i = 0; i < size_pids && pids[i].pid != pid; i++);
  if (i == size_pids) return;
  pids[i].pid = 0;

  view_warn(strconcat(status == 0 ? "done " : "error ", pids[i].msg));
  update_views();
}

extern void received_sigchild()
{
  int pid, status;
  if ((pid = waitpid(-1, &status, WNOHANG)) > 0) received_child_ended(pid, status);
}

/* !! dir must end with a '/' !! */
extern DirList dir_list_with_suffix(char *dir, char *suffix)
{
  struct dirent *entry;
  struct stat entryStat;
  DIR *handle;
  DirList l;
  char tmp[tmpSize], *s;
  int i;

  l.size = listInitSize;
  l.elements = elements_malloc(l.size);

  if ((handle = opendir(dir)) == NULL) {
    warn("bad directory");
    l.size = 0;
    return l;
  }
  
  for (i = 0; (entry = readdir(handle)) != NULL;) {

    if (streq(entry->d_name, ".")) continue;
    if (streq(entry->d_name, "..")) continue;
    if (!strbeginswith(entry->d_name, suffix)) continue;
    if (i == l.size) {
      l.size *= 2;
      elements_realloc(&l.elements, l.size);
    }
    if (!strput(tmp, dir, entry->d_name, tmpSize)) die("too long file name");
    l.elements[i].mode = lstat(tmp, &entryStat) ? 0 : entryStat.st_mode;
    
    if (is_directory(tmp)) {
      s = strconcat(entry->d_name, "/");
    } else {
      s = strdup(entry->d_name);
    }
    l.elements[i++].name = s;
  }
  l.size = i;
  if (l.size) qsort(l.elements, l.size, sizeof(struct elements_), (int (*)(const void *, const void *)) mycmp);
  closedir(handle);
  return l;
}

extern void dir_list_toggle_sort_type()
{
  if (mycmp == name_cmp) mycmp = extension_cmp;
  else if (mycmp == extension_cmp) mycmp = name_cmp;
}

extern void dir_list_free(DirList l)
{
  if (l.elements) {
    int i; 
    for (i = 0; i < l.size; i++) free(l.elements[i].name);
    FREE_NULL(l.elements);
  }
}

extern boolean stat_it(char *file, int *mode)
{
  struct stat entryStat;

  if (stat(file, &entryStat)) return false;
  *mode = entryStat.st_mode;
  return true;
}

extern boolean is_directory(char *dir)
{
  int mode;

  if (!stat_it(dir, &mode)) return false;
  return S_ISDIR(mode);
}

extern boolean is_file(char *name)
{
  int fd;

  fd = open(name, O_RDONLY);
  if (fd == -1) return false;
  close(fd);
  return true;
}

extern char *absolute2relative(char *src, char *base)
{
  char tmp[tmpSize];
  int l, i;

  i = common_chars(src, base);
  for (l = i; l > 0 && src[l - 1] != '/'; l--);
  for (i = 0, base += l; (base = strchr(base, '/')); base++) {
    memmove(tmp + 3, tmp, i);
    memcpy(tmp, "../", 3);
    i += 3;
  }
  strcpy(tmp + i, src + l);
  return strdup(tmp);
}


extern void paste(char *dir_prefix)
{
  int i, j, status, fd, size, *types;
  char *source, *name, *p, **files, dest[tmpSize], *args[tmpSize];

  if (!connect_server(&fd)) die("connect_server");

  send_msg(fd, "get\n");
  
  init_line_by_line_read();
  size = tmpSize;
  files = pchar_malloc(tmpSize);
  types = int_malloc(tmpSize);
  for (i = 0; strlen(source = line_by_line_read(fd)) > 0; i++) {
    if (i == tmpSize) {
      size *= 2;
      pchar_realloc(&files, size);
      int_realloc(&types, size);
    }
    files[i] = strdup(source);
    types[i] = atoi(line_by_line_read(fd));
  }
  size = i;

  for (j = 0; j < size; j++) {
    source = files[j];
    name = strrchr(source, '/');
    if (name && strlen(name) == 1) {
	*name = '\0';
	name = strrchr(source, '/');
    }
    if (name) name++; else name = source;

    strput(dest, dir_prefix, name, tmpSize);
    p = strend(dest);
    for (i = 0; is_file(dest); i++) sprintf(p, "-%d", i);

    switch (types[j]) 
	{
	case TYPE_COPY:
	  args[0] = "cp";
	  args[1] = 
#ifdef SunOS
	    "-R";
#else
	    "-a";
#endif
	  args[2] = source;
	  args[3] = dest;
	  args[4] = NULL;
	  status = exec_it_and_wait(args);
	  if (status) exit(status);	  
	  break;

	case TYPE_CUT:
	  args[0] = "mv";
	  args[1] = source;
	  args[2] = dest;
	  args[3] = NULL;
	  status = exec_it_and_wait(args);
	  if (status) exit(status);	  
	  break;

	case TYPE_SYMBOLIC_LINK_ABS:
	  if (symlink(source, dest) == -1) exit(1);
	  break;

	case TYPE_SYMBOLIC_LINK_REL:
	  if (symlink(absolute2relative(source, dest), dest) == -1) exit(1);
	  break;

	default:
	  break;
	}      
  }
  end_line_by_line_read();
  pchar_free(files, size);
  free(files);
  free(types);
  myexit(0);
}

