/* sfm -- Simple File Manager
   Copyright (C) 1998 Pixel (Pascal Rigaux)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */
#include "sfm_view.h"
#include "menu_pop.h"
#include "filetype.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <dirent.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkentry.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtklist.h>
#include <gtk/gtklistitem.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkselection.h>

static const guint32 forgetTime = 1000; /* in milli-seconds */
static const guint32 forgetTimeFast = 300; /* in milli-seconds */

static SfmView *static_view;

static void sfm_view_new_directory(SfmView *view, char *dir);
static void sfm_view_ask_action(SfmView *view);
static void sfm_view_action(SfmView *view);
static void sfm_view_special_action1(SfmView *view);
static void sfm_view_menu_popup(SfmView *view, guint button, guint32 time);
static void sfm_view_many_actions(SfmView *view);
static void sfm_view_selected_files_choose_action(SfmView *view);
static void sfm_view_active_file_choose_action(SfmView *view);
static void sfm_view_choose_action(SfmView *view, GtkSignalFunc f);
static void sfm_view_up_directory(SfmView *view);
static void sfm_view_set_text(SfmView *view);
static void sfm_view_update_directory(SfmView *view);
static void sfm_view_update_directory_keeping_selection(SfmView *view);
static void sfm_view_update_list(SfmView *view);
static void sfm_view_update_list_keeping_selection(SfmView *view);
static void sfm_view_select_all(SfmView *view);
static void sfm_view_delete(SfmView *view);
static void sfm_view_rename(SfmView *view);
static void sfm_view_copy(SfmView *view);
static void sfm_view_copy_add(SfmView *view);
static void sfm_view_copy_symbolic_link_abs(SfmView *view);
static void sfm_view_copy_symbolic_link_abs_add(SfmView *view);
static void sfm_view_copy_symbolic_link_rel(SfmView *view);
static void sfm_view_copy_symbolic_link_rel_add(SfmView *view);
static void sfm_view_cut(SfmView *view);
static void sfm_view_cut_add(SfmView *view);
static void sfm_view_paste(SfmView *view);

static void sfm_view_select_starting_with(SfmView *view, int c);
static void sfm_view_warn(SfmView *view, char *msg);
static void sfm_view_warn_with_timeout(SfmView *view, char *msg, int forget_time);
static void sfm_view_quit(SfmView *view);
static void sfm_view_quit_fast(SfmView *view);
static void sfm_view_create_directory(SfmView *view); 
static void sfm_view_create_file(SfmView *view); 
static void sfm_view_new_view(SfmView *view, int num); 


static void sfm_view_cancel_reset_starting_word(SfmView *view); 
static int sfm_view_reset_starting_word(SfmView *view); 
static char *get_active_name(SfmView *view); 
static char *get_active_file(SfmView *view); 
static int *get_selected_list(SfmView *view, int *rsize); 

static void when_there(GtkWidget *widget); 
static void sfm_view_ask_text(SfmView *view, char *msg1, char *msg2, GtkSignalFunc f); 
static void sfm_view_ask_really_do_it(SfmView *view, char *msg, GtkSignalFunc f); 
static void sfm_view_choose_an_action(SfmView *view, char **progs, int nb, GtkSignalFunc f); 

static void sfm_view_down_stack_dir_entries(SfmView *view);
static void sfm_view_up_stack_dir_entries(SfmView *view);
static void sfm_view_Xselection_handler(GtkWidget *widget, GtkSelectionData *selection_data, SfmView *view);
static void sfm_view_set_Xselection(SfmView *view, int time);


/* should be nb_ls_colors lines */ 
int colors_init[][3] = { 
  { 140, 140, 140 }, 
  { 170,   0,   0 }, 
  {   0, 170,   0 }, 
  { 180, 150,   0 }, 
  {   0,   0, 170 }, 
  { 170,   0, 170 }, 
  {   0, 170, 170 }, 
  {  60,  60,  60 }, 

  { 170, 170, 170 }, 
  { 255,   0,   0 }, 
  {   0, 200,   0 }, 
  { 230, 190,   0 }, 
  {   0,   0, 255 }, 
  { 255,   0, 255 }, 
  {   0, 200, 200 }, 
  {   0,   0,   0 }, 

  {   0,  60,  60 }, 
  {  60,  60,   0 }, 
  {  60,   0,  60 }, 
  {   0,   0, 255 }, 
  {   0, 255,   0 }, 
  { 255,   0,   0 }, 
  {   0,  60, 170 }, 
  {  60,   0, 170 }, 
  { 170,  60,   0 }, 
  { 170,   0,  60 }, 
  {   0, 170,  60 }, 
  {  60, 170,   0 }, 
}; 

static void sfm_view_invert_selection(SfmView *view) { text_list_invert_selection(view->list); } 
static void sfm_view_dir_entry(SfmView *view) { gtk_widget_grab_focus(view->widgets.dir); } 
static void sfm_view_new_view_1(SfmView *view) { sfm_view_new_view(view, 1); } 
static void sfm_view_new_view_2(SfmView *view) { sfm_view_new_view(view, 2); } 
static void sfm_view_new_view_3(SfmView *view) { sfm_view_new_view(view, 3); } 
static void sfm_view_new_view_4(SfmView *view) { sfm_view_new_view(view, 4); } 
static void sfm_view_toggle_sort_type(SfmView *view) { dir_list_toggle_sort_type(); sfm_view_update_directory(view); } 
static void sfm_view_kill_actions(SfmView *view) { kill_children_left(); }



static menuitem menu_popup_items[] = { 
  { "enter directory",			"Return, Right",	sfm_view_action }, 
  { "up directory",			"BackSpace, Left",	sfm_view_up_directory }, 
  { "rename",				"F2, Ctl+R",		sfm_view_rename }, 
  { "delete",				"Delete",		sfm_view_delete }, 
  { "action",				"Return, Right",	sfm_view_action }, 
  { "quit",				"Ctl+Q, Alt+Q",		sfm_view_quit }, 
  { "",					"",			NULL}, 
  { "copy",				"Ctl+C",		sfm_view_copy }, 
  { "cut",				"Ctl+X",		sfm_view_cut }, 
  { "copy as symlink (rel)",		"Ctl+S",		sfm_view_copy_symbolic_link_rel }, 
  { "copy as symlink (abs)",		"",			sfm_view_copy_symbolic_link_abs }, 
  { "paste",				"Ctl+V",		sfm_view_paste }, 
}; 

static menuitem menu_popup_extended_items[] = { 
  { "all",				"Ctl+A",		sfm_view_select_all }, 
  { "invert",				"Ctl+I",		sfm_view_invert_selection }, 
  { "add to copy",			"Shft+Ctl+C",		sfm_view_copy_add }, 
  { "add to cut",			"Shft+Ctl+X",		sfm_view_cut_add }, 
  { "add to copy as symlink (rel)",	"Shft+Ctl+S",		sfm_view_copy_symbolic_link_rel_add }, 
  { "add to copy as symlink (abs)",	"",			sfm_view_copy_symbolic_link_abs_add }, 
  { "",					"",			NULL}, 
  { "toggle sort type",			"Ctl+T",		sfm_view_toggle_sort_type}, 
  { "",					"",			NULL},
  { "kill actions",			"Ctl+K",		sfm_view_kill_actions },
  { "choose action",			"Shft+Return",		sfm_view_active_file_choose_action }, 
  { "many actions",			"Ctl+Return",		sfm_view_many_actions }, 
  { "precise the action",		"Shft+Ctl+Return",	sfm_view_ask_action }, 
  { "many actions 2",			"Ctl+Alt+Return",	sfm_view_selected_files_choose_action }, 
  { "",					"",			NULL}, 
  { "view #1",				"Ctl+1, Alt+1",		sfm_view_new_view_1 }, 
  { "view #2",				"Ctl+2, Alt+2",		sfm_view_new_view_2 }, 
  { "view #3",				"Ctl+3, Alt+3",		sfm_view_new_view_3 }, 
  { "view #4",				"Ctl+4, Alt+4",		sfm_view_new_view_4 }, 
  { "",					"",			NULL}, 
  { "fast quit",			"Shft+Ctl+Q",		sfm_view_quit_fast }, 
  { "focus dir entry",			"Tab, Ctl+D",		sfm_view_dir_entry }, 
  { "update",				"F5, Ctl+L",		sfm_view_update_directory }, 
}; 

extern void sfm_view_do_it(an_action f) { f(static_view); } 



static int sfm_view_key_press_global(GtkWidget *widget, GdkEventKey *event, SfmView *view) 
{ 
  gboolean b = TRUE; 

  switch (event->keyval)  
    { 
    case '1': 
    case '2': 
    case '3': 
    case '4': 
      if (event->state & GDK_CONTROL_MASK || event->state & GDK_MOD1_MASK) 
	sfm_view_new_view(view, event->keyval - '0'); 
      break; 

    case 'n': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_create_directory(view); 
      break; 
     
    case 'N': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_create_file(view); 
      break; 
     
    case 'q': 
      if (event->state & (GDK_CONTROL_MASK | GDK_MOD1_MASK)) 
	sfm_view_quit(view); 
      break;       

    case 'Q': 
      if (event->state & (GDK_CONTROL_MASK | GDK_MOD1_MASK))
	sfm_view_quit_fast(view); 
      break;       

    default:  
      b = FALSE; 
    } 
  return b; 
} 

static int sfm_view_key_press(GtkWidget *widget, GdkEventKey *event, SfmView *view) 
{ 
  gboolean b = TRUE; 
  GtkObject *obj = GTK_OBJECT(view->widgets.list); 

  switch (event->keyval)  
    { 
    case GDK_Shift_L: 
    case GDK_Shift_R: 
    case GDK_Mode_switch: 
      return FALSE; 
    default: 
      sfm_view_cancel_reset_starting_word(view); 
    } 

  switch (event->keyval)  
    { 
    case GDK_Tab: 
      sfm_view_dir_entry(view); 
      break; 

    case GDK_BackSpace: 
      sfm_view_up_directory(view); 
      break; 
   
    case GDK_Delete: 
	sfm_view_delete(view); 
      break; 

    case GDK_Return: 
      if (event->state & GDK_MOD1_MASK && event->state & GDK_CONTROL_MASK) 
	sfm_view_selected_files_choose_action(view); 
      else if (event->state & GDK_CONTROL_MASK && event->state & GDK_SHIFT_MASK) 
	sfm_view_ask_action(view); 
      else if (event->state & GDK_CONTROL_MASK) 
	sfm_view_many_actions(view); 
      else if (event->state & GDK_SHIFT_MASK) 
	sfm_view_active_file_choose_action(view); 
      else b = FALSE; 
      break; 

    case GDK_F2: 
      sfm_view_rename(view); 
      break; 

    case GDK_F5: 
      sfm_view_update_directory_keeping_selection(view); 
      sfm_view_warn(view, "done updating directory"); 
      break; 

    case GDK_F11:
      sfm_view_special_action1(view);
      break;

    case GDK_Left: 
      sfm_view_up_directory(view); 
      break; 

    case GDK_Right: 
      gtk_signal_emit_by_name(obj, "action"); 
      break; 

    case 'a': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_select_all(view); 
      break; 

    case 'c': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_copy(view); 
      break; 

    case 'C': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_copy_add(view); 
      break; 

    case 'd': 
    case 'D': 
      if (event->state & (GDK_CONTROL_MASK | GDK_MOD1_MASK)) 
	sfm_view_dir_entry(view); 
      break; 

    case 'k': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_kill_actions(view); 
      break; 

    case 'i': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_invert_selection(view); 
      break; 

    case 'l': 
      if (event->state & GDK_CONTROL_MASK) { 
	sfm_view_update_directory_keeping_selection(view); 
	sfm_view_warn(view, "done updating directory"); 
      } 
      break; 

    case 'r': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_rename(view); 
      break; 

    case 's': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_copy_symbolic_link_rel(view); 
      break; 

    case 'S': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_copy_symbolic_link_rel_add(view); 
      break; 

    case 't': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_toggle_sort_type(view); 
      break; 

    case 'v': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_paste(view); 
      break; 

    case 'x': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_cut(view); 
      break; 

    case 'X': 
      if (event->state & GDK_CONTROL_MASK) 
	sfm_view_cut_add(view); 
      break; 

    case 'w':
      if (event->state & (GDK_CONTROL_MASK | GDK_MOD1_MASK))
	sfm_view_set_Xselection(view, event->time);
      break; 

    default:  
      b = FALSE; 
    } 
  if (event->keyval < 0x100 && (event->state == 0 || event->state == GDK_SHIFT_MASK)) 
    sfm_view_select_starting_with(view, event->keyval); 
  else 
    sfm_view_reset_starting_word(view); 

  if (b) gtk_signal_emit_stop_by_name(obj, "key_press_event"); 
  return b;   
} 

static int sfm_view_dir_entry_key_press(GtkWidget *widget, GdkEventKey *event, SfmView *view) 
{ 
  gboolean b = TRUE; 
  GtkObject *obj = GTK_OBJECT(view->widgets.dir); 

  switch (event->keyval)  
    { 
    case GDK_Down: 
      sfm_view_down_stack_dir_entries(view); 
      break; 

    case GDK_Up: 
      sfm_view_up_stack_dir_entries(view); 
      break; 

    case GDK_Return:
      sfm_view_new_directory(view, gtk_entry_get_text(GTK_ENTRY(view->widgets.dir)));
      break;

    default:  
      b = FALSE; 
    } 

  if (b) gtk_signal_emit_stop_by_name(obj, "key_press_event"); 
  return b;   
} 

static void force_focus(GtkWidget *window) 
{ 
  XRaiseWindow(GDK_DISPLAY(), GDK_WINDOW_XWINDOW(window->window)); 
  XSetInputFocus(GDK_DISPLAY(), GDK_WINDOW_XWINDOW(window->window), RevertToPointerRoot, CurrentTime); 
} 
static int received_force_focus(SfmView *view) { force_focus(view->widgets.window); return FALSE; } 

static void sfm_view_set_dir_entry(SfmView *view, char *dir_entry) 
{
  gtk_entry_set_text(GTK_ENTRY(view->widgets.dir), dir_entry); 
  gtk_signal_emit_by_name(GTK_OBJECT(view->widgets.dir), "changed"); 
  gtk_widget_draw(GTK_WIDGET(view->widgets.dir), NULL); 
}

static void sfm_view_update_dir_entry(SfmView *view) 
{ 
  sfm_view_set_dir_entry(view, sfm_doc_set_dir_entry(view->doc));
} 

static void quit() 
{ 
  sfm_view_quit(static_view); 
} 
static void received_sighup(int signal) 
{ 
  /* received when another view has ended, the server gives this signal to another view */ 
  gtk_timeout_add(10, (GtkFunction) received_force_focus, static_view); 
} 

static int update_directory(SfmView *view) { 
  sfm_view_update_directory_keeping_selection(view); 
  return FALSE; 
} 
static void received_sigusr2(int signal) 
{ 
  gtk_timeout_add(10, (GtkFunction) update_directory, static_view); 
} 

static void label_set(SfmView *view, char *p) 
{ 
  FREE_NULL(view->statusbar); 
  view->statusbar = p; 
  gtk_label_set(GTK_LABEL(view->widgets.statusbar), p); 
} 
static int sfm_view_cancel_warn(SfmView *view) 
{ 
  if (view) label_set(view, strdup("")); 
  return FALSE; 
} 

static int real_view_warn(char *msg) { sfm_view_warn(static_view, msg); free(msg); return FALSE; } 
extern void view_warn(char *msg) 
{ 
  gtk_timeout_add(10, (GtkFunction) real_view_warn, msg); 
} 

static void sfm_view_warn(SfmView *view, char *msg) 
{
  sfm_view_warn_with_timeout(view, msg, 2 * forgetTime);
}

static void sfm_view_warn_with_timeout(SfmView *view, char *msg, int forget_time) 
{ 
  if (view) { 
    if (view->statusbarTimeoutId) gtk_timeout_remove(view->statusbarTimeoutId); 
    view->statusbarTimeoutId = gtk_timeout_add(forget_time, (GtkFunction) sfm_view_cancel_warn, view); 
    label_set(view, strdup(msg)); 
  } 
} 


static void real_rename(SfmView *view, char *new_name) 
{ 
  boolean b = sfm_doc_rename(view->doc, get_active_name(view), new_name); 
     
  sfm_view_warn(view, b ? "done renaming" : "error renaming"); 
  if (b) {
    g_list_free (view->list->selection); 
    view->list->selection = NULL; 
    sfm_doc_directory_has_changed(view->doc); 
    text_list_set_active_row(view->list, sfm_doc_find_name(view->doc, new_name)); 
    update_views(); 
  }   
} 


static gint sfm_view_rename_event_handler(GtkWidget *widget, GdkEvent *event, SfmView *view) 
{ 
  boolean b = false; 

  switch (event->type) { 
  case GDK_FOCUS_CHANGE:  
    b = true; 
    break; 

  case GDK_BUTTON_PRESS: 
  case GDK_2BUTTON_PRESS: 
  case GDK_3BUTTON_PRESS: 
    b = event->button.window != GTK_ENTRY(view->list->child)->text_area; 
    break; 

  case GDK_KEY_PRESS: 
    switch (event->key.keyval) { 
    case 'y':
      gtk_selection_convert(GTK_WIDGET(view->list->child), GDK_SELECTION_PRIMARY, gdk_atom_intern ("COMPOUND_TEXT", FALSE), ((GdkEventKey*) event)->time);
      break;
    case GDK_Return: 
      real_rename(view, gtk_entry_get_text(GTK_ENTRY(view->list->child))); 
      /* nobreak */       

    case GDK_Down: 
    case GDK_Up: 
    case GDK_Tab: 
    case GDK_Escape: 
      b = true; 
      break; 
   
    default: 
    } 
 
  default: 
  } 


  if (b) { 
    gtk_signal_disconnect(GTK_OBJECT(view->widgets.window), view->event_signal_handler); 
    GTK_WIDGET_SET_FLAGS(view->widgets.list, GTK_CAN_FOCUS); 
    gtk_container_remove(GTK_CONTAINER(view->list), view->list->child); 
    gtk_widget_grab_focus(view->widgets.list); 
  } 
  return b; 
} 

static void action(TextList *tlist, SfmView *view) { sfm_view_action(view); } 

extern SfmView *sfm_view_new(int number, int *argc, char ***argv) 
{ 
  SfmView *view = malloc(sizeof(SfmView)); 
  GdkColormap *colormap; 
  char title[sizeof(viewTitle)]; 
  int i; 

  gtk_init(argc, argv); 

  view->doc = sfm_doc_new(number, argc, argv); 
  view->Xselection = NULL;
  view->statusbar = NULL; 
  view->statusbarTimeoutId = 0; 

  view->widgets.window = gtk_window_new(GTK_WINDOW_TOPLEVEL); 
  gtk_widget_realize(view->widgets.window); 
  gdk_window_resize(view->widgets.window->window, 190, 250); 
  gtk_signal_connect(GTK_OBJECT(view->widgets.window), "destroy", GTK_SIGNAL_FUNC(quit), NULL);   
  gtk_signal_connect(GTK_OBJECT(view->widgets.window), "delete_event", GTK_SIGNAL_FUNC(quit), NULL);   

  view->widgets.frame = gtk_vbox_new(FALSE, 0); 
  { 
    gtk_container_add(GTK_CONTAINER(view->widgets.window), view->widgets.frame); 

    view->widgets.framedir = gtk_hbox_new(FALSE, 0); 
    { 
      gtk_box_pack_start(GTK_BOX(view->widgets.frame), view->widgets.framedir, FALSE, FALSE, 0); 

      view->widgets.labeldir = gtk_label_new(" Dir "); 
      gtk_box_pack_start(GTK_BOX(view->widgets.framedir), view->widgets.labeldir, FALSE, FALSE, 0); 
      gtk_widget_show(view->widgets.labeldir); 

      view->widgets.dir = gtk_entry_new(); 
      gtk_signal_connect(GTK_OBJECT(view->widgets.dir), "key_press_event", (GtkSignalFunc) sfm_view_dir_entry_key_press, view); 
      gtk_container_add(GTK_CONTAINER(view->widgets.framedir), view->widgets.dir); 
      gtk_widget_show(view->widgets.dir); 

    } gtk_widget_show(view->widgets.framedir); 

    view->widgets.list = text_list_new(); 
    { 
      view->list = TEXTLIST(view->widgets.list); 
      gtk_container_add(GTK_CONTAINER(view->widgets.frame), view->widgets.list); 
      gtk_signal_connect(GTK_OBJECT(view->widgets.list), "key_press_event", (GtkSignalFunc) sfm_view_key_press, view); 
      gtk_signal_connect_after(GTK_OBJECT(view->widgets.window), "key_press_event", (GtkSignalFunc) sfm_view_key_press_global, view); 
      gtk_signal_connect(GTK_OBJECT(view->widgets.list), "action", (GtkSignalFunc) action, view); 
      gtk_signal_connect(GTK_OBJECT(view->widgets.list), "menu_popup", (GtkSignalFunc) sfm_view_menu_popup, view); 
      /*gtk_selection_add_handler (GTK_WIDGET(view->widgets.list), GDK_SELECTION_PRIMARY, gdk_atom_intern("COMPOUND_TEXT", FALSE), (GtkSelectionFunction) sfm_view_Xselection_handler, view);*/

      colormap = gtk_widget_get_colormap(view->widgets.window); 
      for (i = 0; i < nb_ls_colors; i++) { 
	view->colors[i].red   = colors_init[i][0] << 8; 
	view->colors[i].green = colors_init[i][1] << 8; 
	view->colors[i].blue  = colors_init[i][2] << 8; 
	gdk_color_alloc(colormap, &view->colors[i]); 
      } 
      sfm_view_update_list(view); 

      view->widgets.statusbar = gtk_label_new(""); 
      gtk_box_pack_start(GTK_BOX(view->widgets.frame), view->widgets.statusbar, FALSE, FALSE, 2); 
      gtk_widget_show(view->widgets.statusbar); 

      sfm_view_warn(view, "sfm, pleased to meet you"); 

    } gtk_widget_grab_focus(view->widgets.list); 
 
  } gtk_widget_show(view->widgets.frame); 

   
  sprintf(title, viewTitle, view->doc->number); 
  gtk_window_set_title(GTK_WINDOW(view->widgets.window), title); 
  gtk_widget_show(view->widgets.window); 
   

  install_signal_handler(SIGCHLD, received_sigchild); 
  install_signal_handler(SIGHUP, received_sighup); 
  install_signal_handler(SIGUSR2, received_sigusr2); 

  view->startingWord = strdup(""); 
  view->startingWordTimeoutId = -1; 
  static_view = view; 
  return view; 
} 

extern void sfm_view_free(SfmView *view) 
{ 
  sfm_doc_free(view->doc); 
  text_list_clear(view->list); 
  FREE_NULL(view->Xselection); 
  FREE_NULL(view->startingWord); 
  FREE_NULL(view->statusbar); 
  free(view); 
} 

static void sfm_view_new_directory(SfmView *view, char *dir) 
{ 
  sfm_doc_new_directory(view->doc, dir); 
  view->list->active_row = 0; 
  sfm_view_update_list(view); 
} 

static void sfm_view_that_action(GtkWidget *widget, SfmView *view) 
{ 
  char *prog = strdup(gtk_entry_get_text(GTK_ENTRY(view->widgets.entry))); 
  sfm_view_warn(view, "launching"); 
  gtk_widget_destroy(view->widgets.dialog); 
  sfm_doc_that_action(view->doc, strdup(get_active_name(view)), prog); 
  free(prog); 
} 
static void sfm_view_ask_action(SfmView *view) 
{ 
  char *file, *type, *msg; 
  file = get_active_file(view); 
  type = fileType(file); 
  msg = strconcat3("(type: ", type, ")"); 
  free(file); 
  free(type); 
  sfm_view_ask_text(view, "what program do you want to use ?", msg, (GtkSignalFunc) sfm_view_that_action); 
  free(msg); 
} 

static void sfm_view_menu_popup(SfmView *view, guint button, guint32 time) 
{ 
  GtkWidget *menu, *extended; 

  menu = create_menu(view, menu_popup_items, sizeof(menu_popup_items) / sizeof(menuitem)); 
  extended = create_menu(view, menu_popup_extended_items, sizeof(menu_popup_extended_items) / sizeof(menuitem)); 
  add_submenu("more", menu, extended); 
  gtk_menu_popup (GTK_MENU(menu), NULL, NULL, NULL, NULL, button, time); 
} 

static void sfm_view_action(SfmView *view) 
{ 
  char *file; 

  if (text_list_is_empty(view->list)) return; 

  file = get_active_file(view); 
  if (lastchar(file) == '/') chop(file); 

  if (access(file, R_OK) == 0) { 
    if (is_directory(file)) { 
      if (access(file, X_OK) == 0) { 
	*strend(file) = '/'; 
	sfm_view_new_directory(view, file); 
      } 
       else sfm_view_warn(view, "error opening directory"); 
     } else { 
       free(file); 
       file = strdup(get_active_name(view)); 
       if (!sfm_doc_action(view->doc, file)) 
 	sfm_view_ask_action(view); 
     } 
   } else sfm_view_warn(view, "error no read permission"); 
   free(file); 
 } 

 static void sfm_view_special_action1(SfmView *view)
 {
   char *file = strdup(get_active_name(view)); 
   launch_action(&file, 1, "xterm -e less"); 
   free(file); 
 }

 static void sfm_view_many_actions(SfmView *view) 
 { 
   int *tab, size; 

   if (text_list_is_empty(view->list)) return; 

   if ((tab = get_selected_list(view, &size))) { 
     sfm_doc_many_actions(view->doc, tab, size); 
     free(tab); 
   } 
 } 


 static void choosen_action(GtkWidget *widget, SfmView *view) 
 { 
   if (GTK_LIST(widget)->selection) { 
     char *prog = strdup(GTK_LABEL(GTK_BIN(GTK_LIST(widget)->selection->data)->child)->label);     
     char *file = strdup(get_active_name(view)); 
     sfm_view_warn(view, "launching"); 
     gtk_widget_destroy(view->widgets.dialog); 
     printf("\n"); /* for actions like ``ls -l'', acts like a separator */
     launch_action(&file, 1, prog); 
     free(file); 
     free(prog); 
   } 
 } 

 static void choosen_action_selected_files(GtkWidget *widget, SfmView *view) 
 { 
   int *tab, size; 

   if (GTK_LIST(widget)->selection) { 
     char *prog = strdup(GTK_LABEL(GTK_BIN(GTK_LIST(widget)->selection->data)->child)->label);     
     sfm_view_warn(view, "launching"); 
     gtk_widget_destroy(view->widgets.dialog); 

     if ((tab = get_selected_list(view, &size))) { 
       sfm_doc_action_many_files(view->doc, tab, size, prog); 
       free(tab); 
     } 
     free(prog); 
   } 
 } 

 static void sfm_view_selected_files_choose_action(SfmView *view) 
 { 
   if (view->list->selection) sfm_view_choose_action(view, GTK_SIGNAL_FUNC(choosen_action_selected_files)); 
 } 

 static void sfm_view_active_file_choose_action(SfmView *view) 
 { 
   sfm_view_choose_action(view, GTK_SIGNAL_FUNC(choosen_action));
}

static void sfm_view_choose_action(SfmView *view, GtkSignalFunc f)
{
  char **progs, *file;
  int nb;

  file = get_active_file(view);
  progs = sfm_doc_choose_action(view->doc, file, &nb);
  free(file);
  if (progs == NULL) return sfm_view_ask_action(view);

  sfm_view_choose_an_action(view, progs, nb, f);
}

static void sfm_view_choose_an_action(SfmView *view, char **progs, int nb, GtkSignalFunc f)
{
  GtkWidget *cancel, *hbox, *vbox, *label, *gtklist, *item;
  GList *dlist;
  int i;
  
  view->widgets.dialog = gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_signal_connect(GTK_OBJECT(view->widgets.dialog), "delete_event", GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);  
  gtk_signal_connect(GTK_OBJECT(view->widgets.dialog), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);  
  gtk_signal_connect_after(GTK_OBJECT(view->widgets.dialog), "expose_event", GTK_SIGNAL_FUNC(when_there), NULL);  

  vbox = gtk_vbox_new(FALSE, 2);
  {
    gtk_container_add (GTK_CONTAINER(view->widgets.dialog), vbox);

    label = gtk_label_new("Which program do you want to use");
    gtk_container_add (GTK_CONTAINER(vbox), label);
    gtk_widget_show(label);

    gtklist = gtk_list_new();
    gtk_container_add(GTK_CONTAINER(vbox), gtklist);
    dlist=NULL;
    for (dlist = NULL, i = nb - 1; i >= 0; i--) {
      item = gtk_list_item_new_with_label(progs[i]);
      dlist = g_list_prepend(dlist, item);
      gtk_widget_show(item);
    }
    gtk_list_append_items(GTK_LIST(gtklist), dlist);
    gtk_widget_show(gtklist);

    gtk_signal_connect(GTK_OBJECT(gtklist), "selection_changed", f, view);


    gtk_widget_grab_focus(item);
    

    hbox = gtk_hbox_new(FALSE, 5);
    {
      gtk_container_add (GTK_CONTAINER(vbox), hbox);

      cancel = gtk_button_new_with_label ("Cancel");
      gtk_container_add (GTK_CONTAINER(hbox), cancel);
      gtk_signal_connect_object(GTK_OBJECT(cancel), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(view->widgets.dialog));
      gtk_widget_show(cancel);

    } gtk_widget_show(hbox);
  } gtk_widget_show(vbox);
  gtk_grab_add(view->widgets.dialog);
  gtk_widget_show(view->widgets.dialog);  

  free(progs);
}

static void sfm_view_up_directory(SfmView *view)
{
  if (sfm_doc_up_directory(view->doc)) {
    view->list->active_row = 0;    
    sfm_view_update_directory(view);
  }
}

static void sfm_view_update_directory(SfmView *view)
{
  sfm_doc_directory_has_changed(view->doc);
  sfm_view_update_list(view);
}

static void sfm_view_update_directory_keeping_selection(SfmView *view)
{
  char *old_active = strdup(get_active_name(view));
  sfm_doc_directory_has_changed(view->doc);
  if (view->list->active_row >= view->doc->dir_list.size || !streq(old_active, get_active_name(view))) {
    int row = sfm_doc_find_name(view->doc, old_active);
    if (row >= 0) text_list_set_active_row(view->list, row); 
    g_list_free (view->list->selection); 
    view->list->selection = NULL; 
  }
  free(old_active);
  sfm_view_update_list_keeping_selection(view);
}

static void sfm_view_set_text(SfmView *view)
{
  int i;

  text_list_set_size(view->list, view->doc->dir_list.size);

  for (i = 0; i < view->doc->dir_list.size; i++)
    {
      int file_type = sfm_doc_get_color_of_file(view->doc, i);
      text_list_set_text(view->list, i, view->doc->dir_list.elements[i].name);
      if (file_type < nb_ls_colors)
	text_list_set_foreground(view->list, i, &view->colors[file_type]);
    }
}

static void sfm_view_update_list(SfmView *view)
{
  g_list_free (view->list->selection);
  view->list->selection = NULL;
  sfm_view_update_list_keeping_selection(view);
}

static void sfm_view_update_list_keeping_selection(SfmView *view)
{
  sfm_view_set_text(view);
  text_list_first_display(view->list);
  sfm_view_update_dir_entry(view);
  gtk_widget_show(view->widgets.list);
}

static int sfm_view_reset_starting_word(SfmView *view)
{
  FREE_NULL(view->startingWord);
  view->startingWord = strdup("");
  return FALSE;
}
static void sfm_view_cancel_reset_starting_word(SfmView *view)
{
  if (view->startingWordTimeoutId != -1)
    gtk_timeout_remove(view->startingWordTimeoutId);
  view->startingWordTimeoutId = -1;
}
static void sfm_view_schedule_reset_starting_word(SfmView *view)
{
  sfm_view_cancel_reset_starting_word(view);
  view->startingWordTimeoutId = gtk_timeout_add(forgetTime, (GtkFunction) sfm_view_reset_starting_word, view);
}

static void sfm_view_select_starting_with(SfmView *view, int c)
{
  int startIndice = view->list->active_row;
  char *start = char_malloc(2);
  int i;
  start[0] = c; start[1] = '\0';
  
  if (strlen(view->startingWord) == 0 || streq(view->startingWord, start)) { 
    startIndice++; 
  } else {
    char *p = start;
    start = strconcat(view->startingWord, start);
    free(p);
  }
  
  if ((i = sfm_doc_search_starts_with(view->doc, start, startIndice)) >= 0) {
    text_list_set_active_row_and_show(view->list, i, 0.5);
  } else {
    char_replace(&start, strdup(""));
  }
  char_replace(&view->startingWord, start);
  
  sfm_view_schedule_reset_starting_word(view);
}

static void sfm_view_select_all(SfmView *view)
{
  text_list_select_all(view->list);
}

static void sfm_view_real_delete(GtkWidget *widget, SfmView *view)
{
  int *tab, size;

  gtk_widget_destroy(view->widgets.dialog);

  if ((tab = get_selected_list(view, &size))) {
    sfm_doc_delete(view->doc, tab, size);
    free(tab);
  }
}

static void when_there(GtkWidget *widget)
{
  static int i = 1;
  if (i) { i = 0; force_focus(widget); }
}
static void sfm_view_ask_really_do_it(SfmView *view, char *msg, GtkSignalFunc f)
{
  GtkWidget *ok, *cancel, *hbox, *vbox, *label;

  view->widgets.dialog = gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_signal_connect(GTK_OBJECT(view->widgets.dialog), "delete_event", GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);  
  gtk_signal_connect(GTK_OBJECT(view->widgets.dialog), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);  
  gtk_signal_connect_after(GTK_OBJECT(view->widgets.dialog), "expose_event", GTK_SIGNAL_FUNC(when_there), NULL);  

  vbox = gtk_vbox_new(FALSE, 2);
  {
    gtk_container_add (GTK_CONTAINER(view->widgets.dialog), vbox);

    label = gtk_label_new(msg);
    gtk_container_add (GTK_CONTAINER(vbox), label);
    gtk_widget_show(label);

    hbox = gtk_hbox_new(FALSE, 5);
    {
      gtk_container_add (GTK_CONTAINER(vbox), hbox);

      ok = gtk_button_new_with_label ("Ok");
      cancel = gtk_button_new_with_label ("Cancel");
      gtk_container_add (GTK_CONTAINER(hbox), ok);
      gtk_container_add (GTK_CONTAINER(hbox), cancel);
      gtk_signal_connect(GTK_OBJECT(ok), "clicked", f, view);
      gtk_signal_connect_object(GTK_OBJECT(cancel), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(view->widgets.dialog));
      gtk_widget_show(ok);
      gtk_widget_show(cancel);

      gtk_widget_grab_focus(ok);

    } gtk_widget_show(hbox);
  } gtk_widget_show(vbox);
  gtk_grab_add(view->widgets.dialog);
  gtk_widget_show(view->widgets.dialog);
}

static void sfm_view_ask_text(SfmView *view, char *msg1, char *msg2, GtkSignalFunc f)
{
  GtkWidget *ok, *cancel, *hbox, *vbox, *label;

  view->widgets.dialog = gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_signal_connect(GTK_OBJECT(view->widgets.dialog), "delete_event", GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);  
  gtk_signal_connect(GTK_OBJECT(view->widgets.dialog), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroy), NULL);  
  gtk_signal_connect_after(GTK_OBJECT(view->widgets.dialog), "expose_event", GTK_SIGNAL_FUNC(when_there), NULL);  

  vbox = gtk_vbox_new(FALSE, 2);
  {
    gtk_container_add (GTK_CONTAINER(view->widgets.dialog), vbox);

    label = gtk_label_new(msg1);
    gtk_container_add (GTK_CONTAINER(vbox), label);
    gtk_widget_show(label);

    if (msg2) {
      label = gtk_label_new(msg2);
      gtk_container_add (GTK_CONTAINER(vbox), label);
      gtk_widget_show(label);
    }

    view->widgets.entry = gtk_entry_new();
    gtk_container_add(GTK_CONTAINER(vbox), view->widgets.entry);
    gtk_signal_connect(GTK_OBJECT(view->widgets.entry), "activate", f, view);
    gtk_widget_show(view->widgets.entry);
    gtk_widget_grab_focus(view->widgets.entry);

    hbox = gtk_hbox_new(FALSE, 5);
    {
      gtk_container_add (GTK_CONTAINER(vbox), hbox);

      ok = gtk_button_new_with_label ("Ok");
      cancel = gtk_button_new_with_label ("Cancel");
      gtk_container_add (GTK_CONTAINER(hbox), ok);
      gtk_container_add (GTK_CONTAINER(hbox), cancel);
      gtk_signal_connect(GTK_OBJECT(ok), "clicked", f, view);
      gtk_signal_connect_object(GTK_OBJECT(cancel), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(view->widgets.dialog));
      gtk_widget_show(ok);
      gtk_widget_show(cancel);

    } gtk_widget_show(hbox);
  } gtk_widget_show(vbox);
  gtk_grab_add(view->widgets.dialog);
  gtk_widget_show(view->widgets.dialog);
}

static void sfm_view_delete(SfmView *view)
{
  if (g_list_length(view->list->selection) > 0) sfm_view_ask_really_do_it(view, " Really delete files ? ", GTK_SIGNAL_FUNC(sfm_view_real_delete));
}

static void sfm_view_rename(SfmView *view)
{
  GtkWidget *entry;
  gint x, y, height, width;
  
  if (view->list->rows == 0) return;
  sfm_view_warn(view, "renaming");

  text_list_get_coords(view->list, view->list->active_row, 0, &x, &y);
  width = view->list->tlist_window_width + 1;
  height = view->list->row_height + 6;

  entry = gtk_entry_new();
  gtk_container_add(GTK_CONTAINER(view->widgets.list), entry);

  gtk_widget_set_uposition(entry, 1, y - 1);
  gtk_widget_set_usize (entry, width, height);

  gtk_entry_set_text(GTK_ENTRY(entry), get_active_name(view));

  view->event_signal_handler = gtk_signal_connect_after(GTK_OBJECT(view->widgets.window), "event", (GtkSignalFunc) sfm_view_rename_event_handler, view);
  GTK_WIDGET_UNSET_FLAGS(view->widgets.list, GTK_CAN_FOCUS);
  gtk_widget_grab_focus(entry);

  gtk_widget_show(entry);
}

static void sfm_view_copy_or_cut(SfmView *view, TypePaste type_paste, TypeAddOrOverride add_or_override)
{
  int *tab, size;

  sfm_view_warn(view, (add_or_override == TYPE_OVERRIDE ? "set" : "added"));
  if ((tab = get_selected_list(view, &size))) {
    sfm_doc_copy_or_cut(view->doc, type_paste, add_or_override, tab, size);
    free(tab);
  }
}
static void sfm_view_copy(SfmView *view) { sfm_view_copy_or_cut(view, TYPE_COPY, TYPE_OVERRIDE); }
static void sfm_view_copy_add(SfmView *view) { sfm_view_copy_or_cut(view, TYPE_COPY, TYPE_ADD); }
static void sfm_view_copy_symbolic_link_abs(SfmView *view) { sfm_view_copy_or_cut(view, TYPE_SYMBOLIC_LINK_ABS, TYPE_OVERRIDE); }
static void sfm_view_copy_symbolic_link_abs_add(SfmView *view) { sfm_view_copy_or_cut(view, TYPE_SYMBOLIC_LINK_ABS, TYPE_ADD); }
static void sfm_view_copy_symbolic_link_rel(SfmView *view) { sfm_view_copy_or_cut(view, TYPE_SYMBOLIC_LINK_REL, TYPE_OVERRIDE); }
static void sfm_view_copy_symbolic_link_rel_add(SfmView *view) { sfm_view_copy_or_cut(view, TYPE_SYMBOLIC_LINK_REL, TYPE_ADD); }
static void sfm_view_cut(SfmView *view) { sfm_view_copy_or_cut(view, TYPE_CUT, TYPE_OVERRIDE); }
static void sfm_view_cut_add(SfmView *view) { sfm_view_copy_or_cut(view, TYPE_CUT, TYPE_ADD); }

static void sfm_view_paste(SfmView *view)
{
  sfm_doc_paste(view->doc);
}

static void sfm_view_quit(SfmView *view)
{
  if (any_child_left()) sfm_view_warn(view, "an action is in progress");
  else sfm_view_quit_fast(view);
}

static void sfm_view_quit_fast(SfmView *view)
{
  sfm_doc_quit(view->doc);
  gtk_main_quit();
}

static void sfm_view_create_directory(SfmView *view)
{
  sfm_view_warn(view, sfm_doc_create_directory(view->doc) ? "directory created" : "error creating directory");
  update_views();
}

static void sfm_view_create_file(SfmView *view)
{
  sfm_view_warn(view, sfm_doc_create_file(view->doc) ? "file created" : "error creating file");
  update_views();
}

static void sfm_view_new_view(SfmView *view, int num)
{
  if (!sfm_doc_new_view(view->doc, num)) sfm_view_warn(view,  "error creating new view");  
}

static char *get_active_name(SfmView *view)
{
  return view->doc->dir_list.elements[view->list->active_row].name;
}

static char *get_active_file(SfmView *view)
{
  return strconcat(view->doc->dir_prefix, get_active_name(view));
}


static int *get_selected_list(SfmView *view, int *rsize)
{
  GList *l;
  int size, i, *tab;

  l = view->list->selection;
  size = g_list_length(l);
  if (rsize) *rsize = size;
  if (size == 0) return NULL;

  tab = int_malloc(size);
  for (i = 0; i < size; l = l->next, i++) tab[i] = (int) l->data;

  return tab;
}

static void up_down_stack_dir_entries(SfmView *view, char *dir_entry)
{
  if (dir_entry) sfm_view_set_dir_entry(view, dir_entry);
  else sfm_view_warn_with_timeout(view, "history finishes here", forgetTimeFast);
}
static void sfm_view_up_stack_dir_entries(SfmView *view)
{
  up_down_stack_dir_entries(view, sfm_doc_up_stack_dir_entries(view->doc));
}
static void sfm_view_down_stack_dir_entries(SfmView *view)
{
  up_down_stack_dir_entries(view, sfm_doc_down_stack_dir_entries(view->doc));
}

static void sfm_view_Xselection_handler(GtkWidget *widget, GtkSelectionData *selection_data, SfmView *view)
{
  if (view->Xselection) {
    guchar *text;
    GdkAtom encoding;
    gint format, new_length;
  
    gdk_string_to_compound_text(view->Xselection, &encoding, &format, &text, &new_length);
    gtk_selection_data_set(selection_data, encoding, format, text, new_length);
    gdk_free_compound_text(text);
    //gtk_selection_data_set(selection_data, GDK_SELECTION_TYPE_STRING, 8 * sizeof(gchar), "Coucou", 6);
  }
}

static void sfm_view_set_Xselection(SfmView *view, int time)
{
  GList *l;
  char *p, *q;

  gtk_selection_owner_set(GTK_WIDGET(view->list), GDK_SELECTION_PRIMARY, time);
  FREE_NULL(view->Xselection);

  p = NULL;
  for (l = view->list->selection; l; l = l->next) {
    q = view->doc->dir_list.elements[(int) l->data].name;
    if (p) {
      q = strconcat3(p, " ", q);
      free(p);
      p = q;
    } else p = strdup(q);
  }
  view->Xselection = p;
  XStoreBytes(GDK_DISPLAY(), p, strlen(p));
}


