# kill.tcl - a tcl dialog to the kill command
# version 1.3, Feb 15, 1999
# Author: Samy Zafrany  (samy@netanya.ac.il // samy@math.technion.ac.il)
#         www.netanya.ac.il/~samy
#
namespace eval kill {
    variable kill

    # Define the kill array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set kill {
       dialog ""
       user_procs_text ""
       all_procs_text ""
       jobs_text ""
       signal TERM
       pids:list {}
       signal_names 0
       num_to_sig:entry ""
       num_to_sig:textvar ""
       sig_to_num:entry ""
       sig_to_num:textvar ""
    }
}

# kill::create --
#
#   Method to create the dialog box for the kill command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc kill::create { } {
    global tkWorld
    variable kill

    # Put the focus on the kill dialog if it is already open.
    if [winfo exists $kill(dialog)] {
        wm deiconify $kill(dialog)
	focus $kill(dialog)
        raise $kill(dialog)
	return
    } else {
	set kill(dialog) [dialog::create .kill Kill]
    }

    ############################### FIRST TAB

    # The first tab deals with the standard kill command line options.
    set tab1 [tabnotebook::page [dialog::interior $kill(dialog)] "Signal"]
    set Signals {
      HUP INT QUIT ILL TRAP ABRT BUS FPE KILL USR1 SEGV
      USR2 PIPE ALRM TERM STKFLT CHLD CONT STOP TSTP TTIN
      TTOU URG XCPU XFSZ VTALRM PROF WINCH IO PWR UNUSED   
    }

    # Now build the individual radio buttons for the -s (signal) option
    #
    set f1 [frame $tab1.f1 -class TabnotebookFrame]
    set off [frame $tab1.off -class TabnotebookFrame]

    set n 0
    foreach sig $Signals {
        set rb $f1.[string tolower $sig]
        radiobutton $rb \
	    -text "$sig ([expr $n+1])" \
	    -variable kill::kill(signal) \
	    -value "$sig"
        set i [expr $n/4]
        set j [expr $n%4]
        incr n
        grid config $rb -row $i -column $j -padx 2 -pady 2 -sticky "w"

    }
    set rb $off.off
    radiobutton $rb \
        -text "Off" \
        -variable kill::kill(signal) \
        -value ""
    grid $rb -padx 2 -pady 2 -sticky "w"

    # Build the first tab.
    pack $f1 $off \
        -side top \
        -fill x   \
        -padx 5   \
        -pady 5   \
        -ipadx 5  \
        -ipady 5

    ############################### SECOND TAB

    # The second tab has to do with the list of user processes to kill.
    # Only processes that belong to the current user (who runs tkworld)
    # are presented.
    # We use a text widget for presenting this list of processes.
    # The user can select any number of them.
    #
    set tab2 [tabnotebook::page [dialog::interior .kill] "User Processes"]

    set f2 [frame $tab2.f2 -class TabnotebookFrame]
    set t $f2.user_procs_text
    set kill(user_procs_text) $t

    scrollbar $f2.sb -command "$t yview"
    pack $f2.sb -side right -fill y

    text $t -background white \
            -cursor left_ptr  \
            -tabs {2c left 2.5c right} \
            -yscrollcommand "$f2.sb set" \
            -height 20 \
            -width 61
    pack $t -side left -expand yes -fill both
    
    # Now we clear out all default text bindings
    #
    foreach event [bind Text] {
       bind $t $event {break}
    }

    bind $t <Enter> {focus %W}
    bind $t <Button-1> {kill::pid_select %W}
    bind $t <Next>  "$t yview scroll  1 pages"
    bind $t <space> "$t yview scroll  1 pages"
    bind $t <Prior> "$t yview scroll -1 pages"
    bind $t <Up>    "$t yview scroll -1 units"
    bind $t <Down>  "$t yview scroll  1 units"

    pack $f2 \
       -side  top \
       -fill  x   \
       -padx  5   \
       -pady  5   \
       -ipadx 5   \
       -ipady 5

    ############################### THIRD TAB

    # The third tab has to do with the list of all current processes
    # This is useful only for the root user, since normal users cannot
    # terminate processes that do not belong to them.
    # We use a text widget for presenting the available list of processes.
    # The user can select any number of them.
    #
    set tab3 [tabnotebook::page [dialog::interior .kill] "All Processes"]

    set f3 [frame $tab3.f3 -class TabnotebookFrame]
    set t $f3.all_procs_text
    set kill(all_procs_text) $t

    scrollbar $f3.sb -command "$t yview"
    pack $f3.sb -side right -fill y

    text $t -background white \
            -cursor left_ptr  \
            -tabs {2c left 4.5c left} \
            -yscrollcommand "$f3.sb set" \
            -height 20 \
            -width 61
    pack $t -side left -expand yes -fill both
    
    # Now we clear out all default text bindings
    #
    foreach event [bind Text] {
       bind $t $event {break}
    }
    
    bind $t <Enter> {focus %W}
    bind $t <Button-1> {kill::pid_select %W}
    bind $t <Next>  "$t yview scroll  1 pages"
    bind $t <space> "$t yview scroll  1 pages"
    bind $t <Prior> "$t yview scroll -1 pages"
    bind $t <Up>    "$t yview scroll -1 units"
    bind $t <Down>  "$t yview scroll  1 units"

    pack $f3 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    ############################### FOURTH TAB

    # The fourth tab has to do with the list of user jobs.
    # We use a text widget for presenting the available list of jobs.
    # The user can select any number of them.
    #
    set tab4 [tabnotebook::page [dialog::interior .kill] "Jobs"]

    set f4 [frame $tab4.f4 -class TabnotebookFrame]
    set t $f4.jobs_text
    set kill(jobs_text) $t

    scrollbar $f4.sb -command "$t yview"
    pack $f4.sb -side right -fill y

    text $t -background white \
            -cursor left_ptr  \
            -tabs {2c left 4c left} \
            -yscrollcommand "$f4.sb set" \
            -height 20 \
            -width 61
    pack $t -side left -expand yes -fill both
    
    # Now we clear out all default text bindings
    #
    foreach event [bind Text] {
       bind $t $event {break}
    }
    

    bind $t <Enter> {focus %W}
    bind $t <Button-1> {kill::pid_select %W}
    bind $t <Next>  "$t yview scroll  1 pages"
    bind $t <space> "$t yview scroll  1 pages"
    bind $t <Prior> "$t yview scroll -1 pages"
    bind $t <Up>    "$t yview scroll -1 units"
    bind $t <Down>  "$t yview scroll  1 units"

    pack $f4 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Now we fill those text widgets with the process info
    # The info is updated every 10 seconds:
    #
    kill::update

    ############################### FIFTH TAB

    # Processing the -l option of the kill command
    #
    set tab5 \
        [tabnotebook::page [dialog::interior $kill(dialog)] "Other Options"]

    set f5 [frame $tab5.f5 -class TabnotebookFrame]
    radiobutton $f5.signames \
	    -text "Print a list of all signal names" \
	    -variable kill::kill(signal_names) \
	    -value 1
    radiobutton $f5.num_to_sig \
	    -text "Convert a signal number to signal name: " \
	    -variable kill::kill(signal_names) \
	    -value 2
    set kill(num_to_sig:entry) [entry $f5.num_to_sig:entry \
	    -width 12 \
	    -background #d9d9d9 \
	    -state disabled \
	    -textvariable kill::kill(num_to_sig:textvar)]
    radiobutton $f5.sig_to_num \
	    -text "Convert a signal name to signal number: " \
	    -variable kill::kill(signal_names) \
	    -value 3
    set kill(sig_to_num:entry) [entry $f5.sig_to_num:entry \
	    -width 12 \
	    -background #d9d9d9 \
	    -state disabled \
	    -textvariable kill::kill(sig_to_num:textvar)]
    radiobutton $f5.off \
	    -text "Off" \
	    -variable kill::kill(signal_names) \
	    -value 0


    grid $f5.signames \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f5.sig_to_num $f5.sig_to_num:entry \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f5.num_to_sig $f5.num_to_sig:entry \
	    -padx 2 \
	    -pady 2 \
	    -sticky w
    grid $f5.off - \
	    -padx 2 \
	    -pady 2 \
	    -sticky w

    pack $f5 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5
    bind $f5.off        <ButtonPress-1> {kill::toggle_entries 0}
    bind $f5.signames   <ButtonPress-1> {kill::toggle_entries 1}
    bind $f5.num_to_sig <ButtonPress-1> {kill::toggle_entries 2}
    bind $f5.sig_to_num <ButtonPress-1> {kill::toggle_entries 3}

    set mf [frame $tab5.mf -class TabnotebookFrame]
    set m $mf.message
    message $m -aspect 400 -font fixed -text "
NOTE: \
These options are not guaranteed to work in any case. There are different\
versions of the \"kill\" command which depend on the shell (for example\
they work under zsh but not under the standard Bourne shell or even bash).\
There is also an external \"/bin/kill\" command that may differ from the
internal shell \"kill\" commands."

    pack $mf \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5
    pack $m -side top -fill x
}


############################### PROCEDURES

proc kill::refreshUserProcs {} {
   variable kill

   set t $kill(user_procs_text)
   set yv [lindex [$t yview] 0]
   $t configure -state normal
   $t delete 1.0 end
   $t insert end "PID\tCOMMAND\n" title

   foreach line [split [exec ps uxw] \n] {
      if [regexp {[ ]*USER[ ]+PID[ ]+%CPU} $line] {
        set cmd_start [string first COMMAND $line]
        set cut [string first SIZE $line]
        continue
      }
      if [regexp {ps uxw[ ]*$} $line] {continue}
      set short_line [string range $line 0 $cut]
      set pid [lindex $short_line 1]
      set cmd [string range $line $cmd_start [expr $cmd_start+42]]
      $t insert end "$pid\t\t$cmd\n" [list body "pid:$pid"]
   }

   $t tag configure title -font {Helvetica 12 bold}
   $t tag configure title -underline 1

   $t configure -state disabled
   $t yview moveto $yv
}

proc kill::refreshAllProcs {} {
   variable kill

   set t $kill(all_procs_text)
   set yv [lindex [$t yview] 0]
   $t configure -state normal
   $t delete 1.0 end
   $t insert end "PID\tOWNER\tCOMMAND\n" title

   set all_pids {}
   foreach line [split [exec ps auxw] \n] {
      if [regexp {[ ]*USER[ ]+PID[ ]+%CPU} $line] {
        set cut [string first SIZE $line]
        set cmd_start [string first COMMAND $line]
        continue
      }
      if [regexp {ps auxw[ ]*$} $line] {continue}
      set short_line [string range $line 0 $cut]
      set owner [lindex $short_line 0]
      set pid [lindex $short_line 1]
      set cmd [string range $line $cmd_start [expr $cmd_start+42]]
      $t insert end "$pid\t$owner\t$cmd\n" [list body "pid:$pid"]
      lappend all_pids $pid
   }

   $t tag configure title -font {Helvetica 12 bold}
   $t tag configure title -underline 1

   $t configure -state disabled
   $t yview moveto $yv
   
   foreach p $kill(pids:list) {
      if {[lsearch $all_pids $p]==-1} {
        set i [lsearch $kill(pids:list) $p]
        set kill(pids:list) [lreplace $kill(pids:list) $i $i]
      }
   }
}

proc kill::refreshJobs {} {
   variable kill

   set t $kill(jobs_text)
   set yv [lindex [$t yview] 0]
   $t configure -state normal
   $t delete 1.0 end
   $t insert end "JOB\tPID\tCOMMAND\n" title

   set job 0
   set pgid_list {}
   foreach line [split [exec ps jxw] \n] {
      if [regexp {[ ]*PPID[ ]+PID} $line] {
        set cmd_start [string first COMMAND $line]
        set cut [string first TTY $line]
        continue
      }
      if [regexp {ps jxw[ ]*$} $line] {continue}
      set short_line [string range $line 0 $cut]
      set pgid [lindex $short_line 2]
      if {[lsearch $pgid_list $pgid]>=0} {continue}
      lappend pgid_list $pgid
      incr job
      set pid [lindex $short_line 1]
      set cmd [string range $line $cmd_start [expr $cmd_start+42]]
      $t insert end "\[$job\]\t$pid\t$cmd\n" [list body "pid:$pid"]
   }

   $t tag configure title -font {Helvetica 12 bold}
   $t tag configure title -underline 1

   $t configure -state disabled
   $t yview moveto $yv
}

proc kill::pid_select {w} {
   variable kill
   set t1 $kill(user_procs_text)
   set t2 $kill(all_procs_text)
   set t3 $kill(jobs_text)
   set tags [$w tag names current]
   set i [lsearch -glob $tags pid:*]
   if {$i==-1} {return}
   set tg [lindex $tags $i]
   set pid [lindex [split $tg :] 1]
   set i [lsearch $kill(pids:list) $pid]
   if {$i >= 0} {
     $t1 tag configure $tg -background [$t1 cget -background]
     $t2 tag configure $tg -background [$t2 cget -background]
     $t3 tag configure $tg -background [$t3 cget -background]
     set kill(pids:list) [lreplace $kill(pids:list) $i $i]
     return
   }
   $t1 tag configure $tg -background grey75
   $t2 tag configure $tg -background grey75
   $t3 tag configure $tg -background grey75
   lappend kill(pids:list) $pid
}

proc kill::toggle_entries { int } {
    variable kill

    switch $int {
        0 -
	1 {
	    $kill(num_to_sig:entry) delete 0 end
	    $kill(num_to_sig:entry) configure \
		    -bg #d9d9d9 \
		    -state disabled
	    $kill(sig_to_num:entry) delete 0 end
	    $kill(sig_to_num:entry) configure \
		    -bg #d9d9d9 \
		    -state disabled
	}
	2 {
	    $kill(num_to_sig:entry) configure \
		    -bg #ffffff \
		    -state normal
	    $kill(sig_to_num:entry) delete 0 end
	    $kill(sig_to_num:entry) configure \
		    -bg #d9d9d9 \
		    -state disabled
	}
	3 {
	    $kill(num_to_sig:entry) delete 0 end
	    $kill(num_to_sig:entry) configure \
		    -bg #d9d9d9 \
		    -state disabled
	    $kill(sig_to_num:entry) configure \
		    -bg #ffffff \
		    -state normal
	}
    }
}


# kill::ok --
#
#   Method to insert the command the user has created into the CC
#   as a Tcl list.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc kill::ok { } {
   global tkWorld
   variable kill

   # Build the full command line
   #
   set cmd "kill "
   set cmd_done 0

   switch $kill(signal_names) {
      1 {
         append cmd "-l"
         set cmd_done 1
      }
      2 {
         append cmd "-l $kill(num_to_sig:textvar)"
         set cmd_done 1
      }
      3 {
         append cmd "-l $kill(sig_to_num:textvar)"
         set cmd_done 1
      }
   }

   if {$cmd_done==0} {
      set pids [lsort -integer $kill(pids:list)]
      if [string length $kill(signal)] {
         set cmd [eval list kill -s $kill(signal) $pids]
      } else {
         set cmd [eval list kill $pids]
      }
   }



   # Insert the Tcl command list in the Command Center with the
   # proper formatting of a space between each argument on the
   # command line.  If there are no options given by the user,
   # then don't display it in the CC.

   $tkWorld(cmd_center) insert insert $cmd

   # Activate the buttons in the toolbar for the command center.
   toolbar::group_state cmd_center active
   toolbar::button_state $toolbar::toolbar(stop) disabled
}

# kill::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc kill::reset { } {
   variable kill

   set kill(signal) TERM
   set kill(signal_names) 0
   kill::toggle_entries 0
   kill::clear
}

# kill::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc kill::clear { } {
   variable kill

   set kill(sig_to_num:textvar) ""
   set kill(num_to_sig:textvar) ""
   set kill(pids:list) {}
   set t1 $kill(user_procs_text)
   set t2 $kill(all_procs_text)
   set t3 $kill(jobs_text)
   foreach t [list $t1 $t2 $t3] {
      foreach tg [$t tag names] {
         $t tag delete $tg
      }
   }
   kill::refreshUserProcs
   kill::refreshAllProcs
   kill::refreshJobs
}

proc kill::update {} {
   variable kill

   if ![winfo exists $kill(dialog)] {return}
   kill::refreshUserProcs
   kill::refreshAllProcs
   kill::refreshJobs
   after 10000 {kill::update}
}

# kill::help --
#
#   Method to invoke the Kill Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc kill::help { } {
   global tkWorld

   help::create "kill/index.html" "Kill Command Help"
}

# kill::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc kill::close { } {
   variable kill
   
   balloonhelp::cancel
   destroy $kill(dialog)
}
