/*
 * rtp.h  --  RTP header file (RFC 1890)
 */
#ifndef RTP_H
#define RTP_H

#include "config.h"

#if WORDS_BIGENDIAN
#define RTP_BIG_ENDIAN 1
#else
#define RTP_LITTLE_ENDIAN 1
#endif

/*
 * The type definitions below are valid for 32-bit architectures and
 * may have to be adjusted for 16- or 64-bit architectures.
 */
typedef unsigned char  u_int8;
typedef unsigned short u_int16;
typedef unsigned int   u_int32;
typedef signed   int   int32;
typedef signed   short int16;

/*
 * Current protocol version.
 */
#define RTP_VERSION 2

#define RTP_SEQ_MOD (1<<16)
#define RTP_MAX_SDES 255	/* maximum text length for SDES */

#define RTP_TS_MOD  (0xffffffff)

/* Offset from UNIX's epoch to the NTP epoch in seconds (NTP's JAN_1970) */
#define RTP_EPOCH_OFFSET        2208988800UL

#define RTP_M   0x0080  /* Marker: significant event <e.g. frame boundary> */
#define RTP_P   0x2000  /* Padding is present */
#define RTP_X   0x1000  /* Extension Header is present */

/*
 * Parameters controling the RTCP report rate timer.
 */
#define RTCP_SESSION_BW_FRACTION (0.05)
#define RTCP_MIN_RPT_TIME (5.)
#define RTCP_SENDER_BW_FRACTION (0.25)
#define RTCP_RECEIVER_BW_FRACTION (1. - RTCP_SENDER_BW_FRACTION)
#define RTCP_SIZE_GAIN (1./8.)

/*
 * Largest (user-level) packet size generated by our rtp applications.
 * Individual video formats may use smaller mtu's.
 */
#define RTP_MTU 1024

#define MAXHDR 24

typedef enum {
  RTCP_SR   = 200,
  RTCP_RR   = 201,
  RTCP_SDES = 202,
  RTCP_BYE  = 203,
  RTCP_APP  = 204
} rtcp_type_t;

typedef enum {
  RTCP_SDES_END    =  0,
  RTCP_SDES_CNAME  =  1,
  RTCP_SDES_NAME   =  2,
  RTCP_SDES_EMAIL  =  3,
  RTCP_SDES_PHONE  =  4,
  RTCP_SDES_LOC    =  5,
  RTCP_SDES_TOOL   =  6,
  RTCP_SDES_NOTE   =  7,
  RTCP_SDES_PRIV   =  8, 
  RTCP_SDES_IMG    =  9,
  RTCP_SDES_DOOR   = 10,
  RTCP_SDES_SOURCE = 11
} rtcp_sdes_type_t;

/*
 * RTP data header
 */
typedef struct {
#if RTP_BIG_ENDIAN
  unsigned int version:2;  /* protocol version */
  unsigned int p:1;        /* padding flag */
  unsigned int x:1;        /* header extension flag */
  unsigned int cc:4;       /* CSRC count */
  unsigned int m:1;        /* marker bit */
  unsigned int pt:7;       /* payload type */
#elif RTP_LITTLE_ENDIAN
  unsigned int cc:4;       /* CSRC count */
  unsigned int x:1;        /* header extension flag */
  unsigned int p:1;        /* padding flag */
  unsigned int version:2;  /* protocol version */
  unsigned int pt:7;       /* payload type */
  unsigned int m:1;        /* marker bit */
#else
#error Define one of RTP_LITTLE_ENDIAN or RTP_BIG_ENDIAN
#endif
  u_int16 seq;             /* sequence number */
  u_int32 ts;              /* timestamp */
  u_int32 ssrc;            /* synchronization source */
  u_int32 csrc[1];         /* optional CSRC list */
} rtp_hdr_t;

/*
 * RTCP common header word
 */
typedef struct {
#if RTP_BIG_ENDIAN
  unsigned int version:2;  /* protocol version */
  unsigned int p:1;        /* padding flag */
  unsigned int count:5;    /* varies by payload type */
#elif RTP_LITTLE_ENDIAN
  unsigned int count:5;    /* varies by packet type */
  unsigned int p:1;        /* padding flag */
  unsigned int version:2;  /* protocol version */
#else
#error Define one of RTP_LITTLE_ENDIAN or RTP_BIG_ENDIAN
#endif
  unsigned int pt:8;       /* RTCP packet type */
  u_int16 length;          /* packet length in words, without this word */
} rtcp_common_t;

/*
 * Big-endian mask for version, padding bit and packet type pair
 */
#define RTCP_VALID_MASK     (0xc000 | 0x2000 | 0xfe)
#define RTCP_VALID_VALUE    ((RTP_VERSION << 14) | RTCP_SR)

/*
 * Reception report block
 */
typedef struct {
  u_int32 ssrc;            /* data source being reported */
  unsigned int fraction:8; /* fraction lost since last SR/RR */
  int lost:24;             /* cumulative number of packets lost (signed!) */
  u_int32 last_seq;        /* extended last sequence number received */
  u_int32 jitter;          /* interarrival jitter */
  u_int32 lsr;             /* last SR packet from this source */
  u_int32 dlsr;            /* delay since last SR packet */
} rtcp_rr_t;

/*
 * SDES item
 */
typedef struct {
  u_int8 type;             /* type of SDES item (rtcp_sdes_type_t) */
  u_int8 length;           /* length of SDES item (in octets) */
  char data[1];            /* text, not zero-terminated */
} rtcp_sdes_item_t;

/*
 * One RTCP packet
 */
typedef struct {
  rtcp_common_t common;    /* common header */
  union {
    /* sender report (SR) */
    struct rtcp_sr {
      u_int32 ssrc;        /* source this RTCP packet refers to */
      u_int32 ntp_sec;     /* NTP timestamp */
      u_int32 ntp_frac;
      u_int32 rtp_ts;      /* RTP timestamp */
      u_int32 psent;       /* packets sent */
      u_int32 osent;       /* octets sent */ 
      rtcp_rr_t rr[1];     /* variable-length list */
    } sr;

    /* reception report (RR) */
    struct rtcp_rr {
      u_int32 ssrc;        /* source this generating this report */
      rtcp_rr_t rr[1];     /* variable-length list */
    } rr;

    /* source description (SDES) */
    struct rtcp_sdes {
      u_int32 src;              /* first SSRC/CSRC */
      rtcp_sdes_item_t item[1]; /* list of SDES items */
    } sdes;

    /* BYE */
    struct {
      u_int32 src[1];      /* list of sources */
      /* can't express trailing text for reason */
    } bye;
  } r;
} rtcp_t;

typedef struct rtcp_sdes rtcp_sdes_t;
typedef struct rtcp_sr rtcp_sr_t;

/*
 * Per-source state information
 */
/* network infos */
typedef struct {
  u_int16 max_seq;              /* highest seq. number seen */
  u_int32 cycles;               /* shifted count of seq. number cycles */
  u_int32 base_seq;             /* base seq number */
  u_int32 bad_seq;              /* last 'bad' seq number + 1 */
  u_int32 probation;            /* sequ. packets till source is valid */
  u_int32 received;             /* packets received */
  u_int32 expected_prior;       /* packet expected at last interval */
  u_int32 received_prior;       /* packet received at last interval */
  u_int32 transit;              /* relative trans time for prev packet */
  u_int32 jitter;               /* estimated jitter */
} source;

typedef struct {
        u_int32 upper;        /* more significant 32 bits */
        u_int32 lower;        /* less significant 32 bits */
} ntp64;


#endif /* RTP_H */
