/*
 * Interface entre vreng et zbuffer/zrender
 * (c) 1996,1997 Fabrice Bellard
 */

#include <GL/gl.h>

#include "global.h"
#include "defaults.h"
#include "texture.h"
#include "model.h"
#include "walls.h"

#define DEBUG_ZV

#define M_2PI	6.28318530717958647692
#define M_PI_180 0.01745329251994329577

#define SPHERE_SLICES 16
#define SPHERE_STACKS 16

/* camera */
static struct {
  float fovy,near_z,far_z;
} camera_projection;

typedef struct ZVSolid {
  int display_flag;
  M4 pos_mat;
  struct ZVSolid *next, *prev;
  int *display_list;
  int nb_frames;
  int cur_frame;
  V3 bbox_center;
  V3 bbox_size;
  char *optbuf;        /* donnees complementaires */
  int optbuf_size;
} ZVSolid;

typedef struct {
  ZVSolid *first_solid, *last_solid;
  V3 bbox_min;
  V3 bbox_max;
  int first_bbox;
  M4 camera_pos;
  int quality;
} ZVContext;

static ZVContext zv_context;

/*
 * Calcul de la bounding box
 */
void zvVertex3fv(float *v)
{
  int i;
  ZVContext *c = &zv_context;

  if (c->first_bbox) {
    for (i=0;i<3;i++) {
      c->bbox_min.v[i]=v[i];
      c->bbox_max.v[i]=v[i];
    }
    c->first_bbox=0;
  }
  else {
    for (i=0;i<3;i++) {
      if (v[i]<c->bbox_min.v[i]) c->bbox_min.v[i]=v[i];
      if (v[i]>c->bbox_max.v[i]) c->bbox_max.v[i]=v[i];
    }
  }
  glVertex3f(v[0],v[1],v[2]);
}

void zvVertex3f(float x, float y, float z)
{
  float v[3];

  v[0]=x; v[1]=y; v[2]=z;
  zvVertex3fv(v);
}


/*
 * dessin d'un pave (repris de Mesa)
 */
Private
void drawbox(float x0, float x1, float y0, float y1, float z0, float z1, int type, int textures[])
{
    static float n[6][3] = {
	{-1.0, 0.0, 0.0}, {0.0, 1.0, 0.0}, {1.0, 0.0, 0.0},
	{0.0, -1.0, 0.0}, {0.0, 0.0, 1.0}, {0.0, 0.0, -1.0}
    };
    static int faces[6][4] = {
	{ 0, 1, 2, 3 }, { 3, 2, 6, 7 }, { 7, 6, 5, 4 },
	{ 4, 5, 1, 0 }, { 5, 6, 2, 1 }, { 7, 4, 0, 3 }
    };
    static float texcoords[6][4][2] = {
      { { 0,1 }, { 0,0 }, { 1,0 }, { 1,1 } },
      { { 1,1 }, { 1,0 }, { 0,0 }, { 0,1 } },
      { { 0,1 }, { 0,0 }, { 1,0 }, { 1,1 } },
      { { 1,1 }, { 1,0 }, { 0,0 }, { 0,1 } },
      { { 0,0 }, { 0,1 }, { 1,1 }, { 1,0 } },
      { { 0,0 }, { 0,1 }, { 1,1 }, { 1,0 } }
    };
    float v[8][3], tmp;
    int i,j;

    if (x0 > x1) {
	tmp = x0; x0 = x1; x1 = tmp;
    }
    if (y0 > y1) {
	tmp = y0; y0 = y1; y1 = tmp; 
    }
    if (z0 > z1) {
	tmp = z0; z0 = z1; z1 = tmp; 
    }
    v[0][0] = v[1][0] = v[2][0] = v[3][0] = x0;
    v[4][0] = v[5][0] = v[6][0] = v[7][0] = x1;
    v[0][1] = v[1][1] = v[4][1] = v[5][1] = y0;
    v[2][1] = v[3][1] = v[6][1] = v[7][1] = y1;
    v[0][2] = v[3][2] = v[4][2] = v[7][2] = z0;
    v[1][2] = v[2][2] = v[5][2] = v[6][2] = z1;

    for (i = 0; i < 6; i++) {
      if (textures[i] > 0) {
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, textures[i]);
      }
      glBegin(type);
      for (j=0;j<4;j++) {
	if (textures[i] > 0) glTexCoord2fv(&texcoords[i][j][0]);
	glNormal3fv(&n[i][0]);
	zvVertex3fv(&v[faces[i][j]][0]);
      }
      glEnd();
      if (textures[i] > 0)
	glDisable(GL_TEXTURE_2D);
    }
}

/*
 * dessin d'un tore (repris de Mesa)
 */
Private
void drawTorus(float rc, int numc, float rt, int numt)
{
    int i, j, k;
    double s, t;
    double x, y, z;

    for (i = 0; i < numc; i++) {
	glBegin(GL_QUAD_STRIP);
        for (j = 0; j <= numt; j++) {
	    for (k = 1; k >= 0; k--) {
		s = (i + k) % numc + 0.5;
		t = j % numt;

		x = cos(t*M_2PI/numt) * cos(s*M_2PI/numc);
		y = sin(t*M_2PI/numt) * cos(s*M_2PI/numc);
		z = sin(s*M_2PI/numc);
		glNormal3f(x, y, z);

		x = (rt + rc * cos(s*M_2PI/numc)) * cos(t*M_2PI/numt);
		y = (rt + rc * cos(s*M_2PI/numc)) * sin(t*M_2PI/numt);
		z = rc * sin(s*M_2PI/numc);
		zvVertex3f(x, y, z);
	    }
        }
	glEnd();
    }
}

/*
 * Sphre (repris de Mesa)
 */
Private
void drawSphere(float radius, int slices, int stacks)
{
   float rho, drho, theta, dtheta;
   float x, y, z;
   float s, t, ds, dt;
   int i, j, imin, imax;
   int normals;
   float nsign;
  
   normals=1;
   nsign=1;

   drho = M_PI / (float) stacks;
   dtheta = M_2PI / (float) slices;

  /* draw +Z end as a triangle fan */
  glBegin(GL_TRIANGLE_FAN);
  glNormal3f(0.0, 0.0, 1.0);
  glTexCoord2f(0.5,0.0);
  zvVertex3f(0.0, 0.0, nsign * radius);
  for (j=0;j<=slices;j++) {
	 theta = (j==slices) ? 0.0 : j * dtheta;
	 x = -sin(theta) * sin(drho);
	 y = cos(theta) * sin(drho);
	 z = nsign * cos(drho);
	 if (normals)
	   glNormal3f(x*nsign, y*nsign, z*nsign);
	 zvVertex3f(x*radius, y*radius, z*radius);
      }
   glEnd();

      ds = 1.0 / slices;
      dt = 1.0 / stacks;
      t = 1.0;  /* because loop now runs from 0 */
      if (1) {
        imin = 0;
        imax = stacks;
      }
      else {
        imin = 1;
        imax = stacks-1;
      }

      /* draw intermediate stacks as quad strips */
      for (i=imin;i<imax;i++) {
	 rho = i * drho;
	 glBegin(GL_QUAD_STRIP);
         s = 0.0;
	 for (j=0;j<=slices;j++) {
	    theta = (j==slices) ? 0.0 : j * dtheta;
	    x = -sin(theta) * sin(rho);
	    y = cos(theta) * sin(rho);
	    z = nsign * cos(rho);
	    if (normals)
	      glNormal3f(x*nsign, y*nsign, z*nsign);
	    glTexCoord2f(s,1-t);
	    zvVertex3f(x*radius, y*radius, z*radius);
	    x = -sin(theta) * sin(rho+drho);
	    y = cos(theta) * sin(rho+drho);
	    z = nsign * cos(rho+drho);
	    if (normals)
	      glNormal3f(x*nsign, y*nsign, z*nsign);
	    glTexCoord2f(s,1-(t-dt));
            s += ds;
	    zvVertex3f(x*radius, y*radius, z*radius);
	 }
	 glEnd();
	 t -= dt;
      }
    /* draw -Z end as a triangle fan */
    glBegin(GL_TRIANGLE_FAN);
    glNormal3f(0.0, 0.0, -1.0);
      glTexCoord2f(0.5,1.0);
      zvVertex3f(0.0, 0.0, -radius*nsign);
      rho = M_PI - drho;
      s = 1.0;
      t = dt;
      for (j=slices;j>=0;j--) {
	 theta = (j==slices) ? 0.0 : j * dtheta;
	 x = -sin(theta) * sin(rho);
	 y = cos(theta) * sin(rho);
	 z = nsign * cos(rho);
	 if (normals)
	   glNormal3f(x*nsign, y*nsign, z*nsign);
	 glTexCoord2f(s,1-t);
         s -= ds;
	 zvVertex3f(x*radius, y*radius, z*radius);
      }
      glEnd();
}

/*
 * un petit parser pour les noms de solides
 */
static char *parser_str, *parser_ptr;
static int ch;

Private
void solidParserInit(char *str)
{
  parser_str = str;
  parser_ptr = parser_str;
  ch = parser_ptr[0];
}

Private
void solidParserError(char *format,...)
{
  va_list ap;

  va_start(ap, format);
  fprintf(stderr, "solid: parse error: ");
  vfprintf(stderr, format, ap);
  fprintf(stderr, "\n");
  fatal("solid error.");
  va_end(ap);
}

Private
int issep(char c)
{
  return (c == ' ') || (c == ',') || (c == '\0') || (c == ']');
}

Private
void nextCh(void)
{
  if (ch) ch = *++parser_ptr;
}

Private
void skipSpace(void)
{
  while (isspace(ch)) nextCh();
}

Private
void skip(int c)
{
  if (ch != c)
    solidParserError("'%c' expected at position %d in '%s'",
		 c,parser_ptr-parser_str,parser_str);
  nextCh();
}

Private
float getFloat(void) 
{
  char *p;
  char buf[20];

  for (p = buf; isdigit(ch); nextCh())
    *p++ = ch;
  if (ch == '.') {
    *p++ = ch;
    nextCh();
    for ( ; isdigit(ch); nextCh())
      *p++ = ch;
  }
  *p++ = '\0';
  return atof(buf);
}

Private
int getInt(void) 
{
  char *p;
  char buf[20];

  for (p = buf; isdigit(ch); nextCh())
    *p++ = ch;
  *p++ = '\0';
  return atoi(buf);
}
 
#include "solid.h"

char *sid_str[] = {
  "wall",
  "door",
  "user",
  "box",
  "diffuse",
  "ambient",
  "size",
  "torus",
  "sphere",
  "radius",
  "radius2",
  "texture",
  "rect",
  "tex_xn",
  "tex_yp",
  "tex_xp",
  "tex_yn",
  "tex_zp",
  "tex_zn",
  "specular",
  "shininess",
  "statue",
  "geometry",
  "scale",
  "frames",
  "cylinder",
  "cone",
  NULL,
};

Private
int getID(void)
{
  char *p, **q;
  char buf[256];

  for (p = buf; isalnum(ch) || ch == '_'; nextCh())
    *p++ = ch;
  *p++ = '\0';
  for (q = sid_str; *q != NULL && (strcmp(*q,buf) != 0);)
    q++;
  if (*q == NULL) 
    solidParserError("solid: parse error: '%s' in '%s'\n", buf, parser_str);
  return q - sid_str;
}

Private
void getString(char *buf)
{
  char *p;

  for (p = buf; !issep(ch); nextCh())
    *p++ = ch;
  *p++ = '\0';
}

Private
int findChar(char c)
{
  while (issep(ch) && (ch != c))
    nextCh();
  return (ch == c);
}

/*
 * Routines de cration d'un solide
 */

/*
 * TODO: delete textures
 */

Public
void SolidDelete(Solid s)
{
  ZVContext *c = &zv_context;

  if (s) {
    if (s->next)
      s->next->prev = s->prev;
    else
      c->last_solid = s->prev;
    if (s->prev)
      s->prev->next = s->next;
    else
      c->first_solid = s->next;
    if (s->display_list)
      free(s->display_list);
    free(s);
  }
}

Private
int statueParser(V3 *cur_max, V3 *cur_min, int *display_list, int pos)
{
  ZVContext *c = &zv_context;
  int texture = -1, dlist = -1, geom_dlist = -1;
  int i;
  char url_geom[URL_LEN];
  float scale = 1.0;
  int fr_num[MAXFRAMES], nbf = 0;
  Model *mdl;
  
  c->first_bbox=1;
  
  while (1) {
    int id;
    
    while (isspace(ch))
      nextCh();
    if ((ch == '\0') || (ch == ']')) break;
    if (ch==',')
      nextCh();

    id = getID(); skip('=');
    switch (id) {
    case ID_TEXTURE: {
      char url[URL_LEN];
      
      getString(url);
      texture = getTextureFromCache(url);
    } break;
    case ID_GEOMETRY:
      getString(url_geom);
      dlist = 0;
      break;
    case ID_SCALE:
      scale = getFloat();
      break;
    case ID_FRAMES: {
      int cur_frame;
      int end_frame;
      int i = 0;
      nbf = 1;
      
      skip('[');
      while (ch != ']') {
	cur_frame = getInt();
	if (ch == '-') {
	  skip('-');
	  end_frame = getInt();
	}
        else {
	  end_frame = cur_frame;
	  if (ch != ']')
	    skip(',');
	}
	while (cur_frame <= end_frame)
	  fr_num[i++] = cur_frame++;
      }
      fr_num[i] = -1;
      skip(']');
      break;
    }
    default:
      fprintf(stderr, "solid parser: Bad property id=%d\n", id);
      break;
    }
  }
  if ((texture == -1) || (dlist == -1))
    return -1;

  mdl = GetModel(url_geom);
  if (mdl == NULL) 
    return -1;
  
  if (nbf == 0) {
    fr_num[0] = 0;
    fr_num[1] = -1;
  }
  
  for (nbf=0; fr_num[nbf] != -1; ) {
    geom_dlist = GetModelDList(mdl, fr_num[nbf], fr_num[nbf], 0.0, scale);
    if (geom_dlist == -1) {
      FreeModel(mdl);
      return -1;
    }
    dlist = glGenLists(1);
    glNewList(dlist, GL_COMPILE);
    glEnable(GL_TEXTURE_2D);
    glBindTexture(GL_TEXTURE_2D, texture);
    glCallList(geom_dlist);
    glDisable(GL_TEXTURE_2D);
    glEndList();
    display_list[pos + nbf] = dlist;
    /* bbox */
    if (pos == 0) {
      *cur_max = c->bbox_max;
      *cur_min = c->bbox_min;
    }
    else {
      int j;
      
      for (j = 0; j < 3; j++) {
	cur_min->v[j] = (c->bbox_min.v[j] < cur_min->v[j] ?
			 c->bbox_min.v[j] : cur_min->v[j]);
	cur_max->v[j] = (c->bbox_max.v[j] > cur_max->v[j] ?
			 c->bbox_max.v[j] : cur_max->v[j]);
      }    
    }
    nbf++;
  }
  FreeModel(mdl);
  return nbf;
}

Private
int solidParser(int form, V3 *cur_max, V3 *cur_min, int *display_list, int pos)
{
  ZVContext *c = &zv_context;
  int sid, i;
  int list;
  V3 size, v;
  float radius, radius2;
  int texture;
  int box_textures[6];
  char url[URL_LEN];
  static float mat_shininess[] = {20.0};
  static float mat_specular[] = {1.0, 1.0, 1.0, 0.0};
  static float mat_diffuse[] = {1, 1, 1, 0.0};
  static float mat_ambient[] = {1, 1, 1, 0.0};

  c->first_bbox = 1;
  size = V3_New(1,1,1);
  radius = radius2 = 1;
  for (texture = -1, i=0; i<6; box_textures[i++] = -1) ;

  while (1) {
    while (isspace(ch)) nextCh();
    if ((ch == '\0') || (ch == ']')) break;
    if (ch==',') nextCh();

    sid=getID(); skip('=');
    switch (sid) {
    case ID_RADIUS:
      radius = getFloat();
      break;
    case ID_RADIUS2:
      radius2 = getFloat();
      break;
    case ID_DIFFUSE:
    case ID_AMBIENT:
    case ID_SPECULAR:
    case ID_SHININESS:
    case ID_SIZE:
      v.X=getFloat(); skip(',');
      v.Y=getFloat(); skip(',');
      v.Z=getFloat();
      switch (sid) {
      case ID_DIFFUSE:
	for (i=0;i<3;i++) {
	  mat_diffuse[i]=v.v[i];
	  mat_ambient[i]=v.v[i];
	}
	break;
      case ID_AMBIENT:
	for (i=0;i<3;i++) {
	  mat_ambient[i]=v.v[i];
	}
	break;
      case ID_SPECULAR:
	for (i=0;i<3;i++) {
	  mat_specular[i]=v.v[i];
	}
	break;
      case ID_SHININESS:
	  mat_shininess[i]=v.v[0];
	break;
      case ID_SIZE:
	size=v;
	break;
      }
      break;

    case ID_TEXTURE:
      getString(url);
      texture = getTextureFromCache(url);
      break;
    case ID_TEX_XN:
    case ID_TEX_YP:
    case ID_TEX_XP:
    case ID_TEX_YN:
    case ID_TEX_ZP:
    case ID_TEX_ZN:
      getString(url);
      box_textures[sid - ID_TEX_XN] = getTextureFromCache(url);
      break;
    default:
      fprintf(stderr, "solid parser: Bad property sid=%d\n", sid);
      break;
    }
  }
  list = glGenLists(1);
  glNewList(list,GL_COMPILE);
  glCullFace(GL_BACK);
  glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
  glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
  glMaterialfv(GL_FRONT, GL_AMBIENT, mat_ambient);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);

  switch (form) {
  case ID_BOX:
    glShadeModel(GL_FLAT);
    glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
    drawbox(-size.X, size.X, -size.Y, size.Y, -size.Z, size.Z,
	    GL_QUADS, box_textures);
    break;
  case ID_TORUS:
    glShadeModel(GL_SMOOTH);
    drawTorus(radius2,8,radius,20);
    break;
  case ID_SPHERE:
    glShadeModel(GL_SMOOTH);
    if (texture > 0) {
      glEnable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D,texture);
      drawSphere(radius, SPHERE_SLICES, SPHERE_STACKS);
      glDisable(GL_TEXTURE_2D);
    }
    else
      drawSphere(radius, SPHERE_SLICES, SPHERE_STACKS);
    break;
  case ID_RECT:
    glDisable(GL_CULL_FACE);
    if (texture > 0) {
      glEnable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D, texture);
    }
    glBegin(GL_QUADS);
    glTexCoord2f(0,0); zvVertex3f(0,0,0);
    glTexCoord2f(1,0); zvVertex3f(size.X,0,0);
    glTexCoord2f(1,1); zvVertex3f(size.X,size.Y,0);
    glTexCoord2f(0,1); zvVertex3f(0,size.Y,0);
    glEnd();
    if (texture > 0)
      glDisable(GL_TEXTURE_2D);
    glEnable(GL_CULL_FACE);
    break;
  }
  glEndList();

  /* new solid */
  display_list[pos + 0] = list;
  /* bbox */
  if (pos == 0) {
    *cur_max = c->bbox_max;
    *cur_min = c->bbox_min;
  }
  else {
    int j;
    
    for (j = 0; j < 3; j++) {
      cur_min->v[j] = (c->bbox_min.v[j] < cur_min->v[j] ?
		       c->bbox_min.v[j] : cur_min->v[j]);
      cur_max->v[j] = (c->bbox_max.v[j] > cur_max->v[j] ?
		       c->bbox_max.v[j] : cur_max->v[j]);
    }    
  }
  return 1;
}


Public
Solid SolidParser(char *str)
{
  Solid s;
  int sid, pos;
  int nbframes, i, j;
  ZVContext *c = &zv_context;
  V3 cur_max, cur_min;

  solidParserInit(str);
  if (findChar('[')) {
    /* This object has more that one frame */
    skip('[');
    nbframes = getInt();
    skip(']');
  }
  else
    nbframes = 1;

  /* Allocate the new solid */
  s = calloc(1, sizeof(ZVSolid));
  IdM4(&s->pos_mat);
  s->display_flag = 1;
  s->nb_frames = nbframes;
  s->display_list = (int *) malloc(s->nb_frames * sizeof(int));
  s->optbuf = NULL;
  s->optbuf_size = -1;

  s->next=NULL;
  s->prev = c->last_solid;
  if (c->first_solid == NULL)
    c->first_solid = s;
  else 
    c->last_solid->next = s;
  c->last_solid = s;

  for (pos=0; pos < nbframes; ) {
    int res = 0;

    if (nbframes > 1)
      skip('[');
    sid = getID();
    switch (sid) {
    case ID_USER:
    case ID_WALL:
    case ID_DOOR:
    case ID_BOX:
      res = solidParser(ID_BOX, &cur_max, &cur_min, s->display_list, pos);
      break;
    case ID_TORUS:
    case ID_SPHERE:
    case ID_RECT:
      res = solidParser(sid, &cur_max, &cur_min, s->display_list, pos);
      break;
    case ID_STATUE:
      res = statueParser(&cur_max, &cur_min, s->display_list, pos);
      break;
    default:
      solidParserError("Bad object name");
      break;
    }
    if (res == -1) {
      SolidDelete(s);
      return NULL;
    }
    pos += res;
    if (nbframes > 1)
      skip(']');
  }
  for (j = 0; j < 3; j++) {
    s->bbox_center.v[j] = (cur_max.v[j] + cur_min.v[j]) / 2;
    s->bbox_size.v[j] = (cur_max.v[j] - cur_min.v[j]) / 2;
  }
  return s;
}

Public
void SolidSetOptBuf(Solid s, char *buf, int size)
{
  if (s->optbuf_size != size) {
    if (s->optbuf != NULL) free(buf);
    s->optbuf = malloc(size);
    s->optbuf_size = size;
  }
  memcpy(s->optbuf, buf, size);
}

Public
int SolidGetOptBuf(Solid s, char *buf, int size)
{
  if (s->optbuf_size < size)
    size = s->optbuf_size;
  memcpy(buf, s->optbuf, size);
  return size;
}

/***
int SolidIntersect(Solid s1, Solid s2)
{
  return 1;
}
***/

void SolidGetBB(Solid s, V3 *center, V3 *size)
{
  int i, l;
  V3 p[2], v, v1, qmin, qmax;
  
  for (i=0; i < 3; i++) {
    p[0].v[i] = s->bbox_center.v[i] - s->bbox_size.v[i];
    p[1].v[i] = s->bbox_center.v[i] + s->bbox_size.v[i];
  }
  for (l=0; l < 8; l++) {
    v.X = p[l % 2].X;
    v.Y = p[(l/2) % 2].Y;
    v.Z = p[(l/4) % 2].Z;
    MulM4V3(&v1, &s->pos_mat, &v);
    if (l == 0)
      qmin = qmax = v1;
    else {
      for (i=0; i < 3; i++) {
	if (v1.v[i] < qmin.v[i]) qmin.v[i] = v1.v[i];
	if (v1.v[i] > qmax.v[i]) qmax.v[i] = v1.v[i];
      }
    }
  }
  for (i=0; i < 3; i++) {
    center->v[i] = (qmax.v[i]+qmin.v[i])/2;
    size->v[i] = (qmax.v[i]-qmin.v[i])/2;
  }
#ifdef DEBUG_ZV_BB
  trace(DBG_ZV, "boundingbox: center=%g %g %g size=%g %g %g",
	 center->X,center->Y,center->Z, size->X,size->Y,size->Z);
#endif
}

Public
void SolidSetPosition(Solid s, M4 *mpos)
{
  s->pos_mat = *mpos;
}

void SolidGetPosition(Solid s, M4 *mpos)
{
  *mpos = s->pos_mat;
}

Public
void SolidSetVisibilty(Solid s, int flag)
{
  s->display_flag = flag;
}

Public
void SolidSetFrame(Solid s, int frame)
{
  s->cur_frame = frame % s->nb_frames;
}


/*
 * Gestion de la camera
 */
void M4_to_GL(float *gl_matrix, M4 *matrix)
{
  float *m = gl_matrix;
  int i,j;

  for (i=0; i<4; i++)
    for (j=0; j<4; j++)
      *m++ = matrix->m[j][i];
} 

Public
void ZLib_setCameraPosition(M4 *cpos)
{
  ZVContext *c = &zv_context;

  c->camera_pos = *cpos;
}

Public
void ZLib_setCameraProjection(float Fovy, float Near, float Far)
{
  float xmin, xmax, ymin, ymax, ratio;
  int get_viewport[4], xsize, ysize;

  glGetIntegerv(GL_VIEWPORT, get_viewport);
  xsize = get_viewport[2];
  ysize = get_viewport[3];
  camera_projection.fovy = Fovy;
  camera_projection.near_z = Near;
  camera_projection.far_z = Far;
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  ymax = Near * tan (Fovy * M_PI_180);
  ymin = -ymax;
  ratio= (float) xsize / (float) ysize;
  xmax = ymax * ratio;
  xmin = -xmax;
  glFrustum(xmin, xmax, ymin, ymax, Near, Far);
  glTranslatef(0.0, 0.0, -0.5);
}


/*
 * Gestion de l'affichage
 */

void RenderInit(int quality)
{
  ZVContext *c = &zv_context;

  /* glDebug(GL_FALSE); */
  zv_context.quality = quality;
  c->first_solid = NULL;
  c->last_solid = NULL;

  glFrontFace(GL_CCW);
  glEnable(GL_DEPTH_TEST);
  glEnable(GL_LIGHTING);
  glEnable(GL_CULL_FACE);
  glEnable(GL_LIGHT0);
  glEnable(GL_LIGHT1);
  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
   
  /* cache texture */
  initTextureCache(zv_context.quality);

  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
}

Private
void renderSolids()
{
  ZVContext *c = &zv_context;
  ZVSolid *s;
  float m[16];

  for (s = c->first_solid; s != NULL; s = s->next) {
    if (s->display_flag) {
      glLoadName(s->display_list[s->cur_frame]);
      glPushMatrix();
      M4_to_GL(m, &s->pos_mat);
      glMultMatrixf(m);
      glCallList(s->display_list[s->cur_frame]);
      glPopMatrix();
    }
  }
}

Private
void renderLights(void)
{
  float light_diffuse1[4]={ 0.8, 0.8, 0.8, 1};
  /* float light_ambient1[4]={ 0.2, 0.2, 0.2, 1}; */
  float light_ambient1[4]={ 0.9, 0.9, 0.9, 0};
  float light_position1[4]={ 1, 0, -1, 0};
  float light_position2[4]={ -0.5, 0.7, -1, 0};
  float light_position3[4]={ -0.5, 0.7, -1, 0};

  glLightfv(GL_LIGHT0,GL_POSITION,light_position1);
  glLightfv(GL_LIGHT0,GL_DIFFUSE,light_diffuse1);
  glLightfv(GL_LIGHT0,GL_AMBIENT,light_ambient1);
  glLightfv(GL_LIGHT1,GL_POSITION,light_position2);
  glLightfv(GL_LIGHT2,GL_POSITION,light_position3);
}

Public
void Render(void)
{
  ZVContext *c = &zv_context;
  float m[16];

  updateTextures();

  glClearColor(0,0,0,0);
  glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);

  /* eye position */
  glMatrixMode(GL_MODELVIEW);
  M4_to_GL(m, &c->camera_pos);
  glLoadMatrixf(m);

  renderLights();	/* lights */
  render_walls();	/* walls */
  renderSolids();	/* solids */
}

Public
void RenderClose(void)
{
  closeTextureCache();
}

Public
void RenderSetWindow(int xsize, int ysize)
{
  trace(DBG_ZV, "setwindow %d %d", xsize, ysize);
  glXWaitX();
  glViewport(0, 0, xsize, ysize);
  ZLib_setCameraProjection(camera_projection.fovy,
                           camera_projection.near_z,
                           camera_projection.far_z);
}

#define SELECT_BUF_SIZE 1024

Public
Solid RenderGetSelection(int x, int y)
{
  ZVContext *c = &zv_context;
  Solid s;
  int hits, i;
  int xsize, ysize;
  unsigned int zmin, name;
  float xmin, xmax, ymin, ymax, ratio;
  int get_viewport[4];
  float m[16];
  unsigned int selectbuf[SELECT_BUF_SIZE], *psel;

  glGetIntegerv(GL_VIEWPORT, get_viewport);
  xsize=get_viewport[2];
  ysize=get_viewport[3];
  ymax = camera_projection.near_z * tan (camera_projection.fovy * M_PI_180);
  ymin = -ymax;
  ratio = (float) xsize / (float) ysize;
  xmax = ymax * ratio;
  xmin = -xmax;

  glMatrixMode(GL_PROJECTION);
  glPushMatrix(); /* sauvegarde de l'ancienne matrice */
  glLoadIdentity();

  /* on retourne l'axe des y */
  glFrustum(xmin+(x-0.5)*(xmax-xmin)/(xsize-1),
            xmin+(x+0.5)*(xmax-xmin)/(xsize-1),
            ymax-(y+0.5)*(ymax-ymin)/(ysize-1),
            ymax-(y-0.5)*(ymax-ymin)/(ysize-1),
            camera_projection.near_z,
            camera_projection.far_z);
  glTranslatef(0.0, 0.0, -0.5);

  /* selection mode */
  glSelectBuffer(SELECT_BUF_SIZE, selectbuf);
  glRenderMode(GL_SELECT);
  glInitNames();
  glPushName(0);

  /* eye position */
  glMatrixMode(GL_MODELVIEW);
  M4_to_GL(m, &c->camera_pos);
  glLoadMatrixf(m);

  /* draw the solids in the selection buffer */
  renderSolids();

  /* we put the normal mode back */
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  hits = glRenderMode(GL_RENDER);
  trace(DBG_ZV, "selection %d", hits);

  /* parcours du selection buffer, determination du hit le plus proche */
  psel = selectbuf;
  zmin = 0xffffffff;
  name = 0xffffffff;
  for (i=0; i < hits; i++) {
    if (psel[2] < zmin) {
      name = psel[3];
      zmin = psel[2];
    }
    psel += 3 + psel[0];
  }

  /* recherche du handle de solid correspondant */  
  for (s = c->first_solid; s != NULL; s = s->next)
    if (name == s->display_list[s->cur_frame])
      return s;
  return NULL;
}

