/*
 * WMMail - Window Maker Mail
 *
 * Copyright (c) 1996, 1997, 1998  Per Liden
 * Copyright (c) 1997, 1998  Bryan Chan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * mh.c: functions to handle MH-style mail folders
 *
 * $Id: mh.c,v 1.8 1999/01/16 05:42:42 bryan.chan Exp $
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <ctype.h>
#include <dirent.h>
#include <time.h>
#include <unistd.h>
#include "wmmail.h"
#include "wmutil.h"

#ifdef MH_SUPPORT

/* 
 * Supported key(s) in the Options dictionary:
 *
 *   Path
 *
 * This module contains routines to check an MH-style mail folder
 * during the periodic update of wmmail_status. We can classify a
 * given mail folder as being in one of the states listed below.
 *
 * The states and their corresponding effects on wmmail_status are:
 *
 *   - no folder or empty folder                NO_MAIL
 *   - no new mails                             OLD_MAIL
 *   - same number of new mails or fewer        no change
 *   - more new mails than last time            NEW_MAIL
 *
 * We assume that overriding automatically changes wmmail_status
 * to OLD_MAIL.
 *
 */

/* internal function */
static int count_mail(char *, int *, int *);


#define GET_ENTRY(x,y,z) { proplist_t sub_key = PLMakeString(z); \
                           y = PLGetDictionaryEntry(x, sub_key); \
                           PLRelease(sub_key); }

int MH_check(Mailbox *mailbox, int *beep, int *redraw, int *run)
{
  proplist_t  path;
  char       *mailbox_path;
  int         prev_status, 
              prev_new_mail_count;

  prev_status = mailbox->status;
  prev_new_mail_count = mailbox->new_mail_count;

  GET_ENTRY(mailbox->options, path, "Path");
  if (path == NULL)
  {
    croak("mailbox \"%s\" missing option \"Path\"; ignored", mailbox->name);
    return False;
  }
  else if (!PLIsString(path))
  {
    croak("mailbox \"%s\" has invalid path; ignored", mailbox->name);
    return False;
  }
  else
    mailbox_path = expand_path(PLGetString(path));

  if ( count_mail(mailbox_path,
                  &mailbox->total_mail_count,
                  &mailbox->new_mail_count)   )
  {
    if (mailbox->total_mail_count == 0)
    {
      /* there is no mail in the mailbox */
      mailbox->status = NO_MAIL;
    }
    else if (mailbox->new_mail_count == 0)
    {
      /* there are no new mails */
      mailbox->status = OLD_MAIL;
    }
    else if (mailbox->new_mail_count > prev_new_mail_count)
    {
      /* new mails have arrived! */
      if (mailbox->status == NEW_MAIL && always_new_mail_exec)
        *run |= True;
      else if (mailbox->status == NEW_MAIL)
        *run |= False;
      else
        *run |= True;

      *beep = True;

      mailbox->status = NEW_MAIL;
    }

    /* else no change */
  }
  else
  {
    /* no such mailbox */
    croak("error reading MH folder \"%s\"; ignored", mailbox->name);
    wfree(mailbox_path);
    return False;
  }

  *redraw |= (prev_status != mailbox->status);

  mailbox->last_update = time(NULL);

  wfree(mailbox_path);
  return True;
}


static int count_mail(char *path, int *total_mail_count, int *new_mail_count)
{
  struct dirent *entry;
  DIR           *folder;
  FILE          *seq;
  char           buf[512];
  int            new_mail = 0,
                 all_mail = 0,
                 first_unseen = 0,
                 last_unseen = 0;

  sprintf(buf, "%s/.mh_sequences", path);

  /* find the range of seen messages in the folder */
  if (seq = fopen(buf, "r"))
  {
    while (fgets(buf, 511, seq))
    {
      char *end;

      end = strrchr(buf, '\0');

      while (!isspace( *(--end) ))
	*end = '\0';

      /* FIXME: the name of the unseen sequence is not necessarily "unseen";
       *        we should get the sequence name from ~/.mh_profile
       */
      if (!strncmp(buf, "unseen:", 7))
      {
	char *tmp = buf + 7;

	while (*tmp)
	{
          while (*tmp && !isdigit(*tmp))
	    tmp++;

	  if ( !first_unseen || (atoi(tmp) < first_unseen) )
	    first_unseen = atoi(tmp);

	  if ( !last_unseen || (atoi(tmp) > last_unseen) )
	    last_unseen = atoi(tmp);

          while (*tmp && isdigit(*tmp))
	    tmp++;
	}
      }
    }
    fclose(seq);
  }
#if 0
  /* this section commented out because an MH folder does not necessarily
   * have a .mh_sequences; some mail readers delete the file for some reason
   */
  else
  {
    croak(".mh_sequences not found or unreadable in %s", path);
    return False;
  }
#endif

  if ( !(folder = opendir(path)) )
  {
    croak("cannot open directory %s", path);
    return False;
  }

  /* This is a lie; for simplicity, we assume that there are no unseen
   * messages between seen messages (for now)
   *
   * FIXME: can somebody familiar with MH help with this?
   */

  while ((entry = readdir(folder)) != NULL)
  {
    char *tmp = entry->d_name;
    int   is_message = True;

    while (*tmp)
      (void) (isdigit(*tmp++) || (is_message = False));

    if (is_message)
    {
      all_mail++;

      if ( (atoi(entry->d_name) >= first_unseen) &&
	   (atoi(entry->d_name) <= last_unseen) )
	new_mail++;
    }
  }

  closedir(folder);

  *total_mail_count = all_mail;
  *new_mail_count = new_mail;

  return True;
}

#endif
