package biss.jde;

import biss.StringLib;
import biss.awt.Awt;
import biss.awt.PostGraphics;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Toolkit;
import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.util.Enumeration;

/**
 * class to implement pretty printing of Java sources. Generates postscript files
 * of either a whole class or a text portion of it. Requires some pre-formatting
 * (done automatically or on demand by the CUBrowser). Classes and methods have to
 * start in column 0, fields have to be indented
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author J.H.Mehlitz
 */
public class PrettyPrinter
{
	PostGraphics Graph;
	JavaScanner Scan = new JavaScanner( null);
	CompileUnit Cu;
	int HeaderHeight;
	int FooterHeight;
	String CurFunc;
	String CurClass;
	boolean FindFunc;
	String LastFunc;
	final static double XCor = 1.03;
	static Font[] Fonts;
	static FontMetrics[] Metrics;
	final static int FNT_REG = 0;
	final static int FNT_KWD = 1;
	final static int FNT_LIT = 2;
	final static int FNT_CMT = 3;
	final static int FNT_HED = 4;
	String Text;
	final static int TabGrid = 20;
	final static int XOffs = 40;
	byte[] Buf = new byte[255];
	int BufLen = 0;
	byte[] QuoteBuf = new byte[255];
	int QuoteBufLen = 0;
	String PathName;
	boolean InCmt;
	final static int MinFuncLines = 10;

static {
	Fonts = new Font[5];
	Metrics = new FontMetrics[5];

	Toolkit tk = Awt.DefToolkit;

	Fonts[FNT_REG] = new Font( "Courier", Font.PLAIN, 12);
	Metrics[FNT_REG] = tk.getFontMetrics( Fonts[FNT_REG] );

	Fonts[FNT_KWD] = new Font( "Courier", Font.BOLD, 12);
	Metrics[FNT_KWD] = tk.getFontMetrics( Fonts[FNT_KWD] );

	Fonts[FNT_LIT] = new Font( "Courier", Font.PLAIN, 12);
	Metrics[FNT_LIT] = tk.getFontMetrics( Fonts[FNT_LIT] );

	Fonts[FNT_CMT] = new Font( "Courier", Font.ITALIC, 11);
	Metrics[FNT_CMT] = tk.getFontMetrics( Fonts[FNT_CMT] );

	Fonts[FNT_HED] = new Font( "Helvetica", Font.BOLD, 15);
	Metrics[FNT_HED] = tk.getFontMetrics( Fonts[FNT_HED] );
}

public PrettyPrinter ( CompileUnit cu) {
	this( cu.MainTypeName + ".ps");
	Cu = cu;
	HeaderHeight = (int)( 2.835 * 20 / PostGraphics.ScaleY);
	FooterHeight = (int)( 2.835 * 10 / PostGraphics.ScaleY);
}

protected PrettyPrinter ( String pathName) {
	PathName = pathName;
	try { Graph = new PostGraphics( pathName, null); }
	catch ( Throwable t) { System.out.println( t); }
}

public PrettyPrinter ( String text, String pathName) {
	this( pathName);
	Text = text;
}

boolean checkCmt ( int xPos, int yPos) {
	char nc = (char)Scan.C[Scan.I+1];
	if ( Scan.C[Scan.I] == '/') {
		if ( nc == '*')
			InCmt = true;
		if ((nc == '/') || (nc == '*') ){
			xPos = printBuf( xPos, yPos);
			printCmt( xPos, yPos);
			return true;
		}
	}
	else if ( Scan.C[Scan.I] == '*') {
		if ( nc == '/'){
			InCmt = false;
			xPos = printBuf( xPos, yPos);
			printCmt( xPos, yPos);
			return true;
		}
	}

	return false;
}

public boolean print () {
	if ( Text != null) {
		int len = Text.length();
		byte[] ba = new byte[len];
		Text.getBytes( 0, len, ba, 0);
		print( ba);
		return true;
	}
	try {
		ByteArrayOutputStream o = new ByteArrayOutputStream( 40000);
		PrintStream s = new PrintStream(o); 

		Cu.printOn( s);
		print( o.toByteArray() );

		s.close();
		o.close();
		return true;
	}
	catch ( Throwable t) {
		return false;
	}
}

/**
 * print the byte array 
 */
void print ( byte[] data) {
	int lIdx = 1;
	int lHeight	= Metrics[FNT_REG].getHeight() + 3;
	int yPos		= HeaderHeight + 2 * lHeight;
	int mfy			= MinFuncLines * lHeight;
	byte c;

	FindFunc = true;
	Scan.reset( data);

	for ( int idx=0; idx<data.length; idx++) {
		c = data[idx];
		if ( c == '\n') {
			int dy = (Graph.MaxHeight - FooterHeight) - yPos;
			char sc = Scan.character();

			if ( ( dy < lHeight) ||		// page complete
			     ((dy < mfy) &&				// new func
			 (sc != '\t') && (sc != ' ') &&
			 (sc != '}') && (sc != '\n') ) ) {
				printHeader();
				printFooter();
				yPos = HeaderHeight + 2 * lHeight;
				Graph.newPage();
				CurFunc = LastFunc;
				FindFunc = true;
			}
			Scan.C[idx] = Scan.EOT;

			printLine( lIdx++, yPos);
			Scan.I = Scan.I0 = idx+1;
			yPos += lHeight;
		}
	}

	printHeader();	
	printFooter();

	Graph.endDoc();
	Graph.dispose();
}

int printBuf ( int xPos, int yPos) {
	if ( BufLen == 0 )
		return xPos;

	Graph.setFont( Fonts[FNT_REG]);
	Graph.drawBytes( quoteBuf( Buf, 0, BufLen), 0, QuoteBufLen, xPos, yPos);

	int np = (xPos + (int)(XCor * Metrics[FNT_REG].bytesWidth( Buf, 0, BufLen )));
	BufLen = 0;
	return np;
}

int printCmt ( int xPos, int yPos) {
	int len = 0;
	for ( int i=Scan.I; Scan.C[i] != Scan.EOT ;i++, len++);

	if ( len == 0 )
		return xPos;

	Graph.setFont( Fonts[FNT_CMT]);
	Graph.drawBytes( quoteBuf( Scan.C, Scan.I, Scan.I+len), 0, QuoteBufLen, xPos, yPos);

	return (xPos + (int)(XCor * Metrics[FNT_CMT].bytesWidth( Scan.C, Scan.I, len )));
}

boolean printFooter () {

	if ( FooterHeight == 0)
		return false;

	int fh = Metrics[FNT_REG].getHeight();

	String s = "Page " + Graph.CurrentPage;

	int xPos = (Graph.MaxWidth - Metrics[FNT_REG].stringWidth( s)) / 2;
	int yPos = Graph.MaxHeight - (FooterHeight - fh) / 2;

	Graph.drawLine( 0, Graph.MaxHeight - FooterHeight,
	                Graph.MaxWidth, Graph.MaxHeight - FooterHeight);
	Graph.setFont( Fonts[FNT_REG] );
	Graph.drawString ( s, xPos, yPos);

	return true;
}

boolean printHeader () {

	if ( HeaderHeight == 0)
		return false;

	int yPos = HeaderHeight - (HeaderHeight - Metrics[FNT_HED].getHeight()) / 2;
	int ytPos = yPos - Metrics[FNT_HED].getHeight();

	Graph.setFont( Fonts[FNT_REG] );
	Graph.drawString( "CompileUnit:", 0, ytPos);
	Graph.drawString( "Class:", Graph.MaxWidth / 3, ytPos);
	Graph.drawString( "Func:", 2 * Graph.MaxWidth / 3, ytPos);

	Graph.setFont( Fonts[FNT_HED] );
	Graph.drawString( Cu.MainTypeName + ".java" , 0, yPos);
	Graph.drawString( CurClass, Graph.MaxWidth / 3, yPos);
	Graph.drawString( CurFunc, 2 * Graph.MaxWidth / 3, yPos);

	Graph.drawLine( 0, HeaderHeight, Graph.MaxWidth, HeaderHeight);

	return true;
}

int printLine ( int lNum, int yPos) {
	boolean fTag	= true;
	boolean getCls= false;
	String lTok 	= "";
	byte c		= Scan.C[Scan.I0];
	int  xPos	= XOffs;
	int tok		= 1;

	BufLen = 0;

	if ( (c == ' ') || (c == '\t') )
		fTag = false;

	Graph.setFont( Fonts[FNT_REG] );
	Graph.drawString( Integer.toString( lNum), 0, yPos);

	while ( tok != 0 ) {
		while ( ( c = Scan.C[Scan.I]) <= ' ') {
			if ( c == '\t' ){
				xPos = printBuf( xPos, yPos);
				xPos = ((xPos + TabGrid) / TabGrid) * TabGrid;
			}
			else if ( c == ' ')
				Buf[BufLen++] = ' ';
			else if ( c == Scan.EOT )
				return printBuf( xPos, yPos);
			Scan.I++;
		}

		if (( c == '/') || (c == '*') ) {
			if ( checkCmt( xPos, yPos))
				return xPos;
		}
		if ( InCmt) {
			xPos = printBuf( xPos, yPos);
			return printCmt( xPos, yPos);
		}

		if ( (tok = Scan.nextToken() ) != 0 ) {
			if ( Scan.isKeyword(tok) ){
				if ( (tok == Scan.CLASS) || (tok == Scan.INTERFACE) )
					getCls = true;
				xPos = printBuf( xPos, yPos);
				xPos = printScan( Scan.I0, Scan.I, xPos, yPos, FNT_KWD);
			}
			else if ( tok == Scan.SCONST || tok == Scan.CCONST ) {
				xPos = printBuf( xPos, yPos);
				xPos = printScan( Scan.I0, Scan.I, xPos, yPos, FNT_LIT);
			}
			else if ( Scan.I > Scan.I0) {
				if ( fTag) {
					if ( Scan.character() == '(' ){
						LastFunc = lTok;
						if ( FindFunc) {	// first in page
							CurFunc = new String( LastFunc);
							FindFunc = false;
						}
						Graph.drawLine( XOffs, yPos+3, Graph.MaxWidth - XOffs, yPos+3);
					}
					else if ( getCls ) {
						CurClass = Scan.string();
						CurFunc = "init";
						FindFunc = true;
						getCls = false;
					}
					lTok = Scan.string();
				}
				for ( int i=Scan.I0; i<Scan.I; i++)
					Buf[BufLen++] = Scan.C[i];
			}
		}

		c = Scan.C[Scan.I];
		if ( (c == '/') || (c == '*') ) {
			if ( checkCmt( xPos, yPos))
				return xPos;
		}
	}
	return xPos;
}

int printScan ( int start, int end, int xPos, int yPos, int type) {
	if ( start == end )
		return xPos;

	Graph.setFont( Fonts[type]);
	Graph.drawBytes( quoteBuf( Scan.C, start, end), 0, QuoteBufLen, xPos, yPos);

	return (xPos + (int)(XCor * Metrics[type].bytesWidth( Scan.C, start, end-start)));
}

byte[] quoteBuf ( byte[] buf, int start, int end) {
	QuoteBufLen = 0;
	for ( int i=start; i<end; i++) {
		switch( buf[i]) {
		case '\\':
		case '(' :
		case ')' :
			QuoteBuf[QuoteBufLen++] = '\\';
		}
		QuoteBuf[QuoteBufLen++] = buf[i];
	}
	return QuoteBuf;
}
}
