/*
 * Snes9x - Portable Super Nintendo Entertainment System (TM) emulator.
 *
 * (c) Copyright 1996, 1997, 1998, 1999 Gary Henderson (gary@daniver.demon.co.uk) and
 *                                      Jerremy Koot (jkoot@snes9x.com)
 *
 * Super FX C emulator code 
 * (c) Copyright 1997, 1998, 1999 Ivar (Ivar@snes9x.com) and
 *                                Gary Henderson.
 * Super FX assembler emulator code (c) Copyright 1998 zsKnight and _Demo_.
 *
 * DSP1 emulator code (c) Copyright 1998 Ivar, _Demo_ and Gary Henderson.
 * DOS port code contains the works of other authors. See headers in
 * individual files.
 *
 * Snes9x homepage: www.snes9x.com
 *
 * Permission to use, copy, modify and distribute Snes9x in both binary and
 * source form, for non-commercial purposes, is hereby granted without fee,
 * providing that this license information and copyright notice appear with
 * all copies and any derived work.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event shall the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Snes9x is freeware for PERSONAL USE only. Commercial users should
 * seek permission of the copyright holders first. Commercial use includes
 * charging money for Snes9x or software derived from Snes9x.
 *
 * The copyright holders request that bug fixes and improvements to the code
 * should be forwarded to them so everyone can benefit from the modifications
 * in future versions.
 *
 * Super NES and Super Nintendo Entertainment System are trademarks of
 * Nintendo Co., Limited and its subsidiary companies.
 */

#include <string.h>
#include <ctype.h>

#ifdef __linux
#include <unistd.h>
#endif

#ifndef NO_INLINE_SET_GET
#define NO_INLINE_SET_GET
#endif

#include "snes9x.h"
#include "memmap.h"
#include "cpuexec.h"
#include "ppu.h"
#include "display.h"
#include "cheats.h"
#include "apu.h"
#include "sa1.h"

#ifndef ZSNES_FX
#include "fxemu.h"
extern struct FxInit_s SuperFX;
#else
START_EXTERN_C
extern uint8 *SFXPlotTable;
END_EXTERN_C
#endif

extern char *rom_filename;

bool8 CMemory::AllASCII (uint8 *b, int size)
{
    for (int i = 0; i < size; i++)
    {
	if (b[i] < 32 || b[i] > 126)
	    return (FALSE);
    }
    return (TRUE);
}

int CMemory::ScoreHiROM (bool8 skip_header)
{
    int score = 0;
    int o = skip_header ? 0xff00 + 0x200 : 0xff00;

    if ((Memory.ROM [o + 0xdc] + (Memory.ROM [o + 0xdd] << 8) +
	 Memory.ROM [o + 0xde] + (Memory.ROM [o + 0xdf] << 8)) == 0xffff)
	score += 2;

    if (Memory.ROM [o + 0xda] == 0x33)
	score += 2;
    if ((Memory.ROM [o + 0xd5] & 0xf) < 4)
	score += 2;
    if (!(Memory.ROM [o + 0xfd] & 0x80))
	score -= 4;
    if (CalculatedSize > 1024 * 1024 * 3)
	score += 4;
    if ((1 << (Memory.ROM [o + 0xd7] - 7)) > 48)
	score -= 1;
    if (!AllASCII (&Memory.ROM [o + 0xb0], 6))
	score -= 1;
    if (!AllASCII (&Memory.ROM [o + 0xc0], ROM_NAME_LEN - 1))
	score -= 1;

    return (score);
}

int CMemory::ScoreLoROM (bool8 skip_header)
{
    int score = 0;
    int o = skip_header ? 0x7f00 + 0x200 : 0x7f00;

    if ((Memory.ROM [o + 0xdc] + (Memory.ROM [o + 0xdd] << 8) +
	 Memory.ROM [o + 0xde] + (Memory.ROM [o + 0xdf] << 8)) == 0xffff)
	score += 2;

    if (Memory.ROM [o + 0xda] == 0x33)
	score += 2;
    if ((Memory.ROM [o + 0xd5] & 0xf) < 4)
	score += 2;
    if (CalculatedSize <= 1024 * 1024 * 16)
	score += 2;
    if (!(Memory.ROM [o + 0xfd] & 0x80))
	score -= 4;
    if ((1 << (Memory.ROM [o + 0xd7] - 7)) > 48)
	score -= 1;
    if (!AllASCII (&Memory.ROM [o + 0xb0], 6))
	score -= 1;
    if (!AllASCII (&Memory.ROM [o + 0xc0], ROM_NAME_LEN - 1))
	score -= 1;

    return (score);
}
	
char *CMemory::Safe (const char *s)
{
    static char *safe = NULL;
    static int safe_len = 0;

    int len = strlen (s);
    if (!safe || len + 1 > safe_len)
    {
	delete safe;
	safe = new char [safe_len = len + 1];
    }

    for (int i = 0; i < len; i++)
    {
	if (s [i] >= 32 && s [i] < 127)
	    safe [i] = s[i];
	else
	    safe [i] = '?';
    }
    safe [len] = 0;
    return (safe);
}

/**********************************************************************************************/
/* Init()                                                                                     */
/* This function allocates all the memory needed by the emulator                              */
/**********************************************************************************************/
bool8 CMemory::Init ()
{
    RAM	    = new uint8 [0x20000];
    SRAM    = new uint8 [0x10000];
    VRAM    = new uint8 [0x10000];
    ROM     = new uint8 [MAX_ROM_SIZE + 0x200 + 0x8000];
    FillRAM = NULL;
#ifdef ZSNES_FX
    SFXPlotTable = new uint8 [0x100000];
#endif

    IPPU.TileCache [TILE_2BIT] = new uint8 [MAX_2BIT_TILES * 128];
    IPPU.TileCache [TILE_4BIT] = new uint8 [MAX_4BIT_TILES * 128];
    IPPU.TileCache [TILE_8BIT] = new uint8 [MAX_8BIT_TILES * 128];
    
    IPPU.TileCached [TILE_2BIT] = new uint8 [MAX_2BIT_TILES];
    IPPU.TileCached [TILE_4BIT] = new uint8 [MAX_4BIT_TILES];
    IPPU.TileCached [TILE_8BIT] = new uint8 [MAX_8BIT_TILES];
    
    if( !RAM || !SRAM || !VRAM || !ROM || !IPPU.TileCache [TILE_2BIT] ||
	!IPPU.TileCache [TILE_4BIT] || !IPPU.TileCache [TILE_8BIT] ||
	!IPPU.TileCached [TILE_2BIT] || !IPPU.TileCached [TILE_4BIT] ||
	!IPPU.TileCached [TILE_8BIT])
	return (FALSE);
	
    // FillRAM uses first 32K of ROM image area, otherwise space just
    // wasted. Might be read by the SuperFX code.

    FillRAM = ROM;

    // Add 0x8000 to ROM image pointer to stop SuperFX code accessing
    // unallocated memory (can cause crash on some ports).
    ROM += 0x8000;
    ::ROM = ROM;
    ::SRAM = SRAM;
    ::RegRAM = FillRAM;

#ifndef ZSNES_FX
    SuperFX.pvRegisters = &Memory.FillRAM [0x3000];
    SuperFX.nRamBanks = 1;
    SuperFX.pvRam = ::SRAM;
    SuperFX.nRomBanks = (2 * 1024 * 1024) / (32 * 1024);
    SuperFX.pvRom = (uint8 *) ROM;
#endif

    memset (SRAM, 0xaa, 0x10000);
    ZeroMemory (IPPU.TileCached [TILE_2BIT], MAX_2BIT_TILES);
    ZeroMemory (IPPU.TileCached [TILE_4BIT], MAX_4BIT_TILES);
    ZeroMemory (IPPU.TileCached [TILE_8BIT], MAX_8BIT_TILES);
    
    return (TRUE);
}

void CMemory::Deinit ()
{
    ROM -= 0x8000;

    delete RAM;
    delete SRAM;
    delete VRAM;
    delete ROM;

#ifdef ZSNES_FX
    delete SFXPlotTable;
#endif
    delete IPPU.TileCache [TILE_2BIT];
    delete IPPU.TileCache [TILE_4BIT];
    delete IPPU.TileCache [TILE_8BIT];

    delete IPPU.TileCached [TILE_2BIT];
    delete IPPU.TileCached [TILE_4BIT];
    delete IPPU.TileCached [TILE_8BIT];
}

/**********************************************************************************************/
/* LoadROM()                                                                                  */
/* This function loads a Snes-Backup image                                                    */
/**********************************************************************************************/
bool8 CMemory::LoadROM (const char *filename)
{
    unsigned long FileSize = 0;
    int retry_count = 0;
    STREAM ROMFile;
    bool8 Interleaved = FALSE;
    bool8 Tales = FALSE;
    char dir [_MAX_DIR];
    char drive [_MAX_DRIVE];
    char name [_MAX_FNAME];
    char ext [_MAX_EXT];
    char fname [_MAX_PATH];
    int i;

    CalculatedSize = 0;

again:
    _splitpath (filename, drive, dir, name, ext);
    _makepath (fname, drive, dir, name, ext);

#ifdef _WIN32
    memmove (&ext [0], &ext[1], 4);
#endif

    unsigned long TotalFileSize = 0;

#ifdef UNZIP_SUPPORT
    if (strcasecmp (ext, "zip") == 0)
    {
	bool8 LoadZip (const char *, unsigned long *, int *);

	if (!LoadZip (fname, &TotalFileSize, &HeaderCount))
	    return (FALSE);

	strcpy (Memory.ROMFilename, fname);
    }
    else
#endif
    {
	if ((ROMFile = OPEN_STREAM (fname, "rb")) == NULL)
	    return (FALSE);

	strcpy (Memory.ROMFilename, fname);

	HeaderCount = 0;
	uint8 *ptr = ROM;
	bool8 more = FALSE;

	do
	{
	    FileSize = READ_STREAM (ptr, MAX_ROM_SIZE + 0x200 - (ptr - ROM), ROMFile);
	    CLOSE_STREAM (ROMFile);
	    int calc_size = FileSize / 0x2000;
	    calc_size *= 0x2000;

	    if ((FileSize - calc_size == 512 && !Settings.ForceNoHeader) ||
		Settings.ForceHeader)
	    {
		memmove (ptr, ptr + 512, calc_size);
		HeaderCount++;
		FileSize -= 512;
	    }
	    ptr += FileSize;
	    TotalFileSize += FileSize;

	    int len;
	    if (ptr - ROM < MAX_ROM_SIZE + 0x200 &&
		(isdigit (ext [0]) && ext [1] == 0 && ext [0] < '9'))
	    {
		more = TRUE;
		ext [0]++;
#ifdef _WIN32
                memmove (&ext [1], &ext [0], 4);
                ext [0] = '.';
#endif
		_makepath (fname, drive, dir, name, ext);
	    }
	    else
	    if (ptr - ROM < MAX_ROM_SIZE + 0x200 &&
		(((len = strlen (name)) == 7 || len == 8) &&
		 strncasecmp (name, "sf", 2) == 0 &&
		 isdigit (name [2]) && isdigit (name [3]) && isdigit (name [4]) &&
		 isdigit (name [5]) && isalpha (name [len - 1])))
	    {
		more = TRUE;
		name [len - 1]++;
#ifdef _WIN32
                memmove (&ext [1], &ext [0], 4);
                ext [0] = '.';
#endif
		_makepath (fname, drive, dir, name, ext);
	    }
	    else
		more = FALSE;
	} while (more && (ROMFile = OPEN_STREAM (fname, "rb")) != NULL);
    }

    CalculatedSize = (TotalFileSize / 0x2000) * 0x2000;

    if (HeaderCount == 0)
	S9xMessage (S9X_INFO, S9X_HEADERS_INFO, "No ROM file header found.");
    else
    {
	if (HeaderCount == 1)
	    S9xMessage (S9X_INFO, S9X_HEADERS_INFO,
			"Found ROM file header (and ignored it).");
	else
	    S9xMessage (S9X_INFO, S9X_HEADERS_INFO,
			"Found multiple ROM file headers (and ignored them).");
    }

    ZeroMemory (ROM + TotalFileSize, MAX_ROM_SIZE - TotalFileSize);
    int orig_hi_score, orig_lo_score;
    int hi_score, lo_score;

    orig_hi_score = hi_score = ScoreHiROM (FALSE);
    orig_lo_score = lo_score = ScoreLoROM (FALSE);

    if (HeaderCount == 0 &&
	((hi_score > lo_score && ScoreHiROM (TRUE) > hi_score) ||
	 (hi_score <= lo_score && ScoreLoROM (TRUE) > lo_score)))
    {
	S9xMessage (S9X_INFO, S9X_HEADER_WARNING, 
		    "Try specifying the -hd command line option if the game doesn't work\n");
    }

    Interleaved = Settings.ForceInterleaved || Settings.ForceInterleaved2;
    if (Settings.ForceLoROM || (!Settings.ForceHiROM && lo_score >= hi_score))
    {
	Memory.LoROM = TRUE;
	Memory.HiROM = FALSE;

	// Ignore map type byte if not 0x2x or 0x3x
	if ((ROM [0x7fd5] & 0xf0) == 0x20 || (ROM [0x7fd5] & 0xf0) == 0x30)
	{
	    switch (ROM [0x7fd5] & 0xf)
	    {
	    case 1:
		Interleaved = TRUE;
		break;
	    case 2:
		if (!Settings.ForceLoROM &&
		    strncmp ((char *) &ROM [0x7fc0], "SUPER FORMATION SOCCE", 21) != 0)
		{
		    Memory.LoROM = FALSE;
		    Memory.HiROM = TRUE;
		}
		break;
	    case 5:
		Interleaved = TRUE;
		Tales = TRUE;
		break;
	    }
	}
    }
    else
    {
	if ((ROM [0xffd5] & 0xf0) == 0x20 || (ROM [0xffd5] & 0xf0) == 0x30)
	{
	    switch (ROM [0xffd5] & 0xf)
	    {
	    case 0:
	    case 3:
		Interleaved = TRUE;
		break;
	    }
	}
	Memory.LoROM = FALSE;
	Memory.HiROM = TRUE;
    }

    if (!Settings.ForceNotInterleaved && Interleaved)
    {
	S9xMessage (S9X_INFO, S9X_ROM_INTERLEAVED_INFO,
		    "ROM image is in interleaved format - converting...");

	int nblocks = CalculatedSize >> 15;
	int step = 64;

	while (nblocks <= step)
	    step >>= 1;
	    
	nblocks = step;
	uint8 blocks [256];

	if (Tales)
	{
	    nblocks = 0x60;
	    for (i = 0; i < 0x40; i += 2)
	    {
		blocks [i + 0] = (i >> 1) + 0x20;
		blocks [i + 1] = (i >> 1);
	    }
	    for (i = 0; i < 0x80; i += 2)
	    {
		blocks [i + 0x40] = (i >> 1) + 0x80;
		blocks [i + 0x41] = (i >> 1) + 0x40;
	    }
	    Memory.LoROM = FALSE;
	    Memory.HiROM = TRUE;
	}
	else
	if (Settings.ForceInterleaved2)
	{
	    for (i = 0; i < nblocks * 2; i++)
	    {
		blocks [i] = (i & ~0x1e) | ((i & 2) << 2) | ((i & 4) << 2) |
			     ((i & 8) >> 2) | ((i & 16) >> 2);
	    }
	}
	else
	{
	    bool8 t = Memory.LoROM;

	    Memory.LoROM = Memory.HiROM;
	    Memory.HiROM = t;

	    for (i = 0; i < nblocks; i++)
	    {
		blocks [i * 2] = i + nblocks;
		blocks [i * 2 + 1] = i;
	    }
	}

	uint8 *tmp = new uint8 [0x8000];
	for (i = 0; i < nblocks * 2; i++)
	{
	    for (int j = i; j < nblocks * 2; j++)
	    {
		if (blocks [j] == i)
		{
		    memmove (tmp, &ROM [blocks [j] * 0x8000], 0x8000);
		    memmove (&ROM [blocks [j] * 0x8000], 
			     &ROM [blocks [i] * 0x8000], 0x8000);
		    memmove (&ROM [blocks [i] * 0x8000], tmp, 0x8000);
		    uint8 b = blocks [j];
		    blocks [j] = blocks [i];
		    blocks [i] = b;
		    break;
		}
	    }
	}
	delete tmp;
	hi_score = ScoreHiROM (FALSE);
	lo_score = ScoreLoROM (FALSE);

	if ((Memory.HiROM &&
	     (lo_score >= hi_score || orig_lo_score > hi_score ||
	      hi_score < 0)) ||
	    (Memory.LoROM && 
	     (hi_score > lo_score || orig_hi_score > lo_score ||
	      lo_score < 0)))
	{

	    if (retry_count == 0)
	    {
		S9xMessage (S9X_INFO, S9X_ROM_CONFUSING_FORMAT_INFO,
			    "ROM lied about its type! Trying again.");
		Settings.ForceNotInterleaved = TRUE;
		Settings.ForceInterleaved = FALSE;
		retry_count++;
		goto again;
	    }
	}
    }

#ifndef ZSNES_FX
    SuperFX.nRomBanks = CalculatedSize >> 15;
#endif
    Settings.MultiPlayer5Master = Settings.MultiPlayer5;
    Settings.MouseMaster = Settings.Mouse;
    Settings.SuperScopeMaster = Settings.SuperScope;
    Settings.DSP1Master = Settings.ForceDSP1;
    Settings.SuperFX = FALSE;

    ZeroMemory (BlockIsRAM, MEMMAP_NUM_BLOCKS);
    ZeroMemory (BlockIsROM, MEMMAP_NUM_BLOCKS);

    ::SRAM = SRAM;

    if (Memory.HiROM)
    {
	Memory.SRAMSize = ROM [0xffd8];
	strncpy (ROMName, (char *) &ROM[0xffc0], ROM_NAME_LEN - 1);
	ROMSpeed = ROM [0xffd5];
	ROMType = ROM [0xffd6];
	ROMSize = ROM [0xffd7];
	ROMChecksum = ROM [0xffde] + (ROM [0xffdf] << 8);
	ROMComplementChecksum = ROM [0xffdc] + (ROM [0xffdd] << 8);

	// Try to auto-detect the DSP1 chip
	if (!Settings.ForceNoDSP1 &&
	    (ROMType & 0xf) >= 3 && (ROMType & 0xf0) == 0)
	    Settings.DSP1Master = TRUE;

	if ((ROMSpeed & ~0x10) == 0x25)
	{
	    TalesROMMap ();
	}
	else if ((ROMSpeed & ~0x10) == 0x22 && 
		 strncmp (ROMName, "Super Street Fighter", 20) != 0)
	    AlphaROMMap ();
	else
	    HiROMMap ();
    }
    else
    {
	Memory.HiROM = FALSE;
	Memory.SRAMSize = ROM [0x7fd8];
	ROMSpeed = ROM [0x7fd5];
	ROMType = ROM [0x7fd6];
	ROMSize = ROM [0x7fd7];
	ROMChecksum = ROM [0x7fde] + (ROM [0x7fdf] << 8);
	ROMComplementChecksum = ROM [0x7fdc] + (ROM [0x7fdd] << 8);

	strncpy (ROMName, (char *) &ROM[0x7fc0], ROM_NAME_LEN - 1);
	Settings.SuperFX = Settings.ForceSuperFX;

	if ((ROMType & 0xf0) == 0x10)
	    Settings.SuperFX = !Settings.ForceNoSuperFX;

	Settings.SA1 = FALSE;

	// Try to auto-detect the DSP1 chip
	if (!Settings.ForceNoDSP1 &&
	    (ROMType & 0xf) >= 3 && (ROMType & 0xf0) == 0)
	    Settings.DSP1Master = TRUE;

	if (Settings.SuperFX)
	{
	    //::SRAM = ROM + 1024 * 1024 * 4;
	    SuperFXROMMap ();
	    Settings.MultiPlayer5Master = FALSE;
	    Settings.MouseMaster = FALSE;
	    Settings.SuperScopeMaster = FALSE;
	    Settings.DSP1Master = FALSE;
	    Settings.SA1 = FALSE;
	}
	else
	if (Settings.ForceSA1 ||
	    (!Settings.ForceNoSA1 && (ROMSpeed & ~0x10) == 0x23 && 
	     (ROMType & 0xf) > 3 && (ROMType & 0xf0) == 0x30))
	{
	    Settings.SA1 = TRUE;
	    Settings.MultiPlayer5Master = FALSE;
	    Settings.MouseMaster = FALSE;
	    Settings.SuperScopeMaster = FALSE;
	    Settings.DSP1Master = FALSE;
	    SA1ROMMap ();
	}
	else
	if ((ROMSpeed & ~0x10) == 0x25)
	    TalesROMMap ();
	else
	if (strncmp ((char *) &Memory.ROM [0x7fc0], "SOUND NOVEL-TCOOL",17) == 0) 
	{
	    LoROM24MBSMap ();
	    Settings.DSP1Master = FALSE;
	}
	else
	if (strncmp ((char *) &Memory.ROM [0x7fc0], "DERBY STALLION 96",17) == 0)
	{
	    LoROM24MBSMap ();
	    Settings.DSP1Master = FALSE;
	}
	else
	    LoROMMap ();
    }

    int power2 = 0;
    int size = TotalFileSize;

    while (size >>= 1)
	power2++;

    size = 1 << power2;
    int remainder = TotalFileSize - size;

    int sum1 = 0;
    int sum2 = 0;

    for (i = 0; i < size; i++)
	sum1 += ROM [i];

    for (i = 0; i < remainder; i++)
	sum2 += ROM [size + i];

    if (remainder)
	sum1 += sum2 * (size / remainder);

    sum1 &= 0xffff;

    if (Settings.ForceNTSC)
	Settings.PAL = FALSE;
    else
    if (Settings.ForcePAL)
	Settings.PAL = TRUE;
    else
    if (Memory.HiROM)
	// Country code
	Settings.PAL = ROM [0xffd9] >= 2;
    else
	Settings.PAL = ROM [0x7fd9] >= 2;
    
    if (Settings.PAL)
    {
	Settings.FrameTime = Settings.FrameTimePAL;
	Memory.ROMFramesPerSecond = 50;
    }
    else
    {
	Settings.FrameTime = Settings.FrameTimeNTSC;
	Memory.ROMFramesPerSecond = 60;
    }
	
    ROMName[ROM_NAME_LEN - 1] = 0;
    if (strlen (ROMName))
    {
	char *p = ROMName + strlen (ROMName) - 1;

	while (p > ROMName && *(p - 1) == ' ')
	    p--;
	*p = 0;
    }

    if (Settings.SuperFX)
    {
	SRAMMask = 0xffff;
	Memory.SRAMSize = 16;
    }
    else
    {
	SRAMMask = Memory.SRAMSize ?
		    ((1 << (Memory.SRAMSize + 3)) * 128) - 1 : 0;
    }

    IAPU.OneCycle = ONE_APU_CYCLE;

    ApplyROMFixes ();
    sprintf (ROMName, "%s", Safe (ROMName));

    sprintf (String, "\"%s\" [%s] %s, %s, Type: %s, Mode: %s, TV: %s, S-RAM: %s",
	     ROMName,
	     (ROMChecksum + ROMComplementChecksum != 0xffff ||
	      ROMChecksum != sum1) ? "bad checksum" : "checksum ok",
	     MapType (),
	     Size (),
	     KartContents (),
	     MapMode (),
	     TVStandard (),
	     StaticRAMSize ());

    S9xMessage (S9X_INFO, S9X_ROM_INFO, String);
	
    S9xApplyCheats ();
    S9xReset ();
    return TRUE;
}

bool8 CMemory::LoadSRAM (const char *filename)
{
    int size = Memory.SRAMSize ?
	       (1 << (Memory.SRAMSize + 3)) * 128 : 0;

    memset (::SRAM, 0xaa, 0x10000);

    if (size > 0x10000)
	size = 0x10000;
    if (size)
    {
	FILE *file;
	if (file = fopen (filename, "rb"))
	{
	    int len = fread ((char*) ::SRAM, 1, 0x10000, file);
	    fclose (file);
	    if (len - size == 512)
	    {
		// S-RAM file has a header - remove it
		memmove (::SRAM, ::SRAM + 512, size);
	    }
	    return (TRUE);
	}
	return (FALSE);
    }
    return (TRUE);
}

bool8 CMemory::SaveSRAM (const char *filename)
{
    int size = Memory.SRAMSize ?
	       (1 << (Memory.SRAMSize + 3)) * 128 : 0;
    if (size > 0x10000)
	size = 0x10000;

    if (size && *Memory.ROMFilename)
    {
	FILE *file;
	if (file = fopen (filename, "wb"))
	{
	    fwrite ((char *) ::SRAM, size, 1, file);
	    fclose (file);
#if defined(__linux)
	    chown (filename, getuid (), getgid ());
#endif
	    return (TRUE);
	}
	return (FALSE);
    }
    return (TRUE);
}

void CMemory::FixROMSpeed ()
{
    int c;

    for (c = 0x800; c < 0x1000; c++)
    {
	if (BlockIsROM [c])
	    MemorySpeed [c] = CPU.FastROMSpeed;
    }
}

void CMemory::WriteProtectROM ()
{
    memmove ((void *) WriteMap, (void *) Map, sizeof (Map));
    for (int c = 0; c < 0x1000; c++)
    {
	if (BlockIsROM [c])
	    WriteMap [c] = (uint8 *) MAP_NONE;
    }
}

void CMemory::MapRAM ()
{
    int c;

    // Banks 7e->7f, RAM
    for (c = 0; c < 16; c++)
    {
	Map [c + 0x7e0] = RAM;
	Map [c + 0x7f0] = RAM + 0x10000;
	BlockIsRAM [c + 0x7e0] = TRUE;
	BlockIsRAM [c + 0x7f0] = TRUE;
	BlockIsROM [c + 0x7e0] = FALSE;
	BlockIsROM [c + 0x7f0] = FALSE;
    }

    // Banks 70->77, S-RAM
    for (c = 0; c < 0x80; c++)
    {
	Map [c + 0x700] = (uint8 *) MAP_LOROM_SRAM;
	BlockIsRAM [c + 0x700] = TRUE;
	BlockIsROM [c + 0x700] = FALSE;
    }
}

void CMemory::LoROMMap ()
{
    int c;
    int i;

    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x400; c += 16)
    {
	Map [c + 0] = Map [c + 0x800] = RAM;
	Map [c + 1] = Map [c + 0x801] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x800] = TRUE;
	BlockIsRAM [c + 1] = BlockIsRAM [c + 0x801] = TRUE;

	Map [c + 2] = Map [c + 0x802] = (uint8 *) MAP_PPU;
	Map [c + 3] = Map [c + 0x803] = (uint8 *) MAP_PPU;
	Map [c + 4] = Map [c + 0x804] = (uint8 *) MAP_CPU;
	Map [c + 5] = Map [c + 0x805] = (uint8 *) MAP_CPU;
	if (Settings.DSP1Master)
	{
	    Map [c + 6] = Map [c + 0x806] = (uint8 *) MAP_DSP;
	    Map [c + 7] = Map [c + 0x807] = (uint8 *) MAP_DSP;
	}
	else
	{
	    Map [c + 6] = Map [c + 0x806] = (uint8 *) MAP_NONE;
	    Map [c + 7] = Map [c + 0x807] = (uint8 *) MAP_NONE;
	}
	for (i = c + 8; i < c + 16; i++)
	{
	    Map [i] = Map [i + 0x800] = &ROM [c << 11] - 0x8000;
	    BlockIsROM [i] = BlockIsROM [i + 0x800] = TRUE;
	}

	for (i = c; i < c + 16; i++)
	{
	    int ppu = i & 15;
	    
	    MemorySpeed [i] = 
		MemorySpeed [i + 0x800] = ppu >= 2 && ppu <= 3 ? ONE_CYCLE : 8;
	}
    }

    if (Settings.DSP1Master)
    {
	// Banks 30->3f and b0->bf
	for (c = 0x300; c < 0x400; c += 16)
	{
	    for (i = c + 8; i < c + 16; i++)
	    {
		Map [i] = Map [i + 0x800] = (uint8 *) MAP_DSP;
		BlockIsROM [i] = BlockIsROM [i + 0x800] = FALSE;
	    }
	}
    }

    // Banks 40->7f and c0->ff
    for (c = 0; c < 0x400; c += 16)
    {
	for (i = c; i < c + 8; i++)
	    Map [i + 0x400] = Map [i + 0xc00] = &ROM [(c << 11) + 0x200000];

	for (i = c + 8; i < c + 16; i++)
	    Map [i + 0x400] = Map [i + 0xc00] = &ROM [(c << 11) + 0x200000 - 0x8000];

	for (i = c; i < c + 16; i++)
	{
	    MemorySpeed [i + 0x400] = MemorySpeed [i + 0xc00] = 8;
	    BlockIsROM [i + 0x400] = BlockIsROM [i + 0xc00] = TRUE;
	}
    }

    if (Settings.DSP1Master)
    {
	for (c = 0; c < 0x100; c++)
	{
	    Map [c + 0xe00] = (uint8 *) MAP_DSP;
	    MemorySpeed [c + 0xe00] = 8;
	    BlockIsROM [c + 0xe00] = FALSE;
	}
    }
    MapRAM ();
    WriteProtectROM ();
}

void CMemory::HiROMMap ()
{
    int c;
    int i;

    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x400; c += 16)
    {
	Map [c + 0] = Map [c + 0x800] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x800] = TRUE;
	Map [c + 1] = Map [c + 0x801] = RAM;
	BlockIsRAM [c + 1] = BlockIsRAM [c + 0x801] = TRUE;

	Map [c + 2] = Map [c + 0x802] = (uint8 *) MAP_PPU;
	Map [c + 3] = Map [c + 0x803] = (uint8 *) MAP_PPU;
	Map [c + 4] = Map [c + 0x804] = (uint8 *) MAP_CPU;
	Map [c + 5] = Map [c + 0x805] = (uint8 *) MAP_CPU;
	if (Settings.DSP1Master)
	{
	    Map [c + 6] = Map [c + 0x806] = (uint8 *) MAP_DSP;
	    Map [c + 7] = Map [c + 0x807] = (uint8 *) MAP_DSP;
	}
	else
	{
	    Map [c + 6] = Map [c + 0x806] = (uint8 *) MAP_NONE;
	    Map [c + 7] = Map [c + 0x807] = (uint8 *) MAP_NONE;
	}
	    
	for (i = c + 8; i < c + 16; i++)
	{
	    Map [i] = Map [i + 0x800] = &ROM [c << 12];
	    BlockIsROM [i] = BlockIsROM [i + 0x800] = TRUE;
	}

	for (i = c; i < c + 16; i++)
	{
	    int ppu = i & 15;
	    
	    MemorySpeed [i] = 
		MemorySpeed [i + 0x800] = ppu >= 2 && ppu <= 3 ? ONE_CYCLE : 8;
	}
    }

    // Banks 30->3f and b0->bf, address ranges 6000->7fff is S-RAM.
    for (c = 0; c < 16; c++)
    {
	Map [0x306 + (c << 4)] = (uint8 *) MAP_HIROM_SRAM;
	Map [0x307 + (c << 4)] = (uint8 *) MAP_HIROM_SRAM;
	Map [0xb06 + (c << 4)] = (uint8 *) MAP_HIROM_SRAM;
	Map [0xb07 + (c << 4)] = (uint8 *) MAP_HIROM_SRAM;
	BlockIsRAM [0x306 + (c << 4)] = TRUE;
	BlockIsRAM [0x307 + (c << 4)] = TRUE;
	BlockIsRAM [0xb06 + (c << 4)] = TRUE;
	BlockIsRAM [0xb07 + (c << 4)] = TRUE;
    }

    // Banks 40->7f and c0->ff
    for (c = 0; c < 0x400; c += 16)
    {
	for (i = c; i < c + 16; i++)
	{
	    Map [i + 0x400] = Map [i + 0xc00] = &ROM [c << 12];
	    MemorySpeed [i + 0x400] = MemorySpeed [i + 0xc00] = 8;
	    BlockIsROM [i + 0x400] = BlockIsROM [i + 0xc00] = TRUE;
	}
    }

    MapRAM ();
    WriteProtectROM ();
}

void CMemory::TalesROMMap ()
{
    int c;
    int i;

    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x400; c += 16)
    {
	Map [c + 0] = Map [c + 0x800] = RAM;
	Map [c + 1] = Map [c + 0x801] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x800] = TRUE;
	BlockIsRAM [c + 1] = BlockIsRAM [c + 0x801] = TRUE;

	Map [c + 2] = Map [c + 0x802] = (uint8 *) MAP_PPU;
	Map [c + 3] = Map [c + 0x803] = (uint8 *) MAP_PPU;
	Map [c + 4] = Map [c + 0x804] = (uint8 *) MAP_CPU;
	Map [c + 5] = Map [c + 0x805] = (uint8 *) MAP_CPU;
	Map [c + 6] = Map [c + 0x806] = (uint8 *) MAP_NONE;
	Map [c + 7] = Map [c + 0x807] = (uint8 *) MAP_NONE;
	for (i = c + 8; i < c + 16; i++)
	{
	    Map [i] = &ROM [c << 12];
	    Map [i + 0x800] = &ROM [c << 12];
	    BlockIsROM [i] = TRUE;
	    BlockIsROM [i + 0x800] = TRUE;
	}

	for (i = c; i < c + 16; i++)
	{
	    int ppu = i & 15;
	    
	    MemorySpeed [i] = 
		MemorySpeed [i + 0x800] = ppu >= 2 && ppu <= 3 ? ONE_CYCLE : 8;
	}
    }
    
    // Banks 30->3f and b0->bf, address ranges 6000->7ffff is S-RAM.
    for (c = 0; c < 16; c++)
    {
	Map [0x306 + (c << 4)] = (uint8 *) MAP_HIROM_SRAM;
	Map [0x307 + (c << 4)] = (uint8 *) MAP_HIROM_SRAM;
	Map [0xb06 + (c << 4)] = (uint8 *) MAP_HIROM_SRAM;
	Map [0xb07 + (c << 4)] = (uint8 *) MAP_HIROM_SRAM;
	BlockIsRAM [0x306 + (c << 4)] = TRUE;
	BlockIsRAM [0x307 + (c << 4)] = TRUE;
	BlockIsRAM [0xb06 + (c << 4)] = TRUE;
	BlockIsRAM [0xb07 + (c << 4)] = TRUE;
    }

#define OFFSET0 0x100000
#define OFFSET1 0x000000
#define OFFSET2 0x400000
#define OFFSET3 0x200000

    // Banks 40->7f and c0->ff
    for (c = 0; c < 0x400; c += 16)
    {
	for (i = c; i < c + 8; i++)
	{
	    Map [i + 0x400] = &ROM [c << 12];
	    Map [i + 0x408] = &ROM [c << 12];
	    Map [i + 0xc00] = &ROM [c << 12] + OFFSET3;
	    Map [i + 0xc08] = &ROM [c << 12] + OFFSET3;
	    BlockIsROM [i + 0x400] = TRUE;
	    BlockIsROM [i + 0x408] = TRUE;
	    BlockIsROM [i + 0xc00] = TRUE;
	    BlockIsROM [i + 0xc08] = TRUE;
	    MemorySpeed [i + 0x400] = MemorySpeed [i + 0xc00] = 8;
	    MemorySpeed [i + 0x408] = MemorySpeed [i + 0xc08] = 8;
	}
    }
    MapRAM ();

    WriteProtectROM ();
}

void CMemory::AlphaROMMap ()
{
    int c;
    int i;

    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x400; c += 16)
    {
	Map [c + 0] = Map [c + 0x800] = RAM;
	Map [c + 1] = Map [c + 0x801] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x800] = TRUE;
	BlockIsRAM [c + 1] = BlockIsRAM [c + 0x801] = TRUE;

	Map [c + 2] = Map [c + 0x802] = (uint8 *) MAP_PPU;
	Map [c + 3] = Map [c + 0x803] = (uint8 *) MAP_PPU;
	Map [c + 4] = Map [c + 0x804] = (uint8 *) MAP_CPU;
	Map [c + 5] = Map [c + 0x805] = (uint8 *) MAP_CPU;
	Map [c + 6] = Map [c + 0x806] = (uint8 *) MAP_DSP;
	Map [c + 7] = Map [c + 0x807] = (uint8 *) MAP_DSP;

	for (i = c + 8; i < c + 16; i++)
	{
	    Map [i] = Map [i + 0x800] = &ROM [c << 11] - 0x8000;
	    BlockIsROM [i] = TRUE;
	}

	for (i = c; i < c + 16; i++)
	{
	    int ppu = i & 15;
	    
	    MemorySpeed [i] = 
		MemorySpeed [i + 0x800] = ppu >= 2 && ppu <= 3 ? ONE_CYCLE : 8;
	}
    }

    // Banks 40->7f and c0->ff

#undef OFFSET
#define OFFSET 0x000000

    for (c = 0; c < 0x400; c += 16)
    {
	for (i = c; i < c + 16; i++)
	{
	    Map [i + 0x400] = (uint8 *) MAP_DEBUG;
	    Map [i + 0xc00] = &ROM [c << 12] + OFFSET;
	    MemorySpeed [i + 0x400] = MemorySpeed [i + 0xc00] = 8;
	    BlockIsROM [i + 0x400] = BlockIsROM [i + 0xc00] = TRUE;
	}
    }

    MapRAM ();
    WriteProtectROM ();
}

void CMemory::SuperFXROMMap ()
{
    int c;
    int i;
    
    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x400; c += 16)
    {
	Map [c + 0] = Map [c + 0x800] = RAM;
	Map [c + 1] = Map [c + 0x801] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x800] = TRUE;
	BlockIsRAM [c + 1] = BlockIsRAM [c + 0x801] = TRUE;

	Map [c + 2] = Map [c + 0x802] = (uint8 *) MAP_PPU;
	Map [c + 3] = Map [c + 0x803] = (uint8 *) MAP_PPU;
	Map [c + 4] = Map [c + 0x804] = (uint8 *) MAP_CPU;
	Map [c + 5] = Map [c + 0x805] = (uint8 *) MAP_CPU;
	Map [c + 6] = Map [c + 0x806] = (uint8 *) MAP_DSP;
	Map [c + 7] = Map [c + 0x807] = (uint8 *) MAP_DSP;
	for (i = c + 8; i < c + 16; i++)
	{
	    Map [i] = Map [i + 0x800] = &ROM [c << 11] - 0x8000;
	    BlockIsROM [i] = BlockIsROM [i + 0x800] = TRUE;
	}

	for (i = c; i < c + 8; i++)
	{
	    int ppu = i & 15;
	    
	    MemorySpeed [i] = 
		MemorySpeed [i + 0x800] = ppu >= 2 && ppu <= 3 ? ONE_CYCLE : 8;
	}
    }
    
    // Banks 40->7f and c0->ff
    for (c = 0; c < 0x400; c += 16)
    {
	for (i = c; i < c + 16; i++)
	{
	    Map [i + 0x400] = Map [i + 0xc00] = &ROM [c << 12];
	    MemorySpeed [i + 0x400] = MemorySpeed [i + 0xc00] = 8;
	    BlockIsROM [i + 0x400] = BlockIsROM [i + 0xc00] = TRUE;
	}
    }

    // Banks 7e->7f, RAM
    for (c = 0; c < 16; c++)
    {
	Map [c + 0x7e0] = RAM;
	Map [c + 0x7f0] = RAM + 0x10000;
	BlockIsRAM [c + 0x7e0] = TRUE;
	BlockIsRAM [c + 0x7f0] = TRUE;
	BlockIsROM [c + 0x7e0] = FALSE;
	BlockIsROM [c + 0x7f0] = FALSE;
    }

    // Banks 70->71, S-RAM
    for (c = 0; c < 32; c++)
    {
	Map [c + 0x700] = ::SRAM + (((c >> 4) & 1) << 16);
	BlockIsRAM [c + 0x700] = TRUE;
	BlockIsROM [c + 0x700] = FALSE;
    }

    // Banks 00->3f and 80->bf address ranges 6000->7fff is RAM.
    for (c = 0; c < 0x40; c++)
    {
	Map [0x006 + (c << 4)] = (uint8 *) ::SRAM - 0x6000;
	Map [0x007 + (c << 4)] = (uint8 *) ::SRAM - 0x6000;
	Map [0x806 + (c << 4)] = (uint8 *) ::SRAM - 0x6000;
	Map [0x807 + (c << 4)] = (uint8 *) ::SRAM - 0x6000;
	BlockIsRAM [0x006 + (c << 4)] = TRUE;
	BlockIsRAM [0x007 + (c << 4)] = TRUE;
	BlockIsRAM [0x806 + (c << 4)] = TRUE;
	BlockIsRAM [0x807 + (c << 4)] = TRUE;
    }
    // Replicate the first 2Mb of the ROM at ROM + 2MB such that each 32K
    // block is repeated twice in each 64K block.
    for (c = 0; c < 64; c++)
    {
	memmove (&ROM [0x200000 + c * 0x10000], &ROM [c * 0x8000], 0x8000);
	memmove (&ROM [0x208000 + c * 0x10000], &ROM [c * 0x8000], 0x8000);
    }

    WriteProtectROM ();
}

void CMemory::SA1ROMMap ()
{
    int c;
    int i;

    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x400; c += 16)
    {
	Map [c + 0] = Map [c + 0x800] = RAM;
	Map [c + 1] = Map [c + 0x801] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x800] = TRUE;
	BlockIsRAM [c + 1] = BlockIsRAM [c + 0x801] = TRUE;

	Map [c + 2] = Map [c + 0x802] = (uint8 *) MAP_PPU;
	Map [c + 3] = Map [c + 0x803] = (uint8 *) &Memory.FillRAM [0x3000] - 0x3000;
	Map [c + 4] = Map [c + 0x804] = (uint8 *) MAP_CPU;
	Map [c + 5] = Map [c + 0x805] = (uint8 *) MAP_CPU;
	Map [c + 6] = Map [c + 0x806] = (uint8 *) MAP_BWRAM;
	Map [c + 7] = Map [c + 0x807] = (uint8 *) MAP_BWRAM;
	for (i = c + 8; i < c + 16; i++)
	{
	    Map [i] = Map [i + 0x800] = &ROM [c << 11] - 0x8000;
	    BlockIsROM [i] = BlockIsROM [i + 0x800] = TRUE;
	}

	for (i = c; i < c + 16; i++)
	{
	    int ppu = i & 15;
	    
	    MemorySpeed [i] = 
		MemorySpeed [i + 0x800] = ppu >= 2 && ppu <= 3 ? ONE_CYCLE : 8;
	}
    }

    // Banks 40->7f
    for (c = 0; c < 0x400; c += 16)
    {
	for (i = c; i < c + 16; i++)
	    Map [i + 0x400] = (uint8 *) &SRAM [(c << 12) & 0xffff];

	for (i = c; i < c + 16; i++)
	{
	    MemorySpeed [i + 0x400] = 8;
	    BlockIsROM [i + 0x400] = FALSE;
	}
    }

    // c0->ff
    for (c = 0; c < 0x400; c += 16)
    {
	for (i = c;  i < c + 16; i++)
	{
	    Map [i + 0xc00] = &ROM [c << 12];
	    MemorySpeed [i + 0xc00] = 8;
	    BlockIsROM [i + 0xc00] = TRUE;
	}
    }

    for (c = 0; c < 16; c++)
    {
	Map [c + 0x7e0] = RAM;
	Map [c + 0x7f0] = RAM + 0x10000;
	BlockIsRAM [c + 0x7e0] = TRUE;
	BlockIsRAM [c + 0x7f0] = TRUE;
	BlockIsROM [c + 0x7e0] = FALSE;
	BlockIsROM [c + 0x7f0] = FALSE;
    }
    WriteProtectROM ();

    // Now copy the map and correct it for the SA1 CPU.
    memmove ((void *) SA1.WriteMap, (void *) WriteMap, sizeof (WriteMap));
    memmove ((void *) SA1.Map, (void *) Map, sizeof (Map));
    memmove ((void *) SA1.BlockIsRAM, (void *) BlockIsRAM, sizeof (BlockIsRAM));

    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x400; c += 16)
    {
	SA1.Map [c + 0] = SA1.Map [c + 0x800] = &Memory.FillRAM [0x3000];
	SA1.Map [c + 1] = SA1.Map [c + 0x801] = (uint8 *) MAP_NONE;
	SA1.WriteMap [c + 0] = SA1.WriteMap [c + 0x800] = &Memory.FillRAM [0x3000];
	SA1.WriteMap [c + 1] = SA1.WriteMap [c + 0x801] = (uint8 *) MAP_NONE;
	SA1.BlockIsRAM [c + 0] = SA1.BlockIsRAM [c + 0x800] = FALSE;
	SA1.BlockIsRAM [c + 1] = SA1.BlockIsRAM [c + 0x801] = FALSE;
    }

    // Banks 60->6f
    for (c = 0; c < 0x100; c++)
	SA1.Map [c + 0x600] = SA1.WriteMap [c + 0x600] = (uint8 *) MAP_BWRAM_BITMAP;
    
    BWRAM = SRAM;
}

void CMemory::LoROM24MBSMap ()
{
    int c;
    int i;

    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x400; c += 16)
    {
	Map [c + 0] = Map [c + 0x800] = RAM;
	Map [c + 1] = Map [c + 0x801] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x800] = TRUE;
	BlockIsRAM [c + 1] = BlockIsRAM [c + 0x801] = TRUE;

	Map [c + 2] = Map [c + 0x802] = (uint8 *) MAP_PPU;
	Map [c + 3] = Map [c + 0x803] = (uint8 *) MAP_PPU;
	Map [c + 4] = Map [c + 0x804] = (uint8 *) MAP_CPU;
	Map [c + 5] = Map [c + 0x805] = (uint8 *) MAP_CPU;
       Map [c + 6] = Map [c + 0x806] = (uint8 *) MAP_NONE;
       Map [c + 7] = Map [c + 0x807] = (uint8 *) MAP_NONE;

	for (i = c + 8; i < c + 16; i++)
	{
	    Map [i] = Map [i + 0x800] = &ROM [c << 11] - 0x8000;
	    BlockIsROM [i] = BlockIsROM [i + 0x800] = TRUE;
	}

	for (i = c; i < c + 16; i++)
	{
	    int ppu = i & 15;
	    
	    MemorySpeed [i] = 
		MemorySpeed [i + 0x800] = ppu >= 2 && ppu <= 3 ? ONE_CYCLE : 8;
	}
    }

    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x200; c += 16)
    {
	Map [c + 0x800] = RAM;
	Map [c + 0x801] = RAM;
	BlockIsRAM [c + 0x800] = TRUE;
	BlockIsRAM [c + 0x801] = TRUE;

	Map [c + 0x802] = (uint8 *) MAP_PPU;
	Map [c + 0x803] = (uint8 *) MAP_PPU;
	Map [c + 0x804] = (uint8 *) MAP_CPU;
	Map [c + 0x805] = (uint8 *) MAP_CPU;
       Map [c + 0x806] = (uint8 *) MAP_NONE;
	Map [c + 0x807] = (uint8 *) MAP_NONE;

	for (i = c + 8; i < c + 16; i++)
	{
	    Map [i + 0x800] = &ROM [c << 11] - 0x8000 + 0x200000;
	    BlockIsROM [i + 0x800] = TRUE;
	}

	for (i = c; i < c + 16; i++)
	{
	    int ppu = i & 15;
	    
	MemorySpeed [i + 0x800] = ppu >= 2 && ppu <= 3 ? ONE_CYCLE : 8;
	}
    }

    // Banks 40->7f and c0->ff
    for (c = 0; c < 0x400; c += 16)
    {
	for (i = c; i < c + 8; i++)
	    Map [i + 0x400] = Map [i + 0xc00] = &ROM [(c << 11) + 0x200000];

	for (i = c + 8; i < c + 16; i++)
	    Map [i + 0x400] = Map [i + 0xc00] = &ROM [(c << 11) + 0x200000 - 0x8000];

	for (i = c; i < c + 16; i++)
	{
	    MemorySpeed [i + 0x400] = MemorySpeed [i + 0xc00] = 8;
	    BlockIsROM [i + 0x400] = BlockIsROM [i + 0xc00] = TRUE;
	}
    }

    MapRAM ();
    WriteProtectROM ();
}

const char *CMemory::TVStandard ()
{
    return (Settings.PAL ? "PAL" : "NTSC");
}

const char *CMemory::Speed ()
{
    return (ROMSpeed & 0x10 ? "120ns" : "200ns");
}

const char *CMemory::MapType ()
{
    return (HiROM ? "HiROM" : "LoROM");
}

const char *CMemory::StaticRAMSize ()
{
    static char tmp [20];

    if (Memory.SRAMSize > 16)
	return ("Corrupt");
    sprintf (tmp, "%dKb", (SRAMMask + 1) / 1024);
    return (tmp);
}

const char *CMemory::Size ()
{
    static char tmp [20];

    if (ROMSize < 7 || ROMSize - 7 > 23)
	return ("Corrupt");
    sprintf (tmp, "%dMbits", 1 << (ROMSize - 7));
    return (tmp);
}

const char *CMemory::KartContents ()
{
    static char tmp [30];
    static const char *CoPro [16] = {
	"DSP1", "SuperFX", "OBC1", "SA-1", "S-DD1", "CoPro#5", "CoPro#6",
	"CoPro#7", "CoPro#8", "CoPro#9", "CoPro#10", "CoPro#11", "CoPro#12",
	"CoPro#13", "CoPro#14", "CoPro-Custom"
    };
    static const char *Contents [3] = {
	"ROM", "ROM+RAM", "ROM+RAM+BAT"
    };
    if (ROMType == 0)
	return ("ROM only");

    sprintf (tmp, "%s", Contents [(ROMType & 0xf) % 3]);

    if ((ROMType & 0xf) >= 3)
	sprintf (tmp, "%s+%s", tmp, CoPro [(ROMType & 0xf0) >> 4]);

    return (tmp);
}

const char *CMemory::MapMode ()
{
    static char tmp [4];
    sprintf (tmp, "%02x", ROMSpeed & ~0x10);
    return (tmp);
}

const char *CMemory::ROMID ()
{
    return (ROMId);
}

void CMemory::ApplyROMFixes ()
{
    // Breath of Fire 2 has a Hacker Intro that writes behond the limits of
    // the S-RAM the ROM header says the cartridge has and destroys saved
    // games in S-RAM without the hack below.
    if (strcmp (ROMName, "BREATH OF FIRE 2") == 0)
	SRAMMask = 0xffff;

    // XXX: Battle Toads
    Settings.BattleToadsNMIHack = FALSE;

    if (strcmp (ROMName, "TETRIS&Dr.MARIO") == 0 ||
	strcmp (ROMName, "JIGSAW PARTY") == 0)
    {
	Settings.MultiPlayer5Master = FALSE;
	Settings.MouseMaster = FALSE;
	Settings.SuperScopeMaster = FALSE;
    }

    // Games written by the Japanese software company called Human need special
    // SPC700 sound CPU cycle length timing to get them to work...
    if ((Memory.HiROM && strncmp ((char *) &Memory.ROM [0xffb0], "DE", 2) == 0) ||
	(!Memory.HiROM && strncmp ((char *) &Memory.ROM [0x7fb0], "DE", 2) == 0) ||
	strcmp (ROMName, "SUPER FORMATION SOCCE") == 0 ||
	strcmp (ROMName, "HOOK") == 0 ||
	strcmp (ROMName, "SUPER EMPIRE STRIKES") == 0)
    {
	IAPU.OneCycle = ONE_APU_CYCLE_HUMAN;
    }
    if (strcmp (ROMName, "EARTHWORM JIM 2") == 0 ||
	strcmp (ROMName, "ULTIMATE KOMBAT 3") == 0 ||
	strncmp (ROMName, "MORTAL KOMBAT 3", 15) == 0)
	IAPU.OneCycle = 30;

    Settings.DaffyDuckNMIHack = strcmp (ROMName, "DAFFY DUCK: MARV MISS") == 0;
    Settings.StarfoxHack = strcmp (ROMName, "STAR FOX") == 0 ||
			   strcmp (ROMName, "STAR WING") == 0;
    Settings.WinterGold = strcmp (ROMName, "FX SKIING NINTENDO 96") == 0;
    Settings.Seiken3 = strncmp (ROMName, "SeikenDensetsu3", 15) == 0;  
}

#ifdef NO_INLINE_SET_GET
#define INLINE
#include "getset.h"
#endif
