/*
 * X-Mame system independent sound code
 */

/* Hacked to use 16bit sound by Mike Oliphant (oliphant@ling.ed.ac.uk) */
#define __SOUND_C_

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef USE_TIMER
#include <sys/time.h>
#include <signal.h>
static int bytes_per_timer_alarm;

/* Set of signals to block during critical sections */
static sigset_t blockingset;

#endif

#include "xmame.h"
#include "sound.h"
#include "sound/mixer.h"
#include "profiler.h"

/* #define SOUND_DEBUG */

static float master_volume_divider;
static int sound_active = 0;
VOICE_T voices[MIXER_MAX_CHANNELS];
static unsigned char *smallbuf = NULL;
static signed short *buf = NULL;
static signed int *bigbuf = NULL;

/*********** code to quick alloc/free SAMPLE_T items **************
 * malloc/free is a too cpu-expensive process. When using a fixed size
 * malloc/free scheme , it's better to pre-allocate space and use this
 * algorithm
 *
 *   In our code SAMPLE_T is ideal for a quick malloc scheme. Of course
 * this is not applicable to data samples ( not fixed size ... )
 *
 * The SAMPLE_T code caches data blocks, which firstly reduces the
 * number of malloc() calls and secondly means we need not call free()
 * on memory allocated within SAMPLE_T.	 This is absolutely necessary
 * with timer based sound as there is no guarantee that a systems
 * malloc/free library is reentrant so free() must not be called from
 * within a signal handler.
 *
 * The cost of caching data blocks is to increase memory usage
 * slightly.
 *
 **************************************************************************/

/* this should be enough .... */
#define ALLOC_TABLE_SIZE 64

/* Heuristic for deciding whether to shrink an existing allocated data
   block.  This should be a reasonably large number.  */
#define LARGE_SAMPLE 0x10000

static SAMPLE_T SampleTTable[ALLOC_TABLE_SIZE];
static int SampleTIndex[ALLOC_TABLE_SIZE];
static int SampleTPointer;

int InitSampleTTable (void)
{
   int i;

   for (i = 0; i < ALLOC_TABLE_SIZE; i++)
   {
      SampleTIndex[i] = i;
      SampleTTable[i].data = NULL;
      SampleTTable[i].datasize = 0;
   }

   SampleTPointer = (ALLOC_TABLE_SIZE - 1);
   return 0;
}

/* FreeSampleT is called from within the signal handler with timer
   based audio, so care must be taken that it does not modify any data
   that is not appropriately protected from access outside the signal
   handler */

static int FreeSampleT (SAMPLE_T * pt)
{
   int item = (int) (pt - SampleTTable);

   if (SampleTPointer >= (ALLOC_TABLE_SIZE - 1))
   {
      fprintf (stderr_file, "FreeSampleT() consistency error\n");
      exit (1);
   }

   SampleTIndex[++SampleTPointer] = item;

   return 0;
}

static SAMPLE_T *AllocSampleT (size_t blocksize)
{
   SAMPLE_T *pt;

   if (SampleTPointer < 0)
      return (SAMPLE_T *) NULL; /* no items available */

   pt = &SampleTTable[SampleTIndex[SampleTPointer]];

   /* Allocate new memory if the old memory block was too small or
      far larger than we require */

   if (pt->datasize < blocksize || pt->datasize > blocksize + LARGE_SAMPLE)
   {
      signed short *newblock;

#ifdef SOUND_DEBUG
      fprintf (stderr_file,
         "AllocSampleT(): New audio memory block, size %ld (oldsize=%ld)\n",
               (long) blocksize, (long) pt->datasize);
#endif

      newblock = (signed short *) malloc (blocksize);

      if (newblock == NULL)
      {
         /* malloc() failed */

         if (pt->datasize < blocksize)
            /* If malloc() failed and the current block is too small, we
               fail the operation */
            return (SAMPLE_T *) NULL;
      }
      else
      {
         /* malloc() succeeded */
         if (pt->data != NULL)
            free (pt->data);

         pt->data = newblock;
         pt->datasize = blocksize;
      }
   }

   SampleTPointer--;
   return pt;
}

/****************************************************************************
*
* Timer based sound routines
*
* Used in those systems that doesn't allow how many bytes can be send to the
* audio device without blocking
* 
*****************************************************************************/

#ifdef USE_TIMER
struct sigaction sig_action;

int start_timer ()
{
   struct itimerval timer_value;
   void audio_timer (int arg);

   /* set and configure timer catcher */
   if (!play_sound)
      return OSD_OK;
   sig_action.sa_handler = audio_timer;
#ifdef hpux
   sig_action.sa_flags = SA_RESETHAND;
#else
   sig_action.sa_flags = SA_RESTART;
#endif

   if (sigaction (SIGALRM, &sig_action, NULL) < 0)
   {
      fprintf (stderr_file, "Sigaction Failed \n");
      return OSD_NOT_OK;
   }

   /* set and configure realtime alarm */
   timer_value.it_interval.tv_sec =
      timer_value.it_value.tv_sec = 0L;
   timer_value.it_interval.tv_usec =
      timer_value.it_value.tv_usec = 1000000L / audio_timer_freq;

   if (setitimer (ITIMER_REAL, &timer_value, NULL))
   {
      fprintf (stderr_file, "Setting the timer failed.\n");
      return OSD_NOT_OK;
   }

   return OSD_OK;
}

void audio_timer (int arg)
{
   static int in = FALSE;

   if (!play_sound)
      return;
   if (!in)
   {
      in = TRUE;
      osd_audio_update ();
      in = FALSE;
   }                            /* else {
                                   fprintf(stderr_file, ".");
                                   } */
#if defined solaris || defined irix
   /* rearm alarm timer */
   sigaction (SIGALRM, &sig_action, NULL);
#endif
}

#endif /* ifdef USE_TIMER */

/*****************************************************************************
* 
* NON-Timer based sound routines
*
* the idea is simple: I inquire audio driver how many bytes can i send, 
* parse data through the mixer and send it to audio (or if the system is
* timer based, wait for a timer irq to arrive )
*
* Modified and hacked up for mame and sound by
*  Bill Lash - lash@tellabs.com
******************************************************************************/

void osd_audio_update (void)
{
   int i, j;
   long size;
   SAMPLE_T *new_sample;
   signed short *current_data_pt;
   signed short *end_data_pt;
   float vol, pos_frac, freq_fac;
   static int warn = 1;
   int pan;

   if ((play_sound == 0) || (sound_active == 0))
      return;

#ifdef USE_TIMER
   size = bytes_per_timer_alarm;
#else
   if ((size = sysdep_audio_get_freespace ()) <= 256)
      return;

   profiler_mark (PROFILER_SOUND);

   if (size > AUDIO_BUFF_SIZE)
   {
      size = AUDIO_BUFF_SIZE;

      if (warn)
      {
         warn = 0;
         fprintf (stderr_file,
            "WARNING size > AUDIO_BUFF_SIZE in osd_audio_update\n"
            "Sound quality might be worse then usual\n"
            "please mail j.w.r.degoede@et.tudelft.nl about this message\n"
            "include your makefile, machine+os and sounddriver version\n"
            "\n"
            "Don't mail me if you have a sb128 / 64 / ensoniq pci card\n"
            "This is a known problem with the drivers for these cards\n"
            "Try compiling with timer based audio for these cards\n");
      }
   }
#endif
   /* size is in bytes, so better correct it to be in samples. Always do
      it in stereo samples, in mono we just only render the left channel */
   if (!sound_8bit)   size /= 2;
   if (!sound_stereo) size *= 2;
   
   memset (bigbuf, 0, size * sizeof (int));

   sound_active = 0;

   for (j = 0; j < MIXER_MAX_CHANNELS; j++)
   {
      SAMPLE_T *pt = voices[j].sample;

      if (pt)
      {
         current_data_pt = voices[j].current_data_pt;
         pos_frac = voices[j].pos_frac;
         end_data_pt = pt->end_data_pt;
         vol = pt->vol;
         pan = pt->pan;
         freq_fac = pt->freq_fac;
         sound_active = 1;
      }
      else
         continue;

      for (i = 0; i < size; i+=2)
      {
         switch (pan)
         {
            case MIXER_PAN_LEFT:
#ifdef FANCY_SOUND
               bigbuf[i] += (*current_data_pt + (*(current_data_pt + 1) - *current_data_pt)
                          * pos_frac) * vol;
#else
               bigbuf[i] += *current_data_pt * vol;
#endif               
               break;
            case MIXER_PAN_RIGHT:
#ifdef FANCY_SOUND
               bigbuf[i+1] += (*current_data_pt + (*(current_data_pt + 1) - *current_data_pt)
                          * pos_frac) * vol;
#else
               bigbuf[i+1] += *current_data_pt * vol;
#endif               
               break;
            case MIXER_PAN_CENTER:
            {
               int tmp;
#ifdef FANCY_SOUND
               tmp = (*current_data_pt + (*(current_data_pt + 1) - *current_data_pt)
                          * pos_frac) * vol;
#else
               tmp = *current_data_pt * vol;
#endif
               bigbuf[i]   += tmp;
               bigbuf[i+1] += tmp;
               break;
            }
         }

         pos_frac += freq_fac;

         while (pos_frac > 1)
         {
            pos_frac--;
            current_data_pt++;
         }

         if (current_data_pt > end_data_pt)
         {
            if (pt->loop_stream)
            {
               if (pt->next_sample)
               {
                  new_sample = pt->next_sample;
                  FreeSampleT (pt);
                  voices[j].sample = pt = new_sample;
                  end_data_pt = pt->end_data_pt;
                  vol = pt->vol;
               }

               current_data_pt = pt->data;
            }
            else
            {
               osd_stop_sample (j);
               break;
            }
         }  /* if at end of sample */
      }  /* for i=0;i<size;i++) */

      voices[j].current_data_pt = current_data_pt;
      voices[j].pos_frac = pos_frac;
   }  /* for j< MIXER_MAX_CHANNELS */

#ifdef SOUND_DEBUG
   fprintf (stderr_file, "osd_audio_update(): %d avail %d used\n", size, i);
#endif

   if (sound_active)
   {
      int tmp;
      
      switch ((sound_8bit << 4) | sound_stereo)
      {
         case 0x00: /* 16bit mono */
            for (i=0,j=0;i<size;i+=2,j++)
            {
               tmp = bigbuf[i];
               if      (tmp > SAMP_MAX_16) buf[j] = SAMP_MAX_16;
               else if (tmp < SAMP_MIN_16) buf[j] = SAMP_MIN_16;
               else buf[j] = tmp;
            }
            sysdep_audio_play ((unsigned char *) buf, size);
            break;
         case 0x01: /* 16bit stereo */
            for (i=0;i<size;i++)
            {
               tmp = bigbuf[i];
               if      (tmp > SAMP_MAX_16) buf[i] = SAMP_MAX_16;
               else if (tmp < SAMP_MIN_16) buf[i] = SAMP_MIN_16;
               else buf[i] = tmp;
            }
            sysdep_audio_play ((unsigned char *) buf, size*2);
            break;
         case 0x10: /* 8 bit mono */
            for (i=0,j=0;i<size;i+=2,j++)
            {
               tmp = bigbuf[i];
               if      (tmp > SAMP_MAX_16) smallbuf[j] = SAMP_MAX_8;
               else if (tmp < SAMP_MIN_16) smallbuf[j] = SAMP_MIN_8;
               else smallbuf[j] = (tmp >> 8) + 128;
            }
            sysdep_audio_play ((unsigned char *) smallbuf, size/2);
            break;
         case 0x11: /* 8 bit stereo */
            for (i=0;i<size;i++)
            {
               tmp = bigbuf[i];
               if      (tmp > SAMP_MAX_16) smallbuf[i] = SAMP_MAX_8;
               else if (tmp < SAMP_MIN_16) smallbuf[i] = SAMP_MIN_8;
               else smallbuf[i] = (tmp >> 8) + 128;
            }
            sysdep_audio_play ((unsigned char *) smallbuf, size);
            break;
      }
   }

   profiler_mark (PROFILER_END);
}

/****************************************************************************
* 
* General sound routines
*
*****************************************************************************/

int osd_audio_init(void)
{
   int i;

   /* do this before checking if sound is enabled, it's automagicly
      disabled if not available */
   if (sysdep_audio_init()!=OSD_OK) return OSD_NOT_OK;
   
   if (!play_sound)
      return OSD_OK;

   osd_set_mastervolume (attenuation);
#ifdef USE_TIMER
   bytes_per_timer_alarm = options.samplerate / audio_timer_freq;
   /* correct size for stereo & 16bit */
   if (!sound_8bit)  bytes_per_timer_alarm *= 2;
   if (sound_stereo) bytes_per_timer_alarm *= 2;

   if (bytes_per_timer_alarm > AUDIO_BUFF_SIZE)
   {
      fprintf (stderr_file, "Error: bytes/timertick > audio-buf-size\n");
      fprintf (stderr_file,
           "This is a configure error, try reasing fragsize (see -help)\n");
      return OSD_NOT_OK;
   }

   /* Initialise the signal set to block during critical sections */
   sigemptyset (&blockingset);
   sigaddset (&blockingset, SIGALRM);
#if (!defined svgalib) && (!defined svgafx)
   /* in svgalib mode must start timer AFTER open display */
   if (play_sound)
      if (start_timer () == OSD_NOT_OK)
         return (OSD_NOT_OK);
#endif
#endif


   if (sound_8bit)
      smallbuf = malloc (AUDIO_BUFF_SIZE * sizeof(unsigned char));
   else
      buf = malloc (AUDIO_BUFF_SIZE * sizeof(signed short));
   /* size is in bytes, so better correct it to be in samples. Always do
      it in stereo samples, in mono we just only render the left channel */
   i = AUDIO_BUFF_SIZE;
   if (!sound_8bit)   i /= 2;
   if (!sound_stereo) i *= 2;
   
   bigbuf = malloc (i * sizeof(signed int));

   if ((!smallbuf && !buf) || !bigbuf)
   {
      if (buf)
         free (buf);
      if (bigbuf)
         free (buf);
      if (smallbuf)
         free (smallbuf);
      fprintf (stderr_file, "Error: couldn't allocate sound-buffers\n");
      return OSD_NOT_OK;
   }

   InitSampleTTable ();

   for (i = 0; i < MIXER_MAX_CHANNELS; i++)
   {
      voices[i].sample = NULL;
      voices[i].current_data_pt = NULL;
      voices[i].pos_frac = 0;
   }

   return OSD_OK;
}

void osd_audio_close (void)
{
   /* FIXME: SampleT table should be freed */
   int n;
   for (n = 0; n < MIXER_MAX_CHANNELS; n++) osd_stop_sample(n);
   if (play_sound)
      sysdep_audio_close();
   if (buf)
      free (buf);
   if (bigbuf)
      free (bigbuf);
   if (smallbuf)
      free (smallbuf);
}

/* attenuation in dB */
void osd_set_mastervolume (int _attenuation)
{
#ifdef USE_TIMER
   sigset_t oset;

#endif
   int channel;
   float old_master_volume_divider;
   float attenuation_mult = 1.0;

   if (!play_sound)
      return;

   if (_attenuation > 0) _attenuation = 0;
   if (_attenuation < -32) _attenuation = -32;
   
   attenuation = _attenuation;

   while (_attenuation++ < 0)
      attenuation_mult /= 1.122018454;  /* = (10 ^ (1/20)) = 1dB */

   old_master_volume_divider = master_volume_divider;
   master_volume_divider = (float) 100 / attenuation_mult;

#ifdef USE_TIMER
   sigprocmask (SIG_BLOCK, &blockingset, &oset);
#endif

   for (channel = 0; channel < MIXER_MAX_CHANNELS; channel++)
   {
      SAMPLE_T *pt = voices[channel].sample;

      for (; pt; pt = pt->next_sample)
         pt->vol *= old_master_volume_divider / master_volume_divider;
   }

#ifdef USE_TIMER
   if (!sigismember (&oset, SIGALRM))
      sigprocmask (SIG_UNBLOCK, &blockingset, NULL);
#endif
}

int osd_get_mastervolume (void)
{
   /* The core doesn't like it if we return the -48 we use to disable sound */
   if (attenuation < -32) return -32;
   return attenuation;
}

void osd_sound_enable (int enable_it)
{
   static int orig_attenuation;

   if (enable_it)
   {
      osd_set_mastervolume (orig_attenuation);
   }
   else
   {
      if (osd_get_mastervolume() != -48)
      {
         orig_attenuation = osd_get_mastervolume ();
         osd_set_mastervolume (-48);
      }
   }
}


void osd_play_sample_generic (int channel, signed char *data, int len,
                              int freq, int volume, int loop, int bits)
{
#ifdef USE_TIMER
   sigset_t oset;
#endif
   SAMPLE_T *new_samp;

   if ((play_sound == 0) || (channel >= MIXER_MAX_CHANNELS) || (len == 0))
      return;

   if ((freq < 10) || (freq > 100000))
   {
#ifdef SOUND_DEBUG
      fprintf (stderr_file,
               "osd_play_sample() call Ignored: chan:%d frec:%d vol:%d\n",
               channel, freq, volume);
#endif
      return;
   }

   /* backwards compatibility with old 0-255 volume range */
   if (volume > 100)
      volume = volume * 100 / 255;

#ifdef USE_TIMER
   sigprocmask (SIG_BLOCK, &blockingset, &oset);
#endif

   osd_stop_sample (channel);

   if (!(new_samp = AllocSampleT ((len + 1) * 2)))
   {
#ifdef USE_TIMER
      if (!sigismember (&oset, SIGALRM))
         sigprocmask (SIG_UNBLOCK, &blockingset, NULL);
#endif
      return;
   }

   sound_active = 1;

   voices[channel].sample = new_samp;

   voices[channel].current_data_pt = new_samp->data;
   new_samp->end_data_pt = new_samp->data + len - 1;
   voices[channel].pos_frac = 0;
   new_samp->next_sample = (SAMPLE_T *) NULL;
   new_samp->loop_stream = loop;
   new_samp->vol = (float) volume / master_volume_divider;
   new_samp->pan = MIXER_PAN_CENTER;  /* default to center */
   new_samp->freq_fac = (float) freq / options.samplerate;

   if (bits == 8)
   {  
      int i;
      for (i = 0; i < len; i++)
      {
         new_samp->data[i] = data[i] << 8;
      }
   }
   else
   {
      memcpy(new_samp->data, data, len*sizeof(unsigned short));
   }

#ifdef FANCY_SOUND
   /* calculate one sample more as the actual length for interpolation */
   if (loop)
      new_samp->data[len] = new_samp->data[0];
   else
      new_samp->data[len] = 0;
#endif

#ifdef SOUND_DEBUG
   fprintf (stderr_file, "play() chan:%d len:%d frec:%d vol:%d loop:%d\n",
            channel, len, freq, volume, loop);
#endif

#ifdef USE_TIMER
   if (!sigismember (&oset, SIGALRM))
      sigprocmask (SIG_UNBLOCK, &blockingset, NULL);
#endif

   return;
}

void osd_play_sample (int channel, signed char *data, int len, int freq,
                      int volume, int loop)
{
   osd_play_sample_generic (channel, data, len, freq, volume, loop, 8);
}

void osd_play_sample_16 (int channel, signed short *data, int len, int freq,
                         int volume, int loop)
{
   osd_play_sample_generic (channel, (signed char *) data, len / 2, freq, volume,
                            loop, 16);
}

void osd_play_streamed_sample_generic (int channel, signed char *data, int len,
                                    int freq, int volume, int pan, int bits)
{
#ifdef USE_TIMER
   sigset_t oset;
#endif
   SAMPLE_T *new_samp, *last_samp = NULL;

   if ((play_sound == 0) || (channel >= MIXER_MAX_CHANNELS) || (len == 0))
      return;

   if ((freq < 10) || (freq > 100000))
   {
#ifdef SOUND_DEBUG
      fprintf (stderr_file,
            "play_streamed_sample() call Ignored: chan:%d frec:%d vol:%d\n",
               channel, freq, volume);
#endif
      return;
   }

   /* if we have more then 3 samples cueued we're either running unthrottled,
      or we've got out of sync either way drop this sample */
   if ( voices[channel].sample &&
        voices[channel].sample->next_sample &&
        voices[channel].sample->next_sample->next_sample)
      return;

   /* backwards compatibility with old 0-255 volume range */
   if (volume > 100)
      volume = volume * 100 / 255;

#ifdef USE_TIMER
   sigprocmask (SIG_BLOCK, &blockingset, &oset);
#endif

   if (!(new_samp = AllocSampleT ((len + 1) * 2)))
   {
#ifdef USE_TIMER
      if (!sigismember (&oset, SIGALRM))
         sigprocmask (SIG_UNBLOCK, &blockingset, NULL);
#endif
      return;
   }

   sound_active = 1;

   if (voices[channel].sample == NULL)
   {
      voices[channel].sample = new_samp;
      voices[channel].current_data_pt = new_samp->data;
   }
   else
   {
      last_samp = voices[channel].sample;
      while (last_samp->next_sample)
         last_samp = last_samp->next_sample;
      last_samp->next_sample = new_samp;
   }

   new_samp->loop_stream = 1;
   new_samp->next_sample = NULL;
   new_samp->end_data_pt = new_samp->data + len - 1;
   new_samp->vol = (float) volume / master_volume_divider;
   if (sound_stereo)
   {
      new_samp->pan = pan;
   }
   else
   {
      new_samp->pan = MIXER_PAN_LEFT;
      /* we'll probably get both a left and right sample at one channel,
         so half the volume */
      if (pan != MIXER_PAN_CENTER) new_samp->vol /= 2;
   }
   new_samp->freq_fac = (float) freq / options.samplerate;

   if (bits == 8)
   {
      int i;
      for (i = 0; i < len; i++)
      {
         new_samp->data[i] = data[i] << 8;
      }
   }
   else
   {
      memcpy(new_samp->data, data, len*sizeof(unsigned short));
   }

#ifdef FANCY_SOUND
   /* calculate one sample more as the actual length for interpolation */
   new_samp->data[len] = new_samp->data[0];
   if (last_samp)
      *(last_samp->end_data_pt + 1) = new_samp->data[0];
#endif

#ifdef USE_TIMER
   if (!sigismember (&oset, SIGALRM))
      sigprocmask (SIG_UNBLOCK, &blockingset, NULL);
#endif
}

void osd_play_streamed_sample (int channel, signed char *data, int len,
                               int freq, int volume, int pan)
{
   osd_play_streamed_sample_generic (channel, data, len, freq, volume, pan, 8);
}

void osd_play_streamed_sample_16 (int channel, signed short *data, int len,
                                  int freq, int volume, int pan)
{
   osd_play_streamed_sample_generic (channel, (signed char *) data, len / 2,
                                     freq, volume, pan, 16);
}

void osd_set_sample_freq (int channel, int freq)
{
#ifdef USE_TIMER
   sigset_t oset;

#endif
   SAMPLE_T *next_samp = voices[channel].sample;

   if (play_sound == 0 || channel >= MIXER_MAX_CHANNELS)
      return;

   if ((freq < 10) || (freq > 100000))
   {
#ifdef SOUND_DEBUG
      fprintf (stderr_file,
               "osd_set_sample_freq() call Ignored: chan:%d frec:%d\n",
               channel, freq);
#endif
      return;
   }

#ifdef USE_TIMER
   sigprocmask (SIG_BLOCK, &blockingset, &oset);
#endif

   while (next_samp != NULL)
   {
      next_samp->freq_fac = (float) freq / options.samplerate;
      next_samp = next_samp->next_sample;
   }

#ifdef USE_TIMER
   if (!sigismember (&oset, SIGALRM))
      sigprocmask (SIG_UNBLOCK, &blockingset, NULL);
#endif
}

void osd_set_sample_volume (int channel, int volume)
{
#ifdef USE_TIMER
   sigset_t oset;

#endif
   SAMPLE_T *next_samp = voices[channel].sample;

   if (play_sound == 0 || channel >= MIXER_MAX_CHANNELS)
      return;

   /* backwards compatibility with old 0-255 volume range */
   if (volume > 100)
      volume = volume * 100 / 255;

#ifdef USE_TIMER
   sigprocmask (SIG_BLOCK, &blockingset, &oset);
#endif

   while (next_samp != NULL)
   {
      next_samp->vol = (float) volume / master_volume_divider;
      next_samp = next_samp->next_sample;
   }

#ifdef USE_TIMER
   if (!sigismember (&oset, SIGALRM))
      sigprocmask (SIG_UNBLOCK, &blockingset, NULL);
#endif
}

void osd_stop_sample (int channel)
{
#ifdef USE_TIMER
   sigset_t oset;

#endif
   SAMPLE_T *next_samp;
   SAMPLE_T *curr_samp;

   if (play_sound == 0 || channel >= MIXER_MAX_CHANNELS)
      return;

#ifdef USE_TIMER
   sigprocmask (SIG_BLOCK, &blockingset, &oset);
#endif

   if (voices[channel].sample != NULL)
   {
      next_samp = voices[channel].sample;
      voices[channel].sample = NULL;
      voices[channel].current_data_pt = NULL;
      voices[channel].pos_frac = 0;

      while (next_samp != NULL)
      {
         curr_samp = next_samp;
         next_samp = next_samp->next_sample;
         FreeSampleT (curr_samp);
      }
   }

#ifdef USE_TIMER
   if (!sigismember (&oset, SIGALRM))
      sigprocmask (SIG_UNBLOCK, &blockingset, NULL);
#endif
}

void osd_restart_sample (int channel)
{
#ifdef USE_TIMER
   sigset_t oset;

#endif

   if (play_sound == 0 || channel >= MIXER_MAX_CHANNELS)
      return;

#ifdef USE_TIMER
   sigprocmask (SIG_BLOCK, &blockingset, &oset);
#endif

   voices[channel].current_data_pt = voices[channel].sample->data;

#ifdef USE_TIMER
   if (!sigismember (&oset, SIGALRM))
      sigprocmask (SIG_UNBLOCK, &blockingset, NULL);
#endif
}

int osd_get_sample_status (int channel)
{
   int stopped = 0;

   if (play_sound == 0 || channel >= MIXER_MAX_CHANNELS)
      return -1;
   if (voices[channel].sample == NULL)
      stopped = 1;

   return stopped;
}

/* handle ugly dos fm-stuff */
void osd_opl_control (int chip, int reg)
{
}

void osd_opl_write (int chip, int data)
{
}
