/****************************************************************************/
/*                                                                          */
/* IMG* Image Processing Tools Library                                      */
/* Program:   imgSurfacePlot.c                                              */
/* Author:    Simon A.J. Winder                                             */
/* Date:      Thu Oct 20 20:46:00 1994                                      */
/* Copyright (C) 1994 Simon A.J. Winder                                     */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/*                                                                          */
/* Program:   imgSurfacePlot.c                                              */
/* Author:    Simon A.J. Winder                                             */
/* Date:      Mon Feb 15 17:47:26 1993                                      */
/* Function:  Gives 3D projection of a 2D surface                           */
/* Compile:   gcc -Wall -O imgSurfacePlot.c ~sajw/ILib/libbimg.a            */
/*                -I~sajw/ILib -o imgSurfacePlot -lm -lXaw -lXmu -lXt -lX11 */
/*            (sparcs)                                                      */
/* Info:                                                                    */
/*            Input:                                                        */
/*              Input from stdin is either a pgm image file or              */
/*              a floating point array in ASCII format as follows:          */
/*                ILibFloat                                                 */
/*                width height                                              */
/*                0 0 0 0 0 0                                               */
/*                list of width x height floats                             */
/*                (incrementing right and down through the array)           */
/*                                                                          */
/*            Command line options:                                         */
/*            -shade                Produce shaded plot in postscript       */
/*            -width width          Set startup window width (pixels)       */
/*            -height height        Set startup window height (pixels)      */
/*            -landscape            Produce 'landscape' oriented postscript */
/*            -paperwidth width     Set paper width, if not A4 (points)     */
/*            -paperheight height   Set paper height, if not A4 (points)    */
/*            -psfile filename      Set postscript output file, else stdout */
/*            -scale value          Auto-scale magnification factor         */
/*            -fixedscale value     Turn on fixed scaling with factor given */
/*            -pgm                  Turn on fixed scaling for pgm images    */
/*            -rotate value         Set initial viewpoint rotation          */
/*            -elevate value        Set initial viewpoint elevation         */
/*            -mesh                 Draw as a mesh without HL removal       */
/*                                                                          */
/*            Keyboard controls:                                            */
/*            p or C-p              Move viewpoint up by 10 degrees         */
/*            n or C-n              Move viewpoint down by 10 degrees       */
/*            f or C-f              Move viewpoint right by 10 degrees      */
/*            b or C-b              Move viewpoint left by 10 degrees       */
/*            P                     Move viewpoint up by 1 degree           */
/*            N                     Move viewpoint down by 1 degree         */
/*            F                     Move viewpoint left by 1 degree         */
/*            B                     Move viewpoint right by 1 degree        */
/*            Space                 Print the current viewpoint angles      */
/*            Return                Draw current view to Postscript output  */
/*                                                                          */
/*            Mouse controls:                                               */
/*            click right of centre Move viewpoint right                    */
/*            click left of centre  Move viewpoint left                     */
/*            click above centre    Move viewpoint up                       */
/*            click below centre    Move viewpoint down                     */
/*                                                                          */
/****************************************************************************/

#include "tools.h"

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Label.h>

#define PRGNAME "SurfacePlot"
#define ERROR(e) imgError(e,PRGNAME)

/* default values */
#define PAPER_WIDTH 598    /* A4 is approx 598x850 pts */
#define PAPER_HEIGHT 850   /* (pts=points (1/72") */
#define ROTATE_DEG 0       /* Initial rotation angle of view */
#define ELEVATE_DEG 20     /* Initial elevation angle of view */
#define AS_FACTOR 5.0      /* Units across to scale max data value to */
#define Z_FACTOR 1.0       /* Viewing distance as a fraction of diagonal */
#define S_FACTOR 0.95      /* Fraction of screen size to cover with diagram */
#define P_FACTOR 0.8       /* Fraction of paper size to cover with disgram */

/* Angle update macro */
#define Update(val,inc) \
  {\
    val+=(inc);\
    if(val>2.0*M_PI) val-=(2.0*M_PI);\
    else if(val<-2.0*M_PI) val+=(2.0*M_PI);\
  }

/* Forgotten prototype */
extern time_t time();

typedef struct _Points {
  float x,y,z;
} Points;

typedef struct _Polygons {
  Points *vertex[4];
  float z;
} Polygons;

/* Boxes to put globals in! */
typedef struct _Xstuff {
  XtAppContext app_con;
  Display *display;
  GC fgGC,bgGC;
  Pixmap pixmap;
  Widget label;
  int width,height;
} Xstuff;

typedef struct _ResourcedParams {
  Boolean shade_flag,landscape_flag;
  Boolean pgm_flag,mesh_flag;
  int paper_width,paper_height;
  String psfile;
  float auto_scale,fixed_scale;
  int rotate_deg,elevate_deg;
  double rotate,elevate;
} ResourcedParams;

/* Prototypes */
int Sorter(const void *,const void *);
void StartX(int,char **);
void GetImage(void);
void AutoScaleImage(void);
Points *AllocPoints(void);
Polygons *AllocPolygons(void);
void Render(void (*draw_routine)(Polygons *,long));
void DrawRoutineX(Polygons *,long);
void DrawRoutinePS(Polygons *,long);
void OpenOutput(void);
void WritePSTrailer(void);

/* Action callbacks */
void QuitAction(Widget,XEvent *,String *,Cardinal *);
void UpAction(Widget,XEvent *,String *,Cardinal *);
void DownAction(Widget,XEvent *,String *,Cardinal *);
void LeftAction(Widget,XEvent *,String *,Cardinal *);
void RightAction(Widget,XEvent *,String *,Cardinal *);
void UpActionS(Widget,XEvent *,String *,Cardinal *);
void DownActionS(Widget,XEvent *,String *,Cardinal *);
void LeftActionS(Widget,XEvent *,String *,Cardinal *);
void RightActionS(Widget,XEvent *,String *,Cardinal *);
void PrintAction(Widget,XEvent *,String *,Cardinal *);
void MouseAction(Widget,XEvent *,String *,Cardinal *);
void SpaceAction(Widget,XEvent *,String *,Cardinal *);

/* Globals */
static Xstuff xstuff;
static ResourcedParams rp;
static it_image *image;
static int pagenum,wrote_some_postscript;

int main(int argc,char **argv)
{

  IFHELP
    {
      fprintf(stderr,"img%s - Surface viewing tool for X\n",PRGNAME);
      fprintf(stderr,"img%s [options]\n",PRGNAME);
      fprintf(stderr,"  -shade                Produce shaded plot in postscript\n");
      fprintf(stderr,"  -width width          Set startup window width (pixels)\n");
      fprintf(stderr,"  -height height        Set startup window height (pixels)\n");
      fprintf(stderr,"  -landscape            Produce 'landscape' oriented postscript\n");
      fprintf(stderr,"  -paperwidth width     Set paper width, if not A4 (points)\n");
      fprintf(stderr,"  -paperheight height   Set paper height, if not A4 (points)\n");
      fprintf(stderr,"  -psfile filename      Set postscript output file, else stdout\n");
      fprintf(stderr,"  -scale value          Auto-scale magnification factor\n");
      fprintf(stderr,"  -fixedscale value     Turn on fixed scaling with factor given\n");
      fprintf(stderr,"  -pgm                  Turn on fixed scaling for pgm images\n");
      fprintf(stderr,"  -rotate value         Set initial viewpoint rotation\n");
      fprintf(stderr,"  -elevate value        Set initial viewpoint elevation\n");
      fprintf(stderr,"  -mesh                 Draw as a mesh without HL removal\n");
      fprintf(stderr,"  stdin: Float\n");
      exit(0);
    }

  /* Initalise X */
  StartX(argc,argv);

  /* Convert the viewing angles into radians */
  rp.rotate=M_PI/180.0*(double)rp.rotate_deg;
  /* Elevation is couted from face-on, hence the M_PI_2 */
  rp.elevate=M_PI/180.0*(double)rp.elevate_deg-M_PI_2;

  /* If pgm scaling set fixed scale so max value is 256 */
  if(rp.pgm_flag)
    rp.fixed_scale=rp.auto_scale/256.0;

  /* Import image and adjust values to make a nice display */
  GetImage();
  AutoScaleImage();

  /* Draw the thing for the first time */
  Render(DrawRoutineX);

  /* Sit and wait */
  XtAppMainLoop(xstuff.app_con);
  return(0);
}

void GetImage(void)
{
  it_image *i2;

  /* Import the image from stdin */
  if((image=i_read_image_file(stdin,IT_FLOAT | IT_BYTE,IM_CONTIG))==NULL)
    ERROR("can't import image file");

  /* Waste of time sizes */
  if(image->width>128 || image->height>128 ||
     image->width<2 || image->height<2)
    ERROR("image size is out of range");
  
  /* If it is a byte image, convert to float */
  if(image->type==IT_BYTE)
    {
      i2=i_create_image(image->width,image->height,IT_FLOAT,IM_CONTIG);
      if(i2==NULL)
	ERROR("out of memory");
      i_byte_to_float(image,i2);
      i_destroy_image(image);
      image=i2;
    }
}

void AutoScaleImage(void)
{
  double ave,min,max,val,stuff_above,stuff_below,scale;
  long i,pixels;
  it_float *flt_ptr;

  scale=rp.fixed_scale;

  /* Compute average, min and max values from data */
  ave=0.0;
  min=1e99;   /* Let's hope these are big enough */
  max= -1e99;
  pixels=(long) image->width * (long) image->height;
  flt_ptr=im_float_row(image,0);
  for(i=pixels;i--;)
    {
      val= *flt_ptr++;
      if(val<min) min=val;
      if(val>max) max=val;
      ave+=val;
    }
  ave/=(double) pixels;

  /* Auto-scale image so that biggest peaks are AS_FACTOR times */
  /* planar grid separation */
  if(scale==0.0 && min!=max)
  {
    stuff_above=max-ave;
    stuff_below=ave-min;
    if(stuff_above>stuff_below)
      scale=rp.auto_scale/stuff_above;
    else
      scale=rp.auto_scale/stuff_below;
  }

  /* Shift and scale image values */
  flt_ptr=im_float_row(image,0);
  for(i=pixels;i--;)
    {
      val= *flt_ptr;
      *flt_ptr++=scale*(val-ave);
    }
}

void Render(void (*draw_routine)(Polygons *,long))
{
  static Points *points=NULL; /* Not strictly necessary */
  static Polygons *polygons;
  static double distance;
  static x_start,y_start,x_end,y_end;
  int x,y,width,height;
  long num_polys,i;
  double cos_rotate,sin_rotate,cos_elevate,sin_elevate;
  double xx,yy,zz,yyy;
  Points *pts_ptr,*pts_ptr2;
  Polygons *ply_ptr;
  it_float *flt_ptr;

  /* Size of image */
  width=image->width;
  height=image->height;

  /* If we have called here for the first time, do initialisation */
  if(points==NULL)
    {
      /* Get memory */
      points=AllocPoints();
      polygons=AllocPolygons();

      /* Make polygon vertices refer to points data */
      ply_ptr=polygons;
      pts_ptr=points;
      pts_ptr2=points+width;
      for(y=1;y<height;y++,pts_ptr++,pts_ptr2++)
	for(x=1;x<width;x++,ply_ptr++)
	  {
	    ply_ptr->vertex[0]=pts_ptr++;
	    ply_ptr->vertex[1]=pts_ptr2++;
	    ply_ptr->vertex[2]=pts_ptr2;
	    ply_ptr->vertex[3]=pts_ptr;
	  }
      /* Set distance to image plane */
      distance=Z_FACTOR*sqrt((double) (width*width+height*height));
      /* Set a lower limit */
      if(distance<8.0)
	distance=8.0;
      
      /* Constants to be used later */
      x_start= -width/2;
      x_end=width+x_start;
      y_end= -height/2;
      y_start=height+y_end-1;
    }

  /* Work out the rotational scale factors */
  cos_rotate=cos(rp.rotate);
  sin_rotate=sin(rp.rotate);
  cos_elevate=cos(rp.elevate);
  sin_elevate=sin(rp.elevate);

  /* Do the rotations and perspective transformation */
  /* Z is out of the page, X is to the right, Y is up. */
  pts_ptr=points;
  flt_ptr=im_float_row(image,0);
  for(y=y_start;y>=y_end;y--)
    for(x=x_start;x<x_end;x++)
      {
	/* Get the coordinates (image(0,0) is top left) */
	xx=(double) x;
	yy=(double) y;
	zz= *flt_ptr++;

	/* Rotate anticlockwise about Z axis */
	yyy=xx*sin_rotate+yy*cos_rotate;
	xx=xx*cos_rotate-yy*sin_rotate;

	/* Rotate anticlockwise about X axis */
	yy=yyy*cos_elevate-zz*sin_elevate;
	zz=yyy*sin_elevate+zz*cos_elevate;
	
	/* Perspective Transformation */
	pts_ptr->x=xx/(distance-zz);
	pts_ptr->y=yy/(distance-zz);
	pts_ptr->z=zz;
	pts_ptr++;
      }
  
  /* Calculate polygon depths */
  ply_ptr=polygons;
  num_polys=(width-1)*(height-1);
  for(i=num_polys;i--;)
    {
      /* Just average the four point depths---seems to work OK */
      zz=ply_ptr->vertex[0]->z;
      zz+=ply_ptr->vertex[1]->z;
      zz+=ply_ptr->vertex[2]->z;
      zz+=ply_ptr->vertex[3]->z;
      ply_ptr->z=zz/4.0;
      ply_ptr++;
    }

  /* Depth sort polygons (another sort may be quicker?) */
  qsort((void *) polygons,(size_t) num_polys,sizeof(Polygons),Sorter);

  /* Call the drawing routine (either X or Postscript) */
  (draw_routine)(polygons,num_polys);
}

void DrawRoutineX(Polygons *polygons,long num_polys)
{
  long i;
  int x_origin,y_origin;
  double scale,y_scale;
  XPoint points[5];

  /* Clear the pixmap first */
  XFillRectangle(xstuff.display,xstuff.pixmap,xstuff.bgGC,0,0,
		 xstuff.width,xstuff.height);

  /* Make image fit to window size */
  x_origin=xstuff.width/2;
  y_origin=xstuff.height/2;
  if(xstuff.height>xstuff.width)
    scale=xstuff.width*S_FACTOR;
  else
    scale=xstuff.height*S_FACTOR;

  /* Turn upside down as window origin is in top-left */
  y_scale= -scale;

  /* Draw the polygons */
  if(rp.mesh_flag==False)
    {
      for(i=num_polys;i--;polygons++)
	{
	  points[0].x=(int) (polygons->vertex[0]->x * scale)+x_origin;
	  points[0].y=(int) (polygons->vertex[0]->y * y_scale)+y_origin;
	  points[1].x=(int) (polygons->vertex[1]->x * scale)+x_origin;
	  points[1].y=(int) (polygons->vertex[1]->y * y_scale)+y_origin;
	  points[2].x=(int) (polygons->vertex[2]->x * scale)+x_origin;
	  points[2].y=(int) (polygons->vertex[2]->y * y_scale)+y_origin;
	  points[3].x=(int) (polygons->vertex[3]->x * scale)+x_origin;
	  points[3].y=(int) (polygons->vertex[3]->y * y_scale)+y_origin;
	  points[4].x=points[0].x;
	  points[4].y=points[0].y;

	  XFillPolygon(xstuff.display,xstuff.pixmap,xstuff.bgGC,points,4,
		       Complex,CoordModeOrigin);
	  XDrawLines(xstuff.display,xstuff.pixmap,xstuff.fgGC,points,5,
		     CoordModeOrigin);
	}
    }
  else
    {
      for(i=num_polys;i--;polygons++)
	{
	  points[0].x=(int) (polygons->vertex[0]->x * scale)+x_origin;
	  points[0].y=(int) (polygons->vertex[0]->y * y_scale)+y_origin;
	  points[1].x=(int) (polygons->vertex[1]->x * scale)+x_origin;
	  points[1].y=(int) (polygons->vertex[1]->y * y_scale)+y_origin;
	  points[2].x=(int) (polygons->vertex[2]->x * scale)+x_origin;
	  points[2].y=(int) (polygons->vertex[2]->y * y_scale)+y_origin;
	  points[3].x=(int) (polygons->vertex[3]->x * scale)+x_origin;
	  points[3].y=(int) (polygons->vertex[3]->y * y_scale)+y_origin;
	  points[4].x=points[0].x;
	  points[4].y=points[0].y;

	  XDrawLines(xstuff.display,xstuff.pixmap,xstuff.fgGC,points,5,
		     CoordModeOrigin);
	}
    }

  /* Display pixmap */
  XCopyArea(xstuff.display,xstuff.pixmap,XtWindow(xstuff.label),xstuff.fgGC,
	    0,0,xstuff.width,xstuff.height,0,0);
}

Points *AllocPoints(void)
{
  Points *points;
  int num_points;

  /* Memory allocation */
  num_points=image->width * image->height;
  points=(Points *) malloc(num_points*sizeof(Points));
  if(points==NULL)
    ERROR("out of memory");
  return(points);
}

Polygons *AllocPolygons(void)
{
  Polygons *polygons;
  int num_polys;
  
  /* Memory allocation */
  num_polys=(image->width-1) * (image->height-1);
  polygons=(Polygons *) malloc(num_polys*sizeof(Polygons));
  if(polygons==NULL)
    ERROR("out of memory");
  return(polygons);
}

int Sorter(const void *q1,const void *q2)
{
  /* Compare the depths */
  if(((Polygons *)q1)->z > ((Polygons *)q2)->z)
    return(1);
  if(((Polygons *)q1)->z < ((Polygons *)q2)->z)
    return(-1);
  return(0);
}

void StartX(int argc,char **argv)
{
  static float def_scale=AS_FACTOR;
  static float def_fixed=0.0;
  static String fallback_resources[] = {
    "*Label.width: 512",
    "*Label.height: 512",
    "*Label.label: Working...",
    "*Label.borderWidth: 0",
    NULL,
  };
  static XrmOptionDescRec options[] = {
    {"-shade","shade",XrmoptionNoArg,(caddr_t) "True"},
    {"-width",".label.width",XrmoptionSepArg,NULL},
    {"-height",".label.height",XrmoptionSepArg,NULL},
    {"-landscape","landscape",XrmoptionNoArg,(caddr_t) "True"},
    {"-paperwidth","paperwidth",XrmoptionSepArg,NULL},
    {"-paperheight","paperheight",XrmoptionSepArg,NULL},
    {"-psfile","psfile",XrmoptionSepArg,NULL},
    {"-scale","scale",XrmoptionSepArg,NULL},
    {"-fixedscale","fixedscale",XrmoptionSepArg,NULL},
    {"-pgm","pgm",XrmoptionNoArg,"True"},
    {"-rotate","rotate",XrmoptionSepArg,NULL},
    {"-elevate","elevate",XrmoptionSepArg,NULL},
    {"-mesh","mesh",XrmoptionNoArg,(caddr_t) "True"},
  };
  static XtResource resources[] = {
    {"shade","Shade",XtRBoolean,sizeof(Boolean),
       XtOffsetOf(ResourcedParams,shade_flag),
       XtRImmediate,(XtPointer) False},
    {"landscape","Landscape",XtRBoolean,sizeof(Boolean),
       XtOffsetOf(ResourcedParams,landscape_flag),
       XtRImmediate,(XtPointer) False},
    {"paperwidth","Paperwidth",XtRInt,sizeof(int),
       XtOffsetOf(ResourcedParams,paper_width),
       XtRImmediate,(XtPointer) PAPER_WIDTH},
    {"paperheight","Paperheight",XtRInt,sizeof(int),
       XtOffsetOf(ResourcedParams,paper_height),
       XtRImmediate,(XtPointer) PAPER_HEIGHT},
    {"psfile","Psfile",XtRString,sizeof(String),
       XtOffsetOf(ResourcedParams,psfile),
       XtRImmediate,NULL},
    {"scale","Scale",XtRFloat,sizeof(float),
       XtOffsetOf(ResourcedParams,auto_scale),
       XtRFloat,(XtPointer) &def_scale},
    {"fixedscale","Fixedscale",XtRFloat,sizeof(float),
       XtOffsetOf(ResourcedParams,fixed_scale),
       XtRFloat,(XtPointer) &def_fixed},
    {"pgm","Pgm",XtRBoolean,sizeof(Boolean),
       XtOffsetOf(ResourcedParams,pgm_flag),
       XtRImmediate,(XtPointer) False},
    {"rotate","Rotate",XtRInt,sizeof(int),
       XtOffsetOf(ResourcedParams,rotate_deg),
       XtRImmediate,(XtPointer) ROTATE_DEG},
    {"elevate","Elevate",XtRInt,sizeof(int),
       XtOffsetOf(ResourcedParams,elevate_deg),
       XtRImmediate,(XtPointer) ELEVATE_DEG},
    {"mesh","Mesh",XtRBoolean,sizeof(Boolean),
       XtOffsetOf(ResourcedParams,mesh_flag),
       XtRImmediate,(XtPointer) False},
  };
  static String new_translations =
    "<Key>q:QuitAction()\n\
    :<Key>p:UpAction()\n\
    :<Key>n:DownAction()\n\
    :<Key>f:RightAction()\n\
    :<Key>b:LeftAction()\n\
    :<Key>P:UpActionS()\n\
    :<Key>N:DownActionS()\n\
    :<Key>F:RightActionS()\n\
    :<Key>B:LeftActionS()\n\
    <Key>Return:PrintAction()\n\
    <Key>space:SpaceAction()\n\
    <Btn1Down>:MouseAction()\n";
  static XtActionsRec actions[] = {
    { "QuitAction",QuitAction },
    { "UpAction",UpAction },
    { "DownAction",DownAction },
    { "LeftAction",LeftAction },
    { "RightAction",RightAction },
    { "UpActionS",UpActionS },
    { "DownActionS",DownActionS },
    { "LeftActionS",LeftActionS },
    { "RightActionS",RightActionS },
    { "PrintAction",PrintAction },
    { "SpaceAction",SpaceAction },
    { "MouseAction",MouseAction },
  };
  Widget toplevel,label;
  Arg args[5];
  int depth;
  Dimension width,height;
  unsigned long mask;
  Pixel fg,bg;
  XGCValues gcv;

  /* Initalise the application */
  toplevel = XtAppInitialize(&xstuff.app_con,PRGNAME,
			     options,XtNumber(options),&argc,argv,
			     fallback_resources,NULL,0);

  /* Get the resources and put in structure 'rp' */
  XtGetApplicationResources(toplevel,(XtPointer) &rp,
			    resources,XtNumber(resources),NULL,0);

  /* Add all the actions mappings */
  XtAppAddActions(xstuff.app_con,actions,XtNumber(actions));

  /* Create label widget to display in and add the translations */
  label=XtCreateManagedWidget("label",labelWidgetClass,toplevel,NULL,0);
  XtOverrideTranslations(label,XtParseTranslationTable(new_translations));
  XtRealizeWidget(toplevel);

  /* Get info about the window produced */
  xstuff.display=XtDisplay(label);
  XtSetArg(args[0],XtNdepth,&depth);
  XtSetArg(args[1],XtNwidth,&width);
  XtSetArg(args[2],XtNheight,&height);
  XtSetArg(args[3],XtNforeground,&fg);
  XtSetArg(args[4],XtNbackground,&bg);
  XtGetValues(label,args,5);

  /* Create an appropriate pixmap */
  xstuff.pixmap=XCreatePixmap(xstuff.display,XtWindow(label),
			       width,height,depth);
  /* Save the info */
  xstuff.label=label;
  xstuff.width=(int) width;
  xstuff.height=(int) height;

  /* Make GCs to set and clear pixels */
  mask=GCForeground | GCBackground | GCGraphicsExposures;
  gcv.foreground=(unsigned long) fg;
  gcv.background=(unsigned long) bg;
  gcv.graphics_exposures=False;
  xstuff.fgGC=XCreateGC(xstuff.display,XtWindow(label),mask,&gcv);
  gcv.foreground=(unsigned long) bg;
  gcv.background=(unsigned long) fg;
  xstuff.bgGC=XCreateGC(xstuff.display,XtWindow(label),mask,&gcv);

  /* Clear the pixmap */
  XFillRectangle(xstuff.display,xstuff.pixmap,xstuff.bgGC,0,0,width,height);

  /* Tell label widget about pixmap so we get exposure redraws */
  XtSetArg(args[0],XtNbitmap,xstuff.pixmap);
  XtSetValues(xstuff.label,args,1);
}

void QuitAction(Widget w,XEvent *e,String *p,Cardinal *n)
{
  /* If we wrote out Postscript then write the necessary trailer comments */
  if(wrote_some_postscript)
    WritePSTrailer();

  /* Go away! */
  XtDestroyApplicationContext(XtWidgetToApplicationContext(w));
  exit(0);
}

void UpAction(Widget w,XEvent *e,String *p,Cardinal *n)
{
  /* 10 degrees up */
  Update(rp.elevate,M_PI/18.0);
  Render(DrawRoutineX);
}

void DownAction(Widget w,XEvent *e,String *p,Cardinal *n)
{
  /* 10 degrees down */
  Update(rp.elevate,-M_PI/18.0);
  Render(DrawRoutineX);
}

void LeftAction(Widget w,XEvent *e,String *p,Cardinal *n)
{
  /* 10 degrees left */
  Update(rp.rotate,M_PI/18.0);
  Render(DrawRoutineX);
}

void RightAction(Widget w,XEvent *e,String *p,Cardinal *n)
{
  /* 10 degrees right */
  Update(rp.rotate,-M_PI/18.0);
  Render(DrawRoutineX);
}

void UpActionS(Widget w,XEvent *e,String *p,Cardinal *n)
{
  /* 1 degree up */
  Update(rp.elevate,M_PI/180.0);
  Render(DrawRoutineX);
}

void DownActionS(Widget w,XEvent *e,String *p,Cardinal *n)
{
  /* 1 degree down */
  Update(rp.elevate,-M_PI/180.0);
  Render(DrawRoutineX);
}

void LeftActionS(Widget w,XEvent *e,String *p,Cardinal *n)
{
  /* 1 degree left */
  Update(rp.rotate,M_PI/180.0);
  Render(DrawRoutineX);
}

void RightActionS(Widget w,XEvent *e,String *p,Cardinal *n)
{
  /* 1 degree right */
  Update(rp.rotate,-M_PI/180.0);
  Render(DrawRoutineX);
}

void PrintAction(Widget w,XEvent *e,String *p,Cardinal *n)
{
  /* Render image in Postscript to stdout or file */
  fprintf(stderr,"Exporting Postscript\n");
  Render(DrawRoutinePS);
}

void SpaceAction(Widget w,XEvent *e,String *p,Cardinal *n)
{
  /* Print viewing angle */
  fprintf(stderr,"Rotation: %d deg.  Elevation: %d deg.\n",
	  (int)(rp.rotate*180.0/M_PI),90+(int)(rp.elevate*180.0/M_PI));
}

void MouseAction(Widget w,XEvent *event,String *p,Cardinal *n)
{
  double xinc,yinc;
  Arg args[2];
  Dimension width,height;

  /* Move viewpoint with mouse. Click action to stop lots of redraws */
  /* Get current window size */
  XtSetArg(args[0],XtNwidth,&width);
  XtSetArg(args[1],XtNheight,&height);
  XtGetValues(xstuff.label,args,2);

  /* Work out angular increments based on mouse position relative to */
  /* window centre */
  xinc=M_PI_2/400.0*(double)((int)width/2-event->xbutton.x);
  yinc=M_PI_2/400.0*(double)((int)height/2-event->xbutton.y);

  /* Update and draw */
  Update(rp.rotate,xinc);
  Update(rp.elevate,yinc);
  Render(DrawRoutineX);
}

void DrawRoutinePS(Polygons *polys,long num_polys)
{
  long i;
  int v;
  time_t now;
  char *date;
  double xmin,ymin,xmax,ymax,val,scale,scale_y,width,height;
  double pt_width,pt_height,pt_llx,pt_lly,pt_urx,pt_ury;
  double pt_centx,pt_centy,temp;
  Polygons *ptr;
  FILE *fp;

  /* New page number */
  pagenum++;

  /* Open correct stream for output or append */
  /* This must work for sending more than one image */
  if(rp.psfile==NULL)
    fp=stdout;
  else 
    {
      if(pagenum==1)
	{
	  /* Open for the fisrt time */
	  if((fp=fopen(rp.psfile,"w"))==NULL)
	    ERROR("can't open postscript output file");
	}
      else
	if((fp=fopen(rp.psfile,"a+"))==NULL)
	  ERROR("can't open postscript output file");
    }

  /* Find extent of plot */
  xmin=ymin=xmax=ymax=0.0;
  for(ptr=polys,i=num_polys;i--;ptr++)
    for(v=0;v<4;v++)
      {
	val=ptr->vertex[v]->x;
	if(val>xmax) xmax=val;
	if(val<xmin) xmin=val;
	val=ptr->vertex[v]->y;
	if(val>ymax) ymax=val;
	if(val<ymin) ymin=val;
      }

  /* Flip things around if landscape mode */
  if(rp.landscape_flag)
    {
      temp=xmin;
      xmin=ymin;
      ymin= -temp;
      temp=xmax;
      xmax=ymax;
      ymax= -temp;
    }

  /* Calculate the bouding box and scale factor in pts */
  width=xmax-xmin;
  height=ymax-ymin;
  scale=P_FACTOR*(double)rp.paper_width/width;
  scale_y=P_FACTOR*(double)rp.paper_height/height;
  if(scale_y<scale)
    scale=scale_y;
  pt_width=width*scale;
  pt_height=height*scale;
  pt_llx=((double)rp.paper_width-pt_width)/2.0;
  pt_urx=pt_llx+pt_width;
  pt_lly=((double)rp.paper_height-pt_height)/2.0;
  pt_ury=pt_lly+pt_height;
  pt_centx=pt_llx-scale*xmin;
  pt_centy=pt_lly-scale*ymin;

  /* Write postscript header */
  if(pagenum==1)
    {
      time(&now);
      date=ctime(&now);
      fprintf(fp,"%%!PS-Adobe-1.0\n%%%%Title: Surface Plot (%dx%d)\n",
	      image->width,image->height);
      fprintf(fp,"%%%%Creator: img%s\n",PRGNAME);
      fprintf(fp,"%%%%CreationDate: %s",date);
      fprintf(fp,"%%%%Pages: (atend)\n%%%%BoundingBox: %d %d %d %d\n",
	      (int)pt_llx,(int)pt_lly,(int)pt_urx,(int)pt_ury);
      fprintf(fp,"%%%%DocumentFonts:\n%%%%EndComments\n%%%%EndProlog\n");
    }

  /* Write page comment and define a polygon drawing function */
  fprintf(fp,"\n%%%%Page: %d %d\n/v_state save def\n",pagenum,pagenum);
  fprintf(fp,"/p { newpath moveto lineto lineto lineto closepath\n");
  if(rp.mesh_flag==False)
    { /* Draw filled polygons */
      fprintf(fp,"gsave ");
      if(rp.shade_flag==True)   /* Set the shading fraction */
	fprintf(fp,"0.9");
      else
	fprintf(fp,"1.0");
      fprintf(fp," setgray fill grestore ");
    }
  fprintf(fp,"stroke } def\n");
  fprintf(fp,"%d %d translate\n",(int)pt_centx,(int)pt_centy);
  if(rp.landscape_flag)
    fprintf(fp,"-90 rotate\n");
  fprintf(fp,"0.3 setlinewidth\n\n");

  /* Write all the polygons out */
  for(ptr=polys,i=num_polys;i--;ptr++)
    {
      for(v=0;v<4;v++)
	{
	  fprintf(fp,"%g %g ",
		  scale*(ptr->vertex[v]->x),scale*(ptr->vertex[v]->y));
	  if(v==1)
	    fprintf(fp,"\n");
	}
      /* Draw polygon command */
      fprintf(fp," p\n");
    }

  /* Show the page and restore */
  fprintf(fp,"showpage v_state restore\n");

  /* Say we wrote the postscript so that we can add trailer on quitting */
  wrote_some_postscript=1;
  if(rp.psfile!=NULL)
    fclose(fp);
}

void WritePSTrailer(void)
{
  FILE *fp;

  /* Write postscript trailer---info about number of pages output */
  if(rp.psfile==NULL)
    fp=stdout;
  else if((fp=fopen(rp.psfile,"a+"))==NULL)
    ERROR("can't open postscript output file");
  fprintf(fp,"%%%%Trailer\n%%%%Pages: %d\n",pagenum);
  if(rp.psfile!=NULL)
    fclose(fp);
}
/* Version 1.0 (Oct 1994) */
/* Version 1.1 (Nov 1994) */
