/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  double MSslEndMC (double Dx, double Dxx, double Dy, double Dyy)

Purpose:
  Calculate a slope at an end point (for fitting a cubic)

Description:
  This routine calculates a slope for a reference point which is an end point
  for a sequence of points.  This slope can be used to generate piecewise
  monotone cubic interpolants.  Consider three points (x0,y0), (x1,y1) and
  (x2,y2), where x0 < x1 < x2 (a left end point) or x2 < x1 < x0 (a right end
  point).  The slope is calculated at the point (x0,y0).

Parameters:
  <-  double MSslEndMC
      Slope value at the reference point
   -> double Dx
      Increment in abscissa value for the interval next to the reference point.
      This value must be positive, i.e. (x1-x0) for a left end point or (x0-x1)
      for a right end point.
   -> double Dxx
      Increment in abscissa value for the interval one removed from the
      reference point.  This value must be positive, i.e. (x2-x1) for a left
      end point or (x1-x2) for a right end point.
   -> double Dy
      Increment in ordinate value for the interval next to the reference point,
      i.e. (y1-y0)
   -> double Dyy
      Increment in ordinate value for the interval one removed from the
      reference point, i.e. (y2-y1)

Author / revision:
  P. Kabal  Copyright (C) 1997
  $Revision: 1.6 $  $Date: 1997/10/10 19:27:02 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: MSslEndMC.c 1.6 1997/10/10 libtsp-v3r0 $";

#include <assert.h>

#include <libtsp.h>
#include <libtsp/nucleus.h>

#define ABSV(x)		(((x) < 0) ? -(x) : (x))


double
MSslEndMC (Dx, Dxx, Dy, Dyy)

     double Dx;
     double Dxx;
     double Dy;
     double Dyy;

{
  double Sr, Srr, S;
  double Ds;

/* Check for increasing X values */
  assert (Dx > 0.0 && Dxx > 0.0);

/* Sr is the slope to the right of point 0, Srr is the slope to the far
   right of point 0 */
  Sr = Dy / Dx;
  Srr = Dyy / Dxx;

/*
   Set the derivative using a non-centered three-point formula, adjusted to be
   shape-preserving
                                          x1-x0
     d = a Sr + (1-a) Srr , where a = 1 + ----- .
                                          x2-x0

   The factor a modifies the formula to take into account the relative spacing
   of the values (a varies from 2 to 1).
*/
  Ds = Dxx + Dx;
  S = Sr * ((Ds + Dx) / Ds) - Srr * (Dx / Ds);

/* Modify the value if necessary to preserve monotonicity */
  if ((S >= 0.0 && Sr <= 0.0) || (S <= 0.0 && Sr >= 0.0))
    S = 0.0;
  else if ((Sr > 0.0 && Srr < 0.0) || (Sr < 0.0 && Srr > 0.0)) {
    if (ABSV (S) > ABSV (3.0 * Sr))
      S = 3.0 * Sr;
  }

  return S;
}
