
/*
 * DREADERD/READER.C - reader task
 *
 *	Reader task, main control loop(s).
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

Prototype void ReaderTask(int fd, const char *id);
Prototype void NNCommand(Connection *conn);
Prototype void NNCommand2(Connection *conn);
Prototype void NNBadCommandUse(Connection *conn);
Prototype void NNUnknownCommand(Connection *conn);
Prototype void NNTerminate(Connection *conn);
Prototype void NNWriteHello(Connection *conn);
Prototype void NNWaitThread(Connection *conn);
Prototype void NNTPHelp(Connection *conn, char **pptr);
Prototype Connection *InitConnection(ForkDesc *desc, DnsRes *dres);
Prototype void DeleteConnection(Connection *conn);
Prototype void StatusUpdate(Connection *conn, const char *ctl, ...);

Prototype KPDB *KDBActive;

void HandleReaderMsg(ForkDesc *desc);
int makeReaderSlot(void);
void freeReaderSlot(int slot);
void sigSegVReader(int sigNo);

KPDB *KDBActive;
int   TFd = -1;		/* thread's return fd to the parent */
int   NumReaders;
int   *ReaderSlotAry;
int   TerminatePending;

void
ReaderTask(int fd, const char *id)
{
    time_t dtime = 0;
    time_t ltime = 0;
    time_t itime = 0;
    time_t ftime = 0;
    int counter = 0;

    TFd = fd;

    /*
     * [re]open RTStatus
     */
    RTStatusOpen(RTStatus, ThisReaderFork * DiabloReaderThreads + 1, DiabloReaderThreads);

    /*
     * Since we setuid(), we won't core.  This is for debugging
     */
    if (CoreDebugOpt)
	signal(SIGSEGV, sigSegVReader);

    /*
     * Setup thread for passed pipe
     */
    ResetThreads();
    AddThread("reader", fd, -1, THREAD_READER, -1, 0);

    FD_SET(fd, &RFds);

    /*
     * Open KPDB database for active file
     */

    if ((KDBActive = KPDBOpen(PatDbExpand(ReaderDActivePat), O_RDWR)) == NULL) {
	logit(LOG_CRIT, "Unable to open %s", PatDbExpand(ReaderDActivePat));
	sleep(60);
	exit(1);
    }

    /*
     * Selection core
     */

    CheckServerConfig(time(NULL), 1);

    while (TerminatePending == 0 || NReadServAct || NumReaders) {
	/*
	 *  select core
	 */
	struct timeval tv;
	fd_set rfds = RFds;
	fd_set wfds = WFds;
	fd_set read_only_to_find_eof_fds;
	int i, sel_r;

	/*
	 *  Get next scheduled timeout, no more then 2 seconds
	 *  (x 10 counter counts = 20 seconds max for {d,i,f}time 
	 *  check)
	 */

	NextTimeout(&tv, 2 * 1000);

	stprintf("%s readers=%02d spoolsrv=%d/%d postsrv=%d/%d",
	    id,
	    NumReaders,
	    NReadServAct, NReadServers, 
	    NWriteServAct, NWriteServers
	);

	FD_ZERO(&read_only_to_find_eof_fds);
	for (i = 0; i < MaxFds; ++i) {
	    if (FD_ISSET(i, &wfds) && (!(FD_ISSET(i, &rfds)))) {
		FD_SET(i, &rfds);
		FD_SET(i, &read_only_to_find_eof_fds);
	    }
	}

	sel_r = select(MaxFds, &rfds, &wfds, NULL, &tv);
	gettimeofday(&CurTime, NULL);

	if(sel_r < 0)
	    logit(LOG_CRIT,
		  "select error: %s (rfds=0x%x, wfds=0x%x)",
		  strerror(errno),
		  rfds,
		  wfds);

	/*
	 * select is critical, don't make unnecessary system calls.  Only
	 * test the time every 10 selects (20 seconds worst case), and
	 * only check for a new server configuration file every 60 seconds 
	 * after the initial load.  This may rearrange THREAD_SPOOL and
	 * THREAD_POST threads.
	 *
	 * We do not startup spool/post servers for feed-only forks
	 *
	 * However, flush overview cache even for feed-only forks.
	 */

	if (FeedOnlyServer <= 0) {
	    if (++counter == 10) {
		time_t t = CurTime.tv_sec;
		if (ltime) {
		    dtime += t - ltime;
		    itime += t - ltime;
		    ftime += t - ltime;
		}

		/*
		 * Check for server config change once a minute
		 */
		if (dtime < -5 || dtime >= 60) {
		    CheckServerConfig(t, ServersTerminated);
		    dtime = 0;
		}

		/*
		 * Flush overview every 30 seconds to allow dexpireover to work
		 */
		if (ftime < -5 || ftime >= 30) {
		    FlushOverCache();
		    ftime = 0;
		}

		/*
		 * Poll all active descriptors once every 5 minutes.  This
		 * will work around a linux embrionic close bug that
		 * doesn't wakeup select(), and is used to idle-timeout
		 * connections. XXX
		 */
		if (itime < -5 || itime >= 300) {
		    rfds = RFds;
		    itime = 0;
		}
		ltime = t;
		counter = 0;
	    }
	} else {
	    /*
	     * For a feed-only server, we only flush the overview FD
	     * cache every 5 minutes, and with a greater granularity.
	     * It should cycle much faster than that normally, and this
	     * is to prevent idle feed-only forks from keeping locks.
	     */

	    if (++counter == 50) {
		time_t t = CurTime.tv_sec;
		if (ltime) {
		    ftime += t - ltime;
		}

		if (ftime < -5 || ftime >= 300) {
		    FlushOverCache();
		    ftime = 0;
		}
	    }
	}

	for (i = 0; i < MaxFds; ++i) {
	    if (FD_ISSET(i, &rfds) && FD_ISSET(i, &read_only_to_find_eof_fds)) {
		char junk_byte;
		int ret_val;
		/*
		 * This FD is not marked for reading, but select() claims
		 * it has something to say. We don't actually want to read
		 * from it, but we do want to close it if the associated
		 * connection is dead.
		 */
		FD_CLR(i, &rfds);

		/* Use recv() with MSG_PEEK to see if it's closed.
		 * We shouldn't block because we're O_NONBLOCK.
		 */
		ret_val = recv(i, &junk_byte, 1, MSG_PEEK);

		/* If ret_val is zero, this means the socket is closed.
		 * Blast it. Otherwise, ignore it.
		 */
		if(ret_val == 0) {
		    ForkDesc *desc;
		    if((desc = FindThread(i, -1)) != NULL) {
			Connection *conn = desc->d_Data;
			int still_in_use = 0;

			if(conn) {
			    /*
			     * If this client has an outstanding server request,
			     * avoid killing it, because this conn is referenced
			     * elsewhere.
			     */
			    if(conn->co_SReq != NULL)
				still_in_use = 1;

			    if(!still_in_use)
				DeleteConnection(conn);
			}
			if(!still_in_use)
			    DelThread(desc);
		    }
		}
	    }
	}

	for (i = 0; i < MaxFds; ++i) {
	    if (FD_ISSET(i, &rfds) || FD_ISSET(i, &wfds)) {
		ForkDesc *desc;

		if ((desc = FindThread(i, -1)) != NULL) {
		    Connection *conn = desc->d_Data;

		    if (conn) {
			/*
			 * handle output I/O (optimization)
			 */

			MBFlush(conn, &conn->co_TMBuf);
			conn->co_FCounter = 0;
		    }

		    /*
		     * Function dispatch
		     */

		    switch(desc->d_Type) {
		    case THREAD_READER:
			HandleReaderMsg(desc);
			break;
		    case THREAD_NNTP:		/* client	  */
		    case THREAD_SPOOL:		/* spool server	  */
		    case THREAD_POST:		/* posting server */
			conn->co_Func(conn);
			break;
		    default:
			/* panic */
			break;
		    }

		    /*
		     * do not call MBFlush after the function because the
		     * function may be waiting for write data to drain and
		     * we don't want to cause write data to drain here and
		     * then not get a select wakeup later.
		     *
		     * check for connection termination
		     */

		    if (conn) {
			if (conn->co_RMBuf.mh_REof && 
			    conn->co_TMBuf.mh_WEof &&
			    conn->co_TMBuf.mh_MBuf == NULL
			) {
			    DeleteConnection(conn);
			    DelThread(desc);
			}
		    }
		}
	    }
	}
	(void)ScanTimers(1, 0);
    }
}

void
HandleReaderMsg(ForkDesc *desc)
{
    int r;
    DnsRes  dres;
    struct msghdr msg;
    struct iovec  iov;
    struct {
#if FDPASS_USES_CMSG
	struct cmsghdr cmsg;
#endif
	int fd;
    } cmsg;

    bzero(&msg, sizeof(msg));

    iov.iov_base = (void *)&dres;
    iov.iov_len = sizeof(dres);

    msg.msg_iov = &iov;
    msg.msg_iovlen = 1;
#if FDPASS_USES_ACC
    msg.msg_accrights = (caddr_t)&cmsg;
    msg.msg_accrightslen = sizeof(cmsg);
#else
    msg.msg_control = (caddr_t)&cmsg;
    msg.msg_controllen = sizeof(cmsg);
#endif
#if FDPASS_USES_CMSG
    msg.msg_flags = 0;
    cmsg.cmsg.cmsg_len = sizeof(cmsg);
#endif
    cmsg.fd = -1;
    errno = 0;

    /*
     * receive message w/ file descriptor
     */

    if ((r = recvmsg(desc->d_Fd, &msg, MSG_EOR|MSG_WAITALL)) == sizeof(dres)) {
	if (DebugOpt)
	    printf("RECV MSG\n");

	if (cmsg.fd >= MAXFDS) {
	    logit(LOG_WARNING, "fd too large %d/%d, increase MAXFDS for select. Closing fd", cmsg.fd, MAXFDS);
	    /*
	     * Tell the main server that we are done with the connection
	     */
	    fcntl(TFd, F_SETFL, 0);
	    write(TFd, &dres, sizeof(dres));
	    fcntl(TFd, F_SETFL, O_NONBLOCK);
	} else if (cmsg.fd >= 0) {
	    ForkDesc *ndesc;
	    Connection *conn;

	    ndesc = AddThread("client", cmsg.fd, -1, THREAD_NNTP, makeReaderSlot(), 0);
	    ++NumReaders;
	    if (DebugOpt)
		printf("add thread fd=%d\n", cmsg.fd);

	    FD_SET(ndesc->d_Fd, &WFds);	/* will cause immediate effect */
	    conn = InitConnection(ndesc, &dres);
	    if (conn->co_Auth.dr_Flags & DF_FEED)
		conn->co_Flags |= COF_SERVER;
	    RTStatusBase(ndesc->d_Slot, "ACTV %-30s", dres.dr_Host);
	    StatusUpdate(conn, "(startup)");
	    NNWriteHello(conn);
	} else {
	    if (DebugOpt)
		printf("recvmsg(): EOF1\n");
	    DelThread(desc);
	    TerminatePending = 1;
	}
    } 

    /*
     * If recv error, check errno.  If temporary error,
     * leave r negative (select loop).  Set r = 0 to 
     * terminate.
     */

    if (r < 0) {
	if (errno != EINTR &&
	    errno != EWOULDBLOCK &&
	    errno != EAGAIN
	) {
	    r = 0;
	}
    }

    /*
     * EOF (or error)
     */

    if (r == 0) {
	if (DebugOpt)
	    printf("recvmsg(): EOF/error from parent %s\n", strerror(errno));
	DelThread(desc);
	TerminatePending = 1;
    }
}

Connection *
InitConnection(ForkDesc *desc, DnsRes *dres)
{
    MemPool    *pool = NULL;
    Connection *conn = zalloc(&pool, sizeof(Connection));

    desc->d_Data = conn;

    if (dres)
	conn->co_Auth = *dres;

    conn->co_Desc = desc;
    conn->co_MemPool = pool;
    gettimeofday(&conn->co_RateTv, NULL);
    MBInit(&conn->co_TMBuf, desc->d_Fd, &conn->co_MemPool, &conn->co_BufPool);
    MBInit(&conn->co_RMBuf, desc->d_Fd, &conn->co_MemPool, &conn->co_BufPool);
    MBInit(&conn->co_ArtBuf, -1, &conn->co_MemPool, &conn->co_BufPool);

    return(conn);
}

void
DeleteConnection(Connection *conn)
{
    MemPool *mpool = conn->co_MemPool;

    if (conn->co_Desc->d_Type == THREAD_NNTP) {
	RTStatusBase(conn->co_Desc->d_Slot, "CLSD %-30s", conn->co_Auth.dr_Host);
	StatusUpdate(conn, "(closed)");
	freeReaderSlot(conn->co_Desc->d_Slot);
	--NumReaders;
	/*
	 * Inform the main server that we are done with the descriptor
	 * by writing the DnsRes structure back to it, so the main server
	 * can track who from where is connecting to what and when that
	 * connection terminates.
	 */
	conn->co_Auth.dr_ByteCount = conn->co_TMBuf.mh_TotalBytes;
	fcntl(TFd, F_SETFL, 0);
	write(TFd, &conn->co_Auth, sizeof(conn->co_Auth));
	fcntl(TFd, F_SETFL, O_NONBLOCK);
    }

    FreeControl(conn);
    freePool(&conn->co_BufPool);
    freePool(&mpool);		/* includes Connection structure itself */
}

void
NNTerminate(Connection *conn)
{
    conn->co_Func = NNTerminate;
    conn->co_State = "term";
    conn->co_RMBuf.mh_REof = 1;
    conn->co_TMBuf.mh_WEof = 1;
    FD_SET(conn->co_Desc->d_Fd, &WFds);

    /*
     * conn->co_SReq is only non-NULL for a client
     * connection.  Server use of the field will have
     * already NULL'd it out in NNServerTerminate().
     *
     * The problem we have is that the server may be actively
     * using the client connection's MBuf's and be in some 
     * intermediate state.  Therefore, we must change the SReq
     * to point to a sink-NULL client XXX.
     */
    if (conn->co_SReq) {
	if (conn->co_SReq->sr_CConn != conn)
	    fatal("NNTerminate(): server conn had non_NULL co_SReq");
	/* 
	 * Disconnect the co_SReq from the client.  This will cause
	 * the server operation-in-progress to abort, if possible.
	 */
	conn->co_SReq->sr_CConn = NULL;
	conn->co_SReq = NULL;
    }
}

void
NNWriteHello(Connection *conn)
{
    const char *postingOk = "(no posting)";
    const char *noReading = "";
    const char *serverType = "NNTP";

    if (conn->co_Auth.dr_Flags & DF_POST)
	postingOk = "(posting ok)";
    if (conn->co_Flags & COF_SERVER) {
	serverType = "FEED";
    } else if ((conn->co_Auth.dr_Flags & DF_READ) == 0) {
	noReading = "(no reading)";
    }
    if ((conn->co_Auth.dr_Flags & (DF_FEED|DF_READ|DF_POST)) == 0) {
	MBPrintf(
	    &conn->co_TMBuf, 
	    "500 %s Diablo Server, you have no permissions.\r\n",
	    ReportedHostName
	);
	NNTerminate(conn);
	return;
    }

    MBPrintf(
	&conn->co_TMBuf, 
	"200 %s Diablo %s Server ready %s%s.\r\n",
	ReportedHostName,
	serverType,
	postingOk,
	noReading
    );
    NNCommand(conn);
}

/*
 * NNCommand() - general command entry.  Attempt to flush output data
 *		 and then go onto the appropriate command set.
 */

#define CMDF_AUTH	0x00000001
#define CMDF_SERVER	0x00000002
#define CMDF_READER	0x00000004
#define CMDF_NOTFEEDONLY 0x00000008

typedef struct Command {
    const char *cmd_Name;
    int		cmd_Flags;
    void	(*cmd_Func)(Connection *conn, char **pptr);
    const char	*cmd_Help;
} Command;

Command Cmds[] = {
    { 
	"article",
	CMDF_AUTH|CMDF_SERVER|CMDF_READER|CMDF_NOTFEEDONLY,
	NNTPArticle,
	"[MessageID|Number]"
    },
    { 
	"body",
	CMDF_AUTH|CMDF_SERVER|CMDF_READER|CMDF_NOTFEEDONLY,	
	NNTPBody,
	"[MessageID|Number]"
    },
    { 
	"date",		
	0	 |CMDF_SERVER|CMDF_READER,
	NNTPDate,
	""
    },
    { 
	"head",	
	CMDF_AUTH|CMDF_SERVER|CMDF_READER|CMDF_NOTFEEDONLY,
	NNTPHead,
	"[MessageID|Number]"
    },
    { 
	"help",		
	0	 |CMDF_SERVER|CMDF_READER,
	NNTPHelp,
	""
    },
    { 
	"ihave",
	CMDF_AUTH|CMDF_SERVER|CMDF_READER,
	NNTPIHave,
	""
    },
    { 
	"takethis",	
	CMDF_AUTH|CMDF_SERVER|0		 ,
	NNTPTakeThis,
	"MessageID"
    },
    { 
	"check",
	CMDF_AUTH|CMDF_SERVER|0		 ,
	NNTPCheck,
	"MessageID"
    },
    { 
	"mode",
	CMDF_AUTH|CMDF_READER,
	NNTPMode,
	"reader|stream"
    },
    { 
	"mode",
	CMDF_AUTH|CMDF_SERVER,
	NNTPMode ,
	"reader|stream|headfeed"
    },
    { 
	"slave",
	CMDF_AUTH|CMDF_SERVER|CMDF_READER,
	NNTPSlave,
	""
    },
    { 
	"quit",
	0	 |CMDF_SERVER|CMDF_READER,
	NNTPQuit,
	""
    },
    { 
	"group",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPGroup ,
	"newsgroup"
    },
    { 
	"last",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPLast,
	""
    },
    { 
	"next",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPNext,
	""
    },
    { 
	"list",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPList,
	"[active|active.times|newsgroups|distributions|distrib.pats|moderators|overview.fmt|subscriptions]"
    },
    { 
	"listgroup",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPListGroup,
	"newsgroup"
    },
    { 
	"newgroups",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPNewgroups,
	"yymmdd hhmmss [\"GMT\"] [<distributions>]"
    },
    { 
	"newnews",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPNewNews,
	"(not implemented)"
    },
    { 
	"post",	
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPPost,
	""
    },
    { 
	"stat",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPStat,
	"[MessageID|Number]"
    },
    { 
	"xgtitle",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPXGTitle,
	"[group_pattern]"
    },
    { 
	"xhdr",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPXHdr,
	"header [range|MessageID]"
    },
    { 
	"xover",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPXOver,
	"[range]"
    },
    { 
	"xpat",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPXPat,
	"header range|MessageID pat"
    },
    { 
	"xpath",
	CMDF_AUTH|0	     |CMDF_READER,
	NNTPXPath,
	"MessageID"
    },
    { 
	"authinfo",
	0	 |CMDF_SERVER|CMDF_READER,
	NNTPAuthInfo,
	"user Name|pass Password"
    }
};

void 
NNCommand(Connection *conn)
{
    MBFlush(conn, &conn->co_TMBuf);
    NNCommand2(conn);
}

void
NNCommand2(Connection *conn)
{
    char *ptr;
    char *cmd;
    char *buf;
    Command *scan;
    int len;

    conn->co_Func = NNCommand2;
    conn->co_State = "waitcmd";

    /*
     * we have to be careful in regards to recursive operation, nor do
     * we want one descriptor to hog the process.  We can't set RFds
     * because the next command may already be entirely loaded into an
     * MBuf so setting RFds may not unblock us.  Instead, we set WFds
     * which basically forces a wakeup at some point in the future.
     */

    if (conn->co_FCounter) {
	FD_SET(conn->co_Desc->d_Fd, &WFds);
	return;
    }
    ++conn->co_FCounter;

    /*
     * get command
     */

    if ((len = MBReadLine(&conn->co_RMBuf, &buf)) == 0) {
	StatusUpdate(conn, "(idle)");
	return;
    }

    /*
     * check EOF
     */

    if (len < 0) {
	NNTerminate(conn);
	return;
    }

    /*
     * strip CR LF
     */

    ptr = buf;

    if (len > 1 && ptr[len-2] == '\r')
	ptr[len-2] = 0;

    if (DebugOpt)
	printf("command: %s\n", ptr);

    /*
     * extract command (note: StatusUpdate() will limit the line length)
     */

    StatusUpdate(conn, "%s", ptr);

    if ((cmd = parseword(&ptr, " \t")) == NULL) {
	NNCommand(conn);
	return;
    }
    {
	int i;

	for (i = 0; cmd[i]; ++i)
	    cmd[i] = tolower((int)(unsigned char)cmd[i]);
    }

    /*
     * Locate and execute command
     */

    for (scan = &Cmds[0]; scan < &Cmds[arysize(Cmds)]; ++scan) {
	if (strcmp(cmd, scan->cmd_Name) == 0) {
	    if (conn->co_Flags & COF_SERVER) {
		if (scan->cmd_Flags & CMDF_SERVER) {
		    if ((conn->co_Auth.dr_Flags & DF_FEEDONLY) == 0)
			break;
		    if ((scan->cmd_Flags & CMDF_NOTFEEDONLY) == 0)
			break;
		}
	    } else {
		if (scan->cmd_Flags & CMDF_READER)
		    break;
	    }
	}
    }
    if (scan < &Cmds[arysize(Cmds)]) {
	if ((scan->cmd_Flags & CMDF_AUTH) &&
	    (conn->co_Auth.dr_Flags & DF_AUTHREQUIRED)
	) {
	    MBPrintf(&conn->co_TMBuf, "480 Authentication required for command\r\n");
	    NNCommand(conn);
	} else {
	    scan->cmd_Func(conn, &ptr);
	}
    } else {
	NNUnknownCommand(conn);
    }
}

void
NNBadCommandUse(Connection *conn)
{
    MBPrintf(&conn->co_TMBuf, "501 Bad command use\r\n");
    NNCommand(conn);
}

void
NNUnknownCommand(Connection *conn)
{
    MBPrintf(&conn->co_TMBuf, "500 What?\r\n");
    NNCommand(conn);
}

/*
 * NNWaitThread() - wait for some other thread to finish
 *		    our request.  Howeve, we allow writes
 *		    to occur on our descriptor while we
 *		    are waiting in case the other thread
 *		    is shoving stuff out our connection.
 */

void
NNWaitThread(Connection *conn)
{
    conn->co_Func = NNWaitThread;
    conn->co_State = "waitrt";
    FD_CLR(conn->co_Desc->d_Fd, &RFds);
    /*FD_CLR(conn->co_Desc->d_Fd, &WFds);*/
}

void
NNTPHelp(Connection *conn, char **pptr)
{
    Command *scan;

    MBPrintf(&conn->co_TMBuf, "100 Legal commands\r\n");
    for (scan = &Cmds[0]; scan < &Cmds[arysize(Cmds)]; ++scan) {
	int ok = 0;

	if (conn->co_Flags & COF_SERVER) {
	    if (scan->cmd_Flags & CMDF_SERVER) {
		ok = 1;
		if ((scan->cmd_Flags & CMDF_NOTFEEDONLY) &&
		    (conn->co_Auth.dr_Flags & DF_FEEDONLY)
		) {
		    ok = 0;
		}
	    }
	} else {
	    if (scan->cmd_Flags & CMDF_READER)
		ok = 1;
	}
	if (ok)
	    MBPrintf(&conn->co_TMBuf, "  %s %s\r\n", scan->cmd_Name, scan->cmd_Help);
    }
    MBPrintf(&conn->co_TMBuf, ".\r\n");
    NNCommand(conn);
}

void
StatusUpdate(Connection *conn, const char *ctl, ...)
{
    char buf[256];
    va_list va;

    snprintf(buf, sizeof(buf), "%c%c%c %-30s",
	((conn->co_Auth.dr_Flags & DF_FEED) ? 'f' : '-'),
	((conn->co_Auth.dr_Flags & DF_READ) ? 'r' : '-'),
	((conn->co_Auth.dr_Flags & DF_POST) ? 'p' : '-'),
	(conn->co_GroupName ? conn->co_GroupName : "(none)")
    );
    va_start(va, ctl);
    vsnprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), ctl, va);
    va_end(va);
    RTStatusUpdate(conn->co_Desc->d_Slot, "%s", buf);
}

int
makeReaderSlot(void)
{
    static int SlotNo;
    int i;

    if (ReaderSlotAry == NULL) {
	ReaderSlotAry = zalloc(&SysMemPool, sizeof(*ReaderSlotAry) * DiabloReaderThreads);
    }
    for (i = 0; i < DiabloReaderThreads; ++i) {
	SlotNo = (SlotNo + 1) % DiabloReaderThreads;
	if (ReaderSlotAry[SlotNo] == 0) {
	    ReaderSlotAry[SlotNo] = 1;
	    return(SlotNo);
	}
    }
    return(-1);
}

void
freeReaderSlot(int slot)
{
    if (slot >= 0) {
	ReaderSlotAry[slot] = 0;
    }
}

void
sigSegVReader(int sigNo)
{
    if (TFd >= 0)
	close(TFd);
    nice(20);
    for (;;)
	;
}

