
/*
 * LIB/CONFIG.C	- Scan configuration file
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 *
 */

#include "defs.h"

Prototype void LoadDiabloConfig(int ac, char **av);

Prototype int DiabloExpire;
Prototype int DiabloHashMethod;
Prototype int DiabloCompatHashMethod;
Prototype int DiabloHashSize;
Prototype int DiabloActiveEnabled;
Prototype int DiabloRememberDays;
Prototype int DiabloActiveDrop;
Prototype int MaxPerRemote;
Prototype int DiabloXRefSlave;
Prototype int DiabloReaderForks;
Prototype int DiabloReaderThreads;
Prototype int DiabloReaderDns;
Prototype int DiabloReaderFeedForks;
Prototype int DiabloReaderCacheMode;
Prototype int DiabloReaderXOverMode;
Prototype int DiabloFeederMaxSpool;
Prototype int HostCacheRebuildTime;

int SetHashMethod(const char *opt, int *pmethod);

int DiabloExpire;
int DiabloHashMethod;
int DiabloCompatHashMethod = -1;
int DiabloHashSize;
int DiabloXRefSlave;
int DiabloReaderForks = 10;
int DiabloReaderThreads = 40;
int DiabloReaderFeedForks = 4;
int DiabloReaderDns = 5;
int DiabloReaderCacheMode = 1;
int DiabloReaderXOverMode = 1;	/* 0 = off, 1 = on, 2 = trackonly */
int DiabloFeederMaxSpool = 14 * 24 * 6;
int DiabloActiveEnabled;
int DiabloRememberDays = REMEMBERDAYS;
int DiabloActiveDrop;
int MaxPerRemote;
int HostCacheRebuildTime = 60 * 60; 	/* Rebuild host dns cache every hour */

void
LoadDiabloConfig(int ac, char **av)
{
    FILE *fi;
    char buf[256];
    int versOk = 0;
    int exitMe = 0;
    int i;
    const char *diabloConfig;

    /*
     * Figure out path to diablo.config.  Check environment first, then
     * look for -C option to override it.  If neither is given, use the
     * default.
     */

    if ((diabloConfig = getenv("DIABLO_CONFIG_PATH")) == NULL) {
        diabloConfig = "%s/diablo.config";
    }

    for (i = 1; i < ac; ++i) {
	char *ptr = av[i];
	if (ptr[0] == '-' && ptr[1] == 'C') {
	    diabloConfig = ptr[2] ? ptr + 2 : av[++i];
	}
    }
    if (diabloConfig == NULL) {
	fprintf(stderr, "Expected argument to -C option\n");
	exit(1);
    }

    if ((fi = xfopen("r", diabloConfig, NewsHome)) != NULL) {
	DiabloActiveEnabled = 0;

	while (fgets(buf, sizeof(buf), fi) != NULL) {
	    char *cmd;
	    char *opt = NULL;
	    int cmdErr = 1;

	    if (buf[0] == '\n' || buf[0] == '#')
		continue;
	    if ((cmd = strtok(buf, " \t\n")) != NULL &&
		(opt = strtok(NULL, " \t\n")) != NULL
	    ) {
		int optErr = 1;

		cmdErr = 0;
		if (strcasecmp(cmd, "version") == 0) {
		    optErr = 0;
		    if (strtod(opt, NULL) > strtod(VERS, NULL))
			versOk = 0;
		    else
			versOk = 1;
		} else if (versOk == 0) {
		    optErr = 0;
		} else if (strcasecmp(cmd, "expire") == 0) {
		    if (opt) {
			if (strcasecmp(opt, "feeder") == 0) {
			    DiabloExpire = EXPIRE_FEEDER;
			    optErr = 0;
			} else if (strcasecmp(opt, "reader") == 0) {
			    DiabloExpire = EXPIRE_READER;
			    optErr = 0;
			}
		    } 
		} else if (strcasecmp(cmd, "hash") == 0) {
		    if (opt) {
			char *p2;
			if ((p2 = strchr(opt, '/')) != NULL)
			    *p2++ = 0;
			optErr = SetHashMethod(opt, &DiabloHashMethod);
			if (p2)
			    optErr = SetHashMethod(p2, &DiabloCompatHashMethod);
		    }
		} else if (strcasecmp(cmd, "remember") == 0) {
		    if (opt) {
			DiabloRememberDays = strtol(opt, NULL, 0);
			optErr = 0;
		    }
		} else if (strcasecmp(cmd, "maxconnect") == 0) {
		    if (opt) {
			MaxPerRemote = strtol(opt, NULL, 0);
			optErr = 0;
		    }
		} else if (strcasecmp(cmd, "hostcacherebuildtime") == 0) {
		    if (opt) {
			HostCacheRebuildTime = strtol(opt, NULL, 0);
			optErr = 0;
		    }
		} else if (strcasecmp(cmd, "hsize") == 0) {
		    if (opt) {
			int n = strtol(opt, &opt, 0);

			switch(*opt) {
			case 'm':
			case 'M':
			    n *= 1024;
			    /* fall through */
			case 'k':
			case 'K':
			    n *= 1024;
			    optErr = 0;
			    break;
			default:
			    break;
			}
			if (n < 256 * 1024 || n > 128 * 1024 * 1024) {
			    optErr = 1;
			    syslog(LOG_CRIT, "Illegal Hash size: %d", n);
			}
			if ((n ^ (n - 1)) != (n << 1) - 1) {
			    syslog(LOG_CRIT, "Hash size not a power of 2: %d", n);
			    optErr = 1;
			}
			if (optErr == 0)
			    DiabloHashSize = n;
		    }
		} else if (strcasecmp(cmd, "active") == 0) {
		    if (opt) {
			if (strcasecmp(opt, "on") == 0) {
			    optErr = 0;
			    DiabloActiveEnabled = 1;
			} else if (strcasecmp(opt, "off") == 0) {
			    optErr = 0;
			    DiabloActiveEnabled = 0;
			} else if (opt[0] == '/') {
			    optErr = 0;
			    DiabloActiveEnabled = 1;
			    ServerDActivePat = strdup(opt);
			}
		    }
		} else if (strcasecmp(cmd, "activedrop") == 0) {
		    if (opt) {
			if (strcasecmp(opt, "on") == 0) {
			    optErr = 0;
			    DiabloActiveDrop = 1;
			} else if (strcasecmp(opt, "off") == 0) {
			    optErr = 0;
			    DiabloActiveDrop = 0;
			}
		    }
		} else if (strcasecmp(cmd, "maxspool") == 0) {
		    if (opt) {
			DiabloFeederMaxSpool = strtol(opt, NULL, 0);
			if (DiabloFeederMaxSpool < 1)
			    DiabloFeederMaxSpool = 1;
			DiabloFeederMaxSpool *= 24 * 6;
			optErr = 0;
		    }
		} else if (strcasecmp(cmd, "slave") == 0) {
		    if (opt) {
			if (strcasecmp(opt, "on") == 0) {
			    optErr = 0;
			    DiabloXRefSlave = 1;
			} else if (strcasecmp(opt, "off") == 0) {
			    optErr = 0;
			    DiabloXRefSlave = 0;
			}
		    }
		} else if (strcasecmp(cmd, "readerdns") == 0) {
		    if (opt) {
			DiabloReaderDns = strtol(opt, NULL, 0);
			optErr = 0;
		    }
		} else if (strcasecmp(cmd, "readerforks") == 0) {
		    if (opt) {
			DiabloReaderForks = strtol(opt, NULL, 0);
			optErr = 0;
		    }
		} else if (strcasecmp(cmd, "readerfeedforks") == 0) {
		    if (opt) {
			DiabloReaderFeedForks = strtol(opt, NULL, 0);
			optErr = 0;
		    }
		} else if (strcasecmp(cmd, "readerthreads") == 0) {
		    if (opt) {
			DiabloReaderThreads = strtol(opt, NULL, 0);
			optErr = 0;
		    }
		} else if (strcasecmp(cmd, "readercache") == 0) {
		    if (opt) {
			if (strcasecmp(opt, "on") == 0) {
			    optErr = 0;
			    DiabloReaderCacheMode = 1;
			} else if (strcasecmp(opt, "off") == 0) {
			    optErr = 0;
			    DiabloReaderCacheMode = 0;
			}
		    }
		} else if (strcasecmp(cmd, "readerxover") == 0) {
		    if (opt) {
			if (strcasecmp(opt, "on") == 0) {
			    DiabloReaderXOverMode = 1;
			    optErr = 0;
			} else if (strcasecmp(opt, "off") == 0) {
#if 0
			    DiabloReaderXOverMode = 0;	/* NOT YET SUPPORTED */
			    optErr = 0;
#endif
			} else if (strcasecmp(opt, "trackonly") == 0) {
			    DiabloReaderXOverMode = 2;
			    optErr = 0;
			}
		    }
		} else if (strncasecmp(cmd, "path_", 5) == 0) {
		    /*
		     * Paths (defaults already preset in globals)
		     */
		    const char **pptr = NULL;

		    if (strcasecmp(cmd + 5, "home") == 0)
			pptr = &NewsHome;
		    else if (strcasecmp(cmd + 5, "spool") == 0) 
			pptr = &SpoolHomePat;
		    else if (strcasecmp(cmd + 5, "dqueue") == 0) 
			pptr = &DQueueHomePat;
		    else if (strcasecmp(cmd + 5, "group") == 0) 
			pptr = &GroupHomePat;
		    else if (strcasecmp(cmd + 5, "cache") == 0) 
			pptr = &CacheHomePat;
		    else if (strcasecmp(cmd + 5, "log") == 0) 
			pptr = &LogHomePat;
		    else if (strcasecmp(cmd + 5, "feeds") == 0) 
			pptr = &FeedsHomePat;
		    else if (strcasecmp(cmd + 5, "lib") == 0) 
			pptr = &LibHomePat;
		    else if (strcasecmp(cmd + 5, "db") == 0) 
			pptr = &DbHomePat;
		    else if (strcasecmp(cmd + 5, "server_dactive") == 0) 
			pptr = &ServerDActivePat;
		    else if (strcasecmp(cmd + 5, "reader_dactive") == 0) 
			pptr = &ReaderDActivePat;
		    else if (strcasecmp(cmd + 5, "diablo_socket") == 0) 
			pptr = &DiabloSocketPat;
		    else if (strcasecmp(cmd + 5, "dreader_socket") == 0) 
			pptr = &DReaderSocketPat;
		    else if (strcasecmp(cmd + 5, "dexpire") == 0) 
			pptr = &DExpireCtlPat;
		    else if (strcasecmp(cmd + 5, "dcontrol") == 0) 
			pptr = &DControlCtlPat;
		    else if (strcasecmp(cmd + 5, "dhistory") == 0) 
			pptr = &DHistoryPat;
		    else if (strcasecmp(cmd + 5, "diablo_hosts") == 0) 
			pptr = &DiabloHostsPat;
		    else if (strcasecmp(cmd + 5, "dreader_hosts") == 0) 
			pptr = &DReaderHostsPat;
		    else if (strcasecmp(cmd + 5, "dserver_hosts") == 0) 
			pptr = &DServerHostsPat;
		    else if (strcasecmp(cmd + 5, "moderators") == 0) 
			pptr = &ModeratorsPat;
		    else if (strcasecmp(cmd + 5, "spam_cache") == 0) 
			pptr = &SpamCachePat;
		    else if (strcasecmp(cmd + 5, "pcommit_cache") == 0) 
			pptr = &PCommitCachePat;
		    else if (strcasecmp(cmd + 5, "dexpover_list") == 0)
			pptr = &DExpireOverListPat;
		    else if (strcasecmp(cmd + 5, "dhosts_cache") == 0)
			pptr = &DHostsCachePat;
		    else if (strcasecmp(cmd + 5, "dnewsfeeds") == 0) 
			pptr = &DNewsfeedsPat;
		    else if (strcasecmp(cmd + 5, "dnntpspool") == 0) 
			pptr = &DNNTPSpoolCtlPat;
		    else if (strcasecmp(cmd + 5, "distrib_pats") == 0) 
			pptr = &DistribDotPatsPat;
		    else if (strcasecmp(cmd + 5, "distributions") == 0) 
			pptr = &DistributionsPat;
		    else
			cmdErr = 1;

		    /*
		     * XXX check for zero or one '%s' in opt.
		     */
		    if (cmdErr == 0 && opt != NULL) {
			*pptr = strdup(opt);
			optErr = 0;
		    }
		} else {
		    cmdErr = 1;
		    optErr = 0;
		}
		if (optErr) {
		    exitMe = 1;
		    syslog(LOG_CRIT, "Unknown diablo.config command: %s %s",
			cmd, ((opt) ? opt : "")
		    );
		}
	    }
	    if (cmdErr && cmd) {
		exitMe = 1;
		syslog(LOG_CRIT, "Unknown diablo.config command: %s %s",
		    cmd, ((opt) ? opt : "")
		);
	    }
	}
	fclose(fi);
    } else {
	syslog(LOG_CRIT, "Unable to open diablo.config using pattern %s", diabloConfig);
	exitMe = 1;
    }
    if (exitMe) {
	fprintf(stderr, "critical error parsing diablo.config, exiting.  See syslog\n");
	exit(1);
    }
}

int
SetHashMethod(const char *opt, int *pmethod)
{
    int optErr = 0;

    if (strcasecmp(opt, "prime") == 0) {
	*pmethod = HASH_PRIME;
    } else if (strcasecmp(opt, "crc") == 0) {
	*pmethod = HASH_CRC;
    } else if (strcasecmp(opt, "ocrc") == 0) {
	*pmethod = HASH_OCRC;
    } else {
	optErr = 1;
    }
    return(optErr);
}

