#include "defs.h"

#if NEED_TERMIOS
#include <sys/termios.h>
#endif

typedef struct Feed {
    struct Feed *fe_Next;
    char        *fe_Label;
    int         fe_Fd;
    char        *fe_Buf;
    int         fe_BufIdx;
    int         fe_BufMax;
    int         fe_Failed;
} Feed;

Feed    *FeBase;
MemPool *ParProcMemPool;
char    *MySpoolHome;

void writeFeedOut(const char *label, const char *file, const char *msgid, const char *offSize, int rt, int headOnly);
void flushFeedOut(Feed *fe);

int 
fwCallBack(const char *hlabel, const char *msgid, const char *path, const char *offsize, int plfo, int headOnly)
{
    writeFeedOut(hlabel, path, msgid, offsize, ((plfo > 0) ? 1 : 0), headOnly);
    return(0);
}

void
writeFeedOut(const char *label, const char *file, const char *msgid, const char *offSize, int rt, int headOnly)
{
    Feed *fe;

    /*
     * locate feed
     */

    for (fe = FeBase; fe; fe = fe->fe_Next) {
	if (strcmp(label, fe->fe_Label) == 0)
	    break;
    }

    /*
     * allocate feed if not found
     */

    if (fe == NULL) {
	fe = zalloc(&ParProcMemPool, sizeof(Feed) + strlen(label) + 1);
	fe->fe_Label = (char *)(fe + 1);
	strcpy(fe->fe_Label, label);
	fe->fe_Fd = xopen(O_APPEND|O_RDWR|O_CREAT, 0644, "%s/%s", PatExpand(DQueueHomePat), label);
	if (fe->fe_Fd >= 0) {
	    int bsize;

	    fe->fe_Buf = pagealloc(&bsize, 1);
	    fe->fe_BufMax = bsize;
	    fe->fe_BufIdx = 0;
	    fe->fe_Failed = 0;
	}
	fe->fe_Next = FeBase;
	FeBase = fe;
    }

    /*
     * write to buffered feed, flushing buffers if there is not enough
     * room.  If the line is too long, something has gone wrong and we
     * throw it away.
     *
     * note that we cannot fill the buffer to 100%, because the trailing
     * nul (which we do not write) will overrun it.  I temporarily add 4 to
     * l instead of 3 to include the trailing nul in the calculations, but
     * subtract it off after the actual copy operation.
     */

    if (fe->fe_Fd >= 0) {
	int l = strlen(file) + strlen(msgid) + strlen(offSize) + (3 + 32 + 1);

	/*
	 * line would be too long?
	 */
	if (l < fe->fe_BufMax) {
	    /*
	     * line fits in buffer with trailing nul ?
	     */
	    if (l >= fe->fe_BufMax - fe->fe_BufIdx)
		flushFeedOut(fe);

	    sprintf(fe->fe_Buf + fe->fe_BufIdx, "%s %s %s%s\n",
		file, msgid, offSize, 
		(headOnly ? " H" : "")
	    );
	    fe->fe_BufIdx += strlen(fe->fe_Buf + fe->fe_BufIdx);
	    if (rt)
		flushFeedOut(fe);
	}
    }
}

void
flushFeeds(int justClose)
{
    Feed *fe;

    while ((fe = FeBase) != NULL) {
	if (justClose == 0)
	    flushFeedOut(fe);
	if (fe->fe_Buf)
	    pagefree(fe->fe_Buf, 1);
	if (fe->fe_Fd >= 0)
	    close(fe->fe_Fd);
	fe->fe_Fd = -1;
	fe->fe_Buf = NULL;
	FeBase = fe->fe_Next;
	zfree(&ParProcMemPool, fe, sizeof(Feed) + strlen(fe->fe_Label) + 1);
    }
}

void
flushFeedOut(Feed *fe)
{
    if (fe->fe_BufIdx && fe->fe_Buf && fe->fe_Fd >= 0) {
	/*
	 * flush buffer.  If the write fails, we undo it to ensure
	 * that we do not get garbaged feed files.
	 */
	int n = write(fe->fe_Fd, fe->fe_Buf, fe->fe_BufIdx);
	if (n >= 0 && n != fe->fe_BufIdx) {
	    ftruncate(fe->fe_Fd, lseek(fe->fe_Fd, 0L, 1) - n);
	}
	if (n != fe->fe_BufIdx && fe->fe_Failed == 0) {
	    fe->fe_Failed = 1;
	    syslog(LOG_INFO, "failure writing to feed %s", fe->fe_Label);
	}
    }
    fe->fe_BufIdx = 0;
}

int
main(int ac, char **av)
{
    char buf[8192];

    LoadDiabloConfig(ac, av);

     /*
     * Extract the spool home pattern into a real path and place
     * in MySpoolHome
     */

    {
        int l = strlen(PatExpand(SpoolHomePat)) + 1;
        MySpoolHome = malloc(l);
        snprintf(MySpoolHome, l, PatExpand(SpoolHomePat));
    }

    LoadNewsFeed(0, 1, NULL);

    while (fgets(buf, sizeof(buf), stdin) != NULL) {
	char *s1 = strtok(buf, " \t\n");
        if (s1 == NULL)
            continue;

        if (strncmp(s1, "SOUT", 4) == 0) {
            char *path = strtok(NULL, " \t\n");
            const char *msgid = MsgId(strtok(NULL, " \t\n"));
            char *offsize = strtok(NULL, " \t\n");
            char *nglist = strtok(NULL, " \t\n");
            char *dist = strtok(NULL, " \t\n");
            char *npath = strtok(NULL, " \t\n");
            char *headOnly = strtok(NULL, " \t\n");
            if (path && offsize && msgid && nglist) {
                int spamArt = 0;
		if (npath == NULL) npath = "1";
		if (headOnly == NULL) headOnly = "0";
                printf(
                    "SOUTLINE: %s %s %s %s %s HO=%s\n",
                    path, 
                    offsize,
                    msgid,
                    nglist,
                    npath, 
                    headOnly
                );

    		FeedWrite(fwCallBack, msgid, path, offsize, nglist, npath, dist, headOnly, spamArt);
	    }
	}
    }
    exit(0);
}
