/*	
 *   Copyright (C) 1995  Pierre Ficheux (pierre@rd.lectra.fr)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
static char rcsid[] = "$Id: Timer.c,v 1.3 1996/10/04 17:03:05 pierre Exp $";

/*
 * Timer, a digital clock/timer widget
 */

#include <stdio.h>
#include <time.h>

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/Misc.h>
#include <X11/Xaw/XawInit.h>

#include "TimerP.h"

/****************************************************************
 *
 * Full class record constant
 *
 ****************************************************************/

/* Private Data */
static char defaultTranslations[] = "";

#define offset(field) XtOffsetOf(TimerRec, field)
static XtResource resources[] = { 
    {XtNfont,  XtCFont, XtRFontStruct, sizeof(XFontStruct *),
	offset(timer.font),XtRString, XtDefaultFont},
    {XtNclockForeground, XtCClockForeground, XtRPixel, sizeof(Pixel),
	offset(timer.clock_foreground), XtRString, XtDefaultForeground},
    {XtNtimerForeground, XtCTimerForeground, XtRPixel, sizeof(Pixel),
	offset(timer.timer_foreground), XtRString, XtDefaultForeground},
    {XtNmargin, XtCBorderWidth, XtRDimension, sizeof(Dimension),
	offset(timer.margin), XtRImmediate, (XtPointer)2},
    {XtNtimerMode, XtCTimerMode, XtRBoolean, sizeof(Boolean),
	offset(timer.timer_mode), XtRImmediate, (XtPointer)False},
    {XtNmin, XtCMin, XtRInt, sizeof (int),
	offset(timer.min), XtRImmediate, (XtPointer) 0},
    {XtNsec, XtCSec, XtRInt, sizeof (int),
	offset(timer.sec), XtRImmediate, (XtPointer) 0},
};
#undef offset

static void DrawTimer();
static XtTimerCallbackProc RefreshTimer();

static Boolean SetValues();
static void Initialize(), Redisplay();
static void Destroy();
static void ClassInitialize();
static void Realize(), Resize();

TimerClassRec timerClassRec = {
  {
#define superclass			(&widgetClassRec)
    (WidgetClass)superclass,		/* superclass		  */	
    "Timer",				/* class_name		  */
    sizeof(TimerRec),			/* size			  */
    ClassInitialize,			/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    Realize,				/* realize		  */
    NULL,				/* actions		  */
    0,					/* num_actions		  */
    resources,				/* resources		  */
    XtNumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    FALSE,				/* compress_motion	  */
    TRUE,				/* compress_exposure	  */
    TRUE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    Destroy,				/* destroy		  */
    Resize,				/* resize		  */
    Redisplay,				/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    defaultTranslations,		/* tm_table		  */
    XtInheritQueryGeometry,		/* query_geometry	  */
    XtInheritDisplayAccelerator,	/* display_accelerator	  */
    NULL				/* extension		  */
  },  /* CoreClass fields initialization */
};

  /* for public consumption */
WidgetClass timerWidgetClass = (WidgetClass) &timerClassRec;

/****************************************************************
 *
 * Private Procedures
 *
 ****************************************************************/


/* ARGSUSED */
static void 
Initialize(request, new, args, num_args)
Widget request, new;
ArgList args;			/* unused */
Cardinal *num_args;		/* unused */
{
  TimerWidget tw = (TimerWidget) new;
  XGCValues	values;

  values.foreground	= tw->timer.clock_foreground;
  values.background	= tw->core.background_pixel;
  values.font		= tw->timer.font->fid;
  tw->timer.clock_GC = XtGetGC((Widget)tw, (unsigned) GCForeground | GCBackground | GCFont, &values);
  values.foreground	= tw->timer.timer_foreground;
  tw->timer.timer_GC = XtGetGC((Widget)tw, (unsigned) GCForeground | GCBackground | GCFont, &values);

  tw->timer.state = TimerOff;
  tw->timer.min = tw->timer.sec = 0;

  tw->core.width = 2 * tw->timer.margin + 5 * tw->timer.font->max_bounds.width;
  tw->core.height = 2 * tw->timer.margin + tw->timer.font->max_bounds.ascent + tw->timer.font->max_bounds.descent;
}

/***************************
*
*  Action Procedures
*
***************************/

/*
 * Repaint the widget window
 */

/************************
*
*  REDISPLAY (DRAW)
*
************************/

/* ARGSUSED */
static void 
Redisplay(w, event, region)
Widget w;
XEvent *event;
Region region;
{
    TimerWidget tw = (TimerWidget) w;

    /* If clock mode, start now */
    if (!tw->timer.timer_mode && tw->timer.state != TimerOn) {
	tw->timer.id = XtAppAddTimeOut(XtWidgetToApplicationContext((Widget)tw), (unsigned long) 1000, (XtTimerCallbackProc) RefreshTimer, (XtPointer)tw);
	tw->timer.state = TimerOn;
    }

    DrawTimer (tw, True, True);
}


static void 
Destroy(w)
Widget w;
{
  TimerWidget tw = (TimerWidget) w;
  XtReleaseGC (w, tw->timer.timer_GC);
  XtReleaseGC (w, tw->timer.clock_GC);
}

/*
 * Set specified arguments into widget
 */

/* ARGSUSED */
static Boolean 
SetValues (current, request, new)
Widget current, request, new;
{
    TimerWidget curtw = (TimerWidget) current;
    TimerWidget newtw = (TimerWidget) new;
    Boolean redisplay = False;

    if (curtw->timer.font != newtw->timer.font || 
	curtw->timer.clock_foreground != newtw->timer.clock_foreground || 
	curtw->timer.timer_foreground != newtw->timer.timer_foreground || 
	curtw->timer.margin != newtw->timer.margin ||
	curtw->timer.min != newtw->timer.min ||
	curtw->timer.sec != newtw->timer.sec ||
	curtw->timer.timer_mode != newtw->timer.timer_mode) {
	
	if (newtw->timer.timer_mode == False)
	    timerResetTimer (curtw);
	
	redisplay = True;
    }

    return (redisplay);
}

static void ClassInitialize()
{
}

static void Realize(w, valueMask, attributes)
    Widget w;
    Mask *valueMask;
    XSetWindowAttributes *attributes;
{
    XtCreateWindow(w, (unsigned) InputOutput, (Visual *) CopyFromParent,
		   *valueMask, attributes);

}

static void Resize(w)
    Widget w;
{
}

static void DrawTimer(tw, force, redisplay)
TimerWidget tw;
Boolean force, redisplay;
{
    time_t t = time(0);
    struct tm *tm;
    char value[256];

    if (tw->timer.timer_mode) {
	sprintf (value, "%02d:%02d", tw->timer.min, tw->timer.sec);
	if (!redisplay) {
	    tw->timer.sec++;
	    if (tw->timer.sec == 60) {
		tw->timer.sec = 0;
		tw->timer.min++;
	    }
	}
    }
    else {
	if (tw->timer.sec == 60 || force) {
	    tw->timer.sec = 0;
	    tm = localtime (&t);
	    sprintf (value, "%02d:%02d", tm->tm_hour, tm->tm_min);
	}
	else {
	    if (!redisplay)
		tw->timer.sec++;
	    return;
	}
    }

    XDrawImageString(XtDisplay((Widget)tw), XtWindow((Widget)tw), (tw->timer.timer_mode ? tw->timer.timer_GC : tw->timer.clock_GC), tw->timer.margin, tw->timer.margin + tw->timer.font->max_bounds.ascent, value, strlen(value));
}


static XtTimerCallbackProc 
RefreshTimer(tw, id)
TimerWidget tw;
XtIntervalId *id;
{
    if (tw->timer.state == TimerOn) {
	tw->timer.id = XtAppAddTimeOut(XtWidgetToApplicationContext((Widget)tw), (unsigned long) 1000, (XtTimerCallbackProc) RefreshTimer, (XtPointer)tw);

	DrawTimer (tw, False, False);	
    }
}

/*
 * Public functions
 */

void timerStartTimer (w)
Widget w;
{
    TimerWidget tw = (TimerWidget)w;

    if (tw->timer.timer_mode && tw->timer.state == TimerOff) {
	tw->timer.id = XtAppAddTimeOut(XtWidgetToApplicationContext((Widget)tw), (unsigned long) 1000, (XtTimerCallbackProc) RefreshTimer, (XtPointer)tw);
	tw->timer.state = TimerOn;
    }
}

void timerStopTimer (w)
Widget w;
{
    TimerWidget tw = (TimerWidget)w;

    if (tw->timer.timer_mode && tw->timer.state == TimerOn) {
	tw->timer.state = TimerOff;
	XtRemoveTimeOut (tw->timer.id);
    }
}

void timerResetTimer (w)
Widget w;
{
    TimerWidget tw = (TimerWidget)w;

    if (tw->timer.timer_mode) {
	if (tw->timer.state == TimerOn)
	   timerStopTimer (w);   

	tw->timer.min = tw->timer.sec = 0;
	DrawTimer (tw, False, False);
    }
}

