/*
 ** Copyright (c) 1991-1995 Xerox Corporation.  All Rights Reserved.
 **
 ** Unlimited use, reproduction, and distribution of this software is
 ** permitted.  Any copy of this software must include both the above
 ** copyright notice of Xerox Corporation and this paragraph.  Any
 ** distribution of this software must comply with all applicable United
 ** States export control laws.  This software is made available AS IS,
 ** and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 ** INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
 ** AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
 ** PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
 ** THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
 ** CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
 ** XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 */
/* $Id: code.c,v 1.100 1996/03/13 19:40:12 janssen Exp $ */
/* Last edited by Mike Spreitzer January 10, 1996 1:41 am PST */

#include "cstubber.h" 

struct double_s {
  Context c;
  Type t;
  unsigned int id;
  Argument default_arm;
};

static boolean  IncludeComma = FALSE;
static int mCount;

static char   *ArrayName = NULL;
static unsigned  ExceptionIndex = 0;
static unsigned  MethodIndex = 0;

static void classIDs (refany elt, refany rock)
{
  Type            class = (Type) elt;
  Context         context = (Context) rock;
  fprintf(context->file, "\t\"%s\",\n", ur_type(class)->uid);
}

static void generateClassTable (refany elt, refany rock)
{
  Type            class = (Type) elt;
  Context         context = (Context) rock;
  char           *tn;
  Class           o;

  if (type_basic_type(class) != object_Type)
    return;
  tn = c_type_name(class);
  o = class_object(class);

  fprintf(context->file, "ilu_Class _%s__ILUType = NULL;\n", tn);

}

static void 
generateExceptionEntry(refany elt, refany rock)
{
  Exception e = (Exception) elt;
  Context context = (Context) rock;
  if (e->interface == context->interface && e->import == NULL)
    fprintf(context->file,
	    "ILU_C_ExceptionCode _%s__Exception_%s = ILU_NIL;\n",
	    c_interface_name(e->interface), c_simple_name(e->name));
}

static void 
generateExceptionTable(Interface interface, Context context)
{
  list_enumerate(interface->exceptions, generateExceptionEntry,
		 context);
}

static void setExceptionValue (refany elt, refany rock)
{
  Exception       e = (Exception) elt;
  Context         context = (Context) rock;
  FILE           *f = context->file;
  enum PrimitiveTypes t;
  Type            ut = ur_type(e->type);

  if (e->type == NULL)
    return;
  t = type_basic_type(ut);
  fprintf(f, "  else if (stat->returnCode == ex_%s) {\n",
	  c_exception_name(e));
  fprintf(f, "    stat->ptr = (void *) ilu_must_malloc (sizeof (%s));\n",
	  c_type_name(ut));
  if (t == array_Type) {
    char           *rtn = c_return_type(ut);
    fprintf(f, "    memcpy ((void *)((%s)stat->ptr), (void *) va_arg(ap, %s), sizeof(%s));\n",
	    rtn, rtn, c_type_name(ut));
  } else
    fprintf(f, "    *(%s)stat->ptr = %sva_arg (ap, %s);\n",
	    c_parameter_type(ut, Out),
	    ((t == record_Type || t == union_Type
	      || (t == sequence_Type
		  && !(TypeIsString(ut) || TypeIsWString(ut))))
	     ? "*" : ""),
	    c_parameter_type(ut, In));
  if (HasFreeRoutine(ut))
    fprintf(f, "    stat->freeRoutine = (void (*) (void *)) %s__Free;\n",
	    c_type_name(UltimateType(ut)));
  else
    fprintf(f, "    stat->freeRoutine = ((void (*) (void *)) 0);\n");
  fprintf(f, "  }\n");
}

static void generateSignalException (Context  context)
{
  fprintf(context->file, "#include <stdarg.h>\n\n");
  fprintf(context->file, "void %s__BindExceptionValue (ILU_C_ENVIRONMENT *stat, ilu_Exception exception, ...)\n",
	  c_interface_name(context->interface));
  fprintf(context->file, "{\n");
  fprintf(context->file, "  va_list ap;\n");
  fprintf(context->file, "  va_start (ap, exception);\n");
  fprintf(context->file, "  stat->_major = ILU_C_USER_EXCEPTION;\n");
  fprintf(context->file, "  stat->returnCode = exception;\n");
  fprintf(context->file, "  if (exception == NULL)\n");
  fprintf(context->file, "    /* no exception */;\n");
  list_enumerate(context->interface->exceptions, setExceptionValue,
		 context);
  fprintf(context->file, "  else\n");
  fprintf(context->file,
	  "    _ilu_Assert(0, \"bad exn given to %s__BindExceptionValue\");\n",
	  c_interface_name(context->interface));
  fprintf(context->file, "  va_end (ap);\n");
  fprintf(context->file, "}\n\n");
}

static void generateExceptionProcs (Context  context)
{
  if (list_size(context->interface->exceptions) > 0)
    generateSignalException (context);
}

static int DimStop;
static int DimCount;

static void dimHeader (long d, Context context)
{
  if (DimStop >= 0 AND DimCount >= DimStop)
    return;

  fprintf (context->file, "%*.*s{\n%*.*sregister int _i%u;\n%*.*sfor (_i%u = 0;  _i%u < %lu;  _i%u += 1)\n",
	   DimCount*2+2, DimCount*2+2, "", DimCount*2+4, DimCount*2+4, "",
	   DimCount, DimCount*2+4, DimCount*2+4, "",
	   DimCount, DimCount, d, DimCount);
  DimCount += 1;
}

static void dimRef (long d, char * buf)
{
  if (DimStop >= 0 AND DimCount >= DimStop)
    return;

  sprintf (buf + strlen (buf), "[_i%u]", DimCount);
  DimCount += 1;
}

static void dimFooter (long d, Context context)
{
  DimCount -= 1;
  if (DimStop >= 0 AND DimCount < DimStop)
    return;

  fprintf (context->file, "%*.*s}\n", (DimCount - DimStop)*2, (DimCount - DimStop)*2, "");
}

static void outputRecordArg (Argument arg, Context context)
{
  char buf[1000];
  enum PrimitiveTypes t;
  Type ut = ur_type(arg->type);

  t = type_basic_type (ut);
  sprintf (buf, "(%s_val->%s)",     (t == record_Type OR
				     (t == sequence_Type AND !TypeIsString(ut)) OR
				     t == union_Type) ? "&" : "", c_argument_name(arg));
  MarshallValue (context, ut, buf, 2);
}

static void caseConst (ConstantValue val, struct double_s *s)
{
  switch (val->type)
    {
    case integer_Type:
    case shortinteger_Type:
    case cardinal_Type:
    case shortcardinal_Type:
    case byte_Type:
      fprintf (s->c->file, "    case %s%ld:\n", (val->val.i.sign < 0) ? "-" : "", val->val.i.value);
      break;

    case shortcharacter_Type:
      fprintf (s->c->file, "    case %s_%s:\n", c_interface_name (s->t->interface), val->val.s);
      break;

    case boolean_Type:
      fprintf (s->c->file, "    case ilu_%s:\n", val->val.b ? "TRUE" : "FALSE");
      break;

    default:       error ("illegal discriminator value\n");
    }
}

static void outputUnionType (Argument a, struct double_s * s)
{
  char buffer[1000];
  int mkReferent;
  enum PrimitiveTypes t = type_basic_type (ur_type(a->type));
  Type ut = ur_type(a->type);
  char *name;

  if (a->name->base_name != NULL)
    name = (char *) c_simple_name (a->name);
  else
    name = (char *) c_string (type_name (a->type));
  mkReferent = !TypeIsString (ut) && (t == record_Type OR t == union_Type OR t == sequence_Type);
  sprintf (buffer, "%s_val->_u.%s", (mkReferent) ? "&" : "", name);
  if (a == s->default_arm)
    fprintf (s->c->file, "    default:\n");
  else
    list_enumerate (a->values, (void (*)(refany, refany)) caseConst, s);
  MarshallValue (s->c, ut, buffer, 6);
  fprintf (s->c->file, "      break;\n");
}

static void ComputeTotalNumberOfElements (unsigned long dim, unsigned long *total)
{
  *total = *total * dim;
}

static void FindZero (long unsigned int val, boolean *z)
{
  if (val == 0)
    *z = TRUE;
}

static void generateOutputCode (Type type, enum PrimitiveTypes t, Context context)
{
  TypeDescription d = type_description (type);
  char buf[1000];

  fprintf (context->file, "void _%s__Output (ilu_Call _call, %s _val, ilu_Error *_err)\n{\n",
	   c_type_name(type), c_parameter_type (type, In));
  if (t != optional_Type)
    fprintf (context->file, "  if (_val == NULL)\n    return;\n\n");

  if (t == array_Type)
    {
      unsigned long size = 1;
      int dims = list_size(d->structuredDes.array.dimensions);
      boolean zeroarray = FALSE;
      enum PrimitiveTypes t2 = type_basic_type(d->structuredDes.array.type);

      list_enumerate(d->structuredDes.array.dimensions, (void (*)(refany, refany)) FindZero, &zeroarray);

      list_enumerate (d->structuredDes.array.dimensions, (void (*)(refany, refany)) ComputeTotalNumberOfElements, &size);

      if (t2 == byte_Type OR t2 == shortcharacter_Type)
	{
	  DimStop = dims - 1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  sprintf (buf, "_val");
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file, "%*.*silu_Output%s (_call, %s, %lu, _err);\n", dims * 2, dims*2, "",
		   (t2 == byte_Type) ? "Opaque" : "StringVec", buf,
		   (cardinal) list_ref(d->structuredDes.array.dimensions, dims-1));
	  DimCount = dims - 1;
	  DimStop = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  DimStop = -1;
	}
      else if (t2 == character_Type)
	{
	  DimStop = dims - 1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  sprintf (buf, "_val");
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file, "%*.*silu_OutputWStringVec (_call, _val, %lu, NULL, NULL, _err);\n",     dims * 2 + 2, dims*2+2, "", (cardinal) list_ref(d->structuredDes.array.dimensions, dims-1));
	  DimCount = dims - 1;
	  DimStop = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  DimStop = -1;
	}
      else
	{
	  fprintf (context->file, "  ilu_OutputArray (_call, %lu, _err);  if (ILU_ERRNOK(*_err)) return;\n", size);
	  DimStop = -1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  sprintf (buf, "%s_val", (!TypeIsString (d->structuredDes.array.type) &&
				   (t2 == record_Type OR t2 == union_Type OR t2 == sequence_Type)) ? "&" : "");
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  MarshallValue (context, ur_type(d->structuredDes.array.type), buf, 2*dims+4);
	  DimCount = dims;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	}

      fprintf (context->file, "  ilu_EndArray (_call, _err);\n");
    }

  else if (t == sequence_Type)
    {
      Type et = ur_type(d->structuredDes.sequence.type);
      enum PrimitiveTypes st = type_basic_type (ur_type(d->structuredDes.sequence.type));

      if (st == byte_Type)
	{
	  fprintf (context->file, "  ilu_OutputBytes (_call, _val->_buffer, _val->_length, %u, _err);\n",
		   d->structuredDes.sequence.limit);
	}
      else if (st == shortcharacter_Type)
	{
	  fprintf (context->file, "  ilu_OutputString (_call, (ilu_string) _val, _ILU_C_SafeStrlen((ilu_string) _val), %u, _err);\n",
		   d->structuredDes.sequence.limit);
	}
      else if (st == character_Type)
	{
	  fprintf (context->file, "  ilu_OutputWString (_call, _val, _ILU_C_SafeWStrlen(_val), %u, NULL, NULL, _err);\n",
		   d->structuredDes.sequence.limit);
	}
      else
	{
	  fprintf (context->file, "  ilu_OutputSequence (_call, _val->_length, %lu, _err);\n",
		   d->structuredDes.sequence.limit);
	  fprintf (context->file, "  if (ILU_ERRNOK(*_err)) return;\n");
	  fprintf (context->file, "  {\n    %s *p;  int i;\n\n", c_type_name (et));
	  fprintf (context->file, "    for (p = _val->_buffer, i = 0;  i < _val->_length;  p++, i++)\n");
	  sprintf (buf, "%sp",         (st == record_Type OR (st == sequence_Type AND NOT TypeIsString(et)) OR st == union_Type) ? "" : "*");
	  MarshallValue (context, et, buf, 6);
	  fprintf (context->file, "  }\n  ilu_EndSequence (_call, _err);\n");
	}
    }
  else if (t == union_Type)
    {
      struct double_s  s;

      s.c = context;
      s.t = ur_type(d->structuredDes.uniond.discriminator_type);
      s.id = 0;
      s.default_arm = d->structuredDes.uniond.default_arm;
      fprintf (context->file, "  ilu_OutputUnion (_call, _val->_d, sizeof(%s), _err);\n",
	       c_type_name(s.t));
      fprintf (context->file, "  if (ILU_ERRNOK(*_err)) return;\n");
      fprintf (context->file, "  switch (_val->_d) {\n");
      list_enumerate (d->structuredDes.uniond.types, (void (*)(void *,void *)) outputUnionType, &s);
      if (d->structuredDes.uniond.default_arm != NULL)
	; /* handled in outputUnionType */
      else if (d->structuredDes.uniond.others_allowed)
	{
	  fprintf (context->file, "    default:\n      break;\n");
	}
      else
	{
	  fprintf (context->file, "    default:\n");
	  fprintf (context->file, "      fprintf (stderr, \"_%s__Output:  Bad value %%lu in discriminant.\\n\", (unsigned long) _val->_d);\n      break;\n",
		   c_type_name(type));
	}
      fprintf (context->file, "  };\n  ilu_EndUnion (_call, _err);\n");
    }

  else if (t == record_Type)
    {
      fprintf (context->file, "  ilu_OutputRecord (_call, _err);\n");
      fprintf (context->file, "  if (ILU_ERRNOK(*_err)) return;\n");
      list_enumerate (d->structuredDes.record, (void (*)(void *,void *)) outputRecordArg, context);
      fprintf (context->file, "  ilu_EndRecord (_call, _err);\n");
    }
  else if (t == optional_Type)
    {
      boolean needs_deref;
      Type ut2 = ur_type(d->structuredDes.optional);
      enum PrimitiveTypes t2 = type_basic_type(ur_type(d->structuredDes.optional));

      fprintf (context->file, "  ilu_OutputOptional (_call, _val != ILU_NIL, _err);\n");
      fprintf (context->file, "  if (ILU_ERRNOK(*_err) || (_val == ILU_NIL)) return;\n");

      needs_deref = (NOT (t2 == record_Type OR
			  t2 == union_Type OR
			  t2 == array_Type OR
			  t2 == object_Type OR
			  t2 == sequence_Type));
      MarshallValue (context, ut2, needs_deref ? "*_val" : "_val", 2);
    }

  fprintf (context->file, "}\n\n");
}

static void inputRecordArg (Argument  arg,      Context  context)
{
  char  buf[1000];

  sprintf (buf, "%s_val->%s", TypeIsPointer(ur_type(arg->type)) ? "" : "&",     c_argument_name (arg));
  UnmarshallValue (context, ur_type(arg->type), 0, buf, 2, TRUE, FALSE);
}

static void inputUnionType (Argument a, struct double_s *s)
{
  char   buffer[1000];
  Type ut = ur_type(a->type);
  char  *name;

  if (a->name->base_name != NULL)
    name = (char *) c_simple_name (a->name);
  else
    name = (char *) c_string (type_name (a->type));

  if (a == s->default_arm)
    fprintf (s->c->file, "    default:\n");
  else
    list_enumerate (a->values, (void (*)(refany, refany)) caseConst, s);
  sprintf (buffer, "%s_val->_u.%s", TypeIsPointer(ut) ? "" : "&", name);
  UnmarshallValue (s->c, ut, 0, buffer, 6, TRUE, FALSE);
  fprintf (s->c->file, "      break;\n");
}

/*
  We generate these for a small number of types:  arrays, records, sequences, unions,
  and optional.  Everything else is done explicitly.

  For arrays, the Input routine takes an optional argument of type ArrayType, and
  returns an element of type "BaseType *" (since arrays cannot be directly returned).
  For sequences, records, and unions, the Input routine takes an optional argument of
  type Type *, and returns an element of type Type *.  For optional, the Input
  routine takes an optional argument of type Type *, and returns a value of type
  Type.
*/

static void generateInputCode (Type type, enum PrimitiveTypes t, Context context)
{
  char *ret = (char *) c_return_type (type);
  char *parm = c_parameter_type (type, InOut);
  TypeDescription d = type_description (type);
  char *name = c_type_name (type);

  context->class = type;

  fprintf (context->file, "%s _%s__Input (ilu_Call _call, %s _ref, ilu_Error *_err)\n{\n  %s _val;\n\n",
	   (t == array_Type OR TypeIsString(type) OR t == optional_Type) ? ret : parm,
	   c_type_name(type), parm,
	   (t == array_Type) ? ret : parm);
  if (t == array_Type)
    {
      char buf[1000];
      int dims = list_size(d->structuredDes.array.dimensions);
      boolean zeroarray = FALSE;
      Type type2 = ur_type(d->structuredDes.array.type);
      enum PrimitiveTypes t2 = type_basic_type(d->structuredDes.array.type);

      dims = list_size(d->structuredDes.array.dimensions);
      zeroarray = FALSE;
      list_enumerate(d->structuredDes.array.dimensions, (void (*)(refany, refany)) FindZero, &zeroarray);

      fprintf (context->file, "  if (_ref != NULL)\n    _val = (%s *) _ref;\n  else\n    _val = (%s *) ilu_malloc(sizeof(%s));\n",
	       c_type_name(type2), c_type_name(type2), c_type_name(type));
      fprintf (context->file, "  if (_val == NULL)  /* malloc failure */\n");
      fprintf (context->file, "    return ILU_ERR_CONS1(MallocFailure, _err, nbytes, sizeof(%s), _val);\n",
	       c_type_name(type));

      if (t2 == byte_Type OR t2 == shortcharacter_Type)
	{
	  DimStop = dims - 1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  DimCount = 0;
	  buf[0] = '\0';
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file, "%*.*s{\n%*.*s%s _tmp = &(*((%s *)_val))%s[0];\n",     dims*2, dims*2, "", dims*2+2, dims*2+2, "", (t2 == byte_Type) ? "unsigned char *" : "char *", name, buf);
	  fprintf (context->file, "%*.*silu_Input%s (_call, &_tmp, %lu, _err);\n", dims*2+2, dims*2+2, "",     (t2 == byte_Type) ? "Opaque" : "StringVec",     (cardinal) list_ref(d->structuredDes.array.dimensions, dims-1));
	  fprintf (context->file, "%*.*s}\n", dims*2, dims*2, "");
	  DimCount = dims - 1;
	  DimStop = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  DimStop = -1;
	}
      else if (t2 == character_Type)
	{
	  DimStop = dims - 1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  DimCount = 0;
	  buf[0] = '\0';
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file, "%*.*s{\n%*.*silu_character *_tmp = &(*((%s *)_val))%s[0];\n",     dims*2, dims*2, "", dims*2+2, dims*2+2, "", name, buf);
	  fprintf (context->file, "%*.*silu_InputWStringVec (_call, &_tmp, %lu, _err);\n", dims*2+2, dims*2+2, "",     (cardinal) list_ref(d->structuredDes.array.dimensions, dims-1));
	  fprintf (context->file, "%*.*s}\n", dims*2, dims*2, "");
	  DimCount = dims - 1;
	  DimStop = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  DimStop = -1;
	}
      else
	{
	  fprintf (context->file, "  ilu_InputArray (_call, _err);\n  if (ILU_ERRNOK(*_err)) return _val;\n");
	  DimCount = 0;
	  DimStop = -1;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  DimCount = 0;
	  sprintf (buf, "&(*((%s*)_val))", name);
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  UnmarshallValue (context, d->structuredDes.array.type, type->def, buf, dims*2+2, TRUE, FALSE);
	  DimCount = dims;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  fprintf (context->file, "  ilu_EndArray (_call, _err);\n");
	}
    }
  else if (t == sequence_Type)
    {
      Type ut2 = ur_type(d->structuredDes.sequence.type);
      enum PrimitiveTypes t2 = type_basic_type(ur_type(d->structuredDes.sequence.type));

      if (t2 == shortcharacter_Type)
	{
	  fprintf (context->file, "  ilu_cardinal len = 0;\n");
	  fprintf (context->file, "  _val = _ref;\n");
	  fprintf (context->file, "  ilu_InputString (_call, (ilu_string *) _val, &len, %u, _err);\n",
		   d->structuredDes.sequence.limit);
	}
      else if (t2 == character_Type)
	{
	  fprintf (context->file, "  ilu_cardinal len = 0;\n");
	  fprintf (context->file, "  _val = _ref;\n");
	  fprintf (context->file, "  ilu_InputWString (_call, (ilu_wstring *) _val, &len, %u, _err);\n",
		   d->structuredDes.sequence.limit);
	}
      else if (t2 == byte_Type)
	{
	  fprintf (context->file, "  ilu_cardinal len;\n  ilu_bytes b = NULL;\n\n");
	  fprintf (context->file, "  if (_ref == NULL)\n    _val = (%s *) ilu_malloc(sizeof(%s));\n", name, name);
	  fprintf (context->file, "  else\n    _val = _ref;\n");
	  fprintf (context->file, "  if (_val == NULL)  /* malloc failure */\n");
	  fprintf (context->file, "    return ILU_ERR_CONS1(MallocFailure, _err, nbytes, sizeof(%s), _val);\n",
		   c_type_name(type));
	  fprintf (context->file, "  ilu_InputBytes (_call, &b, &len, %lu, _err);\n",
		   d->structuredDes.sequence.limit);
	  fprintf (context->file, "  _val->_maximum = len;\n");
	  fprintf (context->file, "  _val->_length = len;\n");
	  fprintf (context->file, "  _val->_buffer = b;\n");
	}
      else
	{
	  fprintf (context->file, "  ilu_cardinal _count, _index;\n");
	  fprintf (context->file, "  %s _tmp;\n\n", c_type_name (d->structuredDes.sequence.type));
	  fprintf (context->file, "  ilu_InputSequence (_call, &_count, %lu, _err);\n",
		   d->structuredDes.sequence.limit);
	  fprintf (context->file, "  if (ILU_ERRNOK(*_err)) return _val;\n");
	  fprintf (context->file, "  if (_ref != NULL) {\n    _val = _ref;\n");
	  fprintf (context->file, "    %s_Init(_val, 0, NULL);\n    }\n", name);
	  fprintf (context->file, "  else {\n    _val = (%s) ilu_malloc (sizeof (%s));\n", parm, name);
	  fprintf (context->file, "    if (_val == NULL)  /* malloc failure */\n");
	  fprintf (context->file, "      return ILU_ERR_CONS1(MallocFailure, _err, nbytes, sizeof(%s), _val);\n",
		   name);
	  fprintf (context->file, "    _val->_maximum = 0;\n");
	  fprintf (context->file, "    _val->_length = 0;\n");
	  fprintf (context->file, "    _val->_buffer = NULL;\n  }\n");
	  fprintf (context->file, "  for (_index = 0;  _index < _count;  _index++)\n  {\n");
	  UnmarshallValue (context, ut2, type->def,
			   TypeIsPointer(ut2) ? "_tmp" : "&_tmp",
			   4, TRUE, FALSE);
	  fprintf (context->file, "    if (ILU_ERRNOK(*_err)) return _val;\n");
	  fprintf (context->file, "    %s_Append (_val, %s_tmp);\n", c_type_name(type),
		   (t2 == record_Type OR
		    (t2 == sequence_Type AND NOT TypeIsString(ut2)) OR
		    t2 == union_Type) ? "&" : "");
	  fprintf (context->file, "  }\n");
	  fprintf (context->file, "  ilu_EndSequence (_call, _err);\n");
	}
    }
  else if (t == union_Type)
    {
      struct double_s s;

      s.c = context;
      s.t = ur_type(d->structuredDes.uniond.discriminator_type);
      s.id = 0;
      s.default_arm = d->structuredDes.uniond.default_arm;

      fprintf (context->file, "  ilu_cardinal discriminator;\n\n");
      fprintf (context->file, "  ilu_InputUnion (_call, &discriminator, sizeof(%s), _err);\n",
	       c_type_name(s.t));
      fprintf (context->file, "  if (ILU_ERRNOK(*_err)) return _val;\n");
      fprintf (context->file, "  if (_ref != NULL)\n    _val = _ref; \n");
      fprintf (context->file, "  else\n    _val = (%s) ilu_malloc (sizeof (%s));\n", parm, name);
      fprintf (context->file, "  if (_val == NULL)  /* malloc failure */\n");
      fprintf (context->file, "    return ILU_ERR_CONS1(MallocFailure, _err, nbytes, sizeof(%s), _val);\n",
	       name);
      fprintf (context->file, "  switch (discriminator) {\n");
      list_enumerate (d->structuredDes.uniond.types, (void (*)(void *,void *)) inputUnionType, &s);
      if (s.default_arm != NULL)
	;
      else if (d->structuredDes.uniond.others_allowed)
	fprintf (context->file, "    default:\n      break;\n");
      else
	fprintf (context->file, "    default:\n      fprintf (stderr, \"_%s__Input:  Bad value %%lu received for discriminant.\\n\", (unsigned long) discriminator);\n      break;\n", c_type_name(type));
      fprintf (context->file, "  }\n");
      fprintf (context->file, "  _val->_d = (%s) discriminator;\n", c_type_name(s.t));
      fprintf (context->file, "  ilu_EndUnion (_call, _err);\n");
    }
  else if (t == record_Type)
    {
      fprintf (context->file, "  ilu_InputRecord (_call, _err);\n");
      fprintf (context->file, "  if (_ref != NULL)\n");
      fprintf (context->file, "    _val = _ref;\n  else\n    _val = (%s) ilu_malloc (sizeof (%s));\n", parm, name);
      fprintf (context->file, "  if (_val == NULL)  /* malloc failure */\n");
      fprintf (context->file, "    return ILU_ERR_CONS1(MallocFailure, _err, nbytes, sizeof(%s), _val);\n",
	       name);
      list_enumerate (d->structuredDes.record, (void (*)(void *,void *)) inputRecordArg, context);
      fprintf (context->file, "  ilu_EndRecord (_call, _err);\n");
    }
  else if (t == optional_Type)
    {
      Type t2 = ur_type(d->structuredDes.optional);
      char buf[1000];

      fprintf (context->file, "  %s _ptr = ILU_NIL;\n", name);
      fprintf (context->file, "  ilu_boolean _present;\n  ilu_InputOptional (_call, &_present, _err);\n");
      fprintf (context->file, "  if (ILU_ERRNOK(*_err)) return ILU_NIL;\n");
      fprintf (context->file, "  if (_ref != ILU_NIL)\n    _val = _ref;\n  else\n    _val = &_ptr;\n");
      fprintf (context->file, "  if (_present)\n    {\n");
      UnmarshallValue (context, t2, 0, "*_val", 6, TRUE, TRUE);
      fprintf (context->file, "    }\n  else *_val = ILU_NIL;\n");
    }
  else
    {
      fprintf (stderr, "Bad type %s passed to generateInputCode\n", c_type_name(type));
      exit(1);
    }
  if (t == array_Type)
    fprintf (context->file, "  return ((%s) _val);\n}\n\n", c_return_type(type));
  else if (t == optional_Type OR TypeIsString(type))
    fprintf (context->file, "  return (*_val);\n}\n\n");
  else
    fprintf (context->file, "  return (_val);\n}\n\n");
}

static void sizeUnionType (Argument a, struct double_s *s)
{
  char   buffer[1000];
  int  mkReferent;
  char  *name;
  enum PrimitiveTypes t = type_basic_type (ur_type(a->type));
  Type ut = ur_type(a->type);

  if (a->name->base_name != NULL)
    name = (char *) c_simple_name (a->name);
  else
    name = (char *) c_string (type_name (a->type));

  if (a == s->default_arm)
    fprintf (s->c->file, "    default:\n");
  else
    list_enumerate (a->values, (void (*)(refany, refany)) caseConst, s);

  mkReferent = (!TypeIsString (ut) AND (t == record_Type OR t == union_Type OR t == sequence_Type));
  sprintf (buffer, "%s_val->_u.%s", (mkReferent) ? "&" : "", name);
  fprintf (s->c->file, "      size += ");
  SizeValue (s->c, ut, buffer);
  fprintf (s->c->file, ";\n      break;\n");
}

static void sizeRecordArg (Argument arg, Context context)
{
  char buf[1000];
  enum PrimitiveTypes t;
  Type ut = ur_type(arg->type);

  t = type_basic_type (ut);
  sprintf (buf, "(%s_val->%s)",     (t == record_Type OR
				     (t == sequence_Type AND !TypeIsString(ut)) OR
				     t == union_Type) ? "&" : "", c_argument_name(arg));
  fprintf (context->file, "  size += ");
  SizeValue (context, ut, buf);
  fprintf (context->file, ";\n");
}

static void generateSizeofCode (Type type, enum PrimitiveTypes t, Context context)
{
  char buf[1000];
  TypeDescription d = type_description (type);

  fprintf (context->file, "ilu_cardinal _%s__SizeOf (ilu_Call _call, %s _val, ilu_Error *_err)\n{\n",
	   c_type_name(type), c_parameter_type (type, In));
  fprintf (context->file, "  ilu_cardinal size = 0;\n\n");
  if (t == array_Type)
    {
      unsigned long size = 1;
      int dims = list_size(d->structuredDes.array.dimensions);
      boolean zeroarray = FALSE;
      enum PrimitiveTypes t2 = type_basic_type(d->structuredDes.array.type);

      list_enumerate(d->structuredDes.array.dimensions, (void (*)(refany, refany)) FindZero, &zeroarray);
      list_enumerate (d->structuredDes.array.dimensions, (void (*)(refany, refany)) ComputeTotalNumberOfElements, &size);

      if (t2 == byte_Type OR t2 == shortcharacter_Type)
	{
	  DimStop = dims - 1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  sprintf (buf, "_val");
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file, "%*.*ssize += ilu_SizeOf%s (_call, %s, %lu, _err);\n", dims*2, dims*2, "",     (t2 == byte_Type) ? "Opaque" : "StringVec", buf, (cardinal) list_ref(d->structuredDes.array.dimensions, dims-1));
	  DimCount = dims;
	  DimStop = 1;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  DimStop = -1;
	}
      else if (t2 == character_Type)
	{
	  DimStop = dims - 1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  sprintf (buf, "_val");
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file, "%*.*ssize += ilu_SizeOfWStringVec (_call, _val, %lu, NULL, NULL, _err);\n",     dims * 2 + 2, dims*2+2, "", (cardinal) list_ref(d->structuredDes.array.dimensions, dims-1));
	  DimCount = dims;
	  DimStop = 1;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	  DimStop = -1;
	}
      else
	{
	  fprintf (context->file, "  size = ilu_SizeOfArray (_call, %lu, _err);\n", size);
	  fprintf (context->file, "  if (ILU_ERRNOK(*_err)) return 0;\n");
	  DimStop = -1;
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimHeader, context);
	  sprintf (buf, "%s_val", (!TypeIsString (d->structuredDes.array.type) &&
				   (t2 == record_Type OR t2 == union_Type OR t2 == sequence_Type)) ? "&" : "");
	  DimCount = 0;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimRef, buf);
	  fprintf (context->file, "%*.*ssize += ", dims*2+4, dims*2+4, "");
	  SizeValue (context, ur_type(d->structuredDes.array.type), buf);
	  fprintf (context->file, ";\n");
	  DimCount = dims;
	  list_enumerate (d->structuredDes.array.dimensions, (void (*)(void *,void *)) dimFooter, context);
	}

      fprintf (context->file, "  ilu_EndArray (_call, _err);\n");
    }

  else if (t == sequence_Type)
    {
      Type et = d->structuredDes.sequence.type;
      Type ut = ur_type(d->structuredDes.sequence.type);
      enum PrimitiveTypes st = type_basic_type (ur_type(d->structuredDes.sequence.type));

      if (st == byte_Type)
	{
	  fprintf (context->file, "  size = ilu_SizeOfBytes (_call, _val->_buffer, _val->_length, %u, _err);\n",
		   d->structuredDes.sequence.limit);
	}
      else if (st == shortcharacter_Type)
	{
	  fprintf (context->file, "  size = ilu_SizeOfString (_call, (ilu_string) _val, _ILU_C_SafeStrlen((ilu_string) _val), %u, _err);\n",
		   d->structuredDes.sequence.limit);
	}
      else if (st == character_Type)
	{
	  fprintf (context->file, "  size = ilu_SizeOfWString (_call, _val, _ILU_C_SafeWStrlen(_val), %u, NULL, NULL, _err);\n",
		   d->structuredDes.sequence.limit);
	}
      else
	{
	  fprintf (context->file, "  size = ilu_SizeOfSequence (_call, _val->_length, %lu, _err);\n",         d->structuredDes.sequence.limit);
	  fprintf (context->file, "  if (ILU_ERRNOK(*_err)) return 0;\n");
	  fprintf (context->file, "  {\n    %s *p;  int i;\n\n", c_type_name (et));
	  fprintf (context->file, "    for (p = _val->_buffer, i = 0;  i < _val->_length;  p++, i++)\n      size += ");
	  sprintf (buf, "%sp",         (st == record_Type OR (st == sequence_Type AND NOT TypeIsString(ut)) OR st == union_Type) ? "" : "*");
	  SizeValue (context, ut, buf);
	  fprintf (context->file, ";\n  }\n  ilu_EndSequence (_call, _err);\n");
	}
    }

  else if (t == union_Type)
    {
      struct double_s  s;

      s.c = context;
      s.t = ur_type(d->structuredDes.uniond.discriminator_type);
      s.id = 0;
      s.default_arm = d->structuredDes.uniond.default_arm;

      fprintf (context->file, "  size = ilu_SizeOfUnion (_call, _val->_d, sizeof(%s), _err);\n",
	       c_type_name(s.t));
      fprintf (context->file, "  if (ILU_ERRNOK(*_err)) return 0;\n");
      fprintf (context->file, "  switch (_val->_d) {\n");
      list_enumerate (d->structuredDes.uniond.types, (void (*)(void *,void *)) sizeUnionType, &s);
      if (s.default_arm != NULL)
	;
      else if (d->structuredDes.uniond.others_allowed)
	fprintf (context->file, "    default:\n      break;\n");
      else
	fprintf (context->file, "    default:\n      fprintf (stderr, \"_%s__SizeOf:  Bad value %%lu in discriminant.\\n\", (unsigned long) _val->_d);\n      break;\n", c_type_name(type));
      fprintf (context->file, "  };\n  ilu_EndUnion (_call, _err);\n");
    }

  else if (t == record_Type)
    {
      fprintf (context->file, "  size = ilu_SizeOfRecord (_call, _err);\n");
      fprintf (context->file, "  if (ILU_ERRNOK(*_err)) return 0;\n");
      list_enumerate (d->structuredDes.record, (void (*)(void *,void *)) sizeRecordArg, context);
      fprintf (context->file, "  ilu_EndRecord (_call, _err);\n");
    }
  else if (t == optional_Type)
    {
      boolean needs_deref;
      Type ut2 = ur_type(d->structuredDes.optional);
      enum PrimitiveTypes t2 = type_basic_type(ur_type(d->structuredDes.optional));

      fprintf (context->file, "  size = ilu_SizeOfOptional (_call, _val != ILU_NIL, _err);\n");
      fprintf (context->file, "  if (ILU_ERRNOK(*_err)) return 0;\n");
      fprintf (context->file, "  if (_val != ILU_NIL)\n    size += ");

      needs_deref = (NOT (t2 == record_Type OR
			  t2 == union_Type OR
			  t2 == array_Type OR
			  t2 == object_Type OR
			  t2 == sequence_Type));
      SizeValue (context, ut2, needs_deref ? "*_val" : "_val");
      fprintf (context->file, ";\n");
    }
  else if (t == object_Type)
    ; /* done elsewhere */
  else
    {};

  fprintf (context->file, "  return size;\n}\n\n");
}

static void generateSequenceCreateCode (Type  type,  Context  context)
{
  string  name = c_type_name (type);

  fprintf (context->file,     "%s %s_%s_Create (unsigned long count, unsigned char *bytes)\n",     c_return_type (type),     c_interface_name (type->interface),     c_simple_name (type->name));
  fprintf (context->file, "{\n  %s n;\n", name);
  fprintf (context->file,     "  n._length = count;\n");
  fprintf (context->file,     "  n._maximum = count;\n");
  fprintf (context->file,     "  n._buffer = bytes;\n  return (n);\n}\n\n");
}

static void FreeRecordField (Argument field, Context context)
{
  char buf[1000];

  sprintf (buf, "_val->%s", c_argument_name(field));
  FreeValue (field->type, buf, context, 2);
}

static void FreeUnionField (Argument a, struct double_s *s)
{
  char buffer[1000];
  Type ut = ur_type(a->type);
  char *name;

  if (a->name->base_name != NULL)
    name = (char *) c_simple_name (a->name);
  else
    name = (char *) c_string (type_name (a->type));
  sprintf (buffer, "_val->_u.%s", name);
  if (a == s->default_arm)
    fprintf (s->c->file, "    default:\n");
  else
    list_enumerate (a->values, (void (*)(refany, refany)) caseConst, s);
  FreeValue (ut, buffer, s->c, 6);
  fprintf (s->c->file, "      break;\n");
}

static void generateFreeCode (Type type, enum PrimitiveTypes t, Context context)
{
  TypeDescription d = type_description(type);
  /*
   * Generate code to free any allocated storage occupied by this
   * object. (MJS, 96-Jan-9: looks like it actually generates procs
   * to go in the freeRoutine slot of a CORBA_Environment; these
   * procs don't free the object pointed directly at.)
   */

  fprintf(context->file, "void %s__Free (%s _val)\n{\n",
	  c_type_name(type), c_parameter_type(type, Out));
  fprintf(context->file,
	  "  /* What you put in the freeRoutine member of a CORBA_Environment for an exception parameterized by a %s */\n",
	  c_type_name(type));
  fprintf(context->file,
	  "  /* frees allocated storage inside _val (if any), but does not free(_val) */\n");

  if (t == record_Type)
    list_enumerate(type_description(type)->structuredDes.record,
	     (void (*) (void *, void *)) FreeRecordField, context);

  else if (t == union_Type) {
    struct double_s s;

    s.c = context;
    s.t = ur_type(d->structuredDes.uniond.discriminator_type);
    s.id = 0;
    s.default_arm = d->structuredDes.uniond.default_arm;

    fprintf(context->file, "  switch (_val->_d) {\n");
    list_enumerate(d->structuredDes.uniond.types,
		   (void (*) (void *, void *)) FreeUnionField, &s);
    if (d->structuredDes.uniond.default_arm != NULL)
      /* handled in FreeUnionField */
      ;
    else if (d->structuredDes.uniond.others_allowed) {
      fprintf(context->file, "    default:\n      break;\n");
    } else {
      fprintf(context->file, "    default:\n");
      fprintf(context->file, "      fprintf (stderr, \"_%s__Free:  Bad value %%lu in discriminant.\\n\", (unsigned long) _val->_d);\n", c_type_name(type));
      fprintf(context->file, "      break;\n");
    }
    fprintf(context->file, "  };\n");
  } else if (t == sequence_Type) {
    Type            st = ur_type(d->structuredDes.sequence.type);
    enum PrimitiveTypes t2 = type_basic_type(st);
    if (t2 == shortcharacter_Type || t2 == character_Type) {
      fprintf(context->file, "  if (*_val != ILU_NIL)\n");
      fprintf(context->file, "    ilu_free(*_val);\n");
    } else {
      fprintf(context->file, "  if (_val->_buffer != NULL) {\n");
      if (HasFreeRoutine(st)) {
	fprintf(context->file, "    int i;\n");
	fprintf(context->file,
		"    for (i = 0;  i < _val->_length;  i++)\n");
	FreeValue(st, "_val->_buffer[i]", context, 6);
      }
      fprintf(context->file, "    ilu_free(_val->_buffer);\n  };\n");
    }
  } else if (t == array_Type) {

    if (HasFreeRoutine(d->structuredDes.array.type)) {
      char            buf[1000];
      int             dims = list_size(d->structuredDes.array.dimensions);

      DimStop = -1;
      DimCount = 0;
      list_enumerate(d->structuredDes.array.dimensions,
		   (void (*) (void *, void *)) dimHeader, context);
      sprintf(buf, "_val");
      DimCount = 0;
      list_enumerate(d->structuredDes.array.dimensions,
		     (void (*) (void *, void *)) dimRef, buf);
      FreeValue(d->structuredDes.array.type, buf, context,
		2 * dims + 4);
      DimCount = dims;
      list_enumerate(d->structuredDes.array.dimensions,
		   (void (*) (void *, void *)) dimFooter, context);
    } else {
      fprintf(context->file,
	      "  /* simple element type, nothing to free */\n");
    }
  } else if (t == optional_Type) {
    Type            st = ur_type(d->structuredDes.optional);
    TypeKind        t2 = type_kind(st);
    int             indirect = 1;
    switch (t2) {
    case sequence_Type:
      indirect = !(TypeIsString(st) || TypeIsWString(st));
      break;
    case optional_Type:
    case array_Type:
    case object_Type:
      indirect = 0;
    default:;
    }
    fprintf(context->file, "  /* subtype is %s */\n", c_type_name(st));
    fprintf(context->file, "  if (*_val != ILU_NIL) {\n");
    FreeValue(st, (indirect ? "**_val" : "*_val"), context, 4);
    if (indirect)
      fprintf(context->file, "    ilu_free(*_val);\n");
    fprintf(context->file, "  }\n");
  }
  fprintf(context->file, "}\n\n");
}

static void generateBufAllocCode (Type type, enum PrimitiveTypes t, Context context)
{
  char *s = c_type_name(type);

  fprintf (context->file, "%s *CORBA_sequence_%s_bufalloc (CORBA_unsigned_long _count)\n{\n",
	   s, (type->builtIn && strncmp(s, "CORBA_", 6) == 0) ? s + 6 : s);
  fprintf (context->file, "  %s *_p;\n  CORBA_unsigned_long _size = sizeof(%s) * _count;\n\n",
	   s, s);
  fprintf (context->file, "  if ((_p = (%s *) ilu_malloc(_size)) == ILU_NIL)\n", s);
  fprintf (context->file, "    { _ILU_C_MallocFailure(_size); return 0; }\n");
  fprintf (context->file, "  else\n    { memset((void *) _p, 0, _size);  return _p; }\n}\n\n");
}

static void generateAllocCode (Type type, enum PrimitiveTypes t, Context context)
{
  char *s = c_type_name(type);
  char *p = (type_kind(type) == array_Type) ? c_return_type(type) : c_parameter_type(type, Out);

  fprintf (context->file, "%s %s__alloc ()\n{\n", p, s);
  fprintf (context->file, "  return ((%s) CORBA_sequence_%s_bufalloc(1));\n}\n\n", p,
	   (type->builtIn && strncmp(s, "CORBA_", 6) == 0) ? s + 6 : s);
}

static void generateTypeIoCode (Type type, Context context)
{
  enum PrimitiveTypes t = type_basic_type (type);

  if (type->supertype != NULL
      OR type->builtIn
      OR type->interface != context->interface
      OR type->importInterfaceName != NULL)
    return;

  if (type->importInterfaceName == NULL AND
      (t == union_Type OR
       t == record_Type OR
       t == optional_Type OR
       t == sequence_Type OR
       t == array_Type))
    {
      generateOutputCode (type, t, context);
      generateSizeofCode (type, t, context);
      generateInputCode (type, t, context);
    }

  if (HasFreeRoutine(type))
    generateFreeCode (type, t, context);

  generateBufAllocCode(type, t, context);

  if (HasAllocRoutine(type))
    generateAllocCode (type, t, context);
}

static void generateIosCode (Interface  interface,       Context  context)
{
  list_enumerate (interface->types,    (void (*)(void *,void *)) generateTypeIoCode,    context);
}

static void generateGlobalCode (Interface  interface,  Context  context)
{
  generateExceptionTable (interface, context);
  generateExceptionProcs (context);
  generateIosCode (interface, context);
}

static unsigned methodIdx = 47, exnIdx = 86;

static void setupExn (refany elt, refany rock)
{
  Exception       e = (Exception) elt;
  Context         context = (Context) rock;
  fprintf(context->file, "    exns[%u] = ex_%s;\n",
	  exnIdx++, c_exception_name(e));
}

static void generateMethodDef (refany elt, refany rock)
{
  Procedure       m = (Procedure) elt;
  Context         context = (Context) rock;
  FILE           *f = context->file;
  fprintf(f, "  { ilu_Exception\t");
  if (list_size(m->exceptions) > 0) {
    fprintf(f, "exns[%u];\n", list_size(m->exceptions));
    exnIdx = 0;
    list_enumerate(m->exceptions, setupExn, context);
  } else
    /* Don't try C's patience with a 0-length array */
    fprintf(f, "*exns = NULL;\n");
  fprintf(f, "    m = ilu_DefineMethod(cl, %u,\n", methodIdx++);
  fprintf(f, "\t\"%s\",\t/*name*/\n", procedure_name(m));
  fprintf(f, "\t%u,\t/*id*/\n", m->id);
  fprintf(f, "\t%u,\t/*functional*/\n", m->functional != 0);
  fprintf(f, "\t%u,\t/*asynch*/\n", m->asynch != 0);
  fprintf(f, "\t%u,\t/*n exns*/\n", list_size(m->exceptions));
  fprintf(f, "\texns, &lerr);\n");
  fprintf(f, "    if (ILU_ERRNOK(lerr))\n");
  fprintf(f, "      goto fail2;\n");
  fprintf(f, "  }\n");
  return;
}

static void RegisterClass (refany elt, refany rock)
{
  Type            class = (Type) elt;
  Context         context = (Context) rock;
  FILE           *f = context->file;
  char           *tn;
  Class           o;
  TypeDescription des;
  if ((class->builtIn && class->importInterfaceName == NULL)
      || (type_basic_type(class) != object_Type))
    return;
  tn = c_type_name(class);
  o = class_object(class);
  fprintf(f, "  { ilu_string supers[] = {\n");
  list_enumerate(o->superclasses, classIDs, context);
  fprintf(f, "\tNULL};\n");
  fprintf(f, "    cl = ilu_DefineObjectType(\"%s.%s\",\t/*name*/\n",
	  name_base_name(context->interface->name),
	  name_base_name(class->name));
  fprintf(f, "\t\"%s\",\t/*brand*/\n", o->brand == NULL ? "" : o->brand);
  fprintf(f, "\t\"%s\",\t/*uid*/\n", class->uid);
  if (o->singleton == NULL)
    fprintf(f, "\tNULL,\t/*singleton*/\n");
  else
    fprintf(f, "\t\"%s\",\t/*singleton*/\n", o->singleton);
  fprintf(f, "\t%s,\t/* optional */\n\t%s,\t/* collectible */\n",
	  o->optional ? "ilu_TRUE" : "ilu_FALSE",
	  o->collectible ? "ilu_TRUE" : "ilu_FALSE");
  if (o->authentication == NULL)
    fprintf(f, "\tNULL,\t/*auth*/\n");
  else
    fprintf(f, "\t\"%s\",\t/*auth*/\n", o->authentication);
  fprintf(f, "\t%u,\t/*n methods*/\n", list_size(o->methods));
  fprintf(f, "\t%u,\t/*n supers*/\n",
       (o->superclasses == NULL) ? 0 : list_size(o->superclasses));
  fprintf(f, "\tsupers, &lerr);\n");
  fprintf(f, "    if (ILU_ERRNOK(lerr))\n");
  fprintf(f, "      goto fail2;\n");
  fprintf(f, "    _%s__ILUType = cl;\n", tn);
  fprintf(f, "  }\n");
  methodIdx = 0;
  list_enumerate(o->methods, generateMethodDef, context);
  return;
}

static boolean AddGCCallback (Type type, Context context)
{
  TypeDescription des;
  if ((type->builtIn && type->importInterfaceName == NULL)
      OR (type_basic_type(type) != object_Type)
      OR (NOT class_object(type)->collectible))
    return FALSE;
  fprintf(context->file, "  _ILU_C_EnsureGcClient ();\n");
  return TRUE;
}

static void InitializeImportedInterfaces (refany elt, refany rock)
{
  Imported        i = (Imported) elt;
  Context         c = (Context) rock;
  Interface       i2 = GetInterface(i->name, i->filename);
  if (i2 == NULL) {
    fprintf(stderr, "Can't find interface <%s>\n", i->name);
    return;
  } else if (strcmp(i->name, "ilu") == 0);
  else
    fprintf(c->file, "  _%s__GeneralInitialization();\n",
	    c_interface_name(i2));
}

static void genExnDef(refany elt, refany rock)
{
  Exception       e = (Exception) elt;
  Context         context = (Context) rock;
  FILE           *f = context->file;
  if (e->interface != context->interface || e->import != NULL)
    return;
  fprintf(f, "  _%s__Exception_%s = ilu_DefineException(",
	  c_interface_name(e->interface), c_simple_name(e->name));
  if (e->corba_rep_id == NIL)
    fprintf(f, "\"%s\", \"%s\"", name_base_name(e->interface->name),
	    name_base_name(e->name));
  else
    fprintf(f, "ILU_NIL, \"%s\"", e->corba_rep_id);
  fprintf(f, ", &lerr);\n");
  fprintf(f, "  if (ILU_ERRNOK(lerr))\n");
  fprintf(f, "    goto fail1;\n");
  return;
}

static void
generateClientRegistrationCode(Interface interface, Context context)
{
  char           *interface_name = (char *) c_interface_name(interface);
  FILE           *f = context->file;

  fprintf(f,
	  "void _%s__GeneralInitialization (void)\n{\n",
	  interface_name);
  fprintf(f, "  static ilu_boolean initialized = ilu_FALSE;\n");
  fprintf(f, "  ilu_Error lerr = ILU_INIT_NO_ERR;\n");
  fprintf(f, "  ilu_Class cl = ILU_NIL;\n");
  fprintf(f, "  ilu_Method m = ILU_NIL;\n");
  fprintf(f, "  ilu_Mutex otmu = ilu_GetOTMutex();\n");
  fprintf(f, "  if (initialized)\n    return;\n");
  fprintf(f, "  initialized = ilu_TRUE;\n");
  list_enumerate(interface->imports, InitializeImportedInterfaces,
		 context);
  fprintf(f, "  if (!ilu_EnterMutex(otmu, &lerr))\n");
  fprintf(f, "    goto fail1;\n");
  list_enumerate (interface->exceptions, genExnDef, context);
  list_enumerate(interface->classes, RegisterClass, context);
  fprintf(f, "  if (!ilu_ExitMutex(otmu, ilu_TRUE, &lerr))\n");
  fprintf(f, "    goto fail2;\n");
  (void) list_find(interface->classes, (FindProc) AddGCCallback, context);
  fprintf(f, "fail1:\n");
  fprintf(f, "fail2:\n");
  fprintf(f, "  ILU_MUST_BE_SUCCESS(lerr);\n");
  fprintf(f, "}\n\n");
}

static void generateSequenceCode (refany elt, refany rock)
{
  Type            seq = (Type) elt;
  Context         context = (Context) rock;
  boolean         hasReferent;
  char           *rtn;
  Type            sequenceType;
  char           *st;
  enum PrimitiveTypes t;
  char           *tn;

  if (type_basic_type (seq) != sequence_Type OR
      seq->importInterfaceName != NULL OR
      seq->builtIn)
    return;
  sequenceType = type_description (seq)->structuredDes.sequence.type;
  t = type_basic_type (ur_type(sequenceType));
  if (t == shortcharacter_Type OR t == character_Type)
    {
      tn = c_type_name (seq);
      st = c_parameter_type (sequenceType, In);
      rtn = c_type_name (sequenceType);

      fprintf (context->file, "void %s_Every (%s *h, void (*f)(%s *, void *), void * data)\n{\n", tn, tn, rtn);
      fprintf (context->file, "  %s *p;\n  for (p = *h;  *p != 0;  p++) { (*f)(p, data); };\n}\n\n", rtn);

      fprintf (context->file, "void %s_Append (%s *h, %s item)\n{\n", tn, tn, rtn);
      fprintf (context->file, "  _ILU_C_Extend%sString (h, item, (CORBA_boolean) 1);\n}\n\n",
	       (t == shortcharacter_Type) ? "" : "W");

      fprintf (context->file, "void %s_Push (%s *h, %s item)\n{\n", tn, tn, rtn);
      fprintf (context->file, "  _ILU_C_Extend%sString (h, item, (CORBA_boolean) 0);\n}\n\n",
	       (t == shortcharacter_Type) ? "" : "W");

      fprintf (context->file, "void %s_Pop (%s *h, %s *item)\n{\n", tn, tn, rtn);
      fprintf (context->file, "  _ILU_C_Pop%sString (h, item);\n}\n\n",
	       (t == shortcharacter_Type) ? "" : "W");

      fprintf (context->file, "CORBA_unsigned_long %s_Length (%s *h)\n{\n", tn, tn);
      fprintf (context->file, "  if (h == ILU_NIL || *h == ILU_NIL) return 0;\n");
      if (t == shortcharacter_Type)
	fprintf (context->file, "  return (strlen((char *)(*h)));\n}\n\n");
      else
	fprintf (context->file, "  return _ILU_C_SafeWStrlen((ilu_character *)(*h));\n}\n\n");

      fprintf (context->file, "%s * %s_Nth (%s *h, CORBA_unsigned_long n)\n{\n", rtn, tn, tn);
      fprintf (context->file, "  if (h == ILU_NIL || *h == ILU_NIL) return ILU_NIL;\n");
      if (t == shortcharacter_Type)
	fprintf (context->file, "  if (n >= strlen((char *)(*h))) return ILU_NIL; else return &((*h)[n]);\n}\n\n");
      else
	fprintf (context->file, "  if (n >= _ILU_C_SafeWStrlen((ilu_character *)(*h))) return ILU_NIL; else return &((*h)[n]);\n}\n\n");

      fprintf (context->file, "%s %s_Create (CORBA_unsigned_long sz, %s *p)\n{\n", tn, tn, rtn);
      fprintf (context->file, "  %s s;\n\n  if (p == ILU_NIL)\n", tn);
      fprintf (context->file, "    {\n      s = ilu_malloc(sz * sizeof(%s));\n", rtn);
      fprintf (context->file, "      if (s == ILU_NIL) { _ILU_C_MallocFailure(sz * sizeof(%s)); return ILU_NIL; }\n", rtn);
      fprintf (context->file, "      memset((void *) s, 0, sz * sizeof(%s));\n      return s;\n", rtn);
      fprintf (context->file, "    }\n  else\n    return p;\n}\n\n");

      fprintf (context->file, "void %s_Init (%s *s, CORBA_unsigned_long sz)\n", tn, tn, rtn);
      fprintf (context->file, "{\n  return;\n}\n\n");
    }
  else
    {
      tn = c_type_name (seq);
      st = c_parameter_type (sequenceType, In);
      rtn = c_type_name (sequenceType);
      hasReferent = (t == record_Type
		     OR (t == sequence_Type AND NOT TypeIsString(ur_type(sequenceType)))
		     OR t == union_Type);
      fprintf (context->file, "void %s_Every (%s *h, void (*f)(%s%s, void *), void * data)\n{\n", tn, tn, st,         hasReferent ? "" : " *");
      fprintf (context->file, "  _ILU_C_EveryElement ((ILU_C_Sequence) h, (void (*)(void *, void *)) f, sizeof(%s), (void *) data);\n}\n\n", rtn);

      fprintf (context->file, "void %s_Append (%s *h, %s item)\n{\n", tn, tn, st);
      fprintf (context->file, "  _ILU_C_AppendGeneric ((ILU_C_Sequence) h, (char *) %sitem, sizeof(%s));\n}\n\n",         (hasReferent) ? "" : "&", rtn);

      fprintf (context->file, "void %s_Push (%s *h, %s item)\n{\n", tn, tn, st);
      fprintf (context->file, "  _ILU_C_PushGeneric ((ILU_C_Sequence) h, (char *) %sitem, sizeof(%s));\n}\n\n",         (hasReferent) ? "" : "&", rtn);

      fprintf (context->file, "void %s_Pop (%s *h, %s item)\n{\n", tn, tn, st);
      fprintf (context->file, "  _ILU_C_PopGeneric ((ILU_C_Sequence) h, (char *) %sitem, sizeof(%s));\n}\n\n",         hasReferent ? "" : "&", rtn);

      fprintf (context->file, "CORBA_unsigned_long %s_Length (%s *h)\n{\n", tn, tn);
      fprintf (context->file, "  if (h == ILU_NIL) return 0; else return h->_length;\n}\n\n");

      fprintf (context->file, "%s * %s_Nth (%s *h, CORBA_unsigned_long n)\n{\n", rtn, tn, tn);
      fprintf (context->file, "  if (h == ILU_NIL || (n >= h->_length)) return ILU_NIL; else return &(h->_buffer[n]);\n}\n\n");

      fprintf (context->file, "%s %s_%s_Create (CORBA_unsigned_long sz, %s %sp)\n", tn, c_interface_name (context->interface),         c_simple_name (seq->name), st, hasReferent ? "" : "*");
      fprintf (context->file, "{\n  %s s;\n", tn);
      fprintf (context->file, "  if (sz > 0)\n    s._maximum = sz;\n");
      fprintf (context->file, "  else\n    s._maximum = 0;\n");
      fprintf (context->file, "  if (sz > 0 && p != NULL)\n    s._length = sz;\n");
      fprintf (context->file, "  else\n    s._length = 0;\n");
      fprintf (context->file, "  if (sz > 0 && p == NULL) {\n");
      fprintf (context->file, "    s._buffer = (%s *) ilu_malloc (sz * sizeof (%s));\n", rtn, rtn);
      fprintf (context->file, "    if (s._buffer == ILU_NIL) {\n");
      fprintf (context->file, "      s._length = 0;\n");
      fprintf (context->file, "      s._maximum = 0;\n");
      fprintf (context->file, "      _ILU_C_MallocFailure(sz * sizeof(%s)); }}\n", rtn);
      fprintf (context->file, "  else\n    s._buffer = p;\n");
      fprintf (context->file, "  return (s);\n}\n\n");

      fprintf (context->file, "void %s_%s_Init (%s *s, CORBA_unsigned_long sz, %s %sp)\n",         c_interface_name (context->interface), c_simple_name (seq->name), tn, st, (hasReferent ? "" : "*"));
      fprintf (context->file, "{\n");
      fprintf (context->file, "  if (sz == 0 && p != NULL)\n    return;\n");
      fprintf (context->file, "  if (sz > 0)\n    s->_maximum = sz;\n");
      fprintf (context->file, "  else\n    s->_maximum = 0;\n");
      fprintf (context->file, "  if (sz > 0 && p != NULL)\n    s->_length = sz;\n");
      fprintf (context->file, "  else\n    s->_length = 0;\n");
      fprintf (context->file, "  if (sz > 0 && p == NULL) {\n");
      fprintf (context->file, "    s->_buffer = (%s *) ilu_malloc (sz * sizeof (%s));\n", rtn, rtn);
      fprintf (context->file, "    if (s->_buffer == ILU_NIL) {\n");
      fprintf (context->file, "      s->_length = 0;\n");
      fprintf (context->file, "      s->_maximum = 0;\n");
      fprintf (context->file, "      _ILU_C_MallocFailure(sz * sizeof(%s)); }}\n", rtn);
      fprintf (context->file, "  else\n    s->_buffer = p;\n");
      fprintf (context->file, "  return;\n}\n\n");
    }
}

static void listArgumentNames (Argument arg, Context context)
{
  fprintf (context->file, ", %s", c_argument_name(arg));
}

static void generateDispatchClassMethod(refany elt, refany rock)
{
  Procedure m = (Procedure) elt;
  Context c = (Context) rock;
  int             ndx;
  enum PrimitiveTypes t = type_basic_type(ur_type(m->returnType));

  /*
   * this stuff is trying to ensure that we dont generate the same
   * method more than once. this can happen if the current class
   * has more than one super classes and each of those is a
   * subclass of the same superclass. got it!
   * [Probably irrelevant now --- MJS 1/13/95]
   */

  if (methodInList(c_simple_name(m->name))) {
    MethodRecordID++;
    return;
  }
  addMethodToList(c_simple_name(m->name));
  generateProcHeader(m, c, TRUE);
  fprintf(c->file, "  %s (*_f)(%s",
	  (t == void_Type) ? "void" : c_return_type(m->returnType),
	  c_type_name(c->class));
  list_enumerate(m->arguments, (void (*) ()) listArgumentTypes, c);
  fprintf(c->file, ", ILU_C_ENVIRONMENT *);\n");
  fprintf(c->file, "  _f = (%s (*)(%s",
	  (t == void_Type) ? "void" : c_return_type(m->returnType),
	  c_type_name(c->class));
  list_enumerate(m->arguments, (void (*) ()) listArgumentTypes, c);
  ndx = methodNdxOf(m, m->object);
  fprintf(c->file, ", ILU_C_ENVIRONMENT *)) _ILU_C_FindMethod (_handle, _%s__ILUType, %d);\n",
	  c_type_name(ur_type(c->class)),
  /* This would be the wrong ilu_Class for an inherited method! */
	  ndx);
  fprintf(c->file, "  _status->_major = ILU_C_NO_EXCEPTION;\n");
  fprintf(c->file, "  _status->ptr = ILU_NIL;\n");
  fprintf(c->file, "  _status->freeRoutine = (void(*)(void *)) 0;\n");
  fprintf(c->file, "  _status->returnCode = (ilu_Exception) 0;\n");
  if (t != void_Type)
    fprintf(c->file, "  return (");
  else
    fprintf(c->file, "  ");
  fprintf(c->file, "(*_f)(_handle");
  list_enumerate(m->arguments, (void (*) ()) listArgumentNames, c);
  fprintf(c->file, ", _status");
  if (t != void_Type)
    fprintf(c->file, ")");
  fprintf(c->file, ");\n}\n\n");
}

static void generateDispatchMethods (refany elt, refany rock)
{
  Type            pclass = (Type) elt;
  Context         context = (Context) rock;
  Class           od = class_object(ur_type(pclass));
  static int      recursing = 0;

#if 0
  /*
   * Here's some wild stuff. What's happening is that we always
   * want to retain the highest class level as we recurse through
   * the class tree thats why we need to retain this nasty
   * recursing flag so we can know when to reset the context->class
   */

  if (od->superclasses) {
    if (!recursing)
      context->class = ur_type(pclass);
    recursing++;
    list_enumerate(od->superclasses, generateDispatchMethods, context);
    recursing--;
  }
#endif
  /*
   * MJS 1/13/95: Why recurse at all?  We don't want generic
   * functions for inherited methods!  (At least not while their
   * implementations are buggy!)
   */
  if (!recursing)
    context->class = ur_type(pclass);
  if (list_size(od->methods) > 0)
    list_enumerate(od->methods, generateDispatchClassMethod, context);
}

static void generateDispatch (refany elt, refany rock)
{
  Type            class = (Type) elt;
  Context         context = (Context) rock;
  if (type_basic_type(class) != object_Type)
    return;
  MethodRecordID = 0;
  clearMethodList ();
  generateDispatchMethods (class, context);
}

static void generateUserDataAccessors (refany elt, refany rock)
{
  Type            type = (Type) elt;
  Context         context = (Context) rock;
  char           *tn;

  if (type_basic_type(type) != object_Type)
    return;

  tn = c_type_name(type);
  fprintf (context->file, "void %s__SetUserData (%s self, void *userData)\n", tn, tn);
  fprintf (context->file, "{\n  ((ILU_C_Object *) self)->data = userData;\n}\n\n");
  fprintf (context->file, "void *%s__GetUserData (%s self)\n", tn, tn);
  fprintf (context->file, "{\n  return(((ILU_C_Object *) self)->data);\n}\n\n");
}

void generateCommonCode (Interface interface, FILE *file)
{
  struct context_s context;
  char           *pc_interfacename;

  context.file = file;
  context.interface = interface;

  /*
   * get any translation of what the header file for the interface
   * is
   */
  pc_interfacename = interface_header_name(c_interface_name(interface));

  fprintf(file, "#include <stdio.h>\n");
  fprintf(file, "#include <string.h> /* used for error statements */\n");
  fprintf(file, "#include \"%s.h\"\n\n", pc_interfacename);

  list_enumerate(interface->classes, generateClassTable, &context);
  list_enumerate(interface->classes, generateDispatch, &context);
  list_enumerate(interface->classes, generateUserDataAccessors,
		 &context);
  generateGlobalCode(interface, &context);
  list_enumerate(interface->types, generateSequenceCode, &context);
  generateClientRegistrationCode(interface, &context);
}
