/**
 *
 * $Id: PanedW.c,v 1.19 1996/05/02 07:11:07 u27113 Exp $
 *
 * Copyright (C) 1995 Free Software Foundation, Inc.
 *
 * This file is part of the GNU LessTif Library.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **/

static char rcsid[] = "$Id: PanedW.c,v 1.19 1996/05/02 07:11:07 u27113 Exp $";

#include <LTconfig.h>
#include <Xm/XmP.h>
#include <Xm/BaseClassP.h>
#include <Xm/DebugUtil.h>
#include <Xm/PanedWP.h>
#include <Xm/SashP.h>
#include <Xm/SeparatorP.h>
#include <stdio.h>
#include <stdlib.h>

/*
 * Forward Declarations
 */
/* core */
static void class_initialize(void);
static void class_part_initialize(WidgetClass class);
static void initialize(Widget request,
		       Widget new,
		       ArgList args,
		       Cardinal *num_args);
static void destroy(Widget w);
static Boolean set_values(Widget current,
		 	  Widget request,
			  Widget new,
			  ArgList args,
			  Cardinal *num_args);
static void resize(Widget w);
static void expose(Widget w,
		   XEvent *event,
		   Region region);
static void realize(Widget w,
		    XtValueMask *value_mask,
		    XSetWindowAttributes *attributes);

/* composite */
static void insert_child(Widget w);
static XtGeometryResult geometry_manager(Widget w,
					 XtWidgetGeometry *request,
					 XtWidgetGeometry *reply);
static void change_managed(Widget w);

/* constraint */
static void constraint_initialize(Widget request,
				  Widget new,	
				  ArgList args,
				  Cardinal *num_args);
static Boolean constraint_set_values(Widget current,
				     Widget request,	
				     Widget new,
				     ArgList args,
				     Cardinal *num_args);
Cardinal panedw_insert_position(Widget w);

/* helper */
static void  _XmPanedWLayout(Widget w, Boolean ParentResize,
			     Widget pane, Boolean TestMode,
			     XtWidgetGeometry *childgeom);

static XtGeometryResult QueryGeometry(Widget w, XtWidgetGeometry *proposed, XtWidgetGeometry *desired);
/*
 * Resources for the PanedWindow class
 */
#define Offset(field) XtOffsetOf(XmPanedWindowRec, paned_window.field)

static XtResource resources[] = {
    {
	XmNmarginWidth, XmCMarginWidth, XmRHorizontalDimension,
	sizeof(Dimension), Offset(margin_width),
	XmRImmediate, (XtPointer)3
    },
    {
	XmNmarginHeight, XmCMarginHeight, XmRVerticalDimension,
	sizeof(Dimension), Offset(margin_height),
	XmRImmediate, (XtPointer)3
    },
    {
	XmNspacing, XmCSpacing, XmRVerticalDimension,
	sizeof(Dimension), Offset(spacing),
	XmRImmediate, (XtPointer)8
    },
    {
	XmNrefigureMode, XmCBoolean, XmRBoolean,
	sizeof(Boolean), Offset(refiguremode),
	XmRImmediate, (XtPointer)True
    },
    {
	XmNseparatorOn, XmCSeparatorOn, XmRBoolean,
	sizeof(Boolean), Offset(separator_on),
	XmRImmediate, (XtPointer)True
    },
    {
	XmNsashIndent, XmCSashIndent, XmRHorizontalPosition,
	sizeof(Position), Offset(sash_indent),
	XmRImmediate, (XtPointer)-10
    },
    {
	XmNsashWidth, XmCSashWidth, XmRHorizontalDimension,
	sizeof(Dimension), Offset(sash_width),
	XmRImmediate, (XtPointer)10
    },
    {
	XmNsashHeight, XmCSashHeight, XmRVerticalDimension,
	sizeof(Dimension), Offset(sash_height),
	XmRImmediate, (XtPointer)10
    },
    {
	XmNsashShadowThickness, XmCShadowThickness, XmRHorizontalDimension,
	sizeof(Dimension), Offset(sash_shadow_thickness),
	XmRImmediate, (XtPointer)2
    },
    {
	XmNinsertPosition, XmCInsertPosition, XmRFunction,
	sizeof(XtOrderProc), XtOffsetOf(XmPanedWindowRec, composite.insert_position),
	XmRImmediate, (XtPointer)panedw_insert_position
    }
};

static XmSyntheticResource syn_resources[] = {
    {
	XmNmarginWidth,
	sizeof(Dimension), Offset(margin_width),
	_XmFromHorizontalPixels, _XmToHorizontalPixels
    },
    {
	XmNmarginHeight,
	sizeof(Dimension), Offset(margin_height),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
    {
	XmNspacing,
	sizeof(Dimension), Offset(spacing),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
    {
	XmNsashIndent,
	sizeof(Position), Offset(sash_indent),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
    {
	XmNsashWidth,
	sizeof(Dimension), Offset(sash_width),
	_XmFromHorizontalPixels, _XmToHorizontalPixels
    },
    {
	XmNsashHeight,
	sizeof(Dimension), Offset(sash_height),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
    {
	XmNsashShadowThickness,
	sizeof(Dimension), Offset(sash_shadow_thickness),
	_XmFromHorizontalPixels, _XmToHorizontalPixels
    }
};
#undef Offset

#define Offset(field) XtOffsetOf(XmPanedWindowConstraintRec, panedw.field)
static XtResource panedWindowConstraintResources[] = {
    {
	XmNallowResize, XmCBoolean, XmRBoolean,
	sizeof(Boolean), Offset(allow_resize),
	XmRImmediate, (XtPointer)False
    },
    {
	XmNpaneMinimum, XmCPaneMinimum, XmRVerticalDimension,
	sizeof(Dimension), Offset(min),
	XmRImmediate, (XtPointer)1
    }, 
    {
	XmNpaneMaximum, XmCPaneMaximum, XmRVerticalDimension,
	sizeof(Dimension), Offset(max),
	XmRImmediate, (XtPointer)1000
    },
    {
	XmNskipAdjust, XmCBoolean, XmRBoolean,
	sizeof(Boolean), Offset(skip_adjust),
	XmRImmediate, (XtPointer)False
    },
    {
	XmNpositionIndex, XmCPositionIndex, XmRShort,
	sizeof(short), Offset(position_index),
	XmRImmediate, (XtPointer)XmLAST_POSITION
    }
};

static XmSyntheticResource constraint_syn_resources[] = {
    {
	XmNpaneMinimum,
	sizeof(Dimension), Offset(min),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
    {
	XmNpaneMaximum,
	sizeof(Dimension), Offset(max),
	_XmFromVerticalPixels, _XmToVerticalPixels
    },
};

static void SashAction(Widget w, XtPointer client_data, XtPointer call_data);

static CompositeClassExtensionRec panedWCompositeExt = {
    /* next_extension */  NULL,
    /* record_type    */  NULLQUARK,
    /* version        */  XtCompositeExtensionVersion,
    /* record_size    */  sizeof(CompositeClassExtensionRec),
    /* accepts_objects */ True,
#if XtSpecificationRelease >= 6
    /* allows_change_managed_set */ True
#endif
};

static XmBaseClassExtRec _XmPanedWCoreClassExtRec = {
    /* next_extension            */ NULL,
    /* record_type               */ NULLQUARK,                             
    /* version                   */ XmBaseClassExtVersion,
    /* size                      */ sizeof(XmBaseClassExtRec),
    /* initialize_prehook        */ NULL, /* FIXME */
    /* set_values_prehook        */ NULL, /* FIXME */
    /* initialize_posthook       */ NULL, /* FIXME */
    /* set_values_posthook       */ NULL, /* FIXME */
    /* secondary_object_class    */ NULL, /* FIXME */
    /* secondary_object_create   */ NULL, /* FIXME */
    /* get_secondary_resources   */ NULL, /* FIXME */
    /* fast_subclass             */ { 0 }, /* FIXME */
    /* get_values_prehook        */ NULL, /* FIXME */
    /* get_values_posthook       */ NULL, /* FIXME */
    /* class_part_init_prehook   */ NULL, /* FIXME */
    /* class_part_init_posthook  */ NULL, /* FIXME */
    /* ext_resources             */ NULL, /* FIXME */
    /* compiled_ext_resources    */ NULL, /* FIXME */
    /* num_ext_resources         */ 0, /* FIXME */
    /* use_sub_resources         */ FALSE, /* FIXME */
    /* widget_navigable          */ NULL, /* FIXME */
    /* focus_change              */ NULL, /* FIXME */
    /* wrapper_data              */ NULL
};

static XmManagerClassExtRec _XmPanedWMClassExtRec = {
    /* next_extension            */ NULL,
    /* record_type               */ NULLQUARK,
    /* version                   */ XmManagerClassExtVersion,
    /* record_size               */ sizeof(XmManagerClassExtRec),
    /* traversal_children        */ NULL /* FIXME */
};

XmPanedWindowClassRec xmPanedWindowClassRec = {
    /* Core class part */
    {
	/* superclass            */ (WidgetClass) &xmManagerClassRec,
        /* class_name            */ "XmPanedWindow",
	/* widget_size           */ sizeof(XmPanedWindowRec),
	/* class_initialize      */ class_initialize,
	/* class_part_initialize */ class_part_initialize,
	/* class_inited          */ FALSE,
	/* initialize            */ initialize,
	/* initialize_hook       */ NULL,
	/* realize               */ realize,
	/* actions               */ NULL,
	/* num_actions           */ 0,
	/* resources             */ resources,
	/* num_resources         */ XtNumber(resources),
	/* xrm_class             */ NULLQUARK,
	/* compress_motion       */ TRUE,
	/* compress_exposure     */ XtExposeCompressMultiple,
	/* compress_enterleave   */ TRUE,
	/* visible_interest      */ FALSE,
	/* destroy               */ destroy,
	/* resize                */ resize,
	/* expose                */ expose,
	/* set_values            */ set_values,
	/* set_values_hook       */ NULL,
	/* set_values_almost     */ XtInheritSetValuesAlmost,
	/* get_values_hook       */ NULL,
	/* accept_focus          */ NULL,
	/* version               */ XtVersion,
	/* callback offsets      */ NULL,
	/* tm_table              */ XtInheritTranslations,
	/* query_geometry        */	QueryGeometry,
	/* display_accelerator   */ NULL,
	/* extension             */ (XtPointer)&_XmPanedWCoreClassExtRec
    },
    /* Composite class part */
    {
	/* geometry manager */	geometry_manager, 
        /* change_managed   */ change_managed, 
        /* insert_child     */ insert_child,
        /* delete_child     */ XtInheritDeleteChild,
        /* extension        */ &panedWCompositeExt,
    },
    /* Constraint class part */
    {
	/* subresources      */ panedWindowConstraintResources,  
        /* subresource_count */ XtNumber(panedWindowConstraintResources),
        /* constraint_size   */ sizeof(XmPanedWindowConstraintRec),
        /* initialize        */ constraint_initialize,
        /* destroy           */ NULL,  /* FIX ME */
        /* set_values        */ constraint_set_values,
        /* extension         */ NULL,   /* FIX ME */
    },
    /* XmManager class part */
    {
	/* translations                 */ XmInheritTranslations,
        /* syn_resources                */ syn_resources,
        /* num_syn_resources            */ XtNumber(syn_resources),
        /* syn_constraint_resources     */ constraint_syn_resources,
        /* num_syn_constraint_resources */ XtNumber(constraint_syn_resources),
        /* parent_process               */ XmInheritParentProcess,
	/* extension                    */ (XtPointer)&_XmPanedWMClassExtRec
    },
    /* XmPanedWindow part */
    {
	/* extension */ NULL,
    },
};

WidgetClass xmPanedWindowWidgetClass = (WidgetClass)&xmPanedWindowClassRec;

/*
 * handy defines
 */
#define _XmMin(a,b)  ((a) < (b)) ? (a) : (b)
#define _XmMax(a,b)  ((a) < (b)) ? (b) : (a)

/*
 * default number of slots.  8 seems to be a reasonable number.
 */
#define DEFAULT_NUM_SLOTS	8

static void ValidatePaneMin(Widget w)
{
    if (PWC_PaneMinimum(w) < 1) {
	XtWarning("PanedWindow: XmNpaneMinimum must be greater than 0.");
	PWC_PaneMinimum(w) = 1;
    }
    if (PWC_PaneMaximum(w) < PWC_PaneMinimum(w)) {
	_XmWarning(XtParent(w),
		   "XmNpaneMinimum must be less than XmNpaneMaximum.");
	PWC_PaneMaximum(w) = PWC_PaneMinimum(w) + 1;
    }
    if (XtHeight(w) < PWC_PaneMinimum(w))
	XtHeight(w) = PWC_PaneMinimum(w);
}

static void ValidatePaneMax(Widget w)
{
    if(PWC_PaneMaximum(w) < PWC_PaneMinimum(w)) {
	_XmWarning(XtParent(w),
		   "XmNpaneMaximum must be greater than XmNpaneMinimum.");
	PWC_PaneMinimum(w) = PWC_PaneMaximum(w) - 1;
    }
    
    if(XtHeight(w) > PWC_PaneMaximum(w))
	XtHeight(w) = PWC_PaneMaximum(w);
}



static void 
class_initialize()
{
    _XmPanedWCoreClassExtRec.record_type = XmQmotif;
}

static void
class_part_initialize(WidgetClass widget_class)
{
    CompositeClassExtension ext, *extptr;
    XmPanedWindowWidgetClass pclass = (XmPanedWindowWidgetClass)widget_class;

    extptr = (CompositeClassExtension*)_XmGetClassExtensionPtr((XmGenericClassExt*)&(pclass->composite_class.extension),
							       NULLQUARK);

    if (extptr == NULL || *extptr == NULL)
    {
	ext = (CompositeClassExtension) XtNew(CompositeClassExtensionRec);
	if (ext != NULL)
	{
	    ext->next_extension = pclass->composite_class.extension;
	    ext->record_type = NULLQUARK;
	    ext->version = XtCompositeExtensionVersion;
	    ext->record_size = sizeof(CompositeClassExtensionRec);
	    ext->accepts_objects = True;
#if XtSpecificationRelease >= 6
	    ext->allows_change_managed_set = True;
#endif
	    pclass->composite_class.extension = (XtPointer) ext;
	}
    }
    _XmFastSubclassInit(widget_class, XmPANED_WINDOW_BIT);
}

static void
initialize(Widget request,
	   Widget new,
	   ArgList args,
	   Cardinal *num_args)
{
    XmPanedWindowWidget nw = (XmPanedWindowWidget) new;
    int careabout;
    XGCValues gcv;

    PW_StartY(new) = PW_MarginHeight(new);
    PW_PaneCount(new) = 0;
    PW_NumSlots(new) = DEFAULT_NUM_SLOTS;
    PW_NumManagedChildren(new) = 0;
    PW_RecursivelyCalled(new) = False;
    PW_ResizeAtRealize(new) = True;
    PW_TopPane(nw) = NULL;
    PW_BottomPane(nw) = NULL;
    careabout = GCForeground | GCLineWidth | GCSubwindowMode | GCFunction;
    gcv.function = GXxor;
    gcv.line_width = 0;
    gcv.foreground = (1UL << DefaultDepthOfScreen(XtScreen(new))) - 1;
    gcv.subwindow_mode = IncludeInferiors;
    PW_FlipGC(nw) = XtAllocateGC(new, 0, careabout, &gcv, careabout, ~careabout);

    PW_ManagedChildren(nw) = (WidgetList)XtMalloc(sizeof(Widget) *
							PW_NumSlots(new));

    _XmPanedWLayout(new, True, NULL, False, NULL);
}

static void 
constraint_initialize(Widget request, 
		      Widget new, 
		      ArgList args, 
		      Cardinal *num_args)
{
    XdbDebug(__FILE__, new, "PanedWindow constraint initialize\n");

    ValidatePaneMin(new);
    ValidatePaneMax(new);
    PWC_Position(new) = 0;
    PWC_DHeight(new) = 0;
    PWC_DY(new) = 0;
    PWC_OldDY(new) = 0;
    PWC_IsPane(new) = False;
    PWC_Sash(new) = NULL;
    PWC_Separator(new) = NULL;
}


static void
destroy(Widget w)
{
    XtReleaseGC(w, PW_FlipGC(w));
}

static Boolean
set_values(Widget old,
	   Widget request,
	   Widget new,
	   ArgList args,
	   Cardinal *num_args)
{
    XmPanedWindowWidget nw = (XmPanedWindowWidget) new;
    XmPanedWindowWidget ow = (XmPanedWindowWidget) old;
    int i;
    int redisplay = False;
    
    if (PW_SeparatorOn(nw) != PW_SeparatorOn(ow)) {
	for(i = 1;i < PW_NumManagedChildren(nw); i++) {
	    if (PW_SeparatorOn(nw))
		XtManageChild(PWC_Separator(PW_ManagedChildren(nw)[i]));
	    else
		XtUnmanageChild(PWC_Separator(PW_ManagedChildren(nw)[i]));
	}
    }

    if (PW_MarginHeight(nw) != PW_MarginHeight(ow) 
       || PW_MarginWidth(nw) != PW_MarginWidth(ow)
       || PW_Spacing(nw) != PW_Spacing(ow)
       || PW_SashIndent(nw) != PW_SashIndent(ow)) {
	    redisplay = True;
    }

    /*
     * MLM - I don't think this is necessary.  The Layout function will
     * take care of this...  But I'm keeping it until I'm sure.
     */
#if 0
    if (PW_SashWidth(nw) != PW_SashWidth(ow)) {
	for(i = 1; i < PW_PaneCount(nw); i++)
	    XtVaSetValues(PWC_Sash(nw->composite.children[i]), 
			  XmNwidth,  PW_SashWidth(nw), 
			  NULL);
	redisplay = True;
    }

    if (PW_SashHeight(nw) != PW_SashHeight(ow)) {
	for(i = 1;i < PW_PaneCount(nw); i++)
	    XtVaSetValues(PWC_Sash(nw->composite.children[i]), 
			  XmNheight,  PW_SashHeight(nw), 
			  NULL);
	redisplay = True;
    }
#else
    if (PW_SashWidth(nw) != PW_SashWidth(ow) ||
	PW_SashHeight(nw) != PW_SashHeight(ow))
	redisplay = True;
#endif

    if (PW_SashShadowThickness(nw) != PW_SashShadowThickness(ow)) {
	for(i = 1;i < PW_PaneCount(nw); i++)
	    XtVaSetValues(PWC_Sash(nw->composite.children[i]), 
			  XmNshadowThickness, PW_SashShadowThickness(nw), 
			  NULL);
    }

    if (redisplay == True)
	_XmPanedWLayout(new, True, NULL, False, NULL);
    
    return redisplay;
}

static void
expose(Widget w, XEvent *event, Region region)
{
    int i;

    /*
     * make sure panes are lower than sashes and separators 
     * a bit kludgy.  MLM -- it should be ok to do this only on exposes.
     */
    if (XtIsRealized(w)) {
	for(i = 0; i < PW_NumManagedChildren(w); i++) {
 	    XLowerWindow(XtDisplay(w), XtWindow(PW_ManagedChildren(w)[i]));
	}
    }

    _XmRedisplayGadgets(w, event, region);
}

static Boolean 
constraint_set_values(Widget current, 
		      Widget request, 
		      Widget new, 
		      ArgList args, 
		      Cardinal *num_args)
{
    XmPanedWindowWidget pw = (XmPanedWindowWidget) XtParent(current);
    Boolean redisplay = False;
    int i;
    
    if (PWC_PaneMinimum(current) != PWC_PaneMinimum(new)) {
	redisplay = True;
	ValidatePaneMin(new);
    }

    if (PWC_PaneMaximum(current) != PWC_PaneMaximum(new)) {
	redisplay = True;
	ValidatePaneMax(new);
    }
 
    if (PWC_PositionIndex(current) > PWC_PositionIndex(new)) {
	for (i = PWC_PositionIndex(current); i > PWC_PositionIndex(new); i--) {
	    pw->composite.children[i] = pw->composite.children[i-1];
	    PWC_PositionIndex(pw->composite.children[i]) = i;
	}
	pw->composite.children[i] = new;
	if (i == 0) {
	    /*
	     * Give sash and separators to the succesor, because we don't 
	     * want it anymore and the succesor needs them.
	     */
	    PWC_Separator(pw->composite.children[1]) = PWC_Separator(new);
	    PWC_Sash(pw->composite.children[1]) = PWC_Sash(new);
	    PWC_Separator(new) = NULL;
	    PWC_Sash(new) = NULL;
	}
	redisplay = True;
    }
    else if (PWC_PositionIndex(current) < PWC_PositionIndex(new)) {
	for(i = PWC_PositionIndex(current); i < PWC_PositionIndex(new); i++) {
	    pw->composite.children[i] = pw->composite.children[i+1];
	    PWC_PositionIndex(pw->composite.children[i]) = i;
	}
        pw->composite.children[i] = new;
	redisplay = True;
    }

    if (redisplay)
	_XmPanedWLayout(XtParent(new), True, NULL, False, NULL);

    return redisplay;
}

static void
resize(Widget w)
{
    XdbDebug(__FILE__, w, "PanedWindow Resize (%d %d)\n", XtWidth(w), XtHeight(w));

    _XmPanedWLayout(w, False, NULL, False, NULL);
}

static void
realize(Widget w, 
	XtValueMask *value_mask, 
	XSetWindowAttributes *attributes)
{
#define superclass (&xmManagerClassRec)
    (*superclass->core_class.realize)(w, value_mask, attributes);
#undef superclass

    _XmPanedWLayout(w, True, NULL, False, NULL);

    XdbDebug(__FILE__, w, "PanedWindow Realize\n");
}


/*
 * Insert panes before sashes and separators.
 */ 
Cardinal 
panedw_insert_position(Widget w)
{
    XmPanedWindowWidget pw = (XmPanedWindowWidget) XtParent(w);
    int i;

    if (!PWC_IsPane(w)) {
	return pw->composite.num_children;
    }
    else {
	if (PWC_PositionIndex(w) > 0 &&
	    PWC_PositionIndex(w) < PW_PaneCount(pw)) {
	    for(i = PWC_PositionIndex(w); i < PW_PaneCount(pw); i++)
		PWC_PositionIndex(pw->composite.children[i])++;
	}
	else
	    PWC_PositionIndex(w) = PW_PaneCount(pw);
	
	return PWC_PositionIndex(w);
    }
}

static void 
insert_child(Widget w)
{
    XmPanedWindowWidget pw = (XmPanedWindowWidget) XtParent(w);

#define superclass (&xmManagerClassRec)
    if (PW_RecursivelyCalled(pw)) {
	PWC_IsPane(w) = False;
	(*superclass->composite_class.insert_child)(w);
	return;
    }
    else {
	PWC_IsPane(w) = True;
	(*superclass->composite_class.insert_child)(w);
	PW_RecursivelyCalled(pw) = True;
    }
#undef superclass

    /* Create separator and sash if this is not the first widget.
     * We must do this for every widget after the first one, but it should 
     * not be done for widgets with position_index = 0. For these we add'em
     * to the successor.
     */ 
    if (PW_PaneCount(pw)++ > 0) {
	if (PWC_PositionIndex(w) == 0)
	    w = pw->composite.children[1];
	
	PWC_Separator(w) = XtVaCreateWidget("pane_sep", xmSeparatorWidgetClass,
					    (Widget)pw, 
					    XmNshadowThickness, 4,
					    XmNseparatorType,
					    XmSHADOW_ETCHED_OUT,
					    NULL);
	
	PWC_Sash(w) = XtVaCreateWidget("pane_sash", xmSashWidgetClass,
				       (Widget)pw, 
				       XmNshadowThickness,
				       PW_SashShadowThickness(pw),
				       XmNheight, PW_SashHeight(pw),
				       XmNwidth, PW_SashWidth(pw),
				       NULL);

	XtAddCallback(PWC_Sash(w),
		      XmNcallback,
		      SashAction,
		      pw);
    }
    else {
	PWC_Sash(w) = NULL;
	PWC_Separator(w) = NULL;
    }
    PW_RecursivelyCalled(pw) = False;
}

static XtGeometryResult
QueryGeometry(Widget w, XtWidgetGeometry *proposed, XtWidgetGeometry *desired)
{
	XdbDebug(__FILE__, w, "QueryGeometry\n");

	*desired = *proposed;
	return XtGeometryYes;
}

static XtGeometryResult
geometry_manager(Widget w,
		 XtWidgetGeometry *request,
		 XtWidgetGeometry *reply)
{
    int			ask = 0, good = 0;
    XtWidgetGeometry	want;

    XdbDebug(__FILE__, w, "PanedW: geometry manager\n");
#if 1
#define Wants(x)	(request->request_mode & x)
    if ((Wants(CWX) && request->x != XtX(w)) ||
	(Wants(CWY) && request->y != XtY(w)))
	return XtGeometryNo;

    want = *request;

    _XmPanedWLayout(XtParent(w), True, w, True, &want);

    if (Wants(CWWidth)) {
	ask++;
	if ((want.request_mode & CWWidth) && want.width == request->width)
	    good++;
    }
    if (Wants(CWHeight)) {
	ask++;
	if ((want.request_mode & CWHeight) && want.height == request->height)
	    good++;
    }

    if (reply)
	*reply = want;

    if (ask == good)
	return XtGeometryYes;
    if (good == 0)
	return XtGeometryNo;
    else
	return XtGeometryAlmost;
#undef Wants
#endif
    return XtGeometryNo;
}

static int
pi_sort(const void *a, const void *b)
{
    Widget wa = *(WidgetList)a;
    Widget wb = *(WidgetList)b;

    return PWC_PositionIndex(wa) - PWC_PositionIndex(wb);
}

static void
change_managed(Widget w)
{
    XmPanedWindowWidget p = (XmPanedWindowWidget) w;
    int i, j;

	XdbDebug(__FILE__, w, "PanedWindow change_managed\n");

    if (PW_RecursivelyCalled(w)) {
	XdbDebug(__FILE__, w, "ChangeManaged was recursively called\n");
	return;
    }

    /*
     * go through our managed list, checking to see if one of our managed
     * children has been unmanaged.
     */
    for (i = 0; i < PW_NumManagedChildren(w); i++) {
	if (!XtIsManaged(PW_ManagedChildren(w)[i])) {
	    PW_RecursivelyCalled(w) = True;
	    if (PWC_Sash(PW_ManagedChildren(w)[i]))
		XtUnmanageChild(PWC_Sash(PW_ManagedChildren(w)[i]));
	    if (PWC_Separator(PW_ManagedChildren(w)[i]))
		XtUnmanageChild(PWC_Separator(PW_ManagedChildren(w)[i]));
#if 0
	    XSync(XtDisplay(w), False);
#endif
	    PW_RecursivelyCalled(w) = False;

	    if (PW_NumManagedChildren(w) - i > 1) {
		memcpy(&PW_ManagedChildren(w)[i],
		       &PW_ManagedChildren(w)[i+1],
		       sizeof(Widget) * (PW_NumManagedChildren(w) - i - 1));
	    }
	    PW_NumManagedChildren(w)--;
	    i--;
	}
    }

    /*
     * now go through the pane list and see if a previously unmanaged pane
     * has been managed.  Ugly, O(n**2) linear search, but I expect this
     * is exactly what the M*tif guys did.
     */
    for (i = 0; i < PW_PaneCount(p); i++) {
	if (XtIsManaged(p->composite.children[i])) {
	    if (!PWC_IsPane(p->composite.children[i]))
		return;
	    for (j = 0; j < PW_NumManagedChildren(w); j++) {
		if (p->composite.children[i] == PW_ManagedChildren(w)[j])
		    break;
	    }
	    /* not in list if bound met */
	    if (j == PW_NumManagedChildren(w)) {
		if (PW_NumManagedChildren(w) == PW_NumSlots(w)) {
		    PW_NumSlots(w) *= 2;
		    PW_ManagedChildren(w) =
			(WidgetList)XtRealloc((void *)PW_ManagedChildren(w),
					      sizeof(Widget) *
						PW_NumSlots(w));
		}
		PW_ManagedChildren(w)[j] = p->composite.children[i];
		PW_NumManagedChildren(w)++;

		PW_RecursivelyCalled(w) = True;
		if (PWC_Sash(p->composite.children[i]))
		    XtManageChild(PWC_Sash(p->composite.children[i]));

		if (PWC_Separator(p->composite.children[i]))
		    XtManageChild(PWC_Separator(p->composite.children[i]));
#if 0
		XSync(XtDisplay(w), False);
#endif
		PW_RecursivelyCalled(w) = False;
	    }
	}
    }

    /*
     * sort the managed children by position index
     */
    qsort((void *)PW_ManagedChildren(w), PW_NumManagedChildren(w),
	  sizeof(Widget), pi_sort);

    _XmPanedWLayout(w, True, NULL, False, NULL);

    _XmNavigChangeManaged(w);
}

static void 
_XmPanedWLayout(Widget w, Boolean ParentResize,
		Widget rchild, Boolean TestMode, XtWidgetGeometry *childgeom)
{
    Widget sash, separator; 
    Widget child;
    int i;
    Dimension height, nh, curw, curh;
    XtGeometryResult result;

    XdbDebug(__FILE__, w, "_XmPanedWLayout %s %s\n",
	ParentResize ? "ParentResize" : "", TestMode ? "TestMode" : "");

/* Sanity check */
    if (TestMode)
	ParentResize = False;

/* Find widest child and compute total height of PanedWindow */
    curw = curh = 0;
    for (i = 0; i < PW_NumManagedChildren(w); i++) {

	child = PW_ManagedChildren(w)[i];
	curw = curw >= XtWidth(child) ? curw : XtWidth(child);

	if (curh > 0) /* Add spacing between panes */
	    curh += 2 * PW_Spacing(w);

	curh += _XmMax(XtHeight(child), PWC_PaneMinimum(child));
    }
    curh += 2 * PW_MarginHeight(w);
    curw = curw + 2 * PW_MarginWidth(w);
  
/* Resize */
    if (ParentResize) {
	do {
	    result = XtMakeResizeRequest(w, curw, curh, &curw, &curh);
	} while (result == XtGeometryAlmost);

	if (result == XtGeometryYes) {
	    XtWidth(w) = curw;
	    XtHeight(w) = curh;
	}
	if (result == XtGeometryNo) {
	    curw = XtWidth(w);
	    curh = XtHeight(w);
	}
	XdbDebug(__FILE__, w, "_XmPanedWLayout %s (%d %d) => %s\n",
		 XtName(w), curw, curh, XdbGeometryResult2String(result));
    } else {
	curw = XtWidth(w);
	curh = XtHeight(w);
    }

/* now actually do the layout */
    height = PW_MarginHeight(w);
    for(i = 0; i < PW_NumManagedChildren(w); i++) {
	child = PW_ManagedChildren(w)[i];

	if (child == rchild)
		continue;	/* Don't resize the child being specified as parameter */

	XdbDebug2(__FILE__, w, child, "Layout child height %d\n", height);

	if (i > 0) {
	    height = height + PW_Spacing(w);

	    separator  = PWC_Separator(child);
	    if (separator) {
		_XmConfigureObject(separator,
				   0, height - XtHeight(separator)/2,
				   curw, XtHeight(separator),
				   XtBorderWidth(separator));
	    } 

	    sash = PWC_Sash(child);
	    if (sash) {
		XdbDebug(__FILE__, w, "moving sash to : %d %d w = %d h = %d\n", 
			 curw - PW_MarginWidth(w) +
			 PW_SashIndent(w) - PW_SashWidth(w),
			 height - (PW_SashHeight(w)/2),
			 PW_SashWidth(w),
			 PW_SashHeight(w));

		_XmConfigureObject(sash, 
				   curw - PW_MarginWidth(w) +
					PW_SashIndent(w) - PW_SashWidth(w),
				   height - PW_SashHeight(w)/2,
				   PW_SashWidth(w), PW_SashHeight(w),
				   XtBorderWidth(sash));
	    }

	    height = height + PW_Spacing(w);
	}
	XdbDebug(__FILE__, w, "child %s height %d width %d border %d\n",
		 XtName(child), XtHeight(child), XtWidth(child),
		 XtBorderWidth(child));

	if (i == (PW_NumManagedChildren(w) - 1)) {

	    nh = curh - height - PW_MarginHeight(w);

	    _XmConfigureObject(child, 
	    		       PW_MarginWidth(w), height,
			       curw - 2 * PW_MarginWidth(w), 
			       nh,
			       XtBorderWidth(child));
	}
	else {
	    _XmConfigureObject(child, 
	    		       PW_MarginWidth(w), height,
			       curw - 2 * PW_MarginWidth(w), 
			       XtHeight(child),
			       XtBorderWidth(child));
	}
	height = height + XtHeight(child);
    }
}

static int 
MoveBorderUp(Widget pane, int dy, int first)
{
    Widget prev, next=NULL;
    XmPanedWindowWidget pw = (XmPanedWindowWidget) XtParent(pane);
    int sy, my;

    if (PWC_PositionIndex(pane) == 0)
	prev = pane;
    else
	prev =  pw->composite.children[PWC_PositionIndex(pane) - 1];
    if (PWC_PositionIndex(pane) < PW_PaneCount(pw) - 1)
	next = pw->composite.children[PWC_PositionIndex(pane) + 1];

    if (dy == 0) {
	if (first) {
	    PWC_AllowResize(pane) = False;
	}
	if (PWC_PositionIndex(prev) > 0)
	    MoveBorderUp(prev, 0, False);
	PWC_AllowResize(prev) = False;
	sy = my = 0;
    }
    else {
	if (first) /* Don't make current pane larger than pane_maximum */
	    dy = _XmMin(dy, PWC_PaneMaximum(pane) - XtHeight(pane));
	
    
	/* Shrink previous pane */	  
	sy = _XmMin(dy, XtHeight(prev) - PWC_PaneMinimum(prev));
	PWC_AllowResize(prev) = True;
	PWC_DHeight(prev) = XtHeight(prev) - sy;
	
	/* Move previous pane up */
	if (PWC_PositionIndex(prev) > 0) {
	    my = MoveBorderUp(prev, dy - sy, False);
	}
	else 
	    my = 0;
	
	if (first) {
	    PWC_AllowResize(pane) = True;
	    PWC_DHeight(pane) = XtHeight(pane)+ sy + my;
	}
    }
	
    return sy + my;
}


static int
MoveBorderDown(Widget pane, int dy, int first)
{
    Widget prev, next=NULL;
    XmPanedWindowWidget pw = (XmPanedWindowWidget) XtParent(pane);
    int sy, my, my_up = 0;

    if (PWC_PositionIndex(pane) == 0)
        prev = pane;
    else
	prev = pw->composite.children[PWC_PositionIndex(pane) - 1];

    if (PWC_PositionIndex(pane) < PW_PaneCount(pw) - 1)
	next = pw->composite.children[PWC_PositionIndex(pane) + 1];

    if (dy == 0) {
	if (first) {
	    PWC_AllowResize(pane) = False;
	}
	if (next)
	    PWC_AllowResize(next) = False;
	sy = my = 0;
    }
    else {
	if (first) /* Don't make previous pane larger than pane_maximum */
	    dy = _XmMin(dy, PWC_PaneMaximum(prev) - XtHeight(prev));
	
	/* Shrink current pane */
	sy = _XmMin(dy, XtHeight(pane) - PWC_PaneMinimum(pane));
	PWC_AllowResize(pane) = True;
	PWC_DHeight(pane) = XtHeight(pane) - sy;

	/* Move current pane down */
	if (next)
	    my = MoveBorderDown(next, dy - sy, False);
	else 
	    my = 0;

	if (my < dy - sy) {
	    /* Move previous border if we cannot move the lower 
	     * border all the way.
	     */
	    if (first && PWC_PositionIndex(prev) > 0) {
		my_up = MoveBorderUp(prev, dy - sy - my, True);
	    }
	}
	else if (first && PWC_PositionIndex(prev) > 0) {
	    /* Reset previous border to avoid possible incorrect 
	     * border placement as a result of rapid cursor movement.
	     */
	    MoveBorderUp(prev, 0, False); 
	}
	if (first) {
	    PWC_AllowResize(prev) = True;
	    PWC_DHeight(prev) = XtHeight(prev) + sy + my + my_up;
	}
    }

    return sy + my;
}



static void 
MoveBorder(Widget pane, int dy, int first)
{
    if (dy < 0) { 
	MoveBorderUp(pane, - dy, first);
    }
    else if (dy > 0) {
	MoveBorderDown(pane, dy, first);
    }
}

static void 
SashAction(Widget w, XtPointer client_data, XtPointer call_data)
{
    int y, i;
    XmPanedWindowWidget pw = (XmPanedWindowWidget) client_data;
    XEvent *event = ((XmSashCallData *)call_data)->event;
    XButtonEvent *ev = (XButtonEvent *)event;
    Widget child;
    Widget pane = NULL;

    for(i = 0; i < PW_NumManagedChildren(pw); i++) {
	if (PWC_Sash(PW_ManagedChildren(pw)[i]) == w)
	    pane = PW_ManagedChildren(pw)[i];
    }

    switch(event->type) {
    case ButtonPress:
	PW_StartY(pw) = ev->y;
	PWC_DY(pane) = XtY(w) + XtHeight(w) / 2;
	break;
    case ButtonRelease:
	for(i = 0; i < PW_NumManagedChildren(pw); i++) {
	    child = PW_ManagedChildren(pw)[i];
	    if (PWC_AllowResize(child)) {
		_XmConfigureObject(child,
				   XtX(child), XtY(child),
				   XtWidth(child),
			           PWC_DHeight(child),
				   XtBorderWidth(child));
		PWC_AllowResize(child) = False;
	    }
	}
	_XmPanedWLayout((Widget)pw, False, NULL, False, NULL);
	break;
    case MotionNotify:
	PWC_OldDY(pane) = PWC_DY(pane);
	y = ((XMotionEvent *)event)->y - PW_StartY(pw);
	PWC_DY(pane) = XtY(w) + XtHeight(w)/2 + y;

	MoveBorder(pane,  y, True);

	if (PWC_DY(pane) != PWC_OldDY(pane)) {
	    XDrawLine(XtDisplay(pw), XtWindow(pw), PW_FlipGC(pw),
		      0, PWC_OldDY(pane), XtWidth(pw), PWC_OldDY(pane));

	    XDrawLine(XtDisplay(pw), XtWindow(pw), PW_FlipGC(pw),
		      0, PWC_DY(pane), XtWidth(pw), PWC_DY(pane));

	}

	break;
    }
}

Widget
XmCreatePanedWindow(Widget parent, 
		    char *name,
		    Arg *argList,
		    Cardinal argcount)
{
    return XtCreateWidget(name,
			  xmPanedWindowWidgetClass,
			  parent,
			  argList, argcount);
}

