/* Subroutines for insn-output.c for Hitachi H8/500.
   Copyright (C) 1992, 1993, 1995 Free Software Foundation, Inc.
   Contributed by Steve Chamberlain (sac@cygnus.com),
              and  Jim Wilson (wilson@cygnus.com).

This file is part of GNU CC.

GNU CC is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU CC is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU CC; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdio.h>
#include "config.h"
#include "rtl.h"
#include "regs.h"
#include "hard-reg-set.h"
#include "real.h"
#include "flags.h"
#include "insn-config.h"
#include "conditions.h"
#include "insn-flags.h"
#include "output.h"
#include "insn-attr.h"
#include "tree.h"
#include "expr.h"
#include "recog.h"

/* Forward declarations.  */
void print_operand_address ();

/* True if a #pragma interrupt has been seen for the current function.  */
int pragma_interrupt;

/* True if a #pragma saveall has been seen for the current function.  */
int pragma_saveall;

/* True if #pragma no prolog has been seen for the current function.  */
int pragma_noprolog;

/* These have values of 16 or 32.  */
int code_size;
int data_size;

extern int current_function_args_size;

/* REGNO must be saved/restored across calls if this macro is true.  */

#define WORD_REG_USED(regno) 				\
  (regno < 7 &&(pragma_interrupt 					\
   ||   pragma_saveall 					\
   ||  (regno==FRAME_POINTER_REGNUM && frame_pointer_needed)		\
   ||  (regs_ever_live[regno] & !call_used_regs[regno])))

/* Segment register names for all the memory models */

char *all_seg_name[4][FIRST_PSEUDO_REGISTER] = 
{
/* C=16 D=16 */ { 0,0,0,0,0,0,0,0},
/* C=32 D=16 */ { 0,0,0,0,0,0,"tp","tp"},
/* C=16 D=32 */ { 0,0,"dp",0,"ep",0,"tp","tp"},
/* C=32 D=32 */ { 0,0,"dp",0,"ep",0,"tp","tp"}};

/* High parts of a psi */
char *all_high_names[4][FIRST_PSEUDO_REGISTER] = 
{
  /* C=16 D=16 */ {"r0","r1","r2","r3","r4","r5","tp","tp"},
  /* C=32 D=16 */ {"r0","r1","r2","r3","r4","r5","tp","tp"},
  /* C=16 D=32 */ {"r0","r1","dp","r3","ep","r5","tp","tp"},
  /* C=32 D=32 */ {"r0","r1","dp","r3","ep","r5","tp","tp"},
};

/* Low parts of a psi */
char *all_low_names[4][FIRST_PSEUDO_REGISTER] = 
{
  /* C=16 D=16 */ {"r1","r2","r3","r4","r5","fp","fp","sp"},
  /* C=32 D=16 */ {"r1","r2","r3","r4","r5","fp","fp","sp"},
  /* C=16 D=32 */ {"r1","r2","r2","r3","r4","fp","fp","sp"},
  /* C=16 D=32 */ {"r1","r2","r2","r3","r4","fp","fp","sp"},
};

/* This records the row to be used in the above.  */
static int mn;

char *
SEG_NAME (x)
     rtx x;
{
  if (GET_CODE(x)==REG)
    return all_seg_name[mn][REGNO(x)];
  return 0;
}

/*
 * Output assembly language to FILE for the operation OP with operand size
 * SIZE.
 */

/* Output assembly language code for the function prologue.  */
static int push_order[FIRST_PSEUDO_REGISTER] =
{6, 5, 4, 3, 2, 1, 0, -1, -1};
static int pop_order[FIRST_PSEUDO_REGISTER] =
{0, 1, 2, 3, 4, 5, 6, -1, -1};

#define Q (1<<(int)QImode)
#define H (1<<(int)HImode)
#define S (1<<(int)SImode)
#define P (1<<(int)PSImode)
#define F (1<<(int)SFmode)
#define D (1<<(int)DFmode)

int ok_modes[FIRST_PSEUDO_REGISTER] =
{
  /* r0 */ Q | H | S | P | F | D,
  /* r1 */ Q | H,
  /* r2 */ Q | H | S | P | F | D,
  /* r3 */ Q | H,
  /* r4 */ Q | H | S | P | F | D,
  /* r5 */ Q | H,
  /* r6 */ Q | H | S | P | F | D,
  /* r7 */ Q | H | S | P | F | D,
  /* pc */ Q | H | S | P | F | D,
  /* ap */ Q | H | S | P | F | D,
};

/*
 * Frameful function
 *
 * <args> PC FP			<- fp <locals> <saved registers> 	<- sp
 *
 * Frameless function
 *
 * <args> PC <locals> <saved registers>   	<- sp
 *
 */

static int
count_used_regs ()
{
  int pcount = 0;
  int idx;

  /* Count the registers to push */
  for (idx = 0; idx < FIRST_PSEUDO_REGISTER; idx++)
    {
      int regno = push_order[idx];

      if (regno >= 0
	  && WORD_REG_USED (regno)
	  && (regno != FRAME_POINTER_REGNUM
	      || !frame_pointer_needed))
	pcount++;
    }
  return pcount;
}

char *
truncsipsi (operands)
     rtx operands[];
{
  int direction = REGNO (operands[1]) + 1 == REGNO (operands[0]);
  char **ptr;

  if (data_size == 32 && PTR_REG_P (REGNO (operands[0])))
    {
      /* seg <- reg */
      static char *vec[] =
      {"mov.w	%N1,%O0", "ldc	%M1,%S0 ! 1"};
      ptr = vec;
    }
  else
    {
      static char *vec[] =
      {"mov.w	%N1,%O0", "mov.w	%M1,%S0"};
      ptr = vec;
    }


  if (direction)
    {
      output_asm_insn (ptr[0], operands);
      output_asm_insn (ptr[1], operands);
    }
  else
    {
      output_asm_insn (ptr[1], operands);
      output_asm_insn (ptr[0], operands);
    }
  return "";
}

static void
movehi (word, istmpsrc, src, istmpdst, dst)
     int word;
     int istmpsrc;
     rtx src;
     int istmpdst;
     rtx dst;
{
  rtx operands[2];
  char *code;

  int src_code = GET_CODE (src);
  int dst_code = GET_CODE (dst);

  char *src_is_ptrreg = SEG_NAME (src);
  char *dst_is_ptrreg = SEG_NAME (dst);

  operands[0] = dst;
  operands[1] = src;

  if (word == 1)
    {
      output_asm_insn ("mov.w	%O1,%O0", operands);
    }
  else
    {
      /* The msw */
      if (src_is_ptrreg && !istmpsrc && dst_is_ptrreg && !istmpdst)
	{
	  output_asm_insn ("stc	%S1,%O0\n\tldc	%O0,%S0 ! 2", operands);
	}
      else if (src_is_ptrreg && !istmpsrc)
	{
	  if (istmpdst)
	    {
	      output_asm_insn ("stc	%S1,%0", operands);
	    }
	  else
	    {
	      output_asm_insn ("stc	%S1,%S0", operands);
	    }
	}
      else if (dst_is_ptrreg && !istmpdst)
	{
	  if (istmpsrc)
	    {
	      output_asm_insn ("ldc	%1,%S0 !3", operands);
	    }
	  else
	    {
	      output_asm_insn ("ldc	%S1,%S0 !4", operands);
	    }
	}
      else
	{
	  if (istmpsrc)
	    {
	      output_asm_insn ("mov.w	%1,%M0", operands);
	    }
	  else if (istmpdst)
	    {
	      output_asm_insn ("mov.w	%M1,%0", operands);
	    }
	  else
	    {
	      output_asm_insn ("mov.w	%M1,%M0", operands);
	    }
	}
    }
}

char *
movpsi (operands, hastmp)
     rtx operands[];
{
  rtx src = operands[1];
  rtx dst = operands[0];

  int src_code = GET_CODE (src);
  int dst_code = GET_CODE (dst);
  int lowfirst = 0;

  if (push_operand (operands[0], GET_MODE (operands[0])))
    {
      lowfirst = 1;
    }
  if (reg_mentioned_p (dst, src))
    {
      /*
       * something like (reg:PSI 0) (mem/s:PSI (reg:PSI 0 r0)). If
       * this is not a data32 mode then by loading one bit before
       * the other we can win
       */
      if (data_size == 16)
	{
	  rtx r_a = gen_rtx (REG, HImode, REGNO (dst));
	  rtx r_b = gen_rtx (REG, HImode, REGNO (dst) + 1);
	  lowfirst = reg_mentioned_p (r_a, src);
	}
      else if (hastmp && GET_CODE (operands[2]) == REG)
	{
	  /* We'll need to use this to do the work in two parts */
	  rtx tmp = gen_rtx (REG, HImode, REGNO (operands[2]));

	  movehi (lowfirst, 0, operands[1], 1, tmp);
	  movehi (!lowfirst, 0, operands[1], 0, operands[0]);
	  movehi (lowfirst, 1, tmp, 0, operands[0]);

	  return "";
	}
      else
	abort ();
    }

  if (src_code == REG && dst_code == REG)
    {
      /*
       * when mov.d r1,r2 do r2->r3 then r1->r2 when mov.d r1,r0 do
       * r1->r0 then r2->r1
       */

      lowfirst =  REGNO (src) + 1 == REGNO (dst);
    }

  /* output_asm_insn ("!nosplit %1 %0", operands); */
  movehi (lowfirst, 0, operands[1], 0, operands[0]);
  movehi (!lowfirst, 0, operands[1], 0, operands[0]);
  return "";
}

/* Non-zero if stdarg or varargs function.  */
int current_function_anonymous_args;

static int extra_pop;

void
function_prologue (file, size)
     FILE *file;
     int size;
{
  if (!pragma_noprolog)
    {
      int fsize = (size + 1) & -2;
      int idx;
      int regs_to_push = 0;

      int pcount = 0;

      /* Note that `current_function_args_info' is always rounded to a
	 multiple of 2.  If the last named argument is in a register,
	 we need to push that too.  */
      if (current_function_anonymous_args
	  && (current_function_args_info / 2) <= NARG_REGS)
	{
	  regs_to_push = NARG_REGS - (current_function_args_info / 2) + 1;
	  regs_to_push = regs_to_push < NARG_REGS ? regs_to_push : NARG_REGS;
	}

      /* ??? Can this ever be non-zero?  */
      if (current_function_pretend_args_size)
	{
	  regs_to_push += current_function_pretend_args_size / 2;
	}

      if (regs_to_push) {
	int off = 0;
	int n;
	int first_reg; 
	int i;
	extra_pop = regs_to_push * 2;

	/* Make room on stack */
	fprintf(file,"\tsub.w	#%d,sp\n", extra_pop);

	/* Save static chain reg, we need it.
	   We only need to do this if not code32 or data32 because the
	   latter use r0: all args are passed in memory.  */
	if (current_function_needs_context
	    && !(TARGET_CODE32 || TARGET_DATA32))
	  fprintf (file, "\tmov.w	r4,@(%d,sp)\n",
		   TARGET_CODE32 ? 4 : 2);

	/* Move one part of pc */
	fprintf(file,"\tmov.w	@(%d,sp),r4\n",extra_pop);
	fprintf(file,"\tmov.w	r4,@sp\n");
	off+=2;

	if (TARGET_CODE32) 
	  {
	    /* With a 32bit pc, move the other part too */
	    fprintf(file,"\tmov.w	@(%d,sp),r4\n",extra_pop+2);	  
	    fprintf(file,"\tmov.w	r4,@(%d,sp)\n", off);
	    off+=2;
	  }

	/* Restore static chain reg.  */
	if (current_function_needs_context
	    && !(TARGET_CODE32 || TARGET_DATA32))
	  fprintf (file, "\tmov.w	@(%d,sp),r4\n",
		   TARGET_CODE32 ? 4 : 2);

	first_reg = NARG_REGS - regs_to_push;
	for (i = 0; i < regs_to_push; i++)
	  {
	    fprintf(file,"\tmov.w	r%d,@(%d,sp)\n", first_reg, off);
	    off+=2;
	    first_reg++;
	  }
      }
      if (fsize || current_function_args_size ||  current_function_anonymous_args)
	{
	  /* Push fp */
	  if (frame_pointer_needed)
	    fprintf (file, "\tlink	fp,#%d\n", -fsize);
	  else if (fsize)
	    fprintf (file, "\tsub.w	#%d,sp\n", fsize);
	}
      pcount = count_used_regs ();

      /* Now do the push */
      if ((TARGET_SPEED && pcount <= 2)
	  || (TARGET_SPACE && pcount <= 1))
	{
	  /* Push them singly */
	  for (idx = 0; idx < FIRST_PSEUDO_REGISTER; idx++)
	    {
	      int regno = push_order[idx];

	      if (regno >= 0
		  && WORD_REG_USED (regno)
		  && (regno != FRAME_POINTER_REGNUM
		      || !frame_pointer_needed))
		{
		  fprintf (file, "\tmov.w	%s,@-sp\n", reg_names[regno]);
		}
	    }
	}
      else if (pcount)
	{
	  /* push them with a stm */
	  int need_comma = 0;

	  fprintf (file, "\tstm	(");

	  for (idx = 0; idx < FIRST_PSEUDO_REGISTER; idx++)
	    {
	      int regno = push_order[idx];

	      if (regno >= 0
		  && WORD_REG_USED (regno)
		  && (regno != FRAME_POINTER_REGNUM
		      || !frame_pointer_needed))
		{
		  if (need_comma)
		    {
		      fprintf (file, ",");
		    }
		  need_comma = 1;
		  fprintf (file, "%s", reg_names[regno]);
		}
	    }
	  fprintf (file, "),@-sp\n");
	}
    }
}

/* Output assembly language code for the function epilogue.  */

void
function_epilogue (file, size)
     FILE *file;
     int size;
{
  register int regno;

  if (!pragma_noprolog)
    {
      int fsize = (size + 1) & -2;
      int nregs;
      int offset;
      int pcount;
      int idx;
      rtx insn = get_last_insn ();

      /* If the last insn was a BARRIER, we don't have to write any code.  */

      if (GET_CODE (insn) == NOTE)
	insn = prev_nonnote_insn (insn);
      if (insn && GET_CODE (insn) == BARRIER)
	return;

      pcount = count_used_regs ();

      /* Now do the push */
      if ((TARGET_SPEED && pcount <= 2) ||
	  (TARGET_SPACE && pcount <= 1))
	{
	  /* pop them singly */
	  for (idx = 0; idx < FIRST_PSEUDO_REGISTER; idx++)
	    {
	      int regno = pop_order[idx];

	      if (regno >= 0
		  && WORD_REG_USED (regno)
		&& (regno != FRAME_POINTER_REGNUM || !frame_pointer_needed))
		{
		  fprintf (file, "\tmov.w	@sp+,%s\n", reg_names[regno]);
		}
	    }
	}
      else if (pcount)
	{
	  /* pop them with a ldm */
	  int need_comma = 0;

	  fprintf (file, "\tldm	@sp+,(");

	  for (idx = 0; idx < FIRST_PSEUDO_REGISTER; idx++)
	    {
	      int regno = pop_order[idx];

	      if (regno >= 0
		  && WORD_REG_USED (regno)
		  && (regno != FRAME_POINTER_REGNUM
		      || !frame_pointer_needed))
		{
		  if (need_comma)
		    {
		      fprintf (file, ",");
		    }
		  need_comma = 1;
		  fprintf (file, "%s", reg_names[regno]);
		}
	    }
	  fprintf (file, ")\n");
	}
      if (fsize || current_function_args_size || current_function_anonymous_args)
	{
	  if (frame_pointer_needed)
	    fprintf (file, "\tunlk	fp\n");
	  else if (fsize)
	    fprintf (file, "\tadd.w	#%d,sp\n", fsize);
	}
    }

  if (extra_pop)
    {
      /* Move the pc back where it came from */
      fprintf(file,"\tmov.w	@sp,r4\n");      
      fprintf(file,"\tmov.w	r4,@(%d,sp)\n",extra_pop);

      if (TARGET_CODE32)
	{
	  fprintf(file,"\tmov.w	@(%d,sp),r4\n",extra_pop+2);	  
	  fprintf(file,"\tmov.w	r4,@(2,sp)\n");
      }

      fprintf(file, "\tadd.w	#%d,sp\n", extra_pop);
    }

  if (pragma_interrupt)
    fprintf (file, "\trte\n");
  else
    fprintf (file, TARGET_CODE32 ? "\tprts\n" : "\trts\n");

  pragma_interrupt = 0;
  pragma_saveall = 0;
  pragma_noprolog = 0;
  current_function_anonymous_args = 0;
  extra_pop = 0;
}

/* Output assembly code for the start of the file.  */

void
asm_file_start (file)
     FILE *file;
{
  fprintf (file, "!\tGCC For the Hitachi H8/500\n");
  fprintf (file, "!\tBy Hitachi America Ltd and Cygnus Support\n");
  fprintf (file, "!\trelease 0.3\n");
  if (optimize)
    fprintf (file, "!\t-O%d\n", optimize);

  fprintf (file, "!\t%ssegmented code, %ssegmented data\n",
	   TARGET_CODE32 ? "" : "un",
	   TARGET_DATA32 ? "" : "un");

  fprintf (file, "\n\n");
  output_file_directive (file, main_input_filename);
}

/* Handle machine specific pragmas.  */

int
handle_pragma (file)
     FILE *file;
{
  int c;
  char pbuf[20];
  int psize = 0;

  c = getc (file);
  while (c == ' ' || c == '\t')
    c = getc (file);

  if (c == '\n' || c == EOF)
    {
      extern int lineno;
      lineno++;
      return c;
    }
  /* The only pragmas we understand are interrupt and saveall.  */
  while (psize < sizeof (pbuf) - 1
	 && isalpha (c))
    {
      pbuf[psize++] = c;
      c = getc (file);
    }
  pbuf[psize] = 0;

  if (strcmp (pbuf, "interrupt") == 0)
    pragma_interrupt = 1;

  if (strcmp (pbuf, "saveall") == 0)
    pragma_saveall = 1;

  if (strcmp (pbuf, "noprolog") == 0)
    pragma_noprolog = 1;

  return c;
}

/* Return assembly language string which identifies a comparison type.  */

char *
cond_string (code)
     enum rtx_code code;
{
  switch (code)
    {
    case NE:
      return "ne";
    case EQ:
      return "eq";
    case GE:
      return "ge";
    case GT:
      return "gt";
    case LE:
      return "le";
    case LT:
      return "lt";
    case GEU:
      return "hs";
    case GTU:
      return "hi";
    case LEU:
      return "ls";
    case LTU:
      return "lo";
    default:
      abort ();
    }
}

/*
 Print operand X using operand code CODE to assembly language output file
 FILE.

 'B' print "w" if pre_dec onto the stack, else "b".
     This is used to make sure that we never push a byte.

 'L' print operand provided to [p]jsr
 'M' print the msw of a 32 bit value.
 'N' print the lsw of a 32 bit value.

 'O' print the offset portion of a 24 bit address.
 'S' print the segment portion of a 24 bit address.
    'O' and 'S' do different things depending upon the memory model.  Eg
     when in big mode, 'S' on (reg:PSI 4) will produce "EP", 'O' will give
    "r4".  In medium mode, 'S' on r4 will be "r4" and 'O' "r5".

 'j' print condition code name.
 'k' print the reverse cond code name.

 'P' print "p" if in large code mode.  This allows the templates to say
     %P0jsr and produce pjsr or jsr.

 'Q' print the offset portion of a 24 bit address as if inside an
     (x,y).  Useful for peepholes.

 'V' find the set bit, and print its number.  For bitfield ops.
 'W' find the clear bit, and print its number. 'For bitfield ops.

 'R' print an opcode matching match_operator.  Used by peepholes.
 'T' print a .w or a .b depending upon mode.  Used by peepholes.

 'D' print a mov.w if it's an ordinary reg, ldc if seg part
 'E' print a mov.w if it's an ordinary reg, stc if seg part
     When data_size != 32 'D' and 'E' always print mov.w
 */

void
print_operand (file, x, code)
     FILE *file;
     rtx x;
     int code;
{
  int bitint;
  int rn = -1;
  int srn = -1;

  if (GET_CODE (x) == REG)
    {
      rn = REGNO (x);
      if (rn == 2 || rn == 4 || rn == 6 || rn == 7)
	srn = rn;
    }
  else
    {
      srn = -1;
    }

  switch (code)
    {
    case 'D':
      if ((data_size == 32 && srn >= 0)
	  || (srn == 6 || srn == 7))
	{
	  fprintf (file, "ldc");
	}
      else
	{
	  fprintf (file, "mov.w");
	}
      break;
    case 'E':
      if ((data_size == 32 && srn >= 0)
	  || (srn == 6 || srn == 7))
	{
	  fprintf (file, "stc");
	}
      else
	{
	  fprintf (file, "mov.w");
	}
      break;
    case 'R':
      switch (GET_CODE (x))
	{
	case COMPARE:
	  fprintf (file, "cmp");
	  break;
	case NEG:
	  fprintf (file, "neg");
	  break;
	case NOT:
	  fprintf (file, "not");
	  break;
	case SET:
	  fprintf (file, "mov");
	  break;
	case PLUS:
	  fprintf (file, "add");
	  break;
	case MINUS:
	  fprintf (file, "sub");
	  break;
	case AND:
	  fprintf (file, "and");
	  break;
	case IOR:
	  fprintf (file, "or");
	  break;
	case XOR:
	  fprintf (file, "xor");
	  break;
	default:
	  abort ();
	}
      /* Fall through */
    case 'T':
      switch (GET_MODE (x))
	{
	case HImode:
	  fprintf (file, ".w");
	  break;
	case QImode:
	  fprintf (file, ".b");
	  break;
	default:
	  abort ();
	}

      break;
    case 'V':
      bitint = exact_log2 (INTVAL (x));
      if (bitint == -1)
	abort ();
      fprintf (file, "#%d", bitint & 0xf);
      break;
    case 'W':
      bitint = exact_log2 ((~INTVAL (x)) & 0xffff);
      if (bitint == -1)
	abort ();
      fprintf (file, "#%d", bitint & 0xf);
      break;

    case 'B':
      if (GET_CODE (x) == MEM
	  && GET_CODE (XEXP (x, 0)) == PRE_DEC
	  && REGNO (XEXP (XEXP (x, 0), 0)) == 7)
	{
	  fprintf (file, "w");
	}
      else
	{
	  fprintf (file, "b");
	}
      break;
    case 'P':
      if (TARGET_CODE32)
	fprintf (file, "p");
      break;
    case 'L':
      if (GET_CODE (x) == REG)
	{
	  if (data_size == 16 && code_size == 32)
	    {
	      int rn = REGNO (x);
	      fprintf (file, "@%s", reg_names[rn]);
	      break;
	    }
	}
      fprintf (file, "@");
      print_operand_address (file, x);
      break;
    case 'j':
      fprintf (file, cond_string (GET_CODE (x)));
      break;
    case 'k':
      fprintf (file, cond_string (reverse_condition (GET_CODE (x))));
      break;
    case 'Q':
      /* print the offset part of an address inside an () */
      switch (GET_CODE (x))
	{
	case SYMBOL_REF:
	case CONST:
	case LABEL_REF:
	  fprintf (file, "%%off(");
	  print_operand_address (file, x);
	  fprintf (file, ")");
	  break;
	case CONST_INT:
	  print_operand_address (file, x);
	  break;
	default:
	  abort ();
	}

      break;

    case 'O':
      /*
       * print the offset register or the highest word of a PSI
       * thing (low 16 bits of memory)
       */
      if (GET_MODE (x) == HImode)
	{
	  print_operand (file, x, 0);
	}
      else
	switch (GET_CODE (x))
	  {
	  default:
	    abort ();


	  case MEM:
	    if (GET_CODE (XEXP (x, 0)) != PRE_DEC
		&& GET_CODE (XEXP (x, 0)) != POST_INC)
	      {
		x = adj_offsettable_operand (x, 2);
	      }
	    fprintf (file, "@");
	    print_operand_address (file, XEXP (x, 0), 0);
	    break;
	  case REG:
	    fprintf (file, "%s", all_low_names[mn][REGNO (x)]);
	    break;

	  case SYMBOL_REF:
	  case CONST:
	  case LABEL_REF:
	    fprintf (file, "#%%off(");
	    print_operand_address (file, x);
	    fprintf (file, ")");
	    break;
	  case CONST_INT:
	    fprintf (file, "#");
	    print_operand_address (file, x);
	    break;
	  }
      break;
    case 'S':
      /*
       * print the segment register which goes with the reg named,
       * or if memory, the 2nd lowest byte - of a PSI
       */
      if (GET_MODE (x) == HImode)
	print_operand (file, x, 0);
      else
	{
	  switch (GET_CODE (x))
	    {
	    case MEM:
	      if (GET_CODE (XEXP (x, 0)) != PRE_DEC
		  && GET_CODE (XEXP (x, 0)) != POST_INC)
		{
		  x = adj_offsettable_operand (x, 1);
		}
	      fprintf (file, "@");
	      print_operand_address (file, XEXP (x, 0));
	      break;
	    case REG:
	      fprintf (file, all_high_names[mn][REGNO(x)]);
	      break;

	    case CONST_INT:
	    case SYMBOL_REF:
	    case CONST:
	    case LABEL_REF:
	      fprintf (file, "#%%page(");
	      print_operand_address (file, x);
	      fprintf (file, ")");
	      break;
	    default:
	      abort ();
	      break;
	    }
	}
      break;
    case 'N':
      /*
       * Print the next highest register, - the lsw in a 32 bit
       * value
       */
      switch (GET_CODE (x))
	{
	case REG:
	  fprintf (file, "%s", reg_names[REGNO (x) + 1]);
	  break;
	case MEM:
	  if (GET_CODE (XEXP (x, 0)) == PRE_DEC
	      || GET_CODE (XEXP (x, 0)) == POST_INC)
	    {
	      print_operand (file, x, 0);
	    }
	  else
	    {
	      print_operand (file, adj_offsettable_operand (x, 2), 0);
	    }
	  break;

	case CONST_INT:
	  fprintf (file, "#%d", (INTVAL (x)) & 0xffff);
	  break;

	case SYMBOL_REF:
	case CONST:
	case LABEL_REF:
	  fprintf (file, "#%%off(");
	  print_operand_address (file, x);
	  fprintf (file, ")");
	  break;

	default:
	  debug_rtx (x);
	  fprintf (file, "foo");
	}
      break;

    case 'M':
      /* Print the lowest register, - the msw in a 32 bit value */
      switch (GET_CODE (x))
	{
	case SYMBOL_REF:
	case CONST:
	case LABEL_REF:
	  fprintf (file, "#%%hi16(");
	  print_operand_address (file, x);
	  fprintf (file, ")");
	  break;

	case REG:
	  fprintf (file, "%s", reg_names[REGNO (x) + 0]);
	  break;
	case MEM:
	case PRE_DEC:
	case POST_INC:
	  if (GET_CODE (XEXP (x, 0)) == PRE_DEC
	      || GET_CODE (XEXP (x, 0)) == POST_INC)
	    {
	      print_operand (file, x, 0);
	    }
	  else
	    {
	      print_operand (file, adj_offsettable_operand (x, 0), 0);
	    }
	  break;
	case CONST_INT:
	  fprintf (file, "#%d", (INTVAL (x) >> 16) & 0xffff);
	  break;
	default:
	  debug_rtx (x);
	  fprintf (file, "bar");

	}
      break;

    default:
      switch (GET_CODE (x))
	{
	case SUBREG:
	  fprintf (file, "%s", reg_names[REGNO (SUBREG_REG (x))]);
	  if (SUBREG_WORD (x) == 0)
	    abort ();
	  break;
	case REG:
	  fprintf (file, "%s", reg_names[REGNO (x)]);
	  break;

	case MEM:
	  fprintf (file, "@");
	  output_address (XEXP (x, 0));
	  break;

	case CONST_INT:
	case SYMBOL_REF:
	case CONST:
	case LABEL_REF:
	  fprintf (file, "#");
	  print_operand_address (file, x);
	  break;
	}
    }
}

char *
pointy_reg (x)
     rtx x;
{
  int rn = REGNO (x);
  if (data_size == 16 && code_size == 32
      && (rn != 6 && rn != 7))
    return reg_names[rn + 1];

  return reg_names[rn];
}

/* Output assembly language output for the address ADDR to FILE.  */

void
print_operand_address (file, addr)
     FILE *file;
     rtx addr;
{
  switch (GET_CODE (addr))
    {
    case ZERO_EXTEND:
      /* Only found around a subreg */
      {
	rtx inside = XEXP (addr, 0);
	if (GET_CODE (inside) == TRUNCATE)
	  {
	    print_operand_address (file, XEXP (inside, 0));
	  }
      }
      break;
    case LO_SUM:
      fprintf (file, "%s+low(", reg_names[REGNO (XEXP (addr, 0))]);
      print_operand_address (file, XEXP (addr, 1));
      fprintf (file, ")");
      break;
    case REG:
      fprintf (file, "%s", pointy_reg (addr));
      break;

    case PRE_DEC:
      fprintf (file, "-%s", pointy_reg (XEXP (addr, 0)));
      break;

    case POST_INC:
      fprintf (file, "%s+", pointy_reg (XEXP (addr, 0)));
      break;

    case PLUS:
      fprintf (file, "(");
      if (GET_CODE (XEXP (addr, 0)) == REG)
	{
	  /* reg,foo */
	  print_operand_address (file, XEXP (addr, 1));
	  fprintf (file, ",");
	  print_operand_address (file, XEXP (addr, 0));
	}
      else
	{
	  /* foo+k */
	  print_operand_address (file, XEXP (addr, 0));
	  fprintf (file, "+");
	  print_operand_address (file, XEXP (addr, 1));
	}
      fprintf (file, ")");
      break;

    case CONST_INT:
      if (INTVAL (addr) < 0)
	{
	  int v = -INTVAL (addr);

	  fprintf (file, "-%d", v);
	}
      else
	fprintf (file, "%d", INTVAL (addr));
      break;

    default:
      output_addr_const (file, addr);
      break;
    }
}

/*
 * Output to FILE a reference to the user-level label NAME. Strip off the
 * section name if any.  It is separated from the label name by a space.
 */

void
asm_output_labelref (file, name)
     FILE *file;
     char *name;
{
  char *p;

  fputc ('_', file);

  for (p = name; *p; p++)
    {
      if (*p == ' ')
	{
	  /* If we found a space in the name, then we've
	     skipped over the section encoding.  */
	  fputs (p + 1, file);
	  return;
	}
    }

  /* No space, so no section.  */
  fputs (name, file);
}

/* Prepare for a move by ensuring that one of the operands is in a register,
   or the source is constant.  */

int
prep_move_operands (operands, mode)
     rtx operands[];
     enum machine_mode mode;
{
  if (!register_operand (operands[0], mode)
      && !register_operand (operands[1], mode)
      && ! immediate_operand (operands[1], mode))
    {
      /* copy the source to a register */
      operands[1] = copy_to_mode_reg (mode, operands[1]);
    }
  return 0;
}

int
hard_regno_mode_ok (regno, MODE)
     enum machine_mode MODE;
{
  switch (regno)
    {
    case R1_R:
    case R3_R:
    case R5_R:
      return !((MODE == SImode || MODE == SFmode));
    case R0_R:
    case R2_R:
    case R4_R:
    case R6_R:
    case R7_R:
      return 1;
    case PC_R:
      return 0;
    case AP_R:
      return 1;
    }
}

int
call_operand (op, mode)
     rtx op;
     enum machine_mode mode;
{
  if (CONSTANT_P (op) || REG_P (op))
    return 1;

  if (GET_CODE (op) == ZERO_EXTEND && code_size == 16 && 0)
    {
      rtx inside = XEXP (op, 0);
      if (REG_P (inside))
	return 1;
    }
  return 0;
}

/* Update the condition code from the insn.  */
int
notice_update_cc (body, insn)
     rtx body;
     rtx insn;
{
  switch (get_attr_cc (insn))
    {
    case CC_NONE:
      /* Insn does not affect the CC at all */
      break;
    case CC_WHOOPS:
      abort ();
    case CC_SET:
      /*
       * Insn sets CC to recog_operand[0], but overflow 0ed 
       * and carry is untouched.
       */
      CC_STATUS_INIT;
      cc_status.flags |= CC_NO_OVERFLOW;
      cc_status.value1 = recog_operand[0];
      break;

    case CC_ARITH:
      /*
       * Insn sets CC to recog_operand[0], but overflow is
       * bogus and carry is wierd.
       */
      CC_STATUS_INIT;
      cc_status.flags |= CC_NO_OVERFLOW;
      cc_status.value1 = recog_operand[0];
      break;
    case CC_COMPARE:
      /* The insn is a compare instruction */
      CC_STATUS_INIT;
      cc_status.value1 = recog_operand[0];
      cc_status.value1 = recog_operand[1];
      break;
    case CC_TEST:
      /* The insn is a test against 0 */
      CC_STATUS_INIT;
      cc_status.value1 = recog_operand[0];
      cc_status.value1 = const0_rtx;
      break;
    case CC_CLOBBER:
      /* Insn clobbers CC. */
      CC_STATUS_INIT;
      break;
    }
}

/*
 * Shifts.
 *
 * We devote a fair bit of code to getting efficient shifts since we can only
 * shift one bit at a time.
 *
 * Here are some thoughts on what the absolutely positively best code is. "Best"
 * here means some rational trade-off between code size and speed, where
 * speed is more preferred but not at the expense of generating 20 insns.
 *
 * H8/500 QImode shifts 1-4   - do them inline 5-6   - ASHIFT | LSHIFTRT:
 * rotate, mask off other bits ASHIFTRT: loop 7     - ASHIFT | LSHIFTRT:
 * rotate, mask off other bits ASHIFTRT: shll, subx (propagate carry bit to
 * all bits)
 *
 * H8/500 HImode shifts 1-4   - do them inline 5-6   - loop 7     - shift other
 * way once, move byte into place, move carry bit into place 8     - move
 * byte, zero (ASHIFT | LSHIFTRT) or sign extend other (ASHIFTRT) 9     -
 * inline shift 1-4, move byte, set other byte 13-14 - ASHIFT | LSHIFTRT:
 * rotate 3/2, mask, move byte, set other byte to 0 - ASHIFTRT: loop 15    -
 * ASHIFT | LSHIFTRT: rotate 1, mask, move byte, set other byte to 0 -
 * ASHIFTRT: shll, subx, set other byte
 *
 * Don't Panic!!!
 *
 * All of these haven't been implemented.  I've just documented them and
 * provided hooks so they can be.
 */

nshift_operator (x, mode)
     rtx x;
     enum machine_mode mode;
{
  switch (GET_CODE (x))
    {
    case ASHIFTRT:
    case LSHIFTRT:
    case ASHIFT:
      return 1;

    default:
      return 0;
    }
}

int
expand_a_shift (mode, code, operands)
     enum machine_mode mode;
     int code;
     rtx operands[];
{
  extern int rtx_equal_function_value_matters;

  emit_move_insn (operands[0], operands[1]);

  /*
	 * need a loop to get all the bits we want  - we generate the code at
	 * emit time, but need to allocate a scratch reg now
	 */

  emit_insn (gen_rtx
	     (PARALLEL, VOIDmode,
	      gen_rtvec (2, gen_rtx (SET, VOIDmode, operands[0],
			    gen_rtx (code, mode, operands[0], operands[2])),
	       gen_rtx (CLOBBER, VOIDmode, gen_rtx (SCRATCH, HImode, 0)))));

  return 1;
}

/*
 * Shift algorithm determination.
 *
 * There are various ways of doing a shift: SHIFT_INLINE: If the amount is small
 * enough, just generate as many one-bit shifts as we need. SHIFT_ROT_AND: If
 * the amount is large but close to either end, rotate the necessary bits
 * into position and then set the rest to zero. SHIFT_SPECIAL: Hand crafted
 * assembler. SHIFT_LOOP:    If the above methods fail, just loop.
 */

enum shift_alg
{
  SHIFT_INLINE,
  SHIFT_ROT_AND,
  SHIFT_SPECIAL,
  SHIFT_LOOP,
  SHIFT_MAX
};

/* Symbols of the various shifts which can be used as indices.  */

enum shift_type
  {
    SHIFT_ASHIFT, SHIFT_LSHIFTRT, SHIFT_ASHIFTRT
  };

/* Symbols of the various modes which can be used as indices.  */

enum shift_mode
  {
    QIshift, HIshift, SIshift, PSIshift
  };

/*
 * Assembler instruction shift table.
 *
 * These tables are used to look up the basic shifts. They are indexed by cpu,
 * shift_type, and mode.
 */

static const char *const shift_one[3][4] =
{
 /* SHIFT_ASHIFT */
  {
    "shal.b %0",
    "shal.w %0",
    "shal.w %N0\n\trotxl.w	%M0",
    "shal.w %O0"
  },
 /* SHIFT_LSHIFTRT */

  {
    "shlr.b	%0",
    "shlr.w	%0",
    "shlr.w	%M0\n\trotxr.w	%N0",
    "shlr.w	%O0"
  },
 /* SHIFT_ASHIFTRT */
  {
    "shar.b	%0",
    "shar.w	%0",
    "shar.w	%M0\n\trotxr.w	%N0",
    "shar.w	%O0"
  }
};

/* Rotates are organized by which shift they'll be used in implementing.  */

static const char *const rotate_one[3][4] =
{
 /* SHIFT_ASHIFT */
  {
    "rotr.b %0",
    "rotr.w %0",
    0,
    0
  },
 /* SHIFT_LSHIFTRT */
  {
    "rotl.b %0",
    "rotl.w %0",
    0,
    0,
  },
 /* SHIFT_ASHIFTRT */
  {
    "rotr.b %0",
    "rotr.w %0",
    0,
    0,
  }
};


/*
 * Given MODE, SHIFT_TYPE,  and shift count COUNT, determine the best
 * algorithm for doing the shift. The assembler code is stored in ASSEMBLER.
 * We don't achieve maximum efficiency in all cases, but the hooks are here
 * to do so.
 *
 * For now we just use lots of switch statements.  Since we don't even come
 * close to supporting all the cases, this is simplest.  If this function
 * ever gets too big, perhaps resort to a more table based lookup.
 *
 * WARNING: The constraints on insns shiftbyn_QI/HI/SI assume shifts of 1,2,3,4
 * will be inlined (1,2 for SI).
 */

static enum shift_alg
get_shift_alg (shift_type, mode, count, assemblerp)
     enum shift_type shift_type;
     enum machine_mode mode;
     int count;
     const char **assemblerp;
{
  /* The default is to loop.  */
  enum shift_alg alg = SHIFT_LOOP;
  enum shift_mode shift_mode;

  if (count < 0 || count > GET_MODE_BITSIZE (mode))
    abort ();

  switch (mode)
    {
    case QImode:
      shift_mode = QIshift;
      break;
    case PSImode:
      shift_mode = PSIshift;
      break;

    case HImode:
      shift_mode = HIshift;
      break;


    case SImode:
      shift_mode = SIshift;
      break;
    default:
      abort ();
    }

  /* Assume either SHIFT_LOOP or SHIFT_INLINE.  */
  *assemblerp = shift_one[shift_type][shift_mode];

  /* Now look for cases we want to optimize.  There is no need to look
     for shifts that will be done via a loop.  */

  switch (shift_mode)
    {
    case QIshift:
      if (count <= 4)
	return SHIFT_INLINE;
      if (shift_type == SHIFT_ASHIFTRT)
	{
	  if (count == 7)
	    {
	      *assemblerp = "shll %0\t! shar.b(7)\n\tsubx %0,%0\t! end shar.b(7)";
	      return SHIFT_SPECIAL;
	    }
	  return SHIFT_LOOP;
	}
      else
	{
	  *assemblerp = rotate_one[shift_type][shift_mode];
	  return SHIFT_ROT_AND;
	}
      break;


    case HIshift:
      if (count <= 4)
	return SHIFT_INLINE;
      if (count == 8)
	{
	  switch (shift_type)
	    {
	    case SHIFT_ASHIFT:
	      *assemblerp = "swap	%0\n\tclr.b	%0! end shal.w(8)";
	      return SHIFT_SPECIAL;
	    case SHIFT_LSHIFTRT:
	      *assemblerp = "clr.b	%0\n\tswap	%0! end shlr.w(8)";
	      return SHIFT_SPECIAL;
	    case SHIFT_ASHIFTRT:
	      *assemblerp = "swap	%0\n\texts	%0! end shlr.w(8)";
	      return SHIFT_SPECIAL;
	    }
	  abort ();
	}
      if (count == 15)
	{
	  if (shift_type == SHIFT_ASHIFTRT)
	    {
	      *assemblerp = "shll.w %0\t! shar.w(15)\n\tsubx.w %0,%0! shar.w(15)";
	      return SHIFT_SPECIAL;
	    }
	  else
	    {
	      *assemblerp = rotate_one[shift_type][shift_mode];
	      return SHIFT_ROT_AND;
	    }
	}
      break;
    case PSIshift:
      if (count <= 2)
	return SHIFT_INLINE;
      break;

    case SIshift:
      if (count <= 2)
	return SHIFT_INLINE;

      if (count == 31)
	{
	  switch (shift_type)
	    {
	    case SHIFT_ASHIFTRT:
	      *assemblerp = "shll.w %N0\n\tsubx.w\t%N0,%N0\n\tmov.w	%N0,%M0";
	      break;

	    case SHIFT_ASHIFT:
	      *assemblerp = "clr.w %M0\t! shal.l(31)\n\tshlr.w %N0\n\tmov.w %M0,%N0\n\trotxr.w %M0\t! end shal.l(31)";
	      break;
	    case SHIFT_LSHIFTRT:
	      *assemblerp = "clr.w %N0\t! shal.l(31)\n\tshll.w %M0\n\tmov.w %N0,%M0\n\trotxl.w %N0\t! end shal.l(31)";

	      break;

	    default:
	      abort ();
	    }
	  return SHIFT_SPECIAL;

	}
      break;
    default:
      abort ();
    }

  return alg;
}

/* Emit the assembler code for doing shifts.  */

char *
emit_a_shift (insn, operands)
     rtx insn;
     rtx *operands;
{
  static int loopend_lab;
  char *assembler;
  rtx inside = PATTERN (insn);
  rtx shift = operands[3];
  enum machine_mode mode = GET_MODE (shift);
  enum rtx_code code = GET_CODE (shift);
  enum shift_type shift_type;
  enum shift_mode shift_mode;

  loopend_lab++;

  switch (mode)
    {
    case QImode:
      shift_mode = QIshift;
      break;
    case PSImode:
      shift_mode = PSIshift;
      break;
    case HImode:
      shift_mode = HIshift;
      break;
    case SImode:
      shift_mode = SIshift;
      break;
    default:
      abort ();
    }

  switch (code)
    {
    case ASHIFTRT:
      shift_type = SHIFT_ASHIFTRT;
      break;
    case LSHIFTRT:
      shift_type = SHIFT_LSHIFTRT;
      break;
    case ASHIFT:
      shift_type = SHIFT_ASHIFT;
      break;
    default:
      abort ();
    }

  if (GET_CODE (operands[2]) != CONST_INT)
    {
      /* Indexing by reg, so have to loop and test at top */
      output_asm_insn ("mov.w	%2,%4", operands);
      fprintf (asm_out_file, "\tbeq	.Lle%d\n", loopend_lab);
      output_asm_insn ("sub.w	#1,%4", operands);
      /* Get the assembler code to do one shift.  */
      get_shift_alg (shift_type, mode, 1, &assembler);
    }
  else
    {
      int n = INTVAL (operands[2]);
      enum shift_alg alg;

      /* If the count is negative, make it 0.  */
      if (n < 0)
	n = 0;
      /* If the count is too big, truncate it.
         ANSI says shifts of GET_MODE_BITSIZE are undefined - we choose to
	 do the intuitive thing.  */
      else if (n > GET_MODE_BITSIZE (mode))
	n = GET_MODE_BITSIZE (mode);

      alg = get_shift_alg (shift_type, mode, n, &assembler);

      switch (alg)
	{
	case SHIFT_INLINE:
	  while (--n >= 0)
	    output_asm_insn (assembler, operands);
	  return "";
	case SHIFT_ROT_AND:
	  {
	    int m = GET_MODE_BITSIZE (mode) - n;
	    int mask = (shift_type == SHIFT_ASHIFT
			? ((1 << GET_MODE_BITSIZE (mode) - n) - 1) << n
			: (1 << GET_MODE_BITSIZE (mode) - n) - 1);
	    char insn_buf[200];

	    /* Not all possibilities of rotate are supported.  They
	       shouldn't be generated, but let's watch for 'em.  */

	    if (assembler == 0)
	      abort ();
	    while (--m >= 0)
	      output_asm_insn (assembler, operands);
	    switch (mode)
	      {
	      case QImode:
		sprintf (insn_buf, "and.b #%d,%%0\t! end shift %d via rotate+and",
			 mask, n);
		break;
	      case HImode:
		sprintf (insn_buf, "and.w #%d,%%0\n\t! end shift %d via rotate+and",
			 mask, n);
		break;
	      case SImode:
		abort ();
	      }
	    output_asm_insn (insn_buf, operands);
	    return "";
	  }
	case SHIFT_SPECIAL:
	  output_asm_insn (assembler, operands);
	  return "";
	}

      /* Need a loop, move limit to tmp reg */
      fprintf (asm_out_file, "\tmov.w	#%d,r%d\n", n - 1, REGNO (operands[4]));
    }

  fprintf (asm_out_file, ".Llt%d:\n", loopend_lab);
  output_asm_insn (assembler, operands);
  fprintf (asm_out_file, "\tscb/f	r%d,.Llt%d\n",
	   REGNO (operands[4]), loopend_lab);
  fprintf (asm_out_file, ".Lle%d:\n", loopend_lab);

  return "";
}

/* Return a sequence of instructions to perform SI or SF. */

char *
move_two_words (operands, mode)
     rtx operands[];
     enum machine_mode mode;
{
  rtx dst = operands[0];
  rtx src = operands[1];
  int lowfirst;

  /*
   * We sometimes get SI requests for the stack pointer, then use PSI
   * moving instead
   */

  if ((GET_CODE (dst) == REG
       && REGNO (dst) == STACK_POINTER_REGNUM)
      || (GET_CODE (src) == REG
	  && REGNO (src) == STACK_POINTER_REGNUM))
    {
      abort();
      return movpsi (operands, 0);
    }

  if (GET_CODE (dst) == REG
      && GET_CODE (src) == REG)
    {
      /*
       * when mov.d r1,r2 do r2->r3 then r1->r2 when mov.d r1,r0 do
       * r1->r0 then r2->r1
       */

      if (REGNO (src) + 1 != REGNO (dst))
	return "mov.w	%1,%0\n\tmov.w	%N1,%N0 ! cr";
      else
	return "mov.w	%N1,%N0\n\tmov.w	%1,%0 ";

    }
  else if (GET_CODE (src) == MEM)
    {
      int ptrreg1 = -1;
      int ptrreg2 = -1;
      int dreg = REGNO (dst);
      rtx inside = XEXP (src, 0);

      if (GET_CODE (inside) == CONST)
	inside = XEXP (inside, 0);

      if (GET_CODE (inside) == REG)
	{
	  ptrreg1 = REGNO (inside);
	}
      else if (GET_CODE (inside) == PLUS)
	{
	  rtx lhs = XEXP (inside, 0);
	  rtx rhs = XEXP (inside, 1);
	  if (GET_CODE (lhs) == REG)
	    ptrreg1 = REGNO (lhs);
	  if (GET_CODE (rhs) == REG)
	    ptrreg1 = REGNO (rhs);
	}
      else if (GET_CODE (inside) == SYMBOL_REF)
	{
	  /* Nothin needed */
	}
      else if (GET_CODE (inside) == POST_INC)
	{
	  /* mov.b	@r++,r must be done low/high */
	  return "mov.w	%1,%0\n\tmov.w	%1,%N0";
	}
      else if (GET_CODE (inside) == PRE_DEC)
	{
	  /* mov @-r,r must be done high/low */
	  return "mov.w	%1,%N0\n\tmov.w	%1,%M0";
	}
      else
	abort ();

      /* Work out the safe way to copy - normally something like

	 (set (reg:SF 4) (mem/s:SF (plus:PSI (reg:PSI 4 r4)
	 (const_int -4))))

	 needs to be copied r5 first, so that r4 doesn't get trampled
	 and break the indexing.  This is safe since PSI r4 is really ep:r4.

	 When data_size==16 && code_size==32, pointers are still 32
	 bits but we only use the offset, so PSI:r4 is r4:r5, with
	 r4 ignored.

	 So adjust the pointer regs.  */

      if (data_size == 16 && code_size == 32)
	{
	  if (ptrreg1 >= 0)
	    ptrreg1++;
	  if (ptrreg2 >= 0)
	    ptrreg2++;
	}
      if (dreg == ptrreg1)
	{
	  /* Copy into the second half first */
	  return "mov.w	%N1,%N0\n\tmov.w	%M1,%M0 ! cr";
	}
    }

  if (GET_CODE (dst) == MEM)
    {
      rtx inside = XEXP (dst, 0);
      if (GET_CODE (inside) == PRE_DEC)
	{
	  /*
	   * Have to push low first so than high part is in
	   * lower address
	   */
	  return "mov.w	%N1,%0\n\tmov.w	%M1,%0";
	}
      if (GET_CODE (inside) == POST_INC)
	{
	  /* Have to push high first */
	  return "mov.w	%M1,%0\n\tmov.w	%N1,%0";
	}
    }
  return "mov.w	%M1,%M0\n\tmov.w	%N1,%N0";
}

int
h8500_rtx_costs (x, ocode, outer_code)
     rtx x;
     enum rtx_code ocode;
     enum rtx_code outer_code;
{
  enum rtx_code code;
  enum machine_mode mode = GET_MODE (x);
  int msize = GET_MODE_SIZE (mode);
  if (x == 0)
    return 0;
  code = GET_CODE (x);

  switch (code)
    {
    case MINUS:
    case COMPARE:
    case PLUS:
    case AND:
    case IOR:
      return msize *
	(h8500_rtx_costs (XEXP (x, 0), code) + h8500_rtx_costs (XEXP (x, 1), code));
    case ASHIFT:
    case ASHIFTRT:
    case LSHIFTRT:
    case NEG:
    case SIGN_EXTEND:
    case ZERO_EXTEND:
      return 2 * msize * (h8500_rtx_costs (XEXP (x, 0), code));
    case DIV:
    case UDIV:
    case MOD:
    case UMOD:
      if (GET_MODE (x) == HImode || GET_MODE (x) == QImode)
	return 20;
      return 70;
    case MULT:
      if (GET_MODE (x) == HImode || GET_MODE (x) == QImode)
	return 30;
      return 70;
    case NE:
      return 1;
    case IF_THEN_ELSE:
      return -1;
    default:
      return -1;
    case SYMBOL_REF:
      return 3;
    case CONST_INT:
      return 1;
    case CC0:
    case REG:
      return 1;
    case MEM:
      return msize * h8500_rtx_costs (XEXP (x, 0), code);
    case PRE_INC:
      return 1;
    case PRE_DEC:
      return 1;
    }
}

never ()
{
  return 0;
}

binop_operator (x, mode)
     rtx x;
     enum machine_mode mode;
{
  switch (GET_CODE (x))
    {
    case AND:
    case PLUS:
    case MINUS:
    case IOR:
    case XOR:
    case COMPARE:
      return 1;
      break;
    }
  return 0;
}

unop_operator (x, mode)
     rtx x;
     enum machine_mode mode;
{
  switch (GET_CODE (x))
    {
    case SET:
    case NEG:
    case NOT:
      return 1;
      break;
    }
  return 0;
}

void
override_options ()
{
  code_size = TARGET_CODE32 ? 32 : 16;
  data_size = TARGET_DATA32 ? 32 : 16;

  /* FIXME:  This code used to be "MASK_CODE32 && !MASK_DATA32".
     The current code is more meaningful but it crashes building libgcc2
     for -mmedium.  */
  if (0 && TARGET_CODE32 && !TARGET_DATA32)
    {
      /* Can store psi in any reg */
      int i;
      for (i = 0; i < 7; i++)
	ok_modes[i] |= P;
    }
  
  mn = (TARGET_CODE32 != 0) + (TARGET_DATA32 != 0) * 2;
}

/* Trampoline fragments. 

   The nop's are to ensure that the arg addresses
   happen on even boundaries.
   Note that the small model has no free registers.

small code:
	nop			00
	mov.w	#0x1234,r4	5c 12 34
	nop			00
	jmp	@0x1234:16	10 12 34

large code:

	nop			00
	mov.w	#0x1234,r0	58 12 34
	nop			00
	mov.w	#0x5678,r1	59 56 78
	nop			00
	mov.w	#0x1234,r2	5a 12 34
	nop			00
	mov.w	#0x5678,r3	5b 56 78
	nop			00
	pjmp	@r2		11 c2
*/

void
trampoline_template (file)
     FILE *file;
{
  if (data_size == 32 || code_size == 32)
    {
      /* First the context pointer */

      /* To make the address constant aligned */
      fprintf (file, "\tnop\n");
      fprintf (file, "\tmov.w	#0x1234,r0\n");
      fprintf (file, "\tnop\n");
      fprintf (file, "\tmov.w	#0x5678,r1\n");

      /* Then the function pointer */

      /* To make the address constant aligned */
      fprintf (file, "\tnop\n");	
      fprintf (file, "\tmov.w	#0x1234,r2\n");
      fprintf (file, "\tnop\n");
      fprintf (file, "\tmov.w	#0x5678,r3\n");
      if (code_size == 32)
	fprintf (file, "\tpjmp	@r2\n");
      else
	fprintf (file, "\tjmp	@r2\n");
    }
  else
    {
      /* First the context pointer */

      /* To make the address constant aligned */
      fprintf (file, "\tnop\n");
      fprintf (file, "\tmov.w	#0x1234,r4\n");

      /* Then the function pointer */

      /* To make the address constant aligned */
      fprintf (file, "\tnop\n");
      fprintf (file, "\tjmp	@0x1234:16\n");
    }
}

int
trapoline_size ()
{
  if (data_size == 32 || code_size == 32)
    return 20;
  else
    return 8;
}

static void
copy_to_tramp (which, base, src)
     int which;
     rtx base;
     rtx src;
{
  if (data_size == 32 || code_size == 32)
    {
      /* large pointers */
      int off = 2 + which * 8;
      rtx src_in_reg = copy_to_mode_reg (SImode, convert_to_mode (SImode, src, 0));
      emit_move_insn (gen_rtx (MEM, HImode, plus_constant (base, off)),
		      gen_rtx (SUBREG, HImode, src_in_reg, 0));
      emit_move_insn (gen_rtx (MEM, HImode, plus_constant (base, off + 4)),
		      gen_rtx (SUBREG, HImode, src_in_reg, 1));
    }
  else
    {
      /* Small pointers */
      int off = 2 + which * 4;
      rtx src_in_reg = copy_to_mode_reg (HImode, convert_to_mode (HImode, src, 0));
      emit_move_insn (gen_rtx (MEM, HImode, plus_constant (base, off)),
		      src_in_reg);
    }
}

void
initialize_trampoline (TRAMP, FNADDR, CXT)
     rtx TRAMP;
     rtx FNADDR;
     rtx CXT;
{
  copy_to_tramp (0, TRAMP, CXT);
  copy_to_tramp (1, TRAMP, FNADDR);
}
