/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: find.c,v 1.2 94/06/19 11:22:12 nau Exp $";

/*
 * short description:
 * - all pin and via pointers is copied to a single array to have linear access to them
 * - two index fields are build for each layer, one sorted by x1 line coordinates
 *   in an decending order, the other one by x2 line coordinates in asscending order.
 *   The are used to have fewer accesses when searching line-line connections.
 * - lists for pins and vias, for lines and for rectangles are created.
 *   Every object that has to be checked is added to its list.
 * - there's no 'speed-up' mechanism for rectangles because they are not used
 *   as often as lines and are much easier to handle
 * - the maximum distance between line and pin ... would depend on the angle
 *   between them. To speed up computation the limit is set to one third
 *   of the thickness of the objects which is close to 
 *   thickness/2/sqrt(2). The result is that connections between objects
 *   with a distance close to that limit are not recognized.
 *
 * 1. first, the pin or via underneath the given coordinates is looked up
 * 2. all line or rectangle connections to that pin (via) are searched next
 * 3. lookup of all lines (rectangles) connected to the lines (rectangles)
 *    from (2) is done next. This step is repeated until no more new connections
 *    are found.
 * 4. search for all pins (vias) connected to the lines (rectangles) from
 *    (2) and (3)
 * 5. start again with (1) for all new pins (vias) from (4)
 *
 */

/* routines to find connections between pins, vias, lines...
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "global.h"

#include "data.h"
#include "dialog.h"
#include "draw.h"
#include "find.h"
#include "memory.h"
#include "misc.h"
#include "search.h"

/* ---------------------------------------------------------------------------
 * some local types
 */
typedef struct
{
	Cardinal	*Data,			/* pointer to index data */
				Position,		/* currently used position */
				Number;			/* number of objects on stack */
} ListType, *ListTypePtr;

typedef struct					/* holds a copy of all pins and vias */
{								/* plus a pointer to the according element */
	PinTypePtr		Data;
	ElementTypePtr	Element;
} CopyType, *CopyTypePtr;

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	Cardinal		*LineSortedByX1[MAX_LAYER],	/* index to all lines */
						*LineSortedByX2[MAX_LAYER],	/* sorted by X coordinates */
						*PinsAndViasSortedByX,		/* same for pins and vias */
						SortLayer,					/* layer which is currently sorted */
						TotalPinsAndVias;			/* number of all pins and vias */
static	ListType		LineList[MAX_LAYER],		/* all lines to check are put on this stack */
						RectList[MAX_LAYER],		/* all rectangles ... */
						PinAndViaList;
static	CopyTypePtr		AllPinsAndVias;				/* all pins and vias are copied here */

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	int			CompareLineByX1(const void *, const void *);
static	int			CompareLineByX2(const void *, const void *);
static	int			ComparePinsAndViasByX(const void *, const void *);
static	Cardinal	GetLineByX1(Position, Cardinal);
static	Cardinal	GetLineByX2(Position, Cardinal);
static	Cardinal	GetPinOrViaByX(Position);
static	void		DeleteData(void);
static	void		InitFind(void);
static	void		AddPinOrViaToList(Cardinal);
static	void		AddLineToList(Cardinal, Cardinal);
static	void		AddRectToList(Cardinal, Cardinal);
static	void		GetIndexOfLines(Position, Position, Cardinal, Cardinal *, Cardinal **);
static	Cardinal	FindPinOrViaByCoordinates(Position, Position);
static	void		FindLRConnectionsToPinsOrViasOnList(void);
static	void		FindConnectionsToLROnList(void);
static	void		FindPinOrViaConnectionsToLROnList(void);
static	void		FindConnectionsToLine(LineTypePtr, Cardinal);
static	void		FindConnectionsToRect(RectTypePtr, Cardinal);
static	Boolean		IsPinOrViaOnLine(PinTypePtr, LineTypePtr);
static	Boolean		IsLineInRectangle(LineTypePtr, RectTypePtr);
static	Boolean		IsRectInRectangle(RectTypePtr, RectTypePtr);
static	Boolean		FindConnectionsToPinsOfElement(ElementTypePtr, FILE *);

/* ---------------------------------------------------------------------------
 * quicksort compare function for line coordinate X1
 * for sorting the line index field in decending order of X1
 */
static int CompareLineByX1(const void * Index1, const void * Index2)
{
	LineTypePtr	line = PCB->Layer[SortLayer].Line;
	Cardinal	*ptr1 = (Cardinal *) Index1,
				*ptr2 = (Cardinal *) Index2;

	return((int) (line[*ptr2].X1 -line[*ptr1].X1));
}

/* ---------------------------------------------------------------------------
 * quicksort compare function for line coordinate X2
 * for sorting the line index field in ascending order of X2
 */
static int CompareLineByX2(const void * Index1, const void * Index2)
{
	LineTypePtr	line = PCB->Layer[SortLayer].Line;
	Cardinal	*ptr1 = (Cardinal *) Index1,
				*ptr2 = (Cardinal *) Index2;

	return((int) (line[*ptr1].X2 -line[*ptr2].X2));
}

/* ---------------------------------------------------------------------------
 * quicksort compare function for pin (via) coordinate X
 * for sorting the pin (via) index field in ascending order of X
 */
static int ComparePinsAndViasByX(const void * Index1, const void * Index2)
{
	Cardinal	*ptr1 = (Cardinal *) Index1,
				*ptr2 = (Cardinal *) Index2;

	return((int) (AllPinsAndVias[*ptr1].Data->X -AllPinsAndVias[*ptr2].Data->X));
}

/* ---------------------------------------------------------------------------
 * returns a the maximum index which matches
 * X1(line[index]) >= X for all index <= found one
 * the field is sorted in a descending order
 */
static Cardinal GetLineByX1(Position X, Cardinal Layer)
{
	LineTypePtr	line = PCB->Layer[Layer].Line;
	int			left = 0,
				right = PCB->Layer[Layer].LineN -1,
				position;

	while (left < right)
	{
		position = (left +right) /2;
		if (line[LineSortedByX1[Layer][position]].X1 <= X)
			right = position;
		else
			left = position+1;
	}
	return((Cardinal) left);
}

/* ---------------------------------------------------------------------------
 * returns a the maximum index which matches
 * X2(line[index]) <= X for all index <= found one
 * the field is sorted in a ascending order
 */
static Cardinal GetLineByX2(Position X, Cardinal Layer)
{
	LineTypePtr	line = PCB->Layer[Layer].Line;
	int			left = 0,
				right = PCB->Layer[Layer].LineN -1,
				position;

	while (left < right)
	{
		position = (left +right) /2;
		if (line[LineSortedByX2[Layer][position]].X2 >= X)
			right = position;
		else
			left = position+1;
	}
	return((Cardinal) left);
}

/* ---------------------------------------------------------------------------
 * returns a the minimum index which matches
 * X(pin[index]) <= X for all index <= found one
 * the field is sorted in a ascending order
 */
static Cardinal GetPinOrViaByX(Position X)
{
	int	left = 0,
		right = TotalPinsAndVias -1,
		position;

	while (left < right)
	{
		position = (left +right) /2;
		if (AllPinsAndVias[PinsAndViasSortedByX[position]].Data->X >= X)
			right = position;
		else
			left = position+1;
	}
	return((Cardinal) left);
}

/* ---------------------------------------------------------------------------
 * releases all allocated memory
 */
static void DeleteData(void)
{
	Cardinal	i;

	for (i = 0; i < MAX_LAYER; i++)
	{
		MyFree((char **) &LineSortedByX1[i]);
		MyFree((char **) &LineSortedByX2[i]);
		MyFree((char **) &LineList[i].Data);
		MyFree((char **) &RectList[i].Data);
	}
	MyFree((char **) &PinsAndViasSortedByX);
	MyFree((char **) &PinAndViaList.Data);
	MyFree((char **) &AllPinsAndVias);
}

/* ---------------------------------------------------------------------------
 * allocates memory for stacks ...
 * initializes index and sorts it by X1 and X2
 */
static void InitFind(void)
{
	Cardinal		i,
					j,
					pos;
	ElementTypePtr	element;

		/* initialize line and rectangle data, create index fields and sort
		 * them, create empty lists */
	for (i = 0; i < MAX_LAYER; i++)
	{
		if (PCB->Layer[i].LineN)
		{
			SortLayer = i;
			LineSortedByX1[i] = (Cardinal *) MyCalloc(PCB->Layer[i].LineN,
				sizeof(Cardinal), "InitFind()");
			LineSortedByX2[i] = (Cardinal *) MyCalloc(PCB->Layer[i].LineN,
				sizeof(Cardinal), "InitFind()");
			for (j = 0; j < PCB->Layer[i].LineN; j++)
				LineSortedByX1[i][j] = LineSortedByX2[i][j] = j;
			qsort(LineSortedByX1[i], PCB->Layer[i].LineN,
				sizeof(Cardinal), CompareLineByX1);
			qsort(LineSortedByX2[i], PCB->Layer[i].LineN,
				sizeof(Cardinal), CompareLineByX2);
			LineList[i].Data = (Cardinal *) MyCalloc(PCB->Layer[i].LineN,
				sizeof(Cardinal), "InitFind()");
		}

		if (PCB->Layer[i].RectN)
			RectList[i].Data = (Cardinal *) MyCalloc(PCB->Layer[i].RectN,
				sizeof(Cardinal), "InitFind()");

		LineList[i].Position = 0;
		LineList[i].Number = 0;
		RectList[i].Position = 0;
		RectList[i].Number = 0;
	}

		/* pin and via data, start with counting their total number,
		 * then allocate memory and copy the data to a tmp field
		 * set number of the according element in tmp field too
		 */
	TotalPinsAndVias = PCB->ViaN;
	for (i = PCB->ElementN, element = PCB->Element; i; i--, element++)
		TotalPinsAndVias += element->PinN;
	AllPinsAndVias = (CopyTypePtr) MyCalloc(TotalPinsAndVias,
		sizeof(CopyType), "InitFind()");
	PinAndViaList.Data = (Cardinal *) MyCalloc(TotalPinsAndVias,
		sizeof(Cardinal), "InitFind()");
	PinAndViaList.Position = 0;
	PinAndViaList.Number = 0;

		/* copy via data, field is initialized with NULL */
	for (pos = 0; pos < PCB->ViaN; pos++)
		AllPinsAndVias[pos].Data = PCB->Via +pos;
	for (i = 0, element = PCB->Element; i < PCB->ElementN; i++, element++)
		for (j = 0; j < element->PinN; j++, pos++)
		{
			AllPinsAndVias[pos].Data = element->Pin +j;
			AllPinsAndVias[pos].Element = element;
		}

		/* create index field for pins and vias sorted by X in ascending order */
	PinsAndViasSortedByX = (Cardinal *) MyCalloc(TotalPinsAndVias,
		sizeof(Cardinal), "InitFind()");
	for (i = 0; i < TotalPinsAndVias; i++)
		PinsAndViasSortedByX[i] = i;
	qsort(PinsAndViasSortedByX, TotalPinsAndVias, sizeof(Cardinal), ComparePinsAndViasByX);

		/* set new drawing color */
	XSetForeground(Dpy, Output.fgGC, PCB->ConnectedColor);
}

/* ---------------------------------------------------------------------------
 * resets all used flags of pins and vias
 */
void ResetFoundPinsAndVias(void)
{
	Cardinal		i, j;
	ElementTypePtr	element;
	PinTypePtr		ptr;

	for (i = PCB->ViaN, ptr = PCB->Via; i; i--, ptr++)
		CLEAR_FLAG(FOUNDFLAG, ptr);
	for (element = PCB->Element, i = PCB->ElementN; i; i--, element++)
		for (j = element->PinN, ptr = element->Pin; j; j--, ptr++)
			CLEAR_FLAG(FOUNDFLAG, ptr);
}

/* ---------------------------------------------------------------------------
 * resets all used flags of lines and rectangles
 */
void ResetFoundLinesAndRectangles(void)
{
	Cardinal		i, j;
	LineTypePtr		line;
	RectTypePtr		rect;

	for (i = 0; i < MAX_LAYER; i++)
	{
		for (line = PCB->Layer[i].Line, j = PCB->Layer[i].LineN; j; j--, line++)
			CLEAR_FLAG(FOUNDFLAG, line);
		for (rect = PCB->Layer[i].Rect, j = PCB->Layer[i].RectN; j; j--, rect++)
			CLEAR_FLAG(FOUNDFLAG, rect);
	}
}

/* ---------------------------------------------------------------------------
 * adds pin or via to search list and draws it
 */
static void AddPinOrViaToList(Cardinal Index)
{
	PinTypePtr	ptr = AllPinsAndVias[Index].Data;

	SET_FLAG(FOUNDFLAG, ptr);
	PinAndViaList.Data[PinAndViaList.Number++] = Index;
	if (TEST_FLAG(PINFLAG, ptr))
	{
		if (PCB->PinOn)
			DrawPinOrVia(ptr);
	}
	else
	{
		if (PCB->ViaOn)
			DrawPinOrVia(ptr);
	}
}

/* ---------------------------------------------------------------------------
 * adds line to search list and draws it
 */
static void AddLineToList(Cardinal Index, Cardinal Layer)
{
	LineTypePtr	line = &PCB->Layer[Layer].Line[Index];

	SET_FLAG(FOUNDFLAG, line);
	LineList[Layer].Data[LineList[Layer].Number++] = Index;
	if (PCB->Layer[Layer].On)
		DrawLine(line);
}

/* ---------------------------------------------------------------------------
 * adds rectangle to search list and draws it
 */
static void AddRectToList(Cardinal Index, Cardinal Layer)
{
	RectTypePtr	rect = &PCB->Layer[Layer].Rect[Index];

	SET_FLAG(FOUNDFLAG, rect);
	RectList[Layer].Data[RectList[Layer].Number++] = Index;
	if (PCB->Layer[Layer].On)
		DrawRect(rect);
}

/* ---------------------------------------------------------------------------
 * returns a pointer into an sorted index field and it's start index
 * for a range of x coordinates which are used as a upper an lower
 * limit for lines
 */
static void GetIndexOfLines(Position Xlow, Position Xhigh, Cardinal Layer, Cardinal *Index, Cardinal **List)
{
	Cardinal	index1,
				index2;

		/* get index of the first line that may match the coordinates
		 * see GetLineByX1(), GetLineByX2()
		 * take the field with less entries to speed up searching
		 */
	index1 = GetLineByX1(Xhigh, Layer);
	index2 = GetLineByX2(Xlow, Layer);
	if (index1 > index2)
	{
		*Index = index1;
		*List = &LineSortedByX1[Layer][index1];
	}
	else
	{
		*Index = index2;
		*List = &LineSortedByX2[Layer][index2];
	}
}

/* ---------------------------------------------------------------------------
 * finds a pin or via by it's coordinates in the sorted list
 * It's index number or -1 is returned.
 */
static Cardinal FindPinOrViaByCoordinates(Position X, Position Y)
{
	Cardinal	i, limit;
	PinTypePtr	ptr;

	i = GetPinOrViaByX(X -MAX_PINORVIASIZE);
	limit = GetPinOrViaByX(X +MAX_PINORVIASIZE +1);
	while (i <= limit)
	{
		ptr = AllPinsAndVias[PinsAndViasSortedByX[i]].Data;
		if (abs(ptr->X - X) <= ptr->Thickness/4 &&
			abs(ptr->Y - Y) <= ptr->Thickness/4)
			return(PinsAndViasSortedByX[i]);
		i++;
	}
	return(-1);
}

/* ---------------------------------------------------------------------------
 * finds all connections from the pin (via) and the given coordinates
 * returns the number of found pins and vias
 * the 'FOUNDFLAG' is set for all objects found
 */
Cardinal FindConnection(Position X, Position Y)
{
	ElementTypePtr	element;
	PinTypePtr		ptr;
	Cardinal		index;

		/* check if there are any pins or vias at that position */
	if (SearchVia(X, Y) == NULL && !SearchPin(&element, &ptr, X, Y))
		return(0);

	InitFind();

		/* now find the pin (via) again in the tmp field and add it's index
		 * to the list. A 'break' should never happen
		 */
	if ((index = FindPinOrViaByCoordinates(X, Y)) == -1)
	{
		DeleteData();
		return(0);
	}
	AddPinOrViaToList(index);

		/* now loop till we reach the end of the pin and via list */
	while (PinAndViaList.Position < PinAndViaList.Number)
	{
		FindLRConnectionsToPinsOrViasOnList();
		FindConnectionsToLROnList();
	}

		/* we are done */
	DeleteData();
	SetChangedFlag(True);
	if (Settings.RingBellWhenFinished)
		XBell(Dpy, 100);
	return(PinAndViaList.Number);
}

/* ---------------------------------------------------------------------------
 * checks if a pin or via is connected to lines or rectangles
 * if they are, the lines (rectangles) are added to the list and
 * the used flag is set
 */
static void FindLRConnectionsToPinsOrViasOnList(void)
{
	LineTypePtr	line;
	RectTypePtr	rect;
	PinTypePtr	ptr;
	Cardinal	i,
				*sortlist,
				layer;
	Dimension	distance;

		/* loop over all pins (vias) currently on list */
	while(PinAndViaList.Position < PinAndViaList.Number)
	{
			/* get pointer to pin data */
		ptr = AllPinsAndVias[PinAndViaList.Data[PinAndViaList.Position]].Data;

		for (layer = 0; layer < MAX_LAYER; layer++)
		{
				/* get the low and high index of lines which may have
				 * a connection to the pin or via
				 * ### line->X1 <= line->X2 ###
				 */
			distance = (MAX_LINESIZE + ptr->Thickness) /3;

				/* get index of the first line that may
				 * match the coordinates
				 */
			GetIndexOfLines(ptr->X -distance, ptr->X +distance,
				layer, &i, &sortlist);

			for (; i< PCB->Layer[layer].LineN; i++, sortlist++)
			{
				line = PCB->Layer[layer].Line + (*sortlist);
				if (!TEST_FLAG(FOUNDFLAG, line) && IsPinOrViaOnLine(ptr, line))
				{
					AddLineToList(*sortlist, layer);
				}
			}

				/* check rectangles */
			for (i = 0, rect = PCB->Layer[layer].Rect; i < PCB->Layer[layer].RectN; i++, rect++)
				if (!TEST_FLAG(FOUNDFLAG, rect) &&
					VIA_OR_PIN_IN_RECTANGLE(ptr,
					rect->X -ptr->Thickness/3,
					rect->Y -ptr->Thickness/3,
				rect->X +rect->Width +ptr->Thickness/3,
				rect->Y +rect->Height +ptr->Thickness/3))
			{
				AddRectToList(i, layer);
			}
		}
		PinAndViaList.Position++;
	}
}

/* ---------------------------------------------------------------------------
 * find all connections between the pin or via at the current
 * list position and lines or rectangles
 */
static void FindConnectionsToLROnList(void)
{
	Cardinal	i;
	ListTypePtr	list;
	Cardinal	linelistposition[MAX_LAYER],
				rectlistposition[MAX_LAYER];

		/* save current list positions */
	for (i = 0; i< MAX_LAYER; i++)
	{
		linelistposition[i] = LineList[i].Position;
		rectlistposition[i] = RectList[i].Position;
	}
	
		/* loop over all layers */
	for (i = 0; i < MAX_LAYER; i++)
	{
			/* loop over all new lines (rectangles) on the list */
		do
		{
				/* try all new lines */
			for (list = &LineList[i]; list->Position < list->Number; list->Position++)
				FindConnectionsToLine(
					PCB->Layer[i].Line +list->Data[list->Position], i);

				/* try all new rectangle */
			for (list = &RectList[i]; list->Position < list->Number; list->Position++)
				FindConnectionsToRect(
					PCB->Layer[i].Rect +list->Data[list->Position], i);

			/* check if both lists are handled, the second for-loop may have
			 * changed the line list
			 */
		} while (LineList[i].Position < LineList[i].Number ||
				RectList[i].Position < RectList[i].Number);
	}

		/* restore current list positions for the next subroutine */
	for (i = 0; i< MAX_LAYER; i++)
	{
		LineList[i].Position = linelistposition[i];
		RectList[i].Position = rectlistposition[i];
	}
	
		/* search new pin or via connections to the lines and
		 * rectangles just found
		 */
	FindPinOrViaConnectionsToLROnList();
}

/* ---------------------------------------------------------------------------
 * searches for new pins or vias that are connected to NEW lines or NEW
 * rectangles on the list.
 */
static void FindPinOrViaConnectionsToLROnList(void)
{
	LineTypePtr	line;
	RectTypePtr	rect;
	PinTypePtr	ptr;
	Cardinal	i,
				limit,
				layer;
	Dimension	distance;

		/* loop over all layers */
	for (layer = 0; layer < MAX_LAYER; layer++)
	{
			/* check all lines */
		while(LineList[layer].Position < LineList[layer].Number)
		{
			line = PCB->Layer[layer].Line +LineList[layer].Data[LineList[layer].Position];

				/* get the positions in sorted field to speed up searching
				 * ### line->X1 <= line->X2 ###
				 * the '+1' in the second call of GetPinOrViaByX()
				 * makes sure that 'limit' is realy the position outside the
				 * range
				 */
			distance = (MAX_PINORVIASIZE +line->Thickness) /3;
			i = GetPinOrViaByX(line->X1 -distance);
			limit = GetPinOrViaByX(line->X2 +distance +1);
			while (i <= limit)
			{
				ptr = AllPinsAndVias[PinsAndViasSortedByX[i]].Data;
				if (!TEST_FLAG(FOUNDFLAG, ptr) && IsPinOrViaOnLine(ptr, line))
				{
					AddPinOrViaToList(PinsAndViasSortedByX[i]);
				}
				i++;
			}
			LineList[layer].Position++;
		}

			/* now all rectangles */
		while(RectList[layer].Position < RectList[layer].Number)
		{
			rect = PCB->Layer[layer].Rect +RectList[layer].Data[RectList[layer].Position];

				/* get the positions in sorted field to speed up searching */
			distance = MAX_PINORVIASIZE /3;
			i = GetPinOrViaByX(rect->X -distance);
			limit = GetPinOrViaByX(rect->X +rect->Width +distance +1);
			while (i <= limit)
			{
				ptr = AllPinsAndVias[PinsAndViasSortedByX[i]].Data;
				if (!TEST_FLAG(FOUNDFLAG, ptr) && VIA_OR_PIN_IN_RECTANGLE(ptr,
						rect->X -ptr->Thickness/3,
						rect->Y -ptr->Thickness/3,
						rect->X +rect->Width +ptr->Thickness/3,
						rect->Y +rect->Height +ptr->Thickness/3))
				{
					AddPinOrViaToList(PinsAndViasSortedByX[i]);
				}
				i++;
			}
			RectList[layer].Position++;
		}
	}
}

/* ---------------------------------------------------------------------------
 * searches all lines an rectangles that are connected to the given line
 * on the given layer. All found connections are added to the list
 *
 * the notation that is used is:
 * Xij means Xj at line i
 */
static void FindConnectionsToLine(LineTypePtr Line, Cardinal Layer)
{
	LineTypePtr		line1,
					line2;
	RectTypePtr		rect;
	Cardinal		*sortlist;
	Cardinal		index;
	Dimension		distance;
	Boolean			found;
	Position		x, y;

		/* the maximum distance which is possible */
	distance = (Line->Thickness +MAX_LINESIZE) /3;

		/* get index of the first line that may match the coordinates */
	GetIndexOfLines(Line->X1 -distance, Line->X2 +distance, Layer, &index, &sortlist);

		/* no loop till the end of the data is reached
		 * remember, sortlist is an index into the PCBs lines list
		 * DONT RETRY LINES THAT HAVE BEEN FOUND
		 */
	for(found = False; index < PCB->Layer[Layer].LineN; index++, sortlist++)
	{
		line1 = PCB->Layer[Layer].Line + (*sortlist);
		if (TEST_FLAG(FOUNDFLAG, line1))
			continue;

			/* set line1 and line2 so that
			 * Direction(line2) >= Direction(line1)
			 */
		if (line1->Direction >= Line->Direction)
		{
			line2 = line1;
			line1 = Line;
		}
		else
			line2 = Line;

			/* set maximum distance */
		distance = (line1->Thickness +line2->Thickness)/3;
		
			/* check if lines might have a connection */
		if (line2->X1 > line1->X2 +distance ||
			line2->X2 < line1->X1 -distance)
			continue;

		switch(line1->Direction)
		{
			case 0:
				switch(line2->Direction)
				{
					case 0:
							/* both lines are parallel and the x-range has
							 * already been checked -> only compare
							 * y coordinates */
						found = (line2->Y1 <= line1->Y2 +distance &&
							line2->Y2 >= line1->Y1 -distance);
						break;

					case 1:
							/* calculate y which is on line2 at x = x11
							 * y = y21 -(x21 -x11)
							 * and check it's position against line1
							 * x range check is already done
							 */
						y = line2->Y1 -line2->X1 +line1->X1;
						found = (y <= line1->Y2 +distance &&
							y >= line1->Y1 -distance);
						break;

					case 2:
							/* only Y21 (equal to Y22) has to be checked */
						found = (line2->Y1 <= line1->Y2 +distance &&
							line2->Y1 >= line1->Y1 -distance);
						break;

					case 3:
							/* calculate y which is on line2 at x = x11
							 * y = y21 +(x21 -x11)
							 * and check it's position against line1
							 */
						y = line2->Y1 +line2->X1 -line1->X1;
						found = (y <= line1->Y2 +distance &&
							y >= line1->Y1 -distance);
						break;
				 }
				 break;

			case 1:
				switch(line2->Direction)
				{
					case 1:
						/* the x range has already be checked so it's only
						 * necessary to take care of distance between the
						 * two lines. 
						 * The HESSE-FORM is
						 * (x-x1 -(y-y1))/sqrt(2) = 0
						 * and therefor the distance is
						 * d = abs(x11-x21-y11+y21)/sqrt(2)
						 */
					found = (line1->Y2 >= line2->Y1 -distance &&
						line1->Y1 <= line2->Y2 +distance &&
						abs(0.707*(line1->X1 -line2->X1 -line1->Y1 +line2->Y1)) <= (float) distance);
						break;

					case 2:
							/* calculate x which is on line1 at y = y21
							 * x = x11 +(y21 -y11)
							 * and check it's position against line2
							 */
						x = line1->X1 +line2->Y1 -line1->Y1;
						found = (line1->Y1 <= line2->Y1 +distance &&
							line1->Y2 >= line2->Y1 -distance &&
							x <= line2->X2 +distance &&
							x >= line2->X1 -distance);
						break;

					case 3:
							/* calculate a point (x,y) on both lines
							 * and check x against the x coordinates
							 * of all four points of the two lines
							 * check of the x coordinate is good enough
							 * because the point lays on both lines which
							 * causes y to match if x matches the conditions
							 */
						x = (line2->Y1 +line2->X1 -line1->Y1 +line1->X1) /2;
						found = (x <= line1->X2 +line1->Thickness/3 &&
							x >= line1->X1 -line1->Thickness/3 &&
							x <= line2->X2 +line2->Thickness/3 &&
							x >= line2->X1 -line2->Thickness/3);
						break;
				 }
				 break;

			case 2 :
				switch(line2->Direction)
				{
					case 2:
							/* both lines are parallel so only the distance
							 * between the lines has to be less then
							 * the maximum one
							 */
						found = (abs(line2->Y1 -line1->Y1) <= distance);
						break;

					case 3:
							/* calculate x which is on line2 at y = y11
							 * x = x21 +(y21 -y11)
							 * and check it's position against line1
							 */
						x = line2->X1 +line2->Y1 -line1->Y1;
						found = (line2->Y1 >= line1->Y1 -distance &&
							line2->Y2 <= line1->Y1 +distance &&
							x <= line1->X2 +distance &&
							x >= line1->X1 -distance);
						break;
				}
				break;

			case 3:
					/* the x range has already be checked so it's only
					 * necessary to take care of distance between the
					 * two lines. 
					 * The HESSE-FORM is
					 * (x-x1 +(y-y1))/sqrt(2) = 0
					 * and therefor the distance is
					 * d = abs(x11-x21+y11-y21)/sqrt(2)
					 */
				found = (line1->Y1 >= line2->Y2 -distance &&
					line1->Y2 <= line2->Y1 +distance &&
					abs(0.707*(line1->X1 -line2->X1 +line1->Y1 -line2->Y1)) <= (float) distance);
				break;
		}

			/* set used flag and add the line to the list */
		if (found)
		{
			AddLineToList(*sortlist, Layer);
		}
	}

		/* now check all rectangles */
	for(index = 0, rect = PCB->Layer[Layer].Rect; index < PCB->Layer[Layer].RectN; index++, rect++)
		if (!TEST_FLAG(FOUNDFLAG, rect) &&
			IsLineInRectangle(Line, rect))
		{
			AddRectToList(index, Layer);
		}
}

/* ---------------------------------------------------------------------------
 * searches all lines and rectangles that are connected to the given rectangle
 * on the given layer. All found connections are added to the list
 */
static void FindConnectionsToRect(RectTypePtr Rect, Cardinal Layer)
{
	RectTypePtr	rect;
	LineTypePtr	line;
	Cardinal	i,
				*sortlist;

		/* check all rectangles */
	for(i = 0, rect = PCB->Layer[Layer].Rect; i < PCB->Layer[Layer].RectN; i++, rect++)
		if (!TEST_FLAG(FOUNDFLAG, rect) &&
			IsRectInRectangle(rect, Rect))
		{
			AddRectToList(i, Layer);
		}

		/* and now check all lines, first reduce the number of lines by
		 * evaluating the coordinates from the sorted fields
		 */
	GetIndexOfLines(Rect->X -MAX_LINESIZE/3,
		Rect->X +Rect->Width +MAX_LINESIZE/3,
		Layer, &i, &sortlist);

		/* now check all lines that match the condition */
	for(; i< PCB->Layer[Layer].LineN; i++, sortlist++)
	{
		line = PCB->Layer[Layer].Line + (*sortlist);
		if (!TEST_FLAG(FOUNDFLAG, line) && IsLineInRectangle(line, Rect))
		{
			AddLineToList(*sortlist, Layer);
		}
	}
}

/* ---------------------------------------------------------------------------
 * checks if pin is in range of a line
 */
static Boolean IsPinOrViaOnLine(PinTypePtr Ptr, LineTypePtr Line)
{
	Dimension	distance = (Ptr->Thickness +Line->Thickness)/3;

		/* check if pin might be in the same area as the line a
		 * looking at the x coordinates
		 */
	if (Line->X1 -distance <= Ptr->X &&
		Line->X2 +distance >= Ptr->X)
	{
			/* using a maximum distance of one third of the diameter
			 * gives us a good compromise between too may or two less matches
			 */
		distance = (Ptr->Thickness +Line->Thickness)/3;

			/* the compare routines make use of the fact that there
			 * are only 4 different line directions
			 */
		switch(Line->Direction)
		{
			case 0:		/* line straight up */
			case 2:		/* or to the right */
				return(Line->Y1 -distance <= Ptr->Y &&
					Line->Y2 +distance >= Ptr->Y);

			case 1:	
				/* The HESSE-FORM of the line is
				 * (x-x1 -(y-y1))/sqrt(2) = 0
				 * and therefor the distance is
				 * d = abs(x-x1-y+y1)/sqrt(2)
				 * using one third of the diameter is useful instead
				 * of a floating point comparison
				 */
				return(Line->Y1 -distance <= Ptr->Y &&
					Line->Y2 +distance >= Ptr->Y &&
					abs(Ptr->X -Line->X1 -Ptr->Y +Line->Y1) <= distance);

			case 3:
				/* The HESSE-FORM of the line is
				 * (x-x1 +(y-y1))/sqrt(2) = 0
				 * and therefor the distance is
				 * d = abs(x-x1+y-y1)/sqrt(2)
				 * using one third of the diameter is useful instead
				 * of a floating point comparison
				 */
				return(Line->Y1 +distance >= Ptr->Y &&
					Line->Y2 -distance <= Ptr->Y &&
					abs(Ptr->X -Line->X1 +Ptr->Y -Line->Y1) <= distance);
		}
	}
	return(False);
}

/* ---------------------------------------------------------------------------
 * check if line has a connection to the rectangle
 */
static Boolean IsLineInRectangle(LineTypePtr Line, RectTypePtr Rect)
{
	Position	rx1, ry1,
				rx2, ry2,
				x1, y1,
				x2, y2;

		/* initialize some useful identifiers */
	rx1 = Rect->X -Line->Thickness/3;
	ry1 = Rect->Y -Line->Thickness/3;
	rx2 = Rect->X +Rect->Width +Line->Thickness/3;
	ry2 = Rect->Y +Rect->Height +Line->Thickness/3;
	x1 = Line->X1;
	y1 = Line->Y1;
	x2 = Line->X2;
	y2 = Line->Y2;
	switch(Line->Direction)
	{
		case 0:
			return(x1 >= rx1 && x1 <= rx2 && y2 >= ry1 && y1 <= ry2);

		case 1:
				/* clip line with rectangle */
			if (x1 < rx1)
			{
				y1 += (rx1 -x1);
				x1 = rx1;
			}
			if (x2 > rx2)
			{
				y2 -= (x2 -rx2);
				x2 = rx2;
			}
			if (y1 < ry1)
			{
				x1 += (ry1 -y1);
				y1 = ry1;
			}
			if (y2 > ry2)
			{
				x2 -= (y2 -ry2);
				y2 = ry2;
			}
			return(x1 <= x2 && y1 <= y2);

		case 2:
			return(y1 >= ry1 && y1 <= ry2 && x2 >= rx1 && x1 <= rx2);

		case 3:
				/* clip line with rectangle */
			if (x1 < rx1)
			{
				y1 -= (rx1 -x1);
				x1 = rx1;
			}
			if (x2 > rx2)
			{
				y2 += (x2 -rx2);
				x2 = rx2;
			}
			if (y1 > ry2)
			{
				x1 += (y1 -ry2);
				y1 = ry2;
			}
			if (y2 < ry1)
			{
				x2 -= (ry1 -y2);
				y2 = ry1;
			}
			return(x1 <= x2 && y2 <= y1);
	}

		/* this should never happen */
	return(False);
}

/* ---------------------------------------------------------------------------
 * check if rectangle has a connection to the rectangle
 */
static Boolean IsRectInRectangle(RectTypePtr Rect1, RectTypePtr Rect2)
{
	Position	x2_1 = Rect1->X +Rect1->Width,
				y2_1 = Rect1->Y +Rect1->Height,
				x2_2 = Rect2->X +Rect2->Width,
				y2_2 = Rect2->Y +Rect2->Height;

	return(x2_1 >= Rect2->X && Rect1->X <= x2_2 &&
		y2_1 >= Rect2->Y && Rect1->Y <= y2_2);
}

/* ---------------------------------------------------------------------------
 * prints all found connections of an element to FP
 */
static void PrintElementConnenctions(PinTypePtr Pin, FILE *FP)
{
	Cardinal		i,
					indent,
					linelength;
	CopyTypePtr		ptr;
	char			*ename,
					*pname;

	fprintf(FP, "  %s: ", UNKNOWN(Pin->Name));
	linelength = indent = strlen(UNKNOWN(Pin->Name)) +4;
	for (i = 1; i < PinAndViaList.Number; i++)
	{
		ptr = &AllPinsAndVias[PinAndViaList.Data[i]];
		pname = EMPTY(ptr->Data->Name);

			/* get the elements name */
		if (ptr->Element)
		{
			ename = UNKNOWN(TEST_FLAG(CANONICALFLAG, PCB) ?
				ptr->Element->CanonicalName : ptr->Element->NameOnPCB);
		}
		else
			ename = "via";

			/* new line ? */
		if ((linelength += (4 +strlen(pname) +strlen(ename))) >= Settings.CharPerLine)
		{
			linelength = indent +4 +strlen(pname) +strlen(ename);
			fprintf(FP, "\n%*s%s(%s), ", indent, "", pname, ename);
		}
		else
			fprintf(FP, "%s(%s), ", pname, ename);
	}

		/* there might be only one pin in list --> no connections */
	if (i == 1)
		fputs("no connection", FP);
	fputc('\n', FP);
}

/* ---------------------------------------------------------------------------
 * finds all connections to the pins of the passed element.
 * The result is written to filedescriptor 'FP'.
 * Returns True if operation was aborted
 */
static Boolean FindConnectionsToPinsOfElement(ElementTypePtr Element, FILE *FP)
{
	PinTypePtr	ptr = Element->Pin;
	Cardinal	n,
				layer,
				index;

	fprintf(FP, "%s(%s)\n", UNKNOWN(Element->CanonicalName), 
		UNKNOWN(Element->NameOnPCB));

		/* check all pins in element */
	for (n = Element->PinN; n; n--, ptr++)
	{
			/* add pin as first one in list */
		if ((index = FindPinOrViaByCoordinates(ptr->X, ptr->Y)) == -1)
			continue;

			/* pin might has bee checked before */
		if (TEST_FLAG(FOUNDFLAG, AllPinsAndVias[index].Data))
		{
			fprintf(FP, "  %s: has been checked before\n",
				UNKNOWN(ptr->Name));
			continue;
		}
		AddPinOrViaToList(index);

			/* now loop till we reach the end of the pin and via list */
		while (PinAndViaList.Position < PinAndViaList.Number)
		{
			FindLRConnectionsToPinsOrViasOnList();
			FindConnectionsToLROnList();
		}

			/* printout all found connections */
		PrintElementConnenctions(ptr, FP);

			/* reset found lines and rectangles for the next pin */
		for (layer = 0; layer < MAX_LAYER; layer++)
		{
			LineList[layer].Position = LineList[layer].Number = 0;
			RectList[layer].Position = RectList[layer].Number = 0;
		}
/*		
		ResetFoundLinesAndRectangles();
*/
		PinAndViaList.Number = PinAndViaList.Position = 0;
		if (CheckAbort())
		{
			fputs("\n\nABORTED\n", FP);
			return(True);;
		}
	}
	return(False);
}

/* ---------------------------------------------------------------------------
 * find all connections to pins withing one element
 */
void FindElementConnections(ElementTypePtr Element, FILE *FP)
{
		/* reset all currently marked connections */
	InitFind();
	SetupAbortDialog(Output.Toplevel, "press button to abort 'scanning of connections'");
	if (Settings.ResetAfterElement)
	{
		ResetFoundPinsAndVias();
		ResetFoundLinesAndRectangles();
		RedrawOutput();
	}
	FindConnectionsToPinsOfElement(Element, FP);
	if (Settings.ResetAfterElement)
	{
		ResetFoundPinsAndVias();
		ResetFoundLinesAndRectangles();
		SetChangedFlag(True);
	}
	EndAbortDialog();
	RedrawOutput();
	DeleteData();
	if (Settings.RingBellWhenFinished)
		XBell(Dpy, 100);
}

/* ---------------------------------------------------------------------------
 * find all connections to pins withing all element
 */
void FindConnectionsToAllElements(FILE *FP)
{
	ElementTypePtr	element = PCB->Element;
	Cardinal		n = PCB->ElementN,
					i;

		/* reset all currently marked connections */
	InitFind();
	if (Settings.ResetAfterElement)
	{
		ResetFoundPinsAndVias();
		ResetFoundLinesAndRectangles();
		RedrawOutput();
	}
	SetupAbortDialog(Output.Toplevel, "press button to abort 'scanning of connections'");
	for (; n; n--, element++)
	{
		if (FindConnectionsToPinsOfElement(element, FP))
			break;
		for (i = Settings.CharPerLine; i; i--)
			fputc('=', FP);
		fputc('\n', FP);
		if (Settings.ResetAfterElement && n != 1)
		{
			ResetFoundPinsAndVias();
			ResetFoundLinesAndRectangles();
			RedrawOutput();
		}
	}
	if (Settings.ResetAfterElement)
	{
		ResetFoundPinsAndVias();
		ResetFoundLinesAndRectangles();
		SetChangedFlag(True);
	}
	EndAbortDialog();
	RedrawOutput();
	DeleteData();
	if (Settings.RingBellWhenFinished)
		XBell(Dpy, 100);
}
