/*
 * Programm XBLAST V1.2.11 or higher
 * (C) by Oliver Vogel (e-mail: vogel@ikp.uni-koeln.de)
 * March 16th 1996
 * started August 1993
 *
 * File: main.c 
 * gameplay and main programm
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public Licences as published
 * by the Free Software Foundation; either version 2; or (at your option)
 * any later version
 *
 * This program is distributed in the hope that it will entertaining,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILTY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Publis License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define _MAIN_C

#define OLD_SHRINK 0

#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include "include.h"
#include "mytypes.h"
#include "const.h"

#include "block.h"
#include "patchlev.h"
#include "intro.h"
#include "sprite.h"
#include "score.h"
#include "graphics.h"
#include "func.h"
#include "maze.h"
#include "data.h"
#include "event.h"
#include "shrink.h"
#include "status.h"
#include "main.h"
#include "setup.h"

/* 
 * some constants 
 */

#define FRAME_TIME 50000 /* 56000 */
#define BOMB_STEP 2
#define TIME_STEP 48 /* 48 */
#define GAME_TIME (60*TIME_STEP + DEAD_TIME)
#define MIN_RANGE 2
#define EXTRA_RETRY 4

/* 
 * local function prototypes 
 */
#ifdef __STDC__
static void randomize_levels (int *feld);
static int walk_stop (BMPlayer *ps);
static int walk_up (BMPlayer *ps, int flag, int mazex, int mazey);
static int walk_left (BMPlayer *ps, int flag, int mazex, int mazey);
static int walk_down (BMPlayer *ps, int flag, int mazex, int mazey);
static int walk_right (BMPlayer *ps, int flag, int mazex, int mazey);
static int teleport_player (BMPlayer *ps, int mazex, int mazey);
static void do_walk (int player);
static void infect_other_players (void);
static void have_a_gloat (int player);
static void revive_player (int player);
static void do_stunned (int player);
static void do_die (int player);
static void level_end (void);
static void eval_setup (char *prog_name);
#else
static void randomize_levels ();
static int walk_stop ();
static int walk_up ();
static int walk_left ();
static int walk_down ();
static int walk_right ();
static int teleport_player ();
static void do_walk ();
static void infect_other_players ();
static void have_a_gloat ();
static void revive_player ();
static void do_stunned ();
static void do_die ();
static void level_end ();
static void eval_setup ();
#endif

/* 
 * global variables 
 */

/* Player Stats */
BMPlayer player_stat[MAX_PLAYER+1];
PlayerStrings p_string[MAX_PLAYER];
static XBConfig Config;
static XBSettings Setup;

/*
 * local variables
 */

static int num_disp;
static int pause_status = -1;
static int active_player;
static int last_player;
static int level;
static int num_victories = 0;
static int min_range, min_bombs;
static void (*sound_function)();

/*
 * global variables 
 */
int cur_bomb_time;
int defaultBMT;
int buttonBMT;
int game_time;
int num_player;
int game_mode;
int FrameTime;


/* 
 *local function bm_fatal 
 */
#ifdef __STDC__
void 
bm_fatal (char *text,
	  char *prog)
#else
void 
bm_fatal (text,prog)
     char *text,*prog;
#endif
{
  fprintf(stderr,"ERROR : %s.\n",text);
  fprintf(stderr,"Type %s -? for help\n",prog); 
  exit(2);
}






/* 
 * local function randomize_levels 
 */
#ifdef __STDC__
static void 
randomize_levels (int *feld)
#else
static void 
randomize_levels (feld)
     int *feld;
#endif
{
  int i, j, k;
  int hilf;
  int old_end;
  
  old_end = feld[LEVEL_MAX-1];

  for (k=0; k<4; k++) {
    for (i=0; i < LEVEL_MAX; i++) {
      j = (rand()>>4) % LEVEL_MAX;
      hilf = feld[i];
      feld[i] = feld[j];
      feld[j] = hilf;
    }
  }

  if (old_end == feld[0]) {
    j = 1+(rand()>>4) % (LEVEL_MAX-1);
    hilf = feld[0];
    feld[0] = feld[j];
    feld[j] = hilf;
  }
}



/* 
 * global function drop_bomb 
 */
#ifdef __STDC__
void 
drop_bomb (int player,
	   BMPlayer *ps,
	   int type)
#else
void 
drop_bomb(player, ps, type)
     int player;
     BMPlayer *ps;
     int type;
#endif
{
  if (ps->bombs !=0 && ps->illness!=IllEmpty
      && (type == BMTnormal || ps->special_bombs > 0)) {
    if (ps->lives >0) {
      new_explosion(player, (ps->x + BLOCK_WIDTH/2)/BLOCK_WIDTH,
		    (ps->y + BLOCK_HEIGHT + BLOCK_HEIGHT/2)
		    /BLOCK_HEIGHT,
		    (ps->illness == IllMini) ? 1 : ps->range,
		    (ps->remote_control > 0),
		    (ps->illness == IllMalfunction),
		    type, 0, GoDefault );
      ps->bombs --;
      if (type != BMTnormal) {
	ps->special_bombs --;
      }
    }
  }
}



/* 
 * local fucntion walk_stop 
 */
#ifdef __STDC__
static int 
walk_stop (BMPlayer *ps)
#else
static int 
walk_stop (ps)
     BMPlayer *ps;
#endif
{
  if (ps->illness != IllReverse) {
    switch(ps->d_look) {
    case GoDown:
      ps->anime = 0;
      break;
    case GoUp:
      ps->anime = 3;
      break;
    case GoLeft:
      ps->anime = 9;
      break;
    case GoRight:
      ps->anime = 6;
      break;
    }
  }else {
    switch(ps->d_look) {
    case GoDown:
      ps->anime = 3;
      break;
    case GoUp:
      ps->anime = 0;
      break;
    case GoLeft:
      ps->anime = 6;
      break;
    case GoRight:
      ps->anime = 9;
      break;
    }
  }
  return FALSE;
}



/* 
 * local function walk_up 
 */
#ifdef __STDC__
static int 
walk_up (BMPlayer *ps,
	 int flag,
	 int mazex, 
	 int mazey)
#else
static int 
walk_up (ps, flag, mazex, mazey)
     BMPlayer *ps;
     int flag;
     int mazex, mazey;
#endif
{
  int look_to_wall = FALSE;
  
  if ( !(flag && check_maze(mazex, mazey-1))) {
    ps->y -= STEP_VERT;
  } else {
    ps->d_ist = GoStop;
    look_to_wall = TRUE;
  }
  
  if (ps->illness != IllReverse) {
    switch(ps->y % (STEP_VERT*4))  {
    case 0:
    case (2*STEP_VERT):
      ps->anime = 3;
      break;
    case STEP_VERT:
      ps->anime = 4;
      break;
    case (STEP_VERT*3):
      ps->anime = 5;
      break;
    }
  } else {
    switch(ps->y % (STEP_VERT*4)) {
    case 0:
    case (STEP_VERT*2):
      ps->anime = 0;
      break;
    case STEP_VERT:
      ps->anime = 1;
      break;
    case (STEP_VERT*3):
      ps->anime = 2;
      break;
    }
  }
  return look_to_wall;
}



/* 
 * local function walk_left 
 */
#ifdef __STDC__
static int 
walk_left (BMPlayer *ps,
	   int flag,
	   int mazex, 
	   int mazey)
#else
static int 
walk_left (ps, flag, mazex, mazey)
     BMPlayer *ps;
     int flag;
     int mazex, mazey;
#endif
{
  int look_to_wall = FALSE;

  if ( !(flag && check_maze(mazex -1, mazey) )) {
    ps->x -= STEP_HORI;
  } else {
    ps->d_ist = GoStop;
    look_to_wall = TRUE;
  }

  if (ps->illness != IllReverse) { 
    switch(ps->x % (STEP_HORI*4)) {
    case 0:
    case (STEP_HORI*2):
      ps->anime = 9;
      break;
    case STEP_HORI:
      ps->anime = 10;
      break;
    case (STEP_HORI*3):
      ps->anime = 11;
      break;
    }
  } else {
    switch(ps->x % (STEP_HORI*4)) {
    case 0:
    case (STEP_HORI*2):
      ps->anime = 6;
      break;
    case STEP_HORI:
      ps->anime = 7;
      break;
    case (STEP_HORI*3):
      ps->anime = 8;
      break;
    }
  }
  return look_to_wall;
}



/* 
 * local function walk_down 
 */
#ifdef __STDC__
static int 
walk_down (BMPlayer *ps,
	   int flag,
	   int mazex, 
	   int mazey)
#else
static int 
walk_down (ps, flag, mazex, mazey)
     BMPlayer *ps;
     int flag;
     int mazex, mazey;
#endif
{
  int look_to_wall = FALSE;

  if ( !(flag && check_maze(mazex, mazey+1) )) {
    ps->y += STEP_VERT;
  } else {
    look_to_wall = TRUE;
      ps->d_ist = GoStop;
  }
  
  if (ps->illness == IllReverse) {
    switch(ps->y % (STEP_VERT*4))  {
    case 0:
    case (2*STEP_VERT):
      ps->anime = 3;
      break;
    case STEP_VERT:
      ps->anime = 4;
      break;
    case (STEP_VERT*3):
      ps->anime = 5;
      break;
    }
  } else {
    switch(ps->y % (STEP_VERT*4)) {
    case 0:
    case (STEP_VERT*2):
      ps->anime = 0;
      break;
    case STEP_VERT:
      ps->anime = 1;
      break;
    case (STEP_VERT*3):
      ps->anime = 2;
      break;
    }
  }
  return(look_to_wall);
}



/* 
 * local function walk_right 
 */
#ifdef __STDC__
static int 
walk_right (BMPlayer *ps,
	    int flag,
	    int mazex, 
	    int mazey)
#else
static int 
walk_right (ps, flag, mazex, mazey)
     BMPlayer *ps;
     int flag;
     int mazex, mazey;
#endif
{
  int look_to_wall = FALSE;

  if ( !(flag && check_maze(mazex +1, mazey))) {
    ps->x += STEP_HORI;
  } else {
    look_to_wall = TRUE;
    ps->d_ist = GoStop;
  }
  
  if (ps->illness == IllReverse) { 
    switch(ps->x % (STEP_HORI*4)) {
    case 0:
    case (STEP_HORI*2):
      ps->anime = 9;
      break;
    case STEP_HORI:
      ps->anime = 10;
      break;
    case (STEP_HORI*3):
      ps->anime = 11;
      break;
    }
  } else {
    switch(ps->x % (STEP_HORI*4)) {
    case 0:
    case (STEP_HORI*2):
      ps->anime = 6;
      break;
    case STEP_HORI:
      ps->anime = 7;
      break;
    case (STEP_HORI*3):
      ps->anime = 8;
      break;
    }
  }
  return look_to_wall;
}


/*
 * local function teleport_player
 */
#ifdef __STDC__
static int
teleport_player (BMPlayer *ps,
		 int mazex,
		 int mazey) 
#else
static int
teleport_player (ps, mazex, mazey) 
     BMPlayer *ps;
     int mazex, mazey;
#endif
{
  /* teleport dri@eup.siemens-albis.ch */
  /* 'No spare space' bug killed by Garth Denley */
  /* Horizontal/Vertical teleport re-enabled */
  int new_mazex, new_mazey;
  int tele_tries;
  int tele_success;
  
  tele_tries = 25;
  tele_success = FALSE;
  
  do {
    new_mazex = (rand()>>4) % MAZE_W;
    new_mazey = (rand()>>4) % MAZE_H;
    tele_tries--;
    if ((!check_maze(new_mazex,new_mazey))
	&& ((mazex != new_mazex) || (mazey != new_mazey))) {
      tele_success = TRUE;
    }
  } while ((!tele_success) && (tele_tries>0) );
  
  if (tele_success) {
    mazex = new_mazex;
    mazey = new_mazey;
    ps->x = BLOCK_WIDTH * mazex;
    ps->y = BLOCK_HEIGHT * (mazey - 1);
    ps->d_soll = GoStop;
    ps->d_look = GoDown;
  }

  return tele_success;
}



/* 
 * local function do_walk 
 */
#ifdef __STDC__
static void 
do_walk (int player)
#else
static void 
do_walk (player)
      int player;
#endif
{
  int flag;
  int look_to_wall = FALSE;
  int mazex,mazey;
  int i;
  int xalt,yalt;

  BMPlayer *ps;

  ps = player_stat + player;
  xalt = ps->x;
  yalt = ps->y;

  if ( !( (ps->illness == IllSlow) && (ps->illtime & 0x01) ) ) {
    for (i=0; i<=(ps->illness == IllRun); i++) {
      flag = FALSE;
      
      mazex = ps->x / BLOCK_WIDTH;
      mazey = ps->y / BLOCK_HEIGHT + 1;
      
      if ( ( (ps->x % BLOCK_WIDTH) == 0)
	  && ((ps->y % BLOCK_HEIGHT) == 0) ) {
	flag = TRUE;
	
	if (ps->teleport == TELEPORT_TIME) {
	  if (teleport_player(ps, mazex, mazey)) {
	    ps->teleport--;
	  }
	}

	ps->d_ist = ps->d_soll;
	if (ps->d_ist != GoStop)
	  ps->d_look = ps->d_ist;
      }
      
      /* random teleporting */
      if ( (ps->illness == IllTeleport) && (0 == ((rand()>>4)%32)) ) {
	mark_maze( (int)((ps->x+SPRITE_X_OFF )/BLOCK_WIDTH),
		  (int) ((ps->y+SPRITE_Y_OFF)/BLOCK_HEIGHT),
		  (int) ((ps->x+SPRITE_X_OFF+SPRITE_WIDTH-1)/BLOCK_WIDTH),
		  (int) ((ps->y+SPRITE_Y_OFF+SPRITE_HEIGHT-1)/BLOCK_HEIGHT));
	teleport_player(ps, mazex, mazey);
	ps->d_ist = GoStop;
	ps->d_soll = GoStop;
	mark_maze( (int)((ps->x+SPRITE_X_OFF )/BLOCK_WIDTH),
		  (int) ((ps->y+SPRITE_Y_OFF)/BLOCK_HEIGHT),
		  (int) ((ps->x+SPRITE_X_OFF+SPRITE_WIDTH-1)/BLOCK_WIDTH),
		  (int) ((ps->y+SPRITE_Y_OFF+SPRITE_HEIGHT-1)/BLOCK_HEIGHT));
      }
	
      
      switch(ps->d_ist) {
      case GoStop:
	look_to_wall = walk_stop(ps);
	break;
	
      case GoUp:
	look_to_wall = walk_up(ps, flag, mazex, mazey );
	break;
	
      case GoLeft:
	look_to_wall = walk_left(ps, flag, mazex, mazey );
	break;
	
      case GoDown:
	look_to_wall = walk_down(ps, flag, mazex, mazey );
	break;
	
      case GoRight:
	look_to_wall = walk_right(ps, flag, mazex, mazey );
	break;
      }
      
      switch(ps->d_ist) {
      case GoLeft:
	if ( ((ps->x % BLOCK_WIDTH) == (STEP_HORI*BOMB_STEP) )
	    && check_bomb(ps->x/BLOCK_WIDTH, ps->y/BLOCK_HEIGHT+1) ) {
	  if (ps->kick) {
	    move_bomb(ps->x/BLOCK_WIDTH, ps->y/BLOCK_HEIGHT+1, 
		      GoLeft);
	    ps->d_soll = GoStop;
	  }
	  ps->x += STEP_HORI;
	}
	break;

      case GoRight:
	if ( ( (ps->x % BLOCK_WIDTH)
	      == (BLOCK_WIDTH - STEP_HORI*BOMB_STEP) )
	    && check_bomb(ps->x/BLOCK_WIDTH+1, ps->y/BLOCK_HEIGHT+1) ) {
	  if (ps->kick) {
	    move_bomb(ps->x/BLOCK_WIDTH+1, ps->y/BLOCK_HEIGHT+1, 
		      GoRight);
	    ps->d_soll = GoStop;
	  }
	  ps->x -= STEP_HORI;
	}
	break;

      case GoUp:
	if ( ( (ps->y % BLOCK_HEIGHT) == (STEP_VERT*BOMB_STEP) )
	    && check_bomb(ps->x/BLOCK_WIDTH, ps->y/BLOCK_HEIGHT+1)) {
	  if (ps->kick) {
	    move_bomb(ps->x/BLOCK_WIDTH, ps->y/BLOCK_HEIGHT+1, 
		      GoUp);
	    ps->d_soll = GoStop;
	  }
	  ps->y += STEP_VERT;
	}
	break;

      case GoDown:
	if ( ( (ps->y % BLOCK_HEIGHT) == (BLOCK_HEIGHT - STEP_VERT*BOMB_STEP) )
	    && check_bomb(ps->x/BLOCK_WIDTH, ps->y/BLOCK_HEIGHT+2)) {
	  if (ps->kick) {
	    move_bomb(ps->x/BLOCK_WIDTH, ps->y/BLOCK_HEIGHT+2, 
		      GoDown);
	    ps->d_soll = GoStop;
	  }
	  ps->y -= STEP_VERT;
	}
	break;
      }
      
      /* insert get _extra here */
      if ( (ps->x % BLOCK_WIDTH == 0) && (ps->y % BLOCK_HEIGHT == 0) ) {
	switch(get_extra(ps->invincible, ps->x / BLOCK_WIDTH ,
			 ps->y / BLOCK_HEIGHT + 1 ))
	  {
	  case BTBomb:
	    ps->bombs ++;
	    break;
	  case BTRange:
	    if (ps->range < MAX_RANGE)
	      ps->range ++;
	    break;
	  case BTSick:
	    ps->illtime = ILLTIME;
#if 1
	    ps->illness = (rand()>>4)%MAX_ILL+1;
#else
	    ps->illness = IllTeleport;
#endif
	    if (ps->illness == IllInvisible) {
	      ps->next_blink = 7 * ILLTIME / 8 ;
	    }
	    if (ps->illness == IllReverse) {
	      switch (ps->d_ist) {
	      case GoDown:
		ps->d_ist = GoUp;
		break;
	      case GoUp:
		ps->d_ist = GoDown;
		break;
	      case GoLeft:
		ps->d_ist = GoRight;
		break;
	      case GoRight:
		ps->d_ist = GoLeft;
		break;
	      }
	    }
	    break;
	  case BTSpecial:
	    ps->num_extras++;
	    (*special_extra_function)(ps, player, num_disp);
	    break;
	  }
      }
    }
  }

  if (look_to_wall || (ps->teleport > 1) 
      || (ps->d_ist != GoStop) || (ps->invincible > 0)
      || (ps->dying > 0) || (ps->stunned >0) ) {
    mark_maze( (int)((MIN(ps->x,xalt)+ SPRITE_X_OFF )/ BLOCK_WIDTH),
              (int)((MIN(ps->y,yalt) + SPRITE_Y_OFF) / BLOCK_HEIGHT),
              (int)((MAX(ps->x,xalt) + SPRITE_X_OFF + SPRITE_WIDTH -1)
                    / BLOCK_WIDTH),
              (int)((MAX(ps->y,yalt) + SPRITE_Y_OFF + SPRITE_HEIGHT -1)
                    / BLOCK_HEIGHT) );
  }

  if (ps->illness != IllInvisible) {
    if(ps->invincible > 0) {
      add_player_to_sprite_list(player, ps->x, ps->y, ps->anime,
				( ps->invincible-- & 0x01 ));
    } else if ((ps->teleport > 1) && (ps->teleport < TELEPORT_TIME)) {
      ps->teleport--;
      add_player_to_sprite_list(player, ps->x, ps->y, ps->anime,
				((ps->teleport == 1) ||
				 ((ps->teleport < (TELEPORT_TIME - 5)) &&
				  (ps->teleport > 7))));
    } else {
      add_player_to_sprite_list(player, ps->x, ps->y, ps->anime, TRUE);
    }
  }

  if (ps->illness != Healthy) {
    if ( (ps->illtime --) == 0) {
      ps->illness = ps->health;
    }
  }

  if ( (ps->x % BLOCK_WIDTH == 0) && (ps->y % BLOCK_HEIGHT == 0) ) {
    if (ps->illness == IllBomb) {
      if ( ( (rand()>>4) % 4) != 0) {
	drop_bomb(player, ps, defaultBMT);
      }
    }
  }
}

#define JUNKIE_ILL_TIME (ILLTIME)
#define JUNKIE_STUN_TIME 12
#define JUNKIE_TIME_1 360
#define JUNKIE_TIME_2 210
#define JUNKIE_TIME_3 60   /* Speed */

/*
 * local function do_junkie
 */
#ifdef __STDC__
static void
do_junkie (void)
#else
static void
do_junkie()
#endif
{
  BMPlayer *ps1, *ps2;
  int player;
  register int jc;

  /* Junkie contact */
  for (ps1 = player_stat; ps1 < player_stat+num_player; ps1 ++) {
    for (ps2= player_stat; ps2 < player_stat+num_player; ps2 ++) {
      if (ps1 != ps2) {
	if ( (ps1->junkie)
	    && (ABS(ps1->x - ps2->x) < ILL_X)
	    && (ABS(ps1->y - ps2->y) < ILL_Y)) {
	  if (ps2->invincible==0) {
	    ps2->junkie = TRUE;
	    ps2->junkie_time = MAX_JUNKIE_TIME;
	    ps1->junkie_time = MAX_JUNKIE_TIME;
	  
	    player = ps2 - player_stat;
	  } else if (ps1->invincible==0) {
	    ps1->junkie = TRUE;
	    ps1->junkie_time = MAX_JUNKIE_TIME;
	    ps2->junkie_time = MAX_JUNKIE_TIME;
	  
	    player = ps1 - player_stat;
	  }
	}
      }
    }
  }

  /* Junkie countdown */
  for (player=0;player<num_player;player++) {
    ps1=player_stat+player;
    if (ps1->junkie) {
      jc = --(ps1->junkie_time);
      /* Junkie sickness */
      if ((jc == JUNKIE_TIME_1) || (jc == JUNKIE_TIME_2)) {
        /* Give a random illness */
        ps1->illtime = JUNKIE_ILL_TIME;
	ps1->illness = (rand()>>4)%MAX_ILL+1;
      } else if (jc == JUNKIE_TIME_3) {
        /* Stun player and give speed */
        ps1->stunned += JUNKIE_STUN_TIME;
        ps1->illtime = JUNKIE_ILL_TIME;
        ps1->illness = IllRun;
      } else if (jc <= 0) {
        /* Too long! Take a hit. */
        player_stat[player].dying = DEAD_TIME;
        ps1->junkie_time = MAX_JUNKIE_TIME;
      }
    }
  }
}


/*
 * local function infect_other_players
 */
#ifdef __STDC__
static void 
infect_other_players (void)
#else
static void 
infect_other_players ()
#endif
{
  BMPlayer *ps1,*ps2;
  int player;

  for (ps1 = player_stat; ps1 < player_stat+num_player; ps1 ++) {
    for (ps2= ps1+1; ps2 < player_stat+num_player; ps2 ++) {
      if ( (ps1->illness != ps2->illness)
	  && (ABS(ps1->x - ps2->x) < ILL_X)
	  && (ABS(ps1->y - ps2->y) < ILL_Y)) {
	if ( (ps2->invincible==0) &&  (ps1->illtime > ps2->illtime) ) {
	  ps2->illness = ps1->illness;
	  ps2->illtime = ILLTIME;
	  
	  player = ps2 - player_stat;
	} else if ( (ps1->invincible==0) &&  (ps2->illtime > ps1->illtime) ) {
	  ps1->illness = ps2->illness;
	  ps1->illtime = ILLTIME;
	  
	  player = ps1 - player_stat;
	}
      }
    }
  }
}

/*
 * public function print_player_lives
 */
#ifdef __STDC__
void 
print_player_lives (int player, 
		    int flag)
#else
void 
print_player_lives (player, flag)
     int player, flag;
#endif
{
  BMPlayer *ps;
  int disp;
  int x;

  ps = player_stat + player;


  x = (2*player + 1 + (player > 1 ? 7 : 0));
  mark_maze(x, MAZE_H, x, MAZE_H);

  if (flag) {
    for (disp = 0; disp < num_disp; disp ++) {
      draw_score_block(disp, x, SBLives + ps->lives);
    }
    if (ps->lives == 0) {
      mark_maze(x-1, MAZE_H, x-1, MAZE_H);
      for (disp = 0; disp < num_disp; disp ++) {
	draw_score_block(disp, x-1, SBDead + player);
      }
    }
  } else {
    for (disp = 0; disp < num_disp; disp ++) {
      draw_score_block(disp, x, SBLives + ps->victories);
    }
    mark_maze(x-1, MAZE_H, x-1, MAZE_H);
    for (disp = 0; disp < num_disp; disp ++) {
      draw_score_block(disp, x-1, SBPlayer + player);
    }
  }
}


/*
 * local functzion have_a_gloat
 */
#ifdef __STDC__
static void 
have_a_gloat (int player)
#else
static void 
have_a_gloat (player)
     int player;
#endif
{
  int g,gloatpl,gloatpltt;
  
  gloatpl=-1;
  for(g=0; g<6; g++) {
    gloatpltt = rand() % num_player;
    if ((gloatpltt != player) && (player_stat[gloatpltt].lives > 0)) {
      gloatpl = gloatpltt;
      exit;
    }
  }      
  if(gloatpl>-1) {
    set_message(p_string[gloatpl].gloat, FALSE);
  }
}



/*
 * local function revive_player 
 */
#ifdef __STDC__
static void 
revive_player (int player)
#else
static void 
revive_player(player)
     int player;
#endif
{
  BMPlayer *ps;
  PlayerStrings *st;

  ps = player_stat + player;
  st = p_string + player;

  ps->lives --;

  if (ps->lives == 0) {
    active_player --;
    distribute_extras(ps->bombs - min_bombs, 
		      ps->range - min_range,
		      ps->num_extras,
		      ps->special_bombs);
    set_message (st->loselevel, FALSE);
  }  else {
    distribute_extras(0, 0,
		      ps->num_extras,
		      ps->special_bombs);
    set_message (st->loselife, FALSE);
  }

  have_a_gloat(player);

  /* reset values */
  ps->invincible = NEW_INVINCIBLE;
  ps->next_blink = 0;
  ps->dying = 0;
  ps->stunned = 0;
  ps->illness = ps->health;
  ps->illtime = 0;
  ps->teleport = 0;

  /* Note that junkie ISN'T reset (not a bug) */

  /* very important */
  if (ps->remote_control > 0) {
    ignite_players_bombs(player);
  }

  ps->remote_control = 0;
  ps->kick = 0;
  ps->air_button = FALSE;

  /* If special bombs are distributed, then zero the count */
  if (distrib_special()) {
    ps->special_bombs = 0;
  }

  /* Reset extra pickup count */
  ps->num_extras = 0;

  /* reset inital extras */
  if (IF_RC & ps->extra_flags) {
    ps->remote_control = 1;
  }
  if (IF_Teleport & ps->extra_flags) {
    ps->teleport = 1;
  }
  if (IF_Kick & ps->extra_flags) {
    ps->kick = 1;
  }

  mark_maze( (int)((ps->x + SPRITE_X_OFF)/ BLOCK_WIDTH),
            (int)((ps->y +SPRITE_Y_OFF)/ BLOCK_HEIGHT),
            (int)((ps->x + SPRITE_X_OFF + SPRITE_WIDTH -1) / BLOCK_WIDTH),
            (int)((ps->y + SPRITE_Y_OFF + SPRITE_HEIGHT -1) / BLOCK_HEIGHT) );
  
}



/*
 * local function do_stunned
 */
#ifdef __STDC__
static void 
do_stunned (int player)
#else
static void 
do_stunned (player)
     int player;
#endif
{
  switch( (player_stat[player].d_look + player_stat[player].stunned)%4 ) {
  case GoDown:
    player_stat[player].anime = 0;
    break;
  case GoUp:
    player_stat[player].anime = 3;
    break;
  case GoLeft:
    player_stat[player].anime = 9;
    break;
  case GoRight:
    player_stat[player].anime = 6;
    break;
  }
  
  add_player_to_sprite_list
    (player,player_stat[player].x,player_stat[player].y,
     player_stat[player]. anime,TRUE);
  
  mark_maze( (int)((player_stat[player].x + SPRITE_X_OFF)/ BLOCK_WIDTH),
	    (int)((player_stat[player].y + SPRITE_Y_OFF)/ BLOCK_HEIGHT),
	    (int)((player_stat[player].x + SPRITE_X_OFF + SPRITE_WIDTH -1)
		  /BLOCK_WIDTH),
	    (int)((player_stat[player].y + SPRITE_Y_OFF + SPRITE_HEIGHT-1)
		  /BLOCK_HEIGHT)
	    );
  player_stat[player].stunned -- ;
}



/*
 * local function do_die
 */
#ifdef __STDC__
static void 
do_die (int player)
#else
static void 
do_die (player)
     int player;
#endif
{

  if (player_stat[player].lives > 1) {
    add_player_to_sprite_list
      (player,player_stat[player].x,player_stat[player].y,
       13,TRUE);
  } else {
    add_player_to_sprite_list
      (player,player_stat[player].x,player_stat[player].y,
       12,TRUE);
  }
  if (player_stat[player].dying == DEAD_TIME) {
    mark_maze( (int)((player_stat[player].x + SPRITE_X_OFF)/ BLOCK_WIDTH),
              (int)((player_stat[player].y + SPRITE_Y_OFF)/ BLOCK_HEIGHT),
              (int)((player_stat[player].x + SPRITE_X_OFF + SPRITE_WIDTH -1)
                    /BLOCK_WIDTH),
              (int)((player_stat[player].y + SPRITE_Y_OFF + SPRITE_HEIGHT-1)
                    /BLOCK_HEIGHT)
              );
  }
  player_stat[player].dying -- ;
}


/*
 * local function mark_overlapping_players
 *  not very elegant ...
 */
#ifdef __STDC__
static void
mark_overlapping_players (void)
#else
static void
mark_overlapping_players ()
#endif
{
  int p1, p2;
  BMPlayer *ps1, *ps2;

  for (p1 = 0, ps1 = player_stat; p1 < (num_player-1); p1++, ps1++) {
    if ( (0 == (ps1->x % BLOCK_WIDTH)) && (0 == (ps1->y % BLOCK_HEIGHT)) ) {
      for (p2 = p1+1, ps2 = ps1+1; p2 < num_player; p2++, ps2++) {
	if ((0 == (ps2->x % BLOCK_WIDTH)) && (0 == (ps2->y % BLOCK_HEIGHT))) {
	  if ( (ps1->x == ps2->x) && (ps1->y == ps2->y)) {
	    mark_maze (ps1->x/BLOCK_WIDTH, ps1->y/BLOCK_HEIGHT,
		       ps1->x/BLOCK_WIDTH, 1+ps1->y/BLOCK_HEIGHT);
	  }
	}
      }
    }
  }
}

/*
 * public function add_player_num_bomb
 */
#ifdef __STDC__
void 
add_player_num_bomb (int player)
#else
void 
add_player_num_bomb (player)
     int player;
#endif
{
  if (player < MAX_PLAYER) {
    player_stat[player].bombs ++;
  }
}



/* 
 * after one game 
 */
#ifdef __STDC__
static void 
level_end (void)
#else
static void 
level_end ()
#endif
{
  static char *msg_oot  = "Out Of Time";
  static char *msg_draw = "Draw Game";
  char *Message;
  char playerList[1024];
  
  /* create player list for ouput */
  switch(num_player) {
  case 2:
    sprintf(playerList,"{%s} {%s}", p_string[0].name, 
	    p_string[1].name);
    break;
  case 3:
    sprintf(playerList,"{%s} {%s} {%s}", p_string[0].name, 
	    p_string[1].name, p_string[2].name);
    break;
  case 4:
    sprintf(playerList,"{%s} {%s} {%s} {%s}", p_string[0].name, 
	    p_string[1].name, p_string[2].name, p_string[3].name);
    break;
  }

  if (game_time >= (GAME_TIME-1)) {
    Message = msg_oot;
    last_player = MAX_PLAYER;
    
    /* print statistics for draw */
    if (Setup.print_stat) {
      printf("LevelStat {%s} {%s} {%s}\n",get_level_name(level),
	     "Out Of Time", playerList);
    }
  } else {
    if (active_player != 0) {
      for (last_player=0; (last_player<num_player)
	   && (player_stat[last_player].lives == 0); last_player ++);
    } else {
      last_player = MAX_PLAYER;
    }
    if ( (last_player >= num_player) ||
	(player_stat[last_player].dying > 0
	 && (player_stat[last_player].lives == 1))) {
      Message = msg_draw;
      last_player = MAX_PLAYER;
      
      /* print statistics for draw */
      if (Setup.print_stat) {
	printf("LevelStat {%s} {%s} {%s}\n",get_level_name(level),
	       "Draw Game", playerList);
      }
    } else {
      player_stat[last_player].victories ++;
      num_victories = MAX(num_victories,
			  player_stat[last_player].victories);

      Message = p_string[last_player].winlevel;
      
      /* print statistics for winner */
      if (Setup.print_stat)  {
	printf("LevelStat {%s} {%s} {%s}\n",get_level_name(level),
	       p_string[last_player].name, playerList);
      }
    }
  }
  
  clear_sprite_list();

  if (last_player < MAX_PLAYER) {
    add_player_to_sprite_list(last_player, 
			      player_stat[last_player].x,
			      player_stat[last_player].y,
			      WINNER_ANIME, TRUE );
    mark_maze( (int)((player_stat[last_player].x + WINNER_X_OFF)
		     /BLOCK_WIDTH),
	      (int)((player_stat[last_player].y + WINNER_Y_OFF)
		    /BLOCK_HEIGHT),
	      (int)((player_stat[last_player].x 
		     + WINNER_X_OFF + WINNER_WIDTH)
		    /BLOCK_WIDTH),
	      (int)((player_stat[last_player].y 
		     + WINNER_Y_OFF + WINNER_HEIGHT)
		    /BLOCK_HEIGHT)
	      );
  }
  
  wait2_two_text(num_disp, num_player, Message,"Press Space");
  
  fade_out(num_disp);
}


#ifdef __STDC__
int 
check_b_near (int x,
	      int y)
#else
int 
check_b_near (x,y)
     int x,y;
#endif
{
  int player;

  for (player = 0; player < num_player; player ++) {
    if ((ABS(x*BLOCK_WIDTH - player_stat[player].x) < BLOCK_WIDTH)
	&& (ABS(y*BLOCK_HEIGHT-BLOCK_HEIGHT - player_stat[player].y)  
	    < BLOCK_HEIGHT)) {
      return 0;
    }
  }
  return 1;
}



/*
 * local function eval_config
 */
#ifdef __STDC__
static void 
eval_config (char *prog_name)
#else
static void 
eval_config (prog_name)
     char *prog_name;
#endif
{
  int p1, p2;
#ifndef DEBUG
  int p3;
#endif
  int disp;
  
  /* set values derived from setup */
  num_player = Config.num_player;
  if (num_player <= 0) {
    bm_fatal("Number of players not defined",prog_name);
  }
  
  num_disp = Config.num_disp;
  if (num_disp <= 0) {
    bm_fatal("Number of displays not defined",prog_name);
  }

  /* set game mode flag */
  switch (num_player) {
  case 2:
    game_mode = GM_2_Player;
    break;
  case 3:
    game_mode = GM_3_Player;
    break;
  case 4:
    game_mode = GM_4_Player;
    break;
  }
  
#ifdef DEBUG
  fprintf(stderr, "Default Display is %d:%s\n", Config.default_disp,
	  Config.display[Config.default_disp]);
#endif

  /* set control, not very elegant, but it works */
  for (disp=0; disp < num_disp; disp ++) {
    /* find first player for display */
    for (p1 = 0; (p1<num_player) && (Config.pl_at_disp[p1] != disp) ; p1++);
    if (p1 == num_player) {
      fprintf(stderr, "Oops no player for display %d\n",disp);
      exit(1);
    }
    /* look for optional second one */
    for (p2 = p1+1; (p2<num_player) && (Config.pl_at_disp[p2] != disp) ; p2++);
    if (p2 < num_player) {
      /* set two players at display */
      Config.disp_player[disp].num = 2;
      Config.disp_player[disp].p1  = p1;
      Config.disp_player[disp].p2  = p2;
      set_players_for_display(disp, p1, p2);
      /* look for forbidden third one */
#ifndef DEBUG
      for (p3 = p2+1; (p3<num_player) && (Config.pl_at_disp[p3]!=disp) ; p3++);
      if (p3 < num_player) {
	fprintf(stderr, "Error: 3 players for Display %s\n",
		Config.display[disp]);
	exit(1);
      }
#endif
    } else {
      Config.disp_player[disp].num = 1;
      Config.disp_player[disp].p1  = p1;
      Config.disp_player[disp].p2  = p1;
      set_players_for_display(disp, p1, p1);
    }
  }  
}



/*
 * local function eval_setup
 */
#ifdef __STDC__
static void 
eval_setup (char *prog_name)
#else
static void 
eval_setup (prog_name)
     char *prog_name;
#endif
{
  int i, n_level;

  FrameTime = Setup.frame_time;
  if (Setup.sound_flag) {
    sound_function = do_bell;
  } else {
    sound_function = no_bell;
  }
  
  /* correct use_level field */
  n_level = 0;
  for (i=0; i<LEVEL_MAX; i++) {
    if ( game_mode != (game_mode & get_game_mode(i))) {
      /* mode not supported by level */
      if (Setup.use_level[i]) {
	fprintf(stderr, "Level \"%s\" not supported in this game mode.\n",
		get_level_name(i) );
      }
      Setup.use_level[i]=FALSE;
    }
    if (Setup.use_level[i]) {
      n_level ++;
    }
  }
  if (n_level == 0) {
    fprintf (stderr, "No Level left to play in this game mode.\n");
    exit (1);
  }
}

/* main programm what else */
#ifdef __STDC__
int 
main (int argc,
	 char *argv[])
#else
int 
main (argc, argv)
     int argc;
     char *argv[];
#endif
{
  int p, player, disp;
  int last_delay;
  int level_index;
  int level_field[LEVEL_MAX];
  int setup_flag;

  printf("%s\n",c_string);
  printf("Report any bugs to: vogel@ikp.uni-koeln.de\n");

  /* Initialize Random */
  srand(getpid());

  /* choose setup method */
  if (1 != argc) {
    parse_commandline(argc, argv);
    setup_flag = FALSE;
  } else {
    setup_flag = interactive_config(argv[0]);
  }

  /* get config from database */
  config_from_database(&Config);

  /* evaluate config */
  eval_config(argv[0]);

  /* Initialize Displays and X Databases */
  for (player=0; player < num_disp; player++) {
    init_display(player, Config.display[player]);
  }

  /* merge databases fehlt */
  merge_default_databases(&Config);

  /* rest of interacive setup */
  if (setup_flag) {
    interactive_setup(num_player);
  }

  setup_from_database(&Setup);
  eval_setup(argv[0]);

  /* get names and messages */
  player_strings_from_database(p_string);

  /* now save setup if needed */
  save_setup(&Config, &Setup, p_string);

  /* set status board */
  set_victories(Setup.max_victories);

  /* Initialize Graphics */
  for (player=0; player < num_disp; player++) {
    init_graphics(player);
  }

  /* setup status_bar */
  init_status_bar(num_player, num_disp, player_stat, get_level_name(level),
		  FALSE);

  /* Title Screen */
  do_intro(player_stat, p_string, &Config);

  /* the game */

  for (level_index = 0; level_index < LEVEL_MAX; level_index ++) {
    level_field[level_index] = level_index;
  }

  if (Setup.random_mode) {
    level_index = 0;
    randomize_levels(level_field);
  } else {
    level_index = Setup.start_level;
  }


  while( num_victories < Setup.max_victories ) {
    do {
      level = level_field[level_index];
	
      if ( (++ level_index) >= LEVEL_MAX ) {
	level_index = 0;
	if (Setup.random_mode) {
	  randomize_levels(level_field);
	}
      }
    } while (!Setup.use_level[level]);
    
    /* Set up scrambled start positions */
    init_shuffle_startpos();

    /* Introduce level */
    clear_sprite_list();
    level_intro(level, player_stat, &Setup, &Config);
    unload_blocks(num_disp);
    
    /* load maze from data */
    load_maze(player_stat, level, &Config, &Setup, &min_range, &min_bombs);
    active_player = num_player;

    /* Setup shrinks */

    /* exec special_init_functions */
    (*special_init_function)();
    
    /* draw maze in pixmap */
    for (player = 0; player < num_disp; player ++) {
      draw_maze(player);
    }
    reset_status_bar(player_stat, get_level_name(level), TRUE);

    /* mark player positions */
    for (player = 0; player < num_player; player ++) {
      mark_maze( (int)((player_stat[player].x + SPRITE_X_OFF)
		       / BLOCK_WIDTH),
		(int)((player_stat[player].y + SPRITE_Y_OFF)
		      / BLOCK_HEIGHT),
		(int)((player_stat[player].x + SPRITE_X_OFF + SPRITE_WIDTH
		       -1) / BLOCK_WIDTH),
		(int)((player_stat[player].y + SPRITE_HEIGHT + SPRITE_Y_OFF
		       -1) / BLOCK_HEIGHT) );
      print_player_lives(player, TRUE);
    }

    level_start(num_disp);

    /* set up welcome messages */
    welcome(num_player, p_string);

    for (player = 0; player < num_disp; player ++) {
      flush_pixmap(player, num_disp, FALSE);
    }

    game_time = 0;

    init_timer();
    clear_keys(num_player);
    last_delay = 0;
    
    do {
      /* increase game time counter, when not in PAUSE mode */
      if (pause_status < 0) {
	game_time ++;
      }
      /* kill players if we are near the end of game */
      if (game_time == (GAME_TIME - DEAD_TIME + 1)) {
	for (player = 0; player < num_player; player ++) {
	  if (player_stat[player].lives>0) {
	    player_stat[player].lives = 1;
	    player_stat[player].dying = DEAD_TIME;
	  }
	}
      }

      /* check if we are not in PAUSE mode */
      if (pause_status == -1 ) {
	/* clear list of all sprites */
	clear_sprite_list();
	
	/* check player status */
	for (p = 0; p <num_player; p ++) {
	  /* to permute player when drawing and stunning */
	  /* quick and dirty but hopefully it solves some problems */
	  player = (p + game_time) % num_player;
	  if (player_stat[player].lives != 0) {
	    
	    switch(player_stat[player].dying) {
	    case 0:
	      /* player is alive and ... */
	      if (player_stat[player].stunned == 0) {
		/* ... walks around */
		do_walk(player);
	      } else {
		/* ... and stunned */
		do_stunned(player);
	      }
	      break;
	      
	    case 1:
	      /* try to revive player */
	      revive_player(player);
	      break;
	      
	    default:
	      /* player is dying */
	      do_die(player);
	      break;
	    }
	  }
	}
	
	/* check if players stand on the same position */
	mark_overlapping_players();

	/* Shrink functions */
	do_shrink(game_time, num_player, player_stat);

	/* Scramble blocks */
	do_scramble2(game_time, num_player, player_stat);
	
	/* Game functions */
	(*special_game_function)();
	
	/* check bombs */
	do_bombs();
	
	/* sort spites by y-position */
	sort_sprite_list();
	
	/* check explosions */
	if(do_explosions()) {
	  for (disp = 0; disp < num_disp; disp ++) {
	    (*sound_function)(disp);
	  }
	}
	
	/* check if bombs are ignited by other bombs */
	ignite_bombs();
	
	/* stun players hit by other bombs */
	stun_players(player_stat, num_player);
	
	/* check if and Illness infects another player */
	infect_other_players();
	
	/* handle junkies */
	do_junkie();

	for (player = 0; player < num_player; player ++) {
	  if ((player_stat[player].lives !=0  )
	      && (player_stat[player].invincible==0)
	      && check_explosion
	      ( (player_stat[player].x + (BLOCK_WIDTH>>1))/BLOCK_WIDTH,
	       (player_stat[player].y + (BLOCK_HEIGHT>>1))/BLOCK_HEIGHT
	       +1 )) {
	    player_stat[player].dying = DEAD_TIME;
	  }
	}
      }
      /* update status bar */
      update_status_bar (player_stat, game_time);

      game_event(num_disp);
      if (game_eval_keys(num_player, player_stat, p_string, &pause_status)) {
	if (-1 == pause_status) {
	  reset_message();
	  for (disp=0; disp <num_disp; disp ++) {
	    (*sound_function)(disp);
	  }
	} else {
	  set_message("Game Paused", TRUE);
	}
      }

    } while( (game_time<= GAME_TIME)
	    && ( active_player > 1 || number_of_explosions()!=0 ) 
	    && (active_player > 0) );
    
    level_end();
    unload_blocks(num_disp);
    
    clear_sprite_list();
    status_board(last_player, num_victories, player_stat, p_string, 
		 &Setup, &Config);
    unload_blocks(num_disp);
    
  }
  
  if (num_victories == Setup.max_victories) {
    winning_the_game(last_player, player_stat, p_string, &Setup, &Config);
  }
  
  printf("Ready.\n");
  return 0;
}

/*
 * end of file main.c
 */
