/*	From FreeBSD: */
/*
 *      tcpblast - test and estimate TCP thruput
 *
 *      Daniel Karrenberg   <dfk@nic.eu.net>
 */

/*
 *	Changes: Rafal Maszkowski <rzm@pdi.net>
 */
char *verstr="FreeBSD + rzm 960224";

#include <arpa/inet.h>
#include <getopt.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#define DEFBLKSIZE (1024)
#define MAXBLKSIZE (32*1024)

struct	sockaddr_in sock_in;
struct	servent *sp;
struct	hostent *host;

unsigned long starts, startms, stops, stopms, expms;
struct timeval ti; 
struct timezone tiz;

char 	greet[MAXBLKSIZE] = "Hi!";
int 	nblocks, f;
int tcp=0, udp=0, randomb=0, blksize=DEFBLKSIZE, setbufsize=-1, port=9;

/* Long options.  */
static const struct option long_options[] =
{
  { "help", no_argument, NULL, 'h' },
  { "version", no_argument, NULL, 'V' },
  { NULL, 0, NULL, 0 }
};

void usage() {
	fprintf(stderr, "usage: tcpblast [options] destination nblocks\n");
	fprintf(stderr, "Options:\n");
	fprintf(stderr, "-t\tuse TCP\n");
	fprintf(stderr, "-u\tuse UDP\n");
	fprintf(stderr, "-p nn\tuse port #nn instead of default %d\n", port);
	fprintf(stderr, "-r\tsend random data\n");
	fprintf(stderr, "-s nnn\tblock size, default %d bytes\n", blksize);
	fprintf(stderr, "-b nnn\tbuf size, default %d bytes\n", setbufsize);
	fprintf(stderr, "-v\tversion\n");
	fprintf(stderr, "destination\thost name or address\n");
	fprintf(stderr, "nblocks\tnumber of blocks (0..9999)\n");
	exit(1);
}

/* randomize the buffer */
void
randbuff(blksize) 
int blksize;
{
	int i;
	for (i=0; i<blksize; i++) {
		greet[i]=rand() % 256;
	}
}

int
main(argc, argv)
     int argc;
     char **argv;
{	register int i; char optchar;

	/* non-random data - is modem compressing it? */
	bzero(greet, MAXBLKSIZE);
/*	memset(greet, 'a', MAXBLKSIZE); */

	while ((optchar = getopt_long (argc, argv, "tup:rs:b:v", long_options, NULL)) != EOF)
	switch (optchar) {
		case '\0': break;
		case 't': if (tcp==0) tcp=1;	break;
		case 'u': if (udp==0) udp=1;	break;
		case 'r': randomb=1;		break;
		case 's': blksize=atoi(optarg);	break;
		case 'b': setbufsize=atoi(optarg);	break;
		case 'p': port=atoi(optarg);		break;
		case 'v': printf("version: %s\n", verstr);	break;
		default: usage();
	}

/* correctness */
	if (tcp && udp) {
		printf("cannot use both TCP and UDP\n");
		usage();
	}
	{ char *ind;
	/* if neither -t nor -u is chosen use first character of the
	   program name */
		ind=rindex(argv[0], '/');
		if (ind==NULL) ind=argv[0]; else ind++;
		if ( (tcp==0) && (udp==0) && (ind[0]=='t') ) tcp=1;
		if ( (tcp==0) && (udp==0) && (ind[0]=='u') ) udp=1;
	}
	if (!tcp && !udp) {
		printf("must use either TCP or UDP\n");
		usage();
	}

	/* after options processing we need two args left */
	if (argc - optind != 2) {
		printf("give both hostname and block count\n");
		usage();
	}

	nblocks = atoi(argv[optind+1]);
        if (nblocks<=0 || nblocks>=10000) {
		fprintf(stderr, "tcpblast: 1 < nblocks <= 10000 \n");
		exit(1);
	}

	bzero((char *)&sock_in, sizeof (sock_in));
	sock_in.sin_family = AF_INET;
	if (tcp) f = socket(AF_INET, SOCK_STREAM, 0);
	else     f = socket(AF_INET, SOCK_DGRAM, 0);
	if (f < 0) {
		perror("tcpblast: socket");
		exit(3);
	}

	{ int bufsize, size;
	getsockopt(f, SOL_SOCKET, SO_SNDBUF, &bufsize, &size);
	printf("read SO_SNDBUF = %d\n", bufsize);
	if (setbufsize!=-1) {
		setsockopt(f, SOL_SOCKET, SO_SNDBUF, &bufsize, sizeof(bufsize));
		getsockopt(f, SOL_SOCKET, SO_SNDBUF, &bufsize, &size);
		printf("set  SO_SNDBUF = %d\n", bufsize);
	} }

	if (bind(f, (struct sockaddr*)&sock_in, sizeof (sock_in)) < 0) {
		perror("tcp/udpblast: bind");
		exit(1);
	}

	host = gethostbyname(argv[optind]);
	if (host) {
		sock_in.sin_family = host->h_addrtype;
		bcopy(host->h_addr, &sock_in.sin_addr, host->h_length);
	} else {
		sock_in.sin_family = AF_INET;
		sock_in.sin_addr.s_addr = inet_addr(argv[optind]);
		if (sock_in.sin_addr.s_addr == -1) {
			fprintf(stderr, "tcpblast: %s unknown host\n", argv[optind]);
			exit(1);
		}
	}
	sock_in.sin_port = htons(port);

	if (connect(f, (struct sockaddr*)&sock_in, sizeof(sock_in)) <0)
	{
		perror("tcp/udpblast connect:");
		exit(1);
	}

	printf("Sending %s %s data using %d B blocks.\n",
		randomb ? "random":"non-random", tcp ? "TCP":"UDP", blksize);

	if (gettimeofday(&ti, &tiz) < 0)
	{
		perror("tcp/udpblast time:");
		exit(1);
	}
	starts  = ti.tv_sec;
	startms = ti.tv_usec / 1000L;


	for (i=0; i<nblocks; i++)
	{
		if (randomb) randbuff();
		if (write(f, greet, (size_t)blksize) != blksize)
			perror("tcp/udpblast send:");
		write(1, ".", 1);
	}

	if (gettimeofday(&ti, &tiz) < 0)
	{
		perror("tcpblast time:");
		exit(1);
	}
	stops  = ti.tv_sec;
	stopms = ti.tv_usec / 1000L;

	expms = (stops-starts)*1000 + (stopms-startms);
	printf("\n%d KB in %ld msec", (nblocks*blksize)/1024, expms);
	printf("  =  %.1f b/s", ((double) (nblocks*blksize))/expms*8000);
	printf("  =  %.1f B/s", ((double) (nblocks*blksize))/expms*1000);
	printf("  =  %.1f KB/s\n", 
		(double) (nblocks*blksize) / (double) (expms*1024.0) * 1000 );
	return(0);
}
