/* filename: rlpr-common.c
 * project: rlpr
 * author: meem  --  meem@sherilyn.wustl.edu
 * version: $Id: rlpr-common.c,v 1.9 1996/10/12 07:02:14 meem Exp $
 * contents: general-purpose functions for rlpr
 *
 * Time-stamp: <1996/10/12 01:56 -- meem@sherilyn.wustl.edu>
 */

/* copyright (c) 1996 meem, meem@gnu.ai.mit.edu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 1, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 */

#include "config.h"

#include <sys/types.h>		/* for off_t */
#include <stdio.h>
#include <string.h>		/* for strerror() */
#include <ctype.h>
#include <sys/stat.h>		/* for fstat()   */
#include <sys/utsname.h>	/* for uname()   */
#include <unistd.h>		/* for fstat()   */
#include <stdarg.h>		/* for varargs   */
#include <stdlib.h>		/* for malloc()  */
#include "rlpr-common.h"

/* modified slightly from Stevens, "UNIX Network Programming", p. 279 */

int writen(int fd, char *ptr, int nbytes) {
  int nleft, nwritten;

  nleft = nbytes;
  while (nleft > 0) {
    nwritten = write(fd, ptr, nleft);
    if (nwritten <= 0) return nwritten;
    nleft -= nwritten;
    ptr   += nwritten;
  }

  return nbytes;
}

void get_local_hostname(char * dest, size_t dest_sz) {
  struct utsname buf;
  if (uname(&buf) < 0)
    rlpr_fatal("unable to resolve your local hostname!");

  if (strlen(buf.nodename) > dest_sz)
    rlpr_fatal("your hostname is horribly long or hosed!");
  else strcpy(dest, buf.nodename);
}

char* strlower(char *str) {
  char *tmp = str;
  while (*str) {
    *str = tolower(*str);
    str++;
  }
  return tmp;
}  

/* this works with an fd because it is used with temporary
 * files that have already been unlink()'ed.
 */

off_t filesz(int fd) {
  static struct stat st;

  if (fstat(fd, &st) < 0)
    rlpr_fatal("fd %i: %s, make sure %s is writable!", fd, ERRNO, opts_.tmpdir);
  return st.st_size;
}

inline void * rlpr_malloc(size_t sz) {
  void *ptr = malloc(sz);
  check_ptr(ptr);
  return ptr;
}

inline void * rlpr_strdup(char * orig) {
  void *ptr = strdup(orig);
  check_ptr(ptr);
  return ptr;
}

/* these are only used if the host platform doesn't have them */

#ifndef HAVE_STRCSPN
size_t strcspn(const char *s, const char *reject) {
  const char *sp;

  for (sp = s; *sp; sp++)
    if (strchr(reject, *sp) != NULL) break;
  return sp - s;
}
#endif /* NOT HAVE_STRCSPN */

#ifndef HAVE_STRDUP
char * strdup(const char *s) {
  char * ptr = malloc(strlen(s) + 1);
  return ptr ? strcpy(ptr, s) : NULL;
}
#endif /* NOT HAVE_STRDUP */
