/* IluAlarmTech.java */
/* Chris Jacobi, November 4, 1997 11:22 am PST */

/*
 * Copyright (c) 1996 Xerox Corporation.  All Rights Reserved.  
 * Unlimited use, reproduction, and distribution of this software is
 * permitted.  Any copy of this software must include both the above
 * copyright notice of Xerox Corporation and this paragraph.  Any
 * distribution of this software must comply with all applicable United
 * States export control laws.  This software is made available AS IS,
 * and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 * INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
 * PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
 * THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
 * CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
 * XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 */
 
/* 
 * $Id: IluAlarmTech.java,v 1.17 1997/11/04 19:32:02 jacobi Exp $
 * See also IluJava_IluAlarmTech.c
 */
 
package xerox.ilu;

/**
 * Class to implement the Java side of ILU alarms... <br>
 * No Java clients anticipated.
 */
 
/*friendly*/ final
class IluAlarmTech extends java.lang.Thread {
    private static int priority = 
        IluEnvironment.getIntProp(
            "ilu.alarmpriority", java.lang.Thread.MAX_PRIORITY-1);
    private long wakeupAt = 0;
    private int x1; private int x2; //x1 .. x4 represent the proc and the rock 
    private int x3; private int x4;
    private static long count = 0;
    private boolean keepAlive;
    private java.util.Hashtable noGC = new java.util.Hashtable();
   
      
    /** Native code needs an instance of an IluAlarmTech
     *  which will not be garbage collected.
     */
    private native void giveInstance();
    private static IluAlarmTech prototype = null; //must not be gc'ed
    static {
        if (IluInit.doInit()) {
            prototype = new IluAlarmTech();
            prototype.giveInstance();
            xerox.basics.VMExtras.makeGCRoot(prototype);
        }
    }
   
    /* Loads the code and executes static initializations if any */
    public static void init() {
        IluInit.init();
    }

    /* returns and increments the classes object count */
    private static final synchronized long incrementCount() {
        return count++;
    }

 
    /* Constructor; private because real creation is done in createAlarm */
    private IluAlarmTech() {
        super("ILU_Alarm " + incrementCount());
        keepAlive = true;
        noGC.put(this, this);
    }
   
    /** 
     * Real creator; (called from C) to create alarm <p>
     * If you wonder why this isn't static: I couldn't get 
     * invoking static methods to work from the native side,
     * so I made this dynamic.
     */
    IluAlarmTech createAlarm() {
        IluAlarmTech at = new IluAlarmTech();
        at.setDaemon(true);
        at.setPriority(priority);
        at.wakeupAt = 0;
        at.start();
        return at;
    }
   
    /* (called from C side) when implementing ml_set_alarm */
    void setAlarm (int sec, int milli, int x1, int x2, int x3, int x4) {
        synchronized (this) {
            this.x1 = x1; this.x2 = x2;
            this.x3 = x3; this.x4 = x4;
            long time = ((long) sec) * 1000 + milli;
            if (IluDebug.traceAlarms()>1) {
                IluDebug.log.println("! IluAlarmTech setAlarm "  + time + " " + this);
            }
            wakeupAt = time + java.lang.System.currentTimeMillis();
            if (wakeupAt == 0) {wakeupAt = 1;}
            this.notifyAll();
        }
    }
   
    /* (called from C side) when implementing ml_unset_alarm */
    void unsetAlarm () {
        synchronized (this) {
            if (IluDebug.traceAlarms()>1) {
                IluDebug.log.println("! IluAlarmTech unsetAlarm " + this);
            }
            wakeupAt = 0;
            this.x1 = 0; this.x2 = 0;
            this.x3 = 0; this.x4 = 0;
            this.notifyAll();
        }
    }
   
    /* (called from C side) when implementing ml_destroy_alarm */
    void destroyAlarm () {
        keepAlive = false;
        unsetAlarm();
        noGC.remove(this);
    }
   

    /* For java to call C and finally execute the C procedure variable */
    private native void wakeIlu(int x1, int x2, int x3, int x4);
   
    /* The thread which invokes the alarms */
    public void run () {
        boolean call = false;
        int localx1 = 0; int localx2 = 0;
        int localx3 = 0; int localx4 = 0;
        if (IluDebug.traceAlarms()>0) {
            IluDebug.log.println("! IluAlarmTech create " + this);
        }
        while (keepAlive) {
            synchronized (this) {
                while (wakeupAt==0 && keepAlive) {
                    //we are not required to wake up at a certain time
                    try {
                        this.wait();
                    } catch (java.lang.InterruptedException e) {
                    }
                } 
                long time = java.lang.System.currentTimeMillis();
                long difference = wakeupAt - time;
                if (difference>0) {
                    try {
                        this.wait(difference);
                        //Go to the top of the loop to check whether
                        //we need to wait more...
                    } catch (java.lang.InterruptedException e) {
                    }
                } else if (wakeupAt>0) {
                    localx1 = x1; localx2 = x2;
                    localx3 = x3; localx4 = x4;
                    call = true;
                }
            }
            if (call) {
                if (IluDebug.traceAlarms()>1) {
                    IluDebug.log.println("! IluAlarmTech wake it " + this);
                }
                wakeIlu(localx1, localx2, localx3, localx4);
                call = false;
                this.yield();
            } 
        }
    }
   
} // IluAlarmTech

