/* IluPassport.java */
/* Chris Jacobi, September 11, 1997 11:01 am PDT */

/*
 * Copyright (c) 1997 Xerox Corporation.  All Rights Reserved.  
 * Unlimited use, reproduction, and distribution of this software is
 * permitted.  Any copy of this software must include both the above
 * copyright notice of Xerox Corporation and this paragraph.  Any
 * distribution of this software must comply with all applicable United
 * States export control laws.  This software is made available AS IS,
 * and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 * INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
 * PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
 * THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
 * CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
 * XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 */
 
/* 
 * $Id: IluPassport.java,v 1.8 1997/09/11 18:02:58 jacobi Exp $ 
 */


/* 
 * Representation for ilu_Passport. <p>
 * Native code in IluJava_IluPort.c
 */


package xerox.ilu;

/**
 * Like a "wallet" filled with IluIdentity's.<br>
 *
 * Passports can be generated by applications and passed to ILU, or,
 * constructed by ILU (from the other other side of the wire) and passed
 * to the clients.<p>
 *
 * Client generated passports are mutable and stay alive until
 * garbage collected.<br>
 * Ilu generated passports can be used only while the call for which
 * it has been constructed is ongoing.  If used afterwards, an exception 
 * will be raised.<p>
 *
 * A passport can carry at most one IluIdentity of each IluIdentityType.
 *
 * @see IluIdentity
 * @see IluIdentityType
 * @see IluPort
 * @see IluServiceThread
 */
public final class IluPassport extends IluWPBase {
    private long yIluPassport = 0;
    private IluCall call = null;
    /* friendly */ int valid = 0; 
        //  0 -> mutable passport
	// +1 -> valid immutable passport
	// -1 -> disabled immutable passport
    
    /**
     * Creation of a client generated passport (mutable)
     */
    public IluPassport() {
        super();
        nInitClientPassport();
    } //constructor

    /**
     * Creation of an Ilu generated passport (immutable)
     */
     private IluPassport(IluCall call) {
        super();
        this.call = call;
        this.valid = 1;
    } //constructor
    
    private native void nInitClientPassport();
    private native void nFinalizePassport();
    private native int nAddIdentity(IluIdentity ii);
    private native void nFindIdentity(
        IluCall call, IluIdentityType it, IluIdentity iiContainer
        );
    
    protected final void finalize () throws java.lang.Throwable {
        if (valid==0) {nFinalizePassport();}	//returns native memory
        super.finalize();                	//IluWPBase requirement 
    } //finalize
    
    /**
     * Check passport whether it contains an identity of
     * the specified type. 
     */
    public IluIdentity findIdentity(IluIdentityType it) {
        if (it == null) {
            throw new IluSomeSystemException("null IluIdentityType");
        }
        IluIdentity iiContainer = new IluIdentity(it);
        if (iiContainer == null) {
            throw new IluSomeSystemException("problem with IluIdentityType");
        }
        synchronized (this) {
           if (this.valid < 0) {
               throw new IluSomeSystemException(
                   "passport not valid outside of call"
                   );
           }
           nFindIdentity(this.call, it, iiContainer);
        }
        if (iiContainer.yIluIdentity != 0) {
            return iiContainer;
        }
        return null;
    } //findIdentity
    
    /**
     * Add an identity to the passport. <p>
     * Passport must be mutable or an exception is thrown.
     */
    public void addIdentity(IluIdentity ii) {
        if (this.valid <= 0) {
           throw new IluSomeSystemException(
               "passport not mutable"
               );
        }
        if (ii == null) {
            throw new IluSomeSystemException("null identity");
        }
        if (nAddIdentity(ii) < 0) {
            throw new IluSomeSystemException("couldn't add identity");
        }
    } //addIdentity
    

    /**
     * Since native memory disappears when a call is finished
     * we need to be able to disable the passport in time. 
     */
    /*friendly*/ void disable() {
        this.call = null;
        if (this.valid > 0) {
            synchronized (this) {
                this.valid = -1;
            }
        }
    } //disable
    
    
    /**
     * Returns the passport of the current threads skeleton.
     * Can be used from a true method within a call only
     * or an exception is thrown.
     */
    public static IluPassport getSkeletonPassport() {
        java.lang.Thread thread = java.lang.Thread.currentThread();
        if (thread instanceof IluServiceThread) {
            IluCall call = ((IluServiceThread)thread).skeletonCall;
            if (call.skeletonPassport == null) {
                IluPassport p = new IluPassport(call);
                call.skeletonPassport = p;
                //No locking involved: we are using the call of 
                //the currentThread only.
                //
                //We need either a native call to set up the passport, or,
                //use the native passport only while it can be reconstructed
            }
            return call.skeletonPassport;
        } else {
            throw new IluSomeSystemException("Not an ILU thread");
        }
    } //getSkeletonPassport


    /** 
     * Sets the caller IluPassport used for doing ILU calls
     * from this thread. <p>
     *
     * Use null IluPassport to clear. <br>
     * Throws an exception if the current thread is not an IluServiceThread.<br>
     *
     * @see IluServiceThread
     * @see IluServiceThread#cleanAllStubFeatures
     */
    public static void setCurrentPassport(IluPassport passport) {
        IluServiceThread.setCurrentPassport(passport);
    } //setCurrentPassport
    
    
} //IluPassport
