/* IluServer.java */
/* Chris Jacobi, November 4, 1997 11:08 am PST */

/*
 * Copyright (c) 1996, 1997 Xerox Corporation.  All Rights Reserved.  
 * Unlimited use, reproduction, and distribution of this software is
 * permitted.  Any copy of this software must include both the above
 * copyright notice of Xerox Corporation and this paragraph.  Any
 * distribution of this software must comply with all applicable United
 * States export control laws.  This software is made available AS IS,
 * and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 * INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
 * PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
 * THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
 * CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
 * XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 */
 
/* $Id: IluServer.java,v 1.30 1997/11/04 19:32:02 jacobi Exp $ */
 
/* 
 * See IluJava_IluServer.c for native implementation.
 */

package xerox.ilu;
    import java.util.Hashtable;

/**
 * This is the Java representation of the an ilu_Server. <p>
 *
 * Since true servers have associated threads running, don't expect
 * true servers to be garbage collected.<p>
 *
 * @see IluObjectTable
 * @see IluPort
 * @see IluServerRelocation
 */
public final class IluServer extends IluWPBase {
    
    private long yIluServer = 0;
    private long yPtr = 0; //just in case the c side needs a hook
    private java.lang.String id; 
    private xerox.ilu.IluObjectTable jjObjTab = null;
    private xerox.ilu.IluServerRelocationInfo jjRelocatonInfo = null;
    /*friendly*/ java.util.Hashtable portTable; 
    /*friendly*/ java.util.Hashtable retainTable = null; 
        //prevent gc of IluOInt's
    /*friendly*/ int state = 0;
    /*friendly*/ java.lang.Object lock;
    /*friendly*/ boolean deamon = false;
    private boolean hasBeenAsserted = false; /*synchronized*/
    /*friendly*/ boolean promisedPort = false;  
    /*friendly*/ boolean surrogateServer = false;  
    private static IluServer defaultServer0 = null;
    
    /* We report a proto type server because when calling
     * back from native code to java we use only dynamic methods
     */
    /*friendly*/ native void reportIluServerInst();
    /*friendly*/ static IluServer serverProto = null; 
    public static void init() {
        if (serverProto != null) return;
        if (! IluInit.doInit()) return;
        serverProto = new IluServer();
        xerox.basics.VMExtras.makeGCRoot(serverProto);
        serverProto.reportIluServerInst();
    } //init
             
    /**
     * Returns whether this is a surrogate server.
     * If it is not a surrogate server, it is a true server.
     */
    public boolean isSurrogateServer() {
        return surrogateServer;
    } //isSurrogateServer
    
    /**
     * Application visible "constructor" for true server. <p>
     *
     * Excpect the type of objFactory to change; null objFactory is legal 
     * and results in a server not automatically generating true objects.
     */
    public static IluServer createServer(
                java.lang.String id //null -> make one up
                ) throws xerox.ilu.IluSystemException {
        IluServer serv = new IluServer();
        serv.surrogateServer = false;
        if (id == null) id = xerox.ilu.IluRT0.inventID();
        serv.id = id;
        if (IluDebug.tracePCT()>0) {
            IluDebug.log.println("! IluServer: create server for " + id);
        }
        return serv;
    } //createServer
    
    
    /**
     * "accessor" for a true server. <p>
     */
    /*friendly*/ static IluServer defaultServer() 
            throws xerox.ilu.IluSystemException {
        IluServer serv = defaultServer0;
        if (serv == null) {
            serv = createServer(null);
            serv.finishInitialization();
            defaultServer0 = serv;
        }
        return serv;
    } //defaultServer
    

    /**
     * Creates a port and adds it to the server.
     */
    public IluPort createPort(
            java.lang.String protocolInfo,         	//null -> default
            xerox.ilu.IluTransportInfo transportInfo,	//null -> default
            IluPassport passport                 	//null -> default
            ) throws xerox.ilu.IluSystemException {
        IluPort port;
        this.checkTrue();
        this.promisedPort = true; 
        this.finishInitialization();
        port = IluPort.createPort(this, protocolInfo, transportInfo, passport);
        return port;
    } //createPort


    /**
     * Sets the "deamon" flag for ports.<p>
     *
     * The "deamon" flag marks threads to be deamons; deamon threads
     * do not prevent the application from termination. <p>
     *
     * This is a simple flag: Influences ports created in the future.
     * Not synchronized; this is for the server creator only.
     * Returns "this" for conveniance.
     */
    public IluServer setDeamonFlag(boolean deamon)
            throws xerox.ilu.IluSystemException {
        this.deamon = deamon;
        return this;
    } //setDeamonFlag
    

    /* Check that this server is not a surrogate server... */
    private final void checkTrue() throws xerox.ilu.IluSystemException {
        if (this.surrogateServer) {
            throw new xerox.ilu.IluSystemException(
                "don't use surrogate servers"
                );
        }
    } //checkTrue


    /* Check that this server is still in its initialization phase */
    private final void checkInInit() throws xerox.ilu.IluSystemException {
        if (this.hasBeenAsserted) {
            throw new xerox.ilu.IluSystemException(
                "must be called before initialization is asserted"
                );
        }
    } //checkInInit
    
    
    /**
     * Sets the object table for the server. <p>
     *
     * Can be called only before any ports are created.  Will not
     * be advertized unless ports a created or finishInitialization
     * is called directly or indirectly.<p>
     *
     * Not synchronized; this is for the server creator only.
     * An object table can have exactly one server.
     * A server can have at most one object table.
     * Returns "this" for conveniance.<p>
     *
     * Excpect the type of objFactory to change; null objFactory is legal 
     * and results in a server not automatically generating true objects.
     */
    public IluServer setObjectTable(xerox.ilu.IluObjectTable ot)
            throws xerox.ilu.IluSystemException {
        this.checkTrue();
        this.checkInInit();
        if (this.jjObjTab != null) {
            throw new xerox.ilu.IluSystemException(
                "server already has an object table"
                );
        }
        this.jjObjTab = ot;
        ot.setServer(this);
        return this;
    } //setObjectTable
    
    /**
     * Enables IluServerRelocation for this server. <p>
     *
     * Can be called only once at server initialization time.<p>
     * Returns "this" for conveniance.<p>
     */
    public IluServer setServerRelocation(xerox.ilu.IluServerRelocation sr)
            throws xerox.ilu.IluSystemException {
        this.checkTrue();
        this.checkInInit();
        if (sr == null) {
            this.jjRelocatonInfo = null;
        } else {
            this.jjRelocatonInfo = new IluServerRelocationInfo(sr);
        }
        return this;
    } //setServerRelocation
    
    /**
     * Call back to decide whether relocation is necessary.
     * Called by native side only. 
     * Called within server lock. 
     */ 
    IluServerRelocationInfo mustCheckRelocate() {
        IluServerRelocationInfo info = this.jjRelocatonInfo;
        if (info != null) {
            IluServerRelocation checker = info.jjRelocatonChecking;
            if (checker != null) {
                info.pInfoContainer[0] = null;
                info.tInfoContainer[0] = null;
                checker.checkIluServerRelocation(
                    this, info.pInfoContainer, info.tInfoContainer
                    );
                //The containers have been touched by clients; make
                //untouchable safety copies to prevent gc while still
                //continuing with native caller.
                info.jjSaveProtocolInfo = info.pInfoContainer[0];
                info.jjSaveTransportInfo = info.tInfoContainer[0];
                if (info.jjSaveProtocolInfo != null && info.jjSaveTransportInfo != null) {
                   return info;
                }
            }
        }
        return null;
    } //mustCheckRelocate
    

    /**
     * Makes sure enough of the server has been initialized
     * so it can really be used.
     * Returns "this" for conveniance.
     */
    public IluServer finishInitialization(
            ) throws xerox.ilu.IluSystemException {
        this.checkTrue();
        if (! this.hasBeenAsserted) {
            boolean promisedPort = this.promisedPort;
            if (this.state == 29) {
                throw new xerox.ilu.IluSystemException(
                    "use of destroyed server"
                );
            }
            synchronized (this) {
                if (! this.hasBeenAsserted) {
                    if (IluDebug.tracePCT()>0) {
                        IluDebug.log.println("! IluServer: finish init " 
                            + this.id);
                    }
                    this.nativeOfCreateServer(this.id);
                    this.hasBeenAsserted = true;
                }
            }
            if (!this.promisedPort) { 
                this.createPort(null, null, null); 
            }
        }
        if (defaultServer0 == null) {
            defaultServer0 = this;
        }
        return this;
    } //finishInitialization
    
    
    /**
     * Accessor function 
     */
    public final java.lang.String serverId() {
        return this.id;
    } //serverId
    
    
    /*
     * Private<br>
     * Implementation for createServer.
     */
    private native void nativeOfCreateServer(java.lang.String id);
    
    
    /*
     * Constructor private: use createServer.<br>
     * Prevents creation of random servers.
     */
    private IluServer() {
        this.retainTable = new java.util.Hashtable();
        this.portTable = new java.util.Hashtable();
        this.lock = retainTable;
    }  //constructor
    
    /**
     * Destroys true server.<p>
     *
     * (But does not necessarily get rid of existing objects)
     * Once we support applets, this should be security checked.
     */
    public void destroyServer() {
        this.checkTrue();
        if (this == defaultServer0) {
            defaultServer0 = null;
        }
        synchronized (this.lock) {
           if (this.state>=9) return;
           this.state = 19;
        };
        java.util.Hashtable safeCopy = 
            (java.util.Hashtable) this.portTable.clone();
        java.util.Enumeration enum = safeCopy.keys();
        while (enum.hasMoreElements()) {
            IluPort p = (IluPort) enum.nextElement();
            p.closePort();
        }
        this.nativeOfDestroyServer();
        this.state = 29;
        this.hasBeenAsserted = false;
    } //destroyServer
    
    
    private native void nativeOfDestroyServer();
    private native void nativeOfFreeServer();
    
    protected void finalize() throws java.lang.Throwable {
        if (! this.surrogateServer) {
            this.destroyServer();
            this.state = 99;
            this.nativeOfFreeServer();
            super.finalize();
        }
    } //finalize
    
    
    static {
        IluInit.init();
    }
    
} //IluServer

