/* genobj.c */
/* Chris Jacobi, November 11, 1997 2:50 pm PST */

/*
 * Copyright (c) 1991-1997 Xerox Corporation.  All Rights Reserved.
 * Unlimited use, reproduction, and distribution of this software is permitted.
 * Any copy of this software must include both the above copyright notice of
 * Xerox Corporation and this paragraph.  Any distribution of this software
 * must comply with all applicable United States export control laws.  This
 * software is made available AS IS, and XEROX CORPORATION DISCLAIMS ALL
 * WARRANTIES, EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND
 * NOTWITHSTANDING ANY OTHER PROVISION CONTAINED HEREIN, ANY LIABILITY FOR
 * DAMAGES RESULTING FROM THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED,
 * WHETHER ARISING IN CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT
 * LIABILITY, EVEN IF XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGES.
 */

/* $Id: genobj.c,v 1.63 1997/11/12 00:54:34 jacobi Exp $ */

#include <stdio.h>
#include <string.h>
#include "iluptype.h"

#include "shared.h"
#include "stubops.h"
#include "io.h"
#include "name.h"
#include "util.h"
#include "genstub.h"
#include "genrecord.h"
#include "genunion.h" 
#include "context.h"
#include "genobj.h"
#include "genex.h"
#include "hlpcls.h"

static IHandle specialContext = 0;

PUBLIC void init_genobj()
{
    specialContext = getContextC("ilu");
}


PUBLIC boolean obj_is_a(Type t) {
    TypeKind kind;
    t = myUrType(t);
    kind = type_kind(t);
    return (kind==object_Type);
}


PUBLIC Type obj_assert(Type t) {
    t = myUrType(t);
    if (! obj_is_a(t)) fatal("should have been an object");
    return t;
}


PRIVATE char * objShortName(Type t) {
    t = obj_assert(t);
    return easyShortTypeNameCleaned(t);
}

PRIVATE char * baseName(Type t) {
    t = obj_assert(t);
    return packageDotString(getContextT(t), objShortName(t));
}

PRIVATE char * refTypeShortName(Type t) {
    t = obj_assert(t);
    return cat2(objShortName(t), getContextT(t)->p.refSuffix);
}

PUBLIC boolean isCorbaDotObject(Type t) {
    IHandle ih = getContextT(t);
    if (ih==specialContext) {
        if (obj_is_a(t)) {
            char* sn = easyShortTypeNameCleaned(t);
            if (strcmp(sn, "CORBA_Object") == 0) {
                return TRUE;
            }
        }
    }
    return FALSE;
}

PUBLIC char * obj_typeDeclarator(Type t) {
    char* refTypeName;
    IHandle ih;
    t = obj_assert(t);
    ih = getContextT(t);
    if (ih == specialContext) {
       if (isCorbaDotObject(t)) {
           return copy("org.omg.CORBA.Object");
       }
    }
    return packageDotString(ih, refTypeShortName(t));
}


PRIVATE Class
getOClass(Type ot)
{
    ot = obj_assert(ot);
    return type_description(ot)->structuredDes.object;
}


PRIVATE boolean sunStyleHolder(Argument a)
{
     IHandle ih = getContextT(a->type);
     if (ih == specialContext) {
         if (isCorbaDotObject(a->type)) {
             return TRUE;
         }
     }
     return ih->p.genHldCls;
}


PRIVATE char* holderAccess(Argument a)
{
     if (sunStyleHolder(a)) {
        return cat2(argumentName(a), ".value");
     } else {
        return cat2(argumentName(a), "[0]");
     }
}



PRIVATE void
printSkelInElement    (Argument a, void *refany)
/* used to print the calls which unmarshall the arguments to
 * the true method on the server side
 */
{
    char* argName = argumentName(a);
    if (a->direction != In) {
        /* need to create a holder */
        char * htn = holderTypeDeclarator(a->type);
        if (sunStyleHolder(a)) {
            printf("          %s %s = new %s();\n",
                htn, argName, htn);
        } else {
            printf("          %s %s = new %s;\n",
                htn, argName, setDimension(htn, "1"));
        }
    }
    if (a->direction != Out) {
        if (a->direction == In) {
            printf("          %s %s = %s;\n", 
                typeDeclarator(a->type),
                argName, ioInPiece(a->type)
                );
        } else {
            if (sunStyleHolder(a)) {
                printf("          %s.value = %s;\n", 
                    argName, ioInPiece(a->type));
            } else {
                printf("          %s[0] = %s;\n",
                    argName, ioInPiece(a->type));
            }
        }
    }
}


typedef struct {
    Type t;
    IHandle ih;
    int methodidx;        	/* of method currently in the works */
    char* classShortName; 	/* short name of stub class */
    char* iluClassRepField; 	/* name of field in stub class */
    char* otName;        	/* short name of reference interface */
} ObjRockType;


PRIVATE boolean
isVoidType(Type t)
{
    if (t) {
        return (type_ur_kind(t) == void_Type);
    } else {
        return 1;
    }
}


PRIVATE boolean
returnsVoid(Procedure m) 
{
    return isVoidType(m->returnType);
}


PRIVATE boolean
isReallyAsync(Procedure m)
{
    if (m->asynch) {
        if (! returnsVoid(m)) return FALSE;
        if (m->exceptions) return FALSE;
        LOOP_BEGIN(m->arguments, Argument, arg, t1)
            if (arg->direction != In) return FALSE;
        LOOP_END()
        return TRUE;
    }
    return FALSE;
}


static void
printThrowsClause(set exceptions)
{
    boolean addSystemExceptions = FALSE; 
    boolean first = TRUE;
    LOOP_BEGIN(exceptions, Exception, e, t1)
        if (first) printf("\n        throws "); else printf(", ");
        first = FALSE;
        printf("%s", exceptionName(e));
    LOOP_END()
    if (addSystemExceptions) {
        if (first) printf("\n        throws "); else printf(", ");
        printf("xerox.ilu.IluSystemException");
    }
}


static void
printActualParamList(list args)
{
    C_LOOP_BEGIN(args, Argument, a, count, t1)
        if (count) {
            printf(", ");
        }
        /* "if (a->direction == In) { " not necessary
         * name of argument independent from a->direction 
         */
        printf("%s", argumentName(a));
    C_LOOP_END(count)
}


PRIVATE char *
accessIluClass(Type t) 
    /* assumes t is an Object type,
     * returns string which accesses the IluClass
     */
{
    IHandle ih = getContextT(t);
    t = obj_assert(t);
    return cat3(baseName(t), ih->p.stubSuffix, ".iluClass()");
}

 
PRIVATE void printSiblingTest(Argument a, const char* arg, const char* whiteSpace)
    /* arg: accesses argument */  
{
    if (a->sibling) {
        if (obj_is_a(a->type)) {
            printf("%s_call.checkSibling(%s, %s);\n",
                whiteSpace, 
                arg, 
                accessIluClass(a->type)
                );
        }
    }
}



PRIVATE void
defineServantMethod(Procedure m, ObjRockType* r)
/* r contains the index of the method */
{
    char * typeDecl;
    IHandle ih;
    int index;
    boolean async = isReallyAsync(m);

    ih = getContextT(m->object);
    typeDecl = obj_typeDeclarator(m->object);
    
    printf("      case %d: //%s\n", r->methodidx++, procShortNameX(m)); 
        /*?? CHECK: how are indices the same in decl */
    printf("        {\n");
    
    if (! returnsVoid(m)) {
        Type urt = myUrType(m->returnType);
        printf("          %s _r = %s;\n",
               typeDeclarator(urt),
               typeInitializer(type_kind(urt)));
    }
    
    printf("          %s _sob;\n", typeDecl);
    printf("          _call.startReadRequest();\n");
    printf("          _sob = (%s)\n", typeDecl);
    if (getOClass(m->object)->singleton) {
        printf("              _call.getCallSingleton();\n");
    } else {
        printf("              _call.inObject(true, %s.%s);\n", 
           r->classShortName,
           r->iluClassRepField
           );
    }
    list_enumerate(m->arguments, (EnumProc) printSkelInElement, NULL);
    
    printf("          _call.doneReadRequest();\n");
    if (!async) {
        printf("          try {\n");
    }
    printf("            ");
    if (! returnsVoid(m)) {
        printf("_r = ");
    }
    printf("_sob.%s(", procShortName(m));
    printActualParamList(m->arguments);
    printf(");\n");
    
    if (async) {
        printf("      _call.noReply();\n"); 
    } else {
    
        index = 1;
        C_LOOP_BEGIN(m->exceptions, Exception, e, count, t0)
            printf("          } catch (%s _e) {\n", exceptionName(e));
            printf("            %s(_e, %d, _call);\n", 
                fullNameOfWriteProc(e), count+1);
            printf("            return;\n");
        C_LOOP_END(count)
        printf("          } catch (Exception _e) {\n");
        printf("              _call.unexpectedException(_e);\n");
        printf("              return;\n");
        printf("          }\n");
        printf("          if (_call.needsSizing()) {\n");
        printf("            _sz = _call.beginSizingReply();\n");

        if (! returnsVoid(m)) {
            Type urt = myUrType(m->returnType);
            printf("            _sz += %s;\n", 
                ioSzPiece(urt, "_r"));
        }
        LOOP_BEGIN(m->arguments, Argument, a, t1)
            char* arg;
            if (a->direction == Out || a->direction == InOut) {
                arg = holderAccess(a);
                printf("            _sz += %s;\n", ioSzPiece(a->type, arg));
            }
        LOOP_END()
        printf("          }\n");
        printf("          _call.startWriteReply(_sz);\n");
        if (! returnsVoid(m)) {
            Type urt = myUrType(m->returnType);
            printf("          %s;\n", ioOutPiece(urt, "_r"));
        }
        LOOP_BEGIN(m->arguments, Argument, a, t2)
            char *arg;
            if (a->direction == Out || a->direction == InOut) {
                arg = holderAccess(a);
                if (a->sibling) {printSiblingTest(a, arg, "          ");}
                printf("          %s;\n", ioOutPiece(a->type, arg));
            }
        LOOP_END()
        printf("          _call.doneWriteReply();\n");
    }
    printf("        }\n");
    printf("        break;\n");
   
}


static void
printFormalArgList(list args)
{
    C_LOOP_BEGIN(args, Argument, a, count, t1)
        char * argName = argumentName(a);
        char * argDeclarator;
        if (count) {
            printf(", ");
        }
        if (a->direction == In) {
	    argDeclarator = typeDeclarator(a->type);
        } else {
	    argDeclarator = holderTypeDeclarator(a->type);
        }
        printf("%s %s", argDeclarator, argName);
    C_LOOP_END(count)
}



PRIVATE void
printSignatureForOperations(Procedure p, void *refany)
/* Prints one method in the Operations interface */
{
    if (p->doc_string) printDocString("    ", p->doc_string);
    printf("    public ");
    if (p->returnType) {
        printf("%s ", typeDeclarator(p->returnType)); 
    } else {
        printf("void ");
    }
    printf("%s(", procShortName(p));
    printFormalArgList(p->arguments);
    printf(")");
    printThrowsClause(p->exceptions);
    printf(";\n");
}

 
/* Surrogate sizing, marshalling in and out procedures. */
PRIVATE void
printSuperClassRep(Type t, ObjRockType* rock)
/* Purpose: Generate the elements of a superClass classRep array.
 *    Input: t is the class type, rock contains an index
 *    Output: void
 *    Rock must be initialized to an appropriate value before the call.
 */
{
    Class c = class_object(t);
    printf("      ca[%d] = %s%s.iluClass();\n",
           rock->methodidx++, 
           baseName(t),
           getContextT(t)->p.stubSuffix
           );
}


PRIVATE boolean
anyMethodHasExceptions(ObjRockType* rock)
{
    Class c = class_object(rock->t);
    LOOP_BEGIN(c->methods, Procedure, m, t4)
        if (list_size(m->exceptions)) return TRUE;
    LOOP_END()
    return FALSE;
}


PRIVATE void
printMethodRegistration(Procedure m, ObjRockType* rock)
/* rock contains method idx */
{
    /* define the exceptions of the method */
    if (m->exceptions) {
        cardinal i;
        printf("    _xArr = new xerox.ilu.IluExceptionRep[%d];\n", 
            list_size(m->exceptions)
            );
        for (i = 0; i < list_size(m->exceptions); i++) {
            printf("    _xArr[%d] = %s;\n", 
                i, 
                exceptionRep((Exception) list_ref(m->exceptions, i))
                );
        }
    }
    printf("    _%s_methodRep = ", procShortNameX(m));
    printf("xerox.ilu.IluMethodRep.registerMethod(\n");
    printf("        %s, //IluClassRep\n", rock->iluClassRepField);
    printf("        %d, //local idx\n", rock->methodidx++);
    printf("        %s, //name (in isl domain)\n",
        qoString(procedure_name(m)));
    printf("        %d, //remote id\n", m->id);
    printf("        %s, //functional\n", booleanImage(m->functional));
    printf("        %s, //asynch\n", booleanImage(m->asynch));
    if (m->exceptions) {
        printf("        _xArr, //exceptions\n");
    } else {
        printf("        null, //exceptions\n");
    }
    printf("        %d, //no of args\n", list_size(m->arguments));
    if (m->returnType && m->returnType->uid) {
        printf("        %s, //return arg uuid\n", 
            qoString(m->returnType->uid));
    } else {
        printf("        null, //return arg uuid\n");
    }
    printf("        _%s_skeleton);\n", objShortName(m->object));
    
    /* define the arguments of the method */
    C_LOOP_BEGIN(m->arguments, Argument, a, idx, tempa0)
        printf("     _%s_methodRep.defineArg(\n", procShortNameX(m));
        printf("         %d, //argIdx\n", idx);
        printf("         %s, //argName\n", qoString(argumentName(a)));
        printf("         %s, //sibling\n", booleanImage(a->sibling));
        printf("         %d, //direction\n", (int) a->direction);
        printf("         %s); //typeUid\n", qoString(a->type->uid));
    C_LOOP_END(idx) 
    printf("\n");
}

 
PRIVATE char*
shortNameOfObject(Type object)
{
    object = obj_assert(object);
    return easyShortTypeNameCleaned(object);
}


PRIVATE char*
shortNameOfStubClass(Type object)
{
    IHandle ih = getContextT(object);
    return cat2(shortNameOfObject(object), ih->p.stubSuffix);
}


PRIVATE char*
shortNameOfStaticStubProc(Procedure m)
{
    char* sn = procShortName(m);
    return cat5(
        "_",
        shortNameOfObject(m->object),
        "_",
        sn,
        "_S");
}
 
 
PRIVATE void
printDelegationMethod(Procedure m, char* callTo, char* extraArg)
/* Prints a method declaration m so that it calls "callTo" with exactly
 * the same arguments, except possibly a first extra argument "extraArg"
 */
{
    char* returnTypeDecl;
    char* returnStatement; 
    if (stringlength(extraArg)) {
        extraArg = cat2(extraArg, comma0(m->arguments));
    }
    if (returnsVoid(m)) {
        returnTypeDecl = "void";
        returnStatement = "";
    } else {
        returnTypeDecl = typeDeclarator(myUrType(m->returnType));
        returnStatement = "return ";
    }
    printf("    public %s %s(", returnTypeDecl, procShortName(m));
      printFormalArgList(m->arguments);
      printf(")");
      printThrowsClause(m->exceptions);
      printf(" {\n");
    printf("        %s%s(%s",
        returnStatement,
        callTo, 
        extraArg
        );
      printActualParamList(m->arguments);
      printf(");\n");
    printf("    } //%s\n\n", procShortName(m));
}

PRIVATE void
printMethodStub(Procedure m, ObjRockType* rock) 
{
    char* returnTypeDecl;
    char* returnTypeInit;
    char* piece;
    char* methodRep;
    boolean async;
    Type returnType;
  
    async = isReallyAsync(m);
    printDelegationMethod(m, shortNameOfStaticStubProc(m), "this");
    
    if (returnsVoid(m)) {
        returnType = 0;
        returnTypeDecl = "void";
    } else {
        returnType = myUrType(m->returnType);
        returnTypeDecl = typeDeclarator(returnType);
        returnTypeInit = typeInitializer(type_kind(returnType));
    }
    
    printf("    public static %s %s(%s self%s",
           returnTypeDecl,
           shortNameOfStaticStubProc(m), 
           typeDeclarator(m->object), 
           comma0(m->arguments)
           );

      printFormalArgList(m->arguments);
      printf(")");
      printThrowsClause(m->exceptions);
      printf(" {\n");

    printf("        xerox.ilu.IluCall _call;\n");
    printf("        xerox.ilu.IluUserException _userException = null;\n");
    if (! async) {
        printf("        int _ec = 0;\n");
    }

    if (! returnsVoid(m)) {
        printf("        %s _r = %s;\n", returnTypeDecl, returnTypeInit);                    
    }

    printf("        _call = xerox.ilu.IluCall.startCall(\n");
    
    methodRep = cat4(
        shortNameOfStubClass(m->object),
        "._",
        procShortNameX(m),
        "_methodRep"
        );
    printf("            %s,\n", methodRep);
    printf("            (xerox.ilu.IluSurrogateObject) self\n");
    printf("            );\n");

    printf("        try {\n");
    printf("          _retry: do {\n");
    if (getOClass(m->object)->singleton) {
        piece = "0";
    } else {
        piece = cat3(
            "_call.szObject(self, true, ",
            dotCat(
                shortNameOfStubClass(m->object),
                rock->iluClassRepField
                ),
            ")"
            );
    }
    printf("            _call.startWriteRequest(_call.needsSizing() ? (\n");
    printf("                  %s", piece);
    LOOP_BEGIN(m->arguments, Argument, a, t1)
        char* arg;
        if (a->direction == In || a->direction == InOut) {
            if (a->direction == In) {
                arg = argumentName(a);
            } else if (a->direction == InOut) {
                arg = holderAccess(a);
            }
            printf("\n                + %s", ioSzPiece(a->type, arg));
        }
    LOOP_END()
    printf("\n                ) : 0);\n");
    if (! getOClass(m->object)->singleton) {
        printf("            _call.outObject(self, true, %s.%s);\n", 
            rock->classShortName, 
            rock->iluClassRepField
            );
    }
    LOOP_BEGIN(m->arguments, Argument, a, t3)
        char* arg;
        if (a->direction == In || a->direction == InOut) {
            if (a->direction == In) {
                arg = argumentName(a);
            } else if (a->direction == InOut) {
                arg = holderAccess(a);
            }
            if (a->sibling) {printSiblingTest(a, arg, "            ");} 
            printf("            %s;\n", ioOutPiece(a->type, arg));
        }
    LOOP_END()

    printf("            _call.doneWriteRequest();\n");
    
    if (!async) {
        printf("            _ec = _call.startReadReply();\n");
        printf("            if (_ec == 0) {\n");
        if (returnType) {
            printf("                _r = %s;\n", ioInPiece(returnType));
        }
        LOOP_BEGIN(m->arguments, Argument, a, t5)
            char *arg;
            if (a->direction == Out || a->direction == InOut) {
                arg = holderAccess(a);
                printf("                %s = %s;\n", arg, ioInPiece(a->type));
            }
        LOOP_END()
        printf("            } else if (_ec == xerox.ilu.IluCall.retryCode) {\n");
        printf("              continue _retry;\n");
        if (m->exceptions) {
            printf("            } else if (_ec > 0) {\n");  
            printf("                _userException = _call.readException(%s, _ec);\n",
                methodRep
                );  
        }
        printf("            }\n");
        printf("            _call.doneReadReply();\n");
    }
    printf("            break _retry;\n");
    printf("          } while (true);\n");
    printf("        } finally {\n");
    printf("            _call.finishCall();\n");
    printf("        }\n");
    if (!async) {
        printf("        if (_ec != 0) {\n");
        if (m->exceptions) {
            printf("            switch (_ec) {\n");
            C_LOOP_BEGIN(m->exceptions, Exception, e, i, t6)
                printf("              case %d: throw (%s) _userException;\n",
                   i + 1,
                   exceptionName(e)
                   );
            C_LOOP_END(i)
            printf("            }\n");
        }
        printf("            throw xerox.ilu.IluSystemException.fromIluProtocolException(_ec);\n");
        printf("        }\n");
        if (! returnsVoid(m)) {
            printf("        return _r;\n");
        }
    }
    printf("    } //%s\n\n", shortNameOfStaticStubProc(m));
}


PRIVATE void
printSuperClassMethodStub1(Procedure m, void *x)
/* Defines the java methods for a superclass */
{
    IHandle superih = getContextT(m->object);
    char* callTo = cat4(
        baseName(m->object), 
        superih->p.stubSuffix, 
        ".", 
        shortNameOfStaticStubProc(m)
        );
    printDelegationMethod(m, callTo, "this");
}


PRIVATE void
printSuperClassMethodStubs(Type t, void * x)
/* Defines the class methods for all the superclasses
 * t is the type of the object whose superclasses are being
 * considered. 
 * Reuses static stub from superclass.
 */
{
    Class c = class_object(t);
    list_enumerate(c->superclasses, 
        (EnumProc) printSuperClassMethodStubs, 0);
    list_enumerate(c->methods, 
        (EnumProc) printSuperClassMethodStub1, 0);
}


PRIVATE void
printMethodRepDecl(Procedure m, void *x)
/*
 *   declare a var of type xerox.ilu.IluMethodRep for m.
 */
{
    printf("    static xerox.ilu.IluMethodRep _%s_methodRep;\n", 
               procShortNameX(m));
}


PRIVATE void
printeSkeletonClass(ObjRockType* rock)
/* defines the skeleton class in the current stub java file
 */
{
    Type t = rock->t;
    Class c = class_object(t);

    printf("class _%s_skeletonClass ", objShortName(t));
    printf("implements xerox.ilu.IluSkeleton {\n");
    printf("    _%s_skeletonClass() {\n", objShortName(t));
    printf("    }\n\n");

    printf("  public void serveRequest(xerox.ilu.IluCall _call, xerox.ilu.IluMethodRep _m) \n");
    printf("            throws xerox.ilu.IluSystemException {\n");
    printf("    int _sz = 0;\n");
    printf("    switch(_m.methodIdx) {\n");
    
    rock->methodidx = 0;
    list_enumerate(c->methods, (EnumProc) defineServantMethod, rock);
    
    printf("      default:\n");
    printf("    }\n");
    printf("  } //serveRequest\n\n");
    printf("} //_%s_skeletonClass\n\n", objShortName(t));
}


PUBLIC char * obj_ioSzPiece(Type t, const char *arg) {
    t = obj_assert(t);
    return cat5(
        "_call.szObject(", 
        arg, 
        ", false, ", 
        accessIluClass(t), 
        ")"
        );
}


PUBLIC char * obj_ioOutPiece(Type t, const char *arg) {
    t = obj_assert(t);
    return cat5(
        "_call.outObject(", 
        arg, 
        ", false, ", 
        accessIluClass(t), 
        ")"
        );
}


PUBLIC char * obj_ioInPiece(Type t) {
    t = obj_assert(t);
    return cat5(
        "(", 
        typeDeclarator(t), 
        ") _call.inObject(false, ", 
        accessIluClass(t), 
        ")"
        );
}


PRIVATE void
printSuperClassRefs(Type t, void *refany)
/* for the reference class definition */
{
    Class c = class_object(t);
    /* There is no real need to recurse, except that
     * it makes the generated java code easier to read
     */
    list_enumerate(c->superclasses, 
            (EnumProc) printSuperClassRefs, refany);
    printf(", %s", obj_typeDeclarator(t));
}


PRIVATE void
printReferenceInterface(ObjRockType* rock)
{
    Type t = rock->t;
    Class c = class_object(t);
    IHandle ih = rock->ih;
    char* refTypeName = refTypeShortName(t);
    NewJavaFile(t->interface, ih, refTypeName);
    if (c->doc_string) printDocString("", c->doc_string);
    printf("public interface %s ", refTypeName);
    printf("extends xerox.ilu.IluObject");
    list_enumerate(c->superclasses,
            (EnumProc) printSuperClassRefs, 0);
    printf(" {\n");
    generateNestedConstants(ih, easyShortTypeNameCleaned(t));
    list_enumerate(c->methods, (EnumProc) printSignatureForOperations, 0);
    printf("\n}//%s\n\n", refTypeName);
}


PRIVATE void
printDelegationStub(Procedure m, void *x)
{
    char* callTo = cat2("_delegate.", procShortName(m));
    printDelegationMethod(m, callTo, "");
}


PRIVATE void
printSuperDelegationStubs(Type t, void * x)
/* Defines the class methods for all the superclasses
 * t is the type of the object whose superclasses are being
 * considered. 
 * Reuses static stub from superclass.
 */
{
    Class c = class_object(t);
    list_enumerate(c->superclasses, 
        (EnumProc) printSuperDelegationStubs, 0);
    list_enumerate(c->methods, 
        (EnumProc) printDelegationStub, 0);
}


PRIVATE void
printDelegationClass(ObjRockType* rock)
{
    boolean withAnchor = 0;
    Type t = rock->t;
    Class c = class_object(t);
    char* refTypeName;
    char* classShortName = cat2(rock->otName, rock->ih->p.delClassSuffix);
    NewJavaFile(t->interface, rock->ih, classShortName);
    refTypeName = obj_typeDeclarator(t);
    if (c->doc_string) printDocString("", c->doc_string);
    printf("public class %s implements %s", classShortName, refTypeName);
        if (withAnchor) {
            printf(", xerox.ilu.IluAnchorSupport");
        }
        printf(" {\n");
    printf("    private %s _delegate = null;\n", refTypeName);
    if (withAnchor) {
        printf("    private java.lang.Object _internal = null;\n");
    }
    printf("\n");
    printf("    public %s() {\n", classShortName);
    printf("    }\n\n");
    printf("    public %s(%s _to) {\n", classShortName, refTypeName);
    printf("        __setDelegant(_to);\n");
    printf("    }\n\n");
    printf("    public void __setDelegant(%s _to) {\n", refTypeName);
    printf("        _delegate = _to;\n");
    printf("    }\n\n");
    if (withAnchor) {
        printf("    public void rememberAnchor(java.lang.Object _internal) {\n", refTypeName);
        printf("        this._internal = _internal;\n");
        printf("    }\n\n");
    }
    list_enumerate(c->methods, (EnumProc) printDelegationStub, rock);
    list_enumerate(c->superclasses, (EnumProc) printSuperDelegationStubs, t);
    printf("} //%s\n\n", classShortName);
}


PRIVATE void
printStubClass(ObjRockType* rock)
{
    Type t = rock->t;
    IHandle ih = rock->ih;
    Class c = class_object(t);
    char* otName = rock->otName;
    char* refTypeName;
    char* iluClassRepField = rock->iluClassRepField;
    char* classShortName = rock->classShortName;
    
    NewJavaFile(t->interface, ih, classShortName);
    LoadJavaClass(classShortName);
    refTypeName = obj_typeDeclarator(t);
    if (c->doc_string) printDocString("", c->doc_string);
    printf("public class %s extends ", classShortName);
    printf("xerox.ilu.IluSurrogateObject implements %s {\n", refTypeName);


    printf("    static xerox.ilu.IluClassRep %s;\n", iluClassRepField);
    
    printf("    /** returns ilu class implemented by this stub class */\n");
    printf("    static public final xerox.ilu.IluClassRep iluClass() {\n");
    printf("        return %s;\n", iluClassRepField);
    printf("    }//iluClass\n");

    list_enumerate(c->methods, (EnumProc) printMethodRepDecl, NULL);

    if (anyMethodHasExceptions(rock)) {
        printf("    static private xerox.ilu.IluExceptionRep[] _xArr = null;\n");
    }
    printf("  static private _%s_skeletonClass ", otName);
    printf("_%s_skeleton = new _%s_skeletonClass();\n\n", \
           otName, otName);

    printf("  static {\n");
    if (list_size(c->superclasses)>0) {
        printf("    xerox.ilu.IluClassRep[] ca = new xerox.ilu.IluClassRep[%d];\n",
           list_size(c->superclasses));
    }
    printf("    try {\n");
    rock->methodidx = 0;
    if (list_size(c->superclasses)>0) {
        list_enumerate(c->superclasses, (EnumProc) printSuperClassRep, rock);
    }
    
    printf("        %s = xerox.ilu.IluClassRep.setupClass(\n", iluClassRepField);
    printf("            \"%s\", //java reference interface\n",  
        dotCat(packagePrefix(currentIH), otName)
        );
    printf("            \"%s\", //ilu object type\n",
        dotCat(interface_name(currentIfc), otName) 
        );
    printf("            %s, //uuid\n", qoString(t->uid));
    printf("            %d); //method count\n", list_size(c->methods));
    
    if (list_size(c->superclasses)>0) {
        printf("        %s.setSuperClasses(ca);\n", 
            iluClassRepField);
    }
    
    if (c->optional) {
        printf("        %s.setOptional();\n", 
            iluClassRepField);
    }
    
    if (c->collectible) {
        printf("        %s.setCollectable();\n", 
            iluClassRepField);
    }
    
    if (c->brand) {
        printf("        %s.setBrand(%s);\n", 
            iluClassRepField,
            qoString(c->brand));
    }
    
    if (c->singleton) {
        printf("        %s.setSingleton(%s);\n", 
            iluClassRepField,
            qoString(c->singleton));
    }
    
    if (c->doc_string) {
        printf("        %s.setDocString(%s);\n", 
            iluClassRepField,
            qoString(c->doc_string));
    }
    
    if (1) {
        printf("        %s.setSurrClass(\"%s.%s\");\n", 
            iluClassRepField,
            packageName(t->interface, t->scoping),
            classShortName
            );
    }

    if (rock->ih->p.genFactory) {
        printf("        %s.setSurrFactory(new _%s_Factory());\n", 
            iluClassRepField,
            otName);
    }
    
    if (ih->p.genHlp) {
        printf("        %s.setIfName(%s); \n", 
            iluClassRepField,
            qoString(interface_name(currentIfc))
            );
        if (currentIfc->brand) {
            printf("        %s.setIfBrand(%s); \n", 
                iluClassRepField,
                qoString(currentIfc->brand)
                );
        }
        printf("        %s.id(); //makes sure helper class is loaded\n", 
            helperClassShortName(t));
    }
    
    printf("    } catch (xerox.ilu.IluSystemException e) {\n");
    printf("        throw new xerox.ilu.IluRuntimeError(\"error 1 registering %s\", e);\n", otName);
    printf("    }\n");

    rock->methodidx = 0;
    list_enumerate(c->methods, (EnumProc) printMethodRegistration, rock);

    printf("    try {\n");
    printf("        %s.finishClass();\n", iluClassRepField);
    printf("    } catch (xerox.ilu.IluSystemException e) {\n");
    printf("        throw new xerox.ilu.IluRuntimeError(\"error 2 registering %s\", e);\n", otName);
    printf("    }\n");
    printf("  }//static\n\n");

    list_enumerate(c->methods, (EnumProc) printMethodStub, rock);
    list_enumerate(c->superclasses, (EnumProc) printSuperClassMethodStubs, t);
    printf("    public static void registerTrueObject(\n");
    printf("        java.lang.String _ih,\n");
    printf("        %s%s _tob,\n", 
           otName,
           getContextT(t)->p.refSuffix);
    printf("        xerox.ilu.IluServer _s) ");
    printf("throws xerox.ilu.IluSystemException {\n");
    printf("        xerox.ilu.Ilu.registerTrueObject(_ih, _tob, _s, %s, 0);\n",
        iluClassRepField);
    printf("    } //registerTrueObject\n\n");

    printf("    /** ilu use only */\n");
    printf("    public %s() {\n", classShortName);
    printf("    }\n");

    printf("    /** ilu use only */\n");
    printf("    public %s(java.lang.Object arg) {\n", classShortName);
    printf("    }\n");

    printLoadClasses(rock->ih, 0);
    
    printf("} //%s\n\n", classShortName);
    
    /* the file is not yet finished */
}


PRIVATE void
printFactoryClass(ObjRockType* rock)
/*
 * in same file as stubClass
 */
{
    Type t = rock->t;
    IHandle ih = rock->ih;
    char* otName = objShortName(t);
     
    printf("class _%s_Factory extends xerox.ilu.IluFactory {\n", otName);
    printf("     _%s_Factory() { };\n", otName);
    printf("    public xerox.ilu.IluObject createSurrogateObject(java.lang.Object _arg) {\n");
    printf("        return new %s%s(_arg);\n", otName, ih->p.stubSuffix);
    printf("    }\n");
    printf("} // _%s_Factory\n\n", otName);
}


PRIVATE void
printImplClass(ObjRockType* rock)
/* defines an impl class */
{
    boolean withAnchor = 0;
    Type t = rock->t;
    Class c = class_object(t);
    char* classShortName = 
        cat3(rock->ih->p.implPrefix, rock->otName, rock->ih->p.implSuffix);
    char* refTypeName;
    NewJavaFile(t->interface, rock->ih, classShortName);
    refTypeName = obj_typeDeclarator(t);
    if (c->doc_string) printDocString("", c->doc_string);
    printf("public abstract class %s extends org.omg.CORBA.ObjectImpl implements %s", classShortName, refTypeName);
        if (withAnchor) {
            printf(", xerox.ilu.IluAnchorSupport");
        }
        printf(" {\n\n");
    if (withAnchor) {
        printf("    private java.lang.Object _internal = null;\n");
        printf("    /** implements xerox.ilu.IluAnchorSupport */\n");
        printf("    public void rememberAnchor(java.lang.Object _internal) {\n", refTypeName);
        printf("        this._internal = _internal;\n");
        printf("    }\n\n");
    }
    printf("    protected xerox.ilu.IluClassRep _implements() {\n");
    printf("        return %s;\n", accessIluClass(t));
    printf("    } //_implements\n\n");
    printf("} //%s\n\n", classShortName);
}


PRIVATE void
printVisiVarClass(ObjRockType* rock)
/* print a visigenic style var class */
{
    boolean withAnchor = 0;
    Type t = rock->t;
    Class c = class_object(t);
    char* classShortName = cat2(rock->otName, "_var");
    char* refTypeName;
    NewJavaFile(t->interface, rock->ih, classShortName);
    refTypeName = obj_typeDeclarator(t);
    printf("public class %s {\n", classShortName);
    
    printf("    public static %s narrow(xerox.ilu.IluObject _obj) throws xerox.ilu.IluSystemException {\n", rock->otName);
    printf("        return (%s) _obj;\n", rock->otName);
    printf("    } //narrow\n\n");
    
    printf("    public static %s bind(java.lang.String s) throws xerox.ilu.IluSystemException {\n", rock->otName);
    printf("        throw new xerox.ilu.IluSystemException(\"not impl\");\n");
    printf("    } //bind\n\n");
    
    printf("    /* assert classes loaded ... */\n");
    printf("    private static xerox.ilu.IluClassRep _c = %s;\n", accessIluClass(t));
    if (rock->ih->p.genHlp) {
        printf("    private static java.lang.String _i = %s.id();\n", helperClassShortName(t));
    }
    
printf("} //%s\n\n", classShortName);
}


PUBLIC void
obj_helpInnerCallback(Type t)
{
    printf("    public static %s narrow(xerox.ilu.IluObject _x) {\n",
        typeDeclarator(t)
        );
    printf("        return (%s) _x;\n", typeDeclarator(t));
    printf("    } //narrow \n\n");
}


PUBLIC void
obj_defineMain(Type t)
{
    ObjRockType* rock = iluparser_Malloc(sizeof(ObjRockType));
    
    rock->methodidx = 0;
    rock->t = t;
    rock->ih = getContextT(rock->t);
    rock->otName = objShortName(rock->t);
    rock->classShortName = cat2(rock->otName, rock->ih->p.stubSuffix);
    rock->iluClassRepField = copy("_classRep"); 
    
    printReferenceInterface(rock);
    
    if (rock->ih->p.genDel) printDelegationClass(rock);
    if (rock->ih->p.genImpl) printImplClass(rock);
    if (rock->ih->p.genVisiVar) printVisiVarClass(rock);
    
    printStubClass(rock);
    if (rock->ih->p.genFactory) printFactoryClass(rock);
    printeSkeletonClass(rock);
    
    
    
}


/* end */



