// --------------------------------------------------------------------------
// Open Sound System (OSS) / Linux specific audio interface.
// --------------------------------------------------------------------------

#include "AudioDriver.h"

const char AudioDriver::AUDIODEVICE[] = "/dev/dsp";

AudioDriver::AudioDriver()
{
    outOfOrder();
}

AudioDriver::~AudioDriver()
{
    close();
}

void AudioDriver::outOfOrder()
{
    // Reset everything.
    errorString = "None";
    audioHd = (-1);
}

bool AudioDriver::isThere()
{
    return (access(AUDIODEVICE,W_OK)==0);
}

bool AudioDriver::open(const AudioConfig& inConfig)
{
    if ((audioHd=::open(AUDIODEVICE,O_WRONLY,0)) == (-1))
    {
        perror(AUDIODEVICE);
        errorString = "ERROR: Could not open audio device.\n       See standard error output.";
        return false;
    }
    
    // Copy input parameters. May later be replaced with driver defaults.
    config = inConfig;

    if (config.maxFrags!=0 && config.fragSize!=0)
    {
        int maxFrags = config.maxFrags;
        if (maxFrags!=0 && (maxFrags<2))
            maxFrags = 2;  // demand double-buffering at least
        
        int fragSizeExp = 0;
        int fragSize = config.fragSize;
        while (fragSize > 1)
        {
            fragSize >>= 1;
            fragSizeExp++;
        };
        // OSS pguide recommends not to check range.
        // if (fragSizeExp < 7)
        //    fragSizeExp = 7;  // 2^7=128 at least
        
        // N fragments of size (2 ^ S) bytes
        //               NNNNSSSS
        // e.g. frag = 0x0004000e;
        // fragments should be out of [2,3,..,255]
        // fragSizeExp should be out of [7,8,...,17]
        // depending on the kernel audio driver buffer size
        int val = (maxFrags << 16) | fragSizeExp;
        if (ioctl(audioHd,SNDCTL_DSP_SETFRAGMENT,&val) == (-1))
        {
            perror(AUDIODEVICE);
            errorString = "ERROR: Could not set fragmentation parameters.\n       See standard error output.";
            return false;
        }
    }
    
    // Set sample precision and type of encoding.
    int dsp_sampleSize;
    if (config.precision == AudioConfig::BITS_16)
        dsp_sampleSize = 16;
    else  // if (config.precision == AudioConfig::BITS_8)
        dsp_sampleSize = 8;
    if (ioctl(audioHd,SNDCTL_DSP_SAMPLESIZE,&dsp_sampleSize) == (-1))
    {
        perror(AUDIODEVICE);
        errorString = "AUDIO: Could not set sample size.\n       See standard error output.";
        return false;
    }
    // Verify and accept the sample precision the driver accepted.
    if (dsp_sampleSize == 16)
    {
        config.precision = AudioConfig::BITS_16;
        config.encoding = AudioConfig::SIGNED_PCM;
    }
    else if (dsp_sampleSize == 8)
    {
        config.precision = AudioConfig::BITS_8;
        config.encoding = AudioConfig::UNSIGNED_PCM;
    }
    else
    {
        // Can this ever happen?
        errorString = "AUDIO: This error should never happen.";
        return false;
    }

    // Set mono/stereo.
    int dsp_stereo;
    if (config.channels == AudioConfig::STEREO)
        dsp_stereo = 1;
    else  // if (config.channels == AudioConfig::MONO)
        dsp_stereo = 0;
    if (ioctl(audioHd,SNDCTL_DSP_STEREO,&dsp_stereo) == (-1))
    {
        perror(AUDIODEVICE);
        errorString = "AUDIO: Could not set mono/stereo.\n       See standard error output.";
        return false;
    }
    // Verify and accept the number of channels the driver accepted.
    if (dsp_stereo == 1)
        config.channels = AudioConfig::STEREO;
    else if (dsp_stereo == 0)
        config.channels = AudioConfig::MONO;
    else
    {
        // Can this ever happen?
        errorString = "AUDIO: This error should never happen.";
        return false;
    }
    
    // Set frequency.
    int dsp_speed = config.frequency;
    if (ioctl(audioHd,SNDCTL_DSP_SPEED,&dsp_speed) == (-1))
    {
        perror(AUDIODEVICE);
        errorString = "AUDIO: Could not set frequency.\n       See standard error output.";
        return false;
    }
    // Accept the frequency the driver accepted.
    config.frequency = dsp_speed;

    // Get driver fragment size.
    ioctl(audioHd,SNDCTL_DSP_GETBLKSIZE,&config.blockSize);

// Not used yet.
//
//    audio_buf_info myAudInfo;
//    if (ioctl(audioHd,SNDCTL_DSP_GETOSPACE,&myAudInfo) == (-1))
//    {
//        perror(AUDIODEVICE);
//        errorString = "AUDIO: Could not get audio_buf_info.\n       See standard error output.";
//        return false;
//    }
    
    errorString = "OK";
    return true;
}

void AudioDriver::close()
{
    if (audioHd != (-1))
    {
        reset();
        ::close(audioHd);
        outOfOrder();
    }
}

void AudioDriver::play(void* buffer, unsigned long int bufferSize)
{
    if (audioHd != (-1))
    {
        write(audioHd,(char*)buffer,bufferSize);
    }
}
