// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _DtdParser_h_
#define _DtdParser_h_

#include <qobject.h>
#include <qstrlist.h>
#include "Attribute.h"
#include "Dtd.h"
#include "SgmlEntity.h"
#include "SgmlLexer.h"
#include "SgmlNode.h"

//: The DtdParser class parses a DTD received from a Request.
//. The DtdParser tokenizes the a data stream, using an SgmlLexer, and
//. constructs a set of DFAs.  The DFAs are used by the SgmlParser to tell it
//. where to expect optional start and end tags.
class DtdParser : public QObject {
    Q_OBJECT
    int                  _declDepth;
    Dtd*                 _dtd;
    SgmlLexer*           _lexer;
    SgmlEntity*          _currentEntity;
    SgmlNode*            _currentNode;
    QString              _attrName;
    Attribute::ValueType _attrValueType;
    Attribute::Type      _attrType;
    QString              _attrValue;
    QStrIList            _attrEnums;
    QString              _attrDefaultValue;
    bool                 _stagRequired;
    bool                 _etagRequired;
    bool                 _elementList;
    QStrIList            _elementNames;
    QStrIList            _inclNames;
    QStrIList            _exclNames;
    QStrIList            _contentElements;
    enum {
        Attlist, AttlistAttrName, AttlistAttrValueType, AttlistAttrType, AttlistAttrValue,
        Content,
        DeclSubset,
        ElementDeclNames, ElementDeclSTag, ElementDeclETag, ElementDeclModel, ElementDeclIncl, ElementDeclExcl,
        EntityDecl,
        MarkupDeclSkip, MarkupDeclStart
    } _state;

    //. Process the list of elements that this ATTLIST declaration applies to.
    void stateAttlist( SgmlLexer::Token token, const char* text );

    //. Process the name of the attribute.
    void stateAttlistAttrName( SgmlLexer::Token token, const char* text );

    //. Process the attribute's value type (CDATA, NAME, NUMBER).
    void stateAttlistAttrValueType( SgmlLexer::Token token, const char* text );

    //. Process the type of attribute (IMPLIED, FIXED, REQUIRED).
    void stateAttlistAttrType( SgmlLexer::Token token, const char* text );

    //. Process the default value (if any) for an attribute.
    void stateAttlistAttrValue( SgmlLexer::Token token, const char* text );

    //. Create the attribute list for each element.
    void stateAttlistEnd();

    //. Ignore all content, and search for the beginning of a markup
    //. declaration.
    void stateContent( SgmlLexer::Token token, const char* text );

    //. Process a marked section.  A mark section may be INCLUDEd or IGNOREd.
    void stateDeclSubset( SgmlLexer::Token token, const char* text );

    //. Process the list of elements that this declaration applies to.
    void stateElementDeclNames( SgmlLexer::Token token, const char* text );

    //. Process the optional/required start tag specification.
    void stateElementDeclSTag( SgmlLexer::Token token, const char* text );

    //. Process the optional/required end tag specification.
    void stateElementDeclETag( SgmlLexer::Token token, const char* text );

    //. Process the content model declaration for an element.
    void stateElementDeclModel( SgmlLexer::Token token, const char* text );

    //. Process the list of elements that are included on the content model.
    void stateElementDeclIncl( SgmlLexer::Token token, const char* text );

    //. Process the list of elements that are excluded from the content model.
    void stateElementDeclExcl( SgmlLexer::Token token, const char* text );

    //. Do post-processing of an element declaration.  This includes creating
    //. a new SgmlElement, and constructing a ContentModel for it.
    void stateElementDeclEnd();

    //. Process a parameter entity declaration and store it in the DTD.
    void stateEntityDecl( SgmlLexer::Token token, const char* text );

    //. Eat all the tokens until we get a MarkupDeclEnd, and ignore the rest of
    //. this declaration.
    void stateMarkupDeclSkip( SgmlLexer::Token token, const char* text );

    //. Figure out what kind of markup declaration we've got (e.g. ELEMENT,
    //. ATTLIST, ...).
    void stateMarkupDeclStart( SgmlLexer::Token token, const char* text );
public:
    //. Create a new DtdParser, and an SgmlLexer to tokenize the input stream.
    DtdParser( Dtd* dtd );

    //. Destroy the parser.
    ~DtdParser();
public slots:
    //. This slot processes the done signal from the SgmlLexer.  Figure out
    //. what the starting element for the document type is, and emit the done
    //. signal.
    void done();

    //. This slot forwards the data signal from the Request to the SgmlLexer.
    void fwdData( const char* bytes, int length );

    //. This slot forwards the endOfData signal from the Request to the
    //. SgmlLexer.
    void fwdEndOfData();

    //. This slot processes a token emitted from the SgmlLexer.  An appropriate
    //. dispatch function is chosen based on the current state of the parser.
    void token( SgmlLexer::Token token, const char* text );
signals:
    //. This signal is used to forward the end of data signal to the SgmlLexer.
    void data( const char* bytes, int length );

    //. This signal is used to forward the end of data signal to the SgmlLexer.
    void endOfData();

    //. This signal is emitted when both the DTD has been parsed.
    void done( Dtd* dtd );
};

#endif
