// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#ifndef _SgmlNode_h_
#define _SgmlNode_h_

#include <qlist.h>
#include <qstring.h>

class Dtd;

//: The SgmlNode class is a node within the parse tree for an SGML element's content model.
//. There are two classes of nodes: interior and leaf.  Leaf nodes have
//. concrete types, such as Accept, Element, Cdata, Pcdata, and Empty.
//. Interior nodes are operators, such as  Group, And, Or, Star, Qmark, Plus,
//. Seq.  The DFA for the content model is computed from the tree of SgmlNodes.
class SgmlNode {
public:
    enum Type { Accept, Element, Cdata, Pcdata, Empty, Group, And, Or, Star, Qmark, Plus, Seq };
private:
    Type            _type;
    QString         _element;
    SgmlNode*       _left;
    SgmlNode*       _right;
    SgmlNode*       _parent;
    bool            _nullable;
    QList<SgmlNode> _firstpos;
    QList<SgmlNode> _lastpos;
    QList<SgmlNode> _followpos;
public:
    SgmlNode( Type type, SgmlNode* parent=0 );
    SgmlNode( SgmlNode* node, SgmlNode* parent=0 );

    //. Destroy the node and its descendents.
    ~SgmlNode();

    //. Return the node's element name.
    const QString& element() { return _element; }

    //. Set the node's element name.
    void element( const QString& element ) { _element = element; }

    //. Return the node's left subtree.
    SgmlNode* left() { return _left; }

    //. Set the node's left subtree.
    void left( SgmlNode* left ) { _left = left; }

    //. Return the node's parent.
    SgmlNode* parent() { return _parent; }

    //. Set the node's parent.
    void parent( SgmlNode* parent ) { _parent = parent; }

    //. Return the node's right subtree.
    SgmlNode* right() { return _right; }

    //. Set the node's right subtree.
    void right( SgmlNode* right ) { _right = right; }

    //. Return the node's type.
    Type type() { return _type; }

    //. Set the type of the node.
    void type( Type type ) { _type = type; }

    //. Calculate the nullable(), firstpos(), lastpos(), and followpos()
    //. functions.
    void calculate();

    //. Returns TRUE if this entire subtree can be nulled out.
    bool nullable() { return _nullable; }

    //. Return the list of nodes that can appear at the beginning of this
    //. subtree.
    QList<SgmlNode>& firstpos() { return _firstpos; }

    //. Return the list of nodes that can appear at the end of this subtree.
    QList<SgmlNode>& lastpos() { return _lastpos; }

    //. Return the list of nodes that can directly follow the subtree rooted at
    //. this node.
    QList<SgmlNode>& followpos() { return _followpos; }
};

#endif
